(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards.insights');

    app.constant('WebAppInsightHandler', {
        name: 'Webapp',
        desc: 'Display webapp',
        icon: 'icon-code',
        color: 'notebook',

        getSourceId: function(insight) {
            return insight.params.webAppSmartId;
        },
        sourceType: 'WEB_APP',
        hasEditTab: false,
        defaultTileParams: {

        },
        defaultTileShowTitleMode: 'NO',
        defaultTileDimensions: [18, 12]
    });

    app.controller('WebAppViewCommon', function($scope, $stateParams, $controller, $q, DataikuAPI, Logger, WebAppsService) {
        $scope.resolvedWebApp = resolveObjectSmartId($scope.insight.params.webAppSmartId, $stateParams.projectKey);

        const baseType = WebAppsService.getBaseType($scope.insight.params.webAppType);
        if (baseType == 'STANDARD') {
            $controller('StandardWebAppController', { $scope: $scope });
        } else if (baseType == 'BOKEH') {
            $controller('BokehWebAppController', { $scope: $scope });
        } else if (baseType == 'DASH') {
            $controller('DashWebAppController', { $scope: $scope });
        } else if (baseType == 'SHINY') {
            $controller('ShinyWebAppController', { $scope: $scope });
        } else if (baseType == 'CODE_STUDIO_AS_WEBAPP') {
            $controller('CodeStudioWebAppController', { $scope: $scope });
        } else {
            Logger.error('Unknown app type: ', $scope.insight.params.webAppType);
        }

    });

    app.directive('webAppInsightTile', function($controller, $q, DataikuAPI, $timeout, DashboardUtils, StateUtils, TileLoadingState, TileLoadingBehavior, ChartFilters, DSSVisualizationThemeUtils){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                editable: '=',
                filters: '<',
                filtersParams: '<',
                theme: '<'
            },
            link: function($scope, element, attrs){
                let iframe;
                $scope.element = element;
                $scope.StateUtils = StateUtils;

                $scope.iframe;

                $controller('WebAppViewCommon', { $scope: $scope });

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                $scope.load = function(resolve, reject) {
                    $scope.loading = true;
                    const app = {
                        projectKey: $scope.insight.projectKey,
                        id: $scope.insight.params.webAppSmartId
                    };
                    const promises = [];

                    promises.push($scope.getViewURL(app).then(function(url) {
                        $scope.iFrameUrl = url;
                    }));

                    promises.push(DataikuAPI.webapps.getTrustedCodeReport($scope.insight.projectKey, $scope.insight.params.webAppSmartId).spinner(false)
                        .success(trustedCodeReport => {
                            $scope.trustedCodeReport = trustedCodeReport;
                        }));

                    Promise.allSettled(promises).then(function() {
                        DashboardUtils.setLoaded.bind([$scope, resolve])();
                        const timeoutInSeconds = Math.min($scope.tile.tileParams.loadTimeoutInSeconds, 240);
                        if (timeoutInSeconds > 0) {
                            $timeout(function() {
                                $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                            }, timeoutInSeconds * 1000);
                            return TileLoadingBehavior.DELAYED_COMPLETE;
                        }
                    });
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                function watchFilterChanges(iframe) {
                    if (!_.isNil($scope.unregisterFilterWatchGroup)) {
                        $scope.unregisterFilterWatchGroup();
                    }
                    $scope.unregisterFilterWatchGroup = $scope.$watchGroup(['filters', 'filtersParams'], function([newFilters, newFiltersParams], [oldFilters, oldFiltersParams]) {
                        if (newFilters !== oldFilters || newFiltersParams !== oldFiltersParams) {
                            sendFiltersToIframe(iframe);
                        }
                    });
                }

                function sendFiltersToIframe(iframe) {
                    if (!_.isNil($scope.filtersParams) && !_.isNil($scope.filters)) {
                        const message = {
                            type: 'filters',
                            filters: ChartFilters.getCleanFiltersForSharing($scope.filters),
                            filtersParams: ChartFilters.getCleanFiltersParamsForSharing($scope.filtersParams)
                        };
                        iframe.contentWindow.postMessage(message, '*');
                    }
                }

                $scope.onIframeLoaded = function(_iframe) {
                    if (!$scope.iFrameUrl) {
                        return;
                    }
                    iframe = _iframe;
                    sendFiltersToIframe(_iframe);
                    watchFilterChanges(_iframe);
                };

                $scope.$on('$destroy', function() {
                    if (!_.isNil($scope.unregisterFilterWatchGroup)) {
                        $scope.unregisterFilterWatchGroup();
                    }
                });
            }
        };
    });

    app.directive('webAppInsightView', function($controller, StateUtils, DataikuAPI, TaggableObjectsService){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_view.html',
            scope: {
                insight: '=',
                onError: '&'
            },
            link: function($scope, element) {
                $scope.element = element;
                $scope.StateUtils = StateUtils;

                $controller('WebAppViewCommon', { $scope: $scope });
                const app = {
                    projectKey: $scope.insight.projectKey,
                    id: $scope.insight.params.webAppSmartId,
                    versionTag: $scope.insight.params.versionTag
                };

                $scope.onIframeLoaded = function(iframe) {
                    if(!$scope.iFrameUrl) {
                        return;
                    }
                    const content = iframe.contentDocument.body.innerText;
                    if (content.startsWith('{"errorType":') && $scope.onError) {
                        try {
                            $scope.onError({ error: JSON.parse(content) });
                        } catch (e) {
                            // Not a JSON error object
                        }
                    } else {
                        TaggableObjectsService.checkAndUpdateThumbnailData(
                            { ...app, type: 'WEB_APP' },
                            'body',
                            iframe.contentDocument
                        );
                    }
                };

                $scope.getViewURL(app).then(function(url) {
                    $scope.iFrameUrl = url;
                });

                DataikuAPI.webapps.getTrustedCodeReport($scope.insight.projectKey, $scope.insight.params.webAppSmartId)
                    .success(trustedCodeReport => $scope.trustedCodeReport = trustedCodeReport);
            }
        };
    });

    app.directive('webAppInsightTileParams', function(){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_tile_params.html',
            scope: {
                tileParams: '='
            },
            link: function($scope, element, attrs){
                // Used when creating a new tile to correctly initialize the timeout value in editor.
                $scope.$watch('tileParams', function(nv) {
                    if (nv && nv.loadTimeoutInSeconds === undefined) {
                        nv.loadTimeoutInSeconds = 0;
                    }
                });
                if ($scope.tileParams.loadTimeoutInSeconds === undefined) {
                    $scope.tileParams.loadTimeoutInSeconds = 0;
                }
            }
        };
    });

    app.directive('webAppInsightCreateForm', function(DataikuAPI){
        return {
            templateUrl: '/templates/dashboards/insights/web_app/web_app_create_form.html',
            scope: true,
            link: function($scope, element, attrs){
                $scope.hook.defaultName = 'Webapp';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = nv.label;
                    $scope.insight.params.webAppType = nv.subtype;
                });
            }
        };
    });

})();
