(function() {
    'use strict';

    const app = angular.module('dataiku.projectdeployer');

    app.service('DeploymentUtils', function() {
        return {
            sanitizeForK8S: function(id) {
                return id.replaceAll(/[^.a-z0-9-_]/ig, '-');
            },
            sanitizeInfraId: function(id) {
                return id.replaceAll(/[^a-z0-9-_]/ig, '-');
            }
        }
    });

    app.controller('_ProjectDeployerDeploymentsBaseController', function(TopNav) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments');
    });

    app.controller('ProjectDeployerDeploymentsController', function($scope, $controller) {
        $controller('_ProjectDeployerDeploymentsBaseController', {$scope: $scope});
    });

    app.service('ProjectDeployerAsyncHeavyStatusLoader', function(DataikuAPI, Logger, DeployerUtils) {
        return {
            newLoader: function(infraStatusList, heavyStatusPerDeploymentId) {
                const loader = {};
                let loading = true;
                loader.stillRefreshing = () => loading

                let canLoadStatus = true;
                loader.stopLoading = function() {
                    canLoadStatus = false;
                },

                loader.loadHeavyStatus = function() {
                    if (!infraStatusList.length || !canLoadStatus) {
                        loading = false;
                        return;
                    }

                    const infraStatus = infraStatusList.shift();
                    const infraId = infraStatus.infraBasicInfo.id;
                    const deploymentIds = infraStatus.deployments.map(_ => _.id);
                    Logger.info("Sending heavy status list request for deployments of infra " + infraId);
                    DataikuAPI.projectdeployer.deployments.listHeavyStatus(infraId)
                        .success(function(heavyStatusList) {
                            Logger.info("Got heavy status list for infra " + infraId);
                            heavyStatusList.forEach((heavyStatus) => {
                                heavyStatusPerDeploymentId[heavyStatus.deploymentId] = heavyStatus;
                            });
                            loader.loadHeavyStatus();
                        }).error(function(a,b,c) {
                            Logger.warn("Failed to load heavy status list for infra " + infraId);
                            deploymentIds.forEach((deploymentId) => {
                                heavyStatusPerDeploymentId[deploymentId] = {
                                    health: "LOADING_FAILED",
                                    healthMessages: DeployerUtils.getFailedHeavyStatusLoadMessage(getErrorDetails(a,b,c))
                                };
                            });
                            loader.loadHeavyStatus();
                        });
                }
                return loader;
            },
        }
    });

    app.directive('projectDeploymentCard', function(DeployerUtils, ProjectDeployerDeploymentUtils, DeployerDeploymentTileService, $state) {
        return {
            scope: {
                lightStatus: '=',
                heavyStatus: '=',
                onRightClick: '&',
                customRightClick: '@',
            },
            templateUrl: '/templates/project-deployer/deployment-card.html',
            replace: true,
            link: function(scope, elem, attrs) {
                scope.dashboardTile = attrs.hasOwnProperty("deploymentDashboardTile");
                function onLightStatusChanged() {
                    if (!scope.dashboardTile) {
                        scope.automationInfo = {};
                        scope.automationInfo.deploymentInfoList = [];
                        scope.automationInfo.automationProjectKey = ProjectDeployerDeploymentUtils.getAutomationProject(scope.lightStatus.deploymentBasicInfo);
                        if (scope.lightStatus.infraBasicInfo.automationNodeExternalUrl) {
                            scope.isMultiAutomationNodeInfra = false;
                            const url = ProjectDeployerDeploymentUtils.getAutomationProjectUrl(scope.lightStatus.infraBasicInfo.automationNodeExternalUrl, scope.automationInfo.automationProjectKey);
                            scope.automationInfo.deploymentInfoList.push({ url });
                        } else if (scope.lightStatus.infraBasicInfo.automationNodes) {
                            scope.isMultiAutomationNodeInfra = true;
                            scope.lightStatus.infraBasicInfo.automationNodes.forEach(node => {
                                const url = ProjectDeployerDeploymentUtils.getAutomationProjectUrl(node.automationNodeExternalUrl, scope.automationInfo.automationProjectKey);
                                const nodeId = node.automationNodeId;
                                scope.automationInfo.deploymentInfoList.push({ url, nodeId });
                            });
                        }
                    }
                }

                scope.outcome = {
                    FAILED: "failed",
                    WARNING: "with warnings",
                    RUNNING: "currently running",
                    ABORTED: "aborted",
                    SUCCESS: "successful"
                };

                function getAutomationProjectMonitoringSummary(monitoring) {
                    if (!monitoring) {
                        return {
                            unreachable: true
                        }
                    }
                    if (!monitoring.hasScenarios) {
                        return {
                            noScenarios: true
                        };
                    }
                    if (!monitoring.hasActiveScenarios) {
                        return {
                            noActiveScenarios: true
                        };
                    }

                    const failedRuns = monitoring.failed.length;
                    const warningRuns = monitoring.warning.length;
                    const stillRunningRuns = monitoring.running.length;
                    const abortedRuns = monitoring.aborted.length;
                    const successRuns = monitoring.successful.length;
                    const total = failedRuns + warningRuns + stillRunningRuns + abortedRuns + successRuns;
                    const highestSeverity = {};

                    if (failedRuns) {
                        highestSeverity.value = scope.outcome.FAILED;
                        highestSeverity.scenarios = monitoring.failed;
                    } else if (warningRuns) {
                        highestSeverity.value = scope.outcome.WARNING;
                        highestSeverity.scenarios = monitoring.warning;
                    } else if (stillRunningRuns) {
                        highestSeverity.value = scope.outcome.RUNNING;
                        highestSeverity.scenarios = monitoring.running;
                    } else if (abortedRuns) {
                        highestSeverity.value = scope.outcome.ABORTED;
                        highestSeverity.scenarios = monitoring.aborted;
                    } else if (successRuns) {
                        highestSeverity.value = scope.outcome.SUCCESS;
                        highestSeverity.scenarios = monitoring.successful;
                    }

                    const summaryLine = getScenarioLastRunsSummaryLine(highestSeverity, total);
                    const highestSeverityNames = displayScenarioNamesOfHighestSeverity(highestSeverity, total);

                    return { total, highestSeverity, summaryLine, highestSeverityNames };
                }

                const getScenarioLastRunsSummaryLine = function(highestSeverity, total) {
                    if (!highestSeverity.scenarios) return;
                    if (highestSeverity.scenarios.length < total) {
                        return `${total} (${highestSeverity.scenarios.length} ${highestSeverity.value})`;
                    }
                    return `${total}, all ${highestSeverity.value}`;
                }

                const displayScenarioNamesOfHighestSeverity = function(highestSeverity, total, cap=15) {
                    const scenarioNames = highestSeverity.scenarios;
                    if (!scenarioNames || scenarioNames.length === total) return '';
                    let tooltipMsg = scenarioNames.slice(0, cap).join(", ");
                    const hiddenRuns = scenarioNames.length - cap;
                    if (hiddenRuns > 0) {
                        tooltipMsg += " and " + hiddenRuns + " more";
                    }
                    return tooltipMsg;
                }

                function getWebappInfoList(heavyStatus) {
                    const isMultiAutomationNodeInfra = heavyStatus.type === "MULTI_AUTOMATION_NODE";
                    if (isMultiAutomationNodeInfra) {
                        if (!heavyStatus.heavyStatusPerNode) return [];
                        return Object.entries(heavyStatus.heavyStatusPerNode)
                            .filter(e => typeof e[1].webappBackendInfoList !== "undefined")
                            .map(e => e[1].webappBackendInfoList.map(webapp => ({nodeId: e[0], ...webapp})))
                            .flat();
                    } else {
                        if (!heavyStatus.webappBackendInfoList) return [];
                        return heavyStatus.webappBackendInfoList;
                    }
                };

                const translateWebappsSummaryToText = function(totalCount, autostartEnabledCount, backendNotReadyCount) {
                    if (autostartEnabledCount) {
                        if (backendNotReadyCount) {
                            return `${autostartEnabledCount} (${backendNotReadyCount} stopped)`;
                        }
                        return `${autostartEnabledCount} (all running)`;
                    }
                    if (totalCount) {
                        return 'No auto-started webapps';
                    }
                    return 'No webapps';
                }

                const getWebappsStatusTooltipText = function(backendNotReadyWebapps, cap=12) {
                    if (backendNotReadyWebapps.length) {
                        const names = backendNotReadyWebapps.slice(0, cap).map(w => w.name).join(', ');
                        return `Auto-started webapps with stopped backend: ${names}`;
                    }
                    return 'All auto-started webapps are up and running';
                }

                const getWebappsSummary = function(webappInfoList) {
                    const totalCount = webappInfoList.length;

                    const autostartEnabledWebapps = webappInfoList.filter(i => i.isBackendAutostartOn);
                    const autostartEnabledCount = autostartEnabledWebapps.length;

                    const backendNotReadyWebapps = autostartEnabledWebapps.filter(w => w.status !== 'BACKEND_READY');
                    const backendNotReadyCount = backendNotReadyWebapps.length;

                    const text = translateWebappsSummaryToText(totalCount, autostartEnabledCount, backendNotReadyCount);
                    const tooltipText = getWebappsStatusTooltipText(backendNotReadyWebapps);

                    return { autostartEnabledCount, backendNotReadyCount, text, tooltipText };
                };

                scope.redirect = function() {
                    if (attrs.hasOwnProperty("redirectToDeploymentPage")) {
                        $state.go('projectdeployer.deployments.deployment.status', {deploymentId: scope.lightStatus.deploymentBasicInfo.id});
                    }
                }

                scope.handleRightClick = function(event) {
                    scope.onRightClick({$event: event});
                }

                scope.shouldHandleRightClick = function() {
                    return scope.customRightClick ? "true" : "false";
                }

                scope.$watch('lightStatus', function() {
                    if (scope.lightStatus) {
                        onLightStatusChanged();
                    }
                }, true);

                scope.$watch("heavyStatus", function() {
                    scope.deploymentStatus = DeployerDeploymentTileService.getDeploymentHealth(scope.heavyStatus);
                    if (scope.heavyStatus) {
                        scope.scenarioLastRuns = getAutomationProjectMonitoringSummary(scope.heavyStatus.monitoring);
                        const webappInfoList = getWebappInfoList(scope.heavyStatus);
                        scope.webappsSummary = getWebappsSummary(webappInfoList);
                    } else {
                        delete scope.scenarioLastRuns;
                        delete scope.webappsSummary;
                    }
                })
            }
        }
    });

    app.directive('projectDeploymentsListWidget', function($state) {
        return {
            scope: {
                deployments: '=projectDeploymentsListWidget',
                statusPage: '=',
                healthMap: '='
            },
            templateUrl: '/templates/project-deployer/deployment-list.html',
            replace: true,
            link: function(scope) {
                scope.redirect = function(deployment) {
                    $state.go('projectdeployer.deployments.deployment.status', {deploymentId: deployment.deploymentBasicInfo.id});
                };
            }
        };
    });

    app.controller('ProjectDeployerDeploymentDashboardController', function($scope, $controller, $state, WT1,
    ProjectDeployerDeploymentService, ProjectDeployerAsyncHeavyStatusLoader) {
        $controller('_DeployerDeploymentDashboardController', {$scope});

        if ($scope.isFeatureLocked) return;

        $scope.uiState.query.healthStatusMap = [
            'HEALTHY',
            'WARNING',
            'OUT_OF_SYNC',
            'UNHEALTHY',
            'ERROR',
            'UNKNOWN',
            'LOADING_FAILED'
        ].map(hs => ({
            id: hs,
            $selected: false
        }));

        $scope.uiState.query.typeBadgesMap = [
            '-',
            'AGENT',
            'LLM',
            'ML',
        ].map(gac => ({
            id: gac,
            $selected: false
        }));

        $scope.orderByExpression = ['projectBasicInfo.id', 'deploymentBasicInfo.bundleId', 'deploymentBasicInfo.id'];

        $scope.clearAllAndRefreshFilter = function() {
            $scope.clearSelectedItems($scope.uiState.query.projects);
            $scope.uiState.query.tags = [];
            $scope.clearSelectedItems($scope.uiState.query.healthStatusMap);
            $scope.clearSelectedItems($scope.uiState.query.typeBadgesMap);
            $scope.filterDeployments();
        }

        $scope.hasSelection = function() {
            return $scope.uiState.query.tags.length > 0 ||
                $scope.uiState.query.projects.some(elem => elem.$selected) ||
                $scope.uiState.query.healthStatusMap.some(elem => elem.$selected) ||
                $scope.uiState.query.typeBadgesMap.some(elem => elem.$selected);
        }

        function filterOnSearchBarQuery(lightStatus) {
            if (!$scope.uiState.query.q) return true;
            const query = $scope.uiState.query.q.toLowerCase();
            return lightStatus.deploymentBasicInfo.publishedProjectKey.toLowerCase().includes(query)
                || lightStatus.deploymentBasicInfo.deployedProjectKey && lightStatus.deploymentBasicInfo.deployedProjectKey.toLowerCase().includes(query)
                || lightStatus.deploymentBasicInfo.bundleId.toLowerCase().includes(query)
                || lightStatus.deploymentBasicInfo.id.toLowerCase().includes(query)
                || lightStatus.deploymentBasicInfo.infraId.toLowerCase().includes(query)
                || lightStatus.projectBasicInfo.name.toLowerCase().includes(query);
        }

        $scope.deploymentIsInUI = function(lightStatus) {
            const selectedProjects = $scope.uiState.query.projects.filter(project => project.$selected);
            const selectedStatuses = $scope.uiState.query.healthStatusMap.filter(hs => hs.$selected);
            const deploymentHealthStatus = $scope.heavyStatusPerDeploymentId[lightStatus.deploymentBasicInfo.id];
            const selectedTypeBadges = $scope.uiState.query.typeBadgesMap.filter(badge => badge.$selected);

            return filterOnSearchBarQuery(lightStatus) &&
                (!selectedProjects.length || selectedProjects.find(project => project.projectBasicInfo.id === lightStatus.deploymentBasicInfo.publishedProjectKey)) &&
                (!$scope.uiState.query.tags.length || $scope.uiState.query.tags.find(tag => lightStatus.deploymentBasicInfo.tags.find(deplTag => deplTag === tag))) &&
                (!selectedStatuses.length || selectedStatuses.find(hs => deploymentHealthStatus && deploymentHealthStatus.health === hs.id)) &&
                (!selectedTypeBadges.length || selectedTypeBadges.find(tb => {
                    if (tb.id === '-') {
                        return !lightStatus.deploymentBasicInfo.typeBadges || lightStatus.deploymentBasicInfo.typeBadges.length === 0;
                    } else {
                        return lightStatus.deploymentBasicInfo.typeBadges.find(deplTb => deplTb === tb.id)
                    }
                }));
        };

        $scope.showRightClickMenu = function($event, cell) {
            const url = $state.href('projectdeployer.deployments.deployment.status', {deploymentId: cell.deploymentBasicInfo.id});
            $scope.showOpenInANewTabMenu($event, url);
        }

        $scope.getFilteredDeploymentCount = function(deployments) {
            return deployments.filter(deployment => $scope.deploymentIsInUI(deployment)).length;
        };

        $scope.startCreateDeployment = function() {
            ProjectDeployerDeploymentService.startCreateDeployment().then(function(newDeployment) {
                $state.go('projectdeployer.deployments.deployment.status', {deploymentId: newDeployment.id});
                WT1.event('project-deployer-deployment-create', {deploymentType: 'PROJECT' });
            });
        };

        let loader;
        $scope.$watch("infraStatusList", function(nv) {
            if (!nv) return;
            loader = ProjectDeployerAsyncHeavyStatusLoader.newLoader([].concat(nv), $scope.heavyStatusPerDeploymentId);
            loader.loadHeavyStatus();
        });

        $scope.stillRefreshing = function() {
            return !$scope.globalLightStatusLoaded || !loader || loader.stillRefreshing();
        };

        $scope.$on("$destroy", function() {
            loader.stopLoading();
        });

        $scope.$watch('projectStatusList', function(nv) {
            if (!nv) return;

            $scope.uiState.query.projects = angular.copy($scope.projectStatusList);
        });
    });


    app.controller('ProjectDeployerPackagesPanelController', function($scope, $controller, ProjectDeployerProjectsService, DeployerUtils) {
        $controller("_DeployerPackagesPanelController", {$scope});

        $scope.getPackageDeployments = function(deployments, bundleId) {
            return deployments.filter(d => d.bundleId === bundleId);
        };

        $scope.deployBundle = function(projectStatus, bundleId) {
            ProjectDeployerProjectsService.deployBundle(projectStatus, bundleId, DeployerUtils.DEPLOY_SOURCE.PACKAGE_PANEL);
        };

        $scope.$watch('projectStatusList', function(nv) {
            if (!nv) return;

            $scope.uiState.fullProjectList = $scope.computeFullList(nv);
        });
    });

    app.controller('_ProjectDeployerEditDeploymentController', function($scope, $controller,  DataikuAPI) {
        $controller('_ProjectDeployerDeploymentsBaseController', {$scope: $scope});

        $scope.uiState = $scope.uiState || {};

        // --- Target folder setup ---
        $scope.projectFolderHierarchy = {};

        $scope.setFolderHierarchy = function(callback) {
            $scope.projectFolderHierarchy = {};

            if ($scope.deploymentSettings.infraId) {
                DataikuAPI.projectdeployer.infras.getProjectFolderHierarchy($scope.deploymentSettings.infraId).success(rootFolder => {
                    $scope.projectFolderHierarchy = rootFolder;

                    // add necessary elements for using browse-path directive
                    function fixupTree(tree) {
                        const children = tree.children;
                        // add pathElts
                        tree.directory = true;
                        tree.fullPath = tree.id;

                        // remove parent and children
                        const treeWithoutChildren = Object.assign({}, tree, { children: [] });
                        children.forEach(child => {
                            child.parent = treeWithoutChildren;
                            fixupTree(child);
                        })
                    }

                    fixupTree($scope.projectFolderHierarchy);

                    if (typeof callback === 'function') {
                        callback();
                    }
                });
            }
        };

        DataikuAPI.projectdeployer.publishedProjects.listLightStatus()
        .success(projects => {
            $scope.projects = projects;
        })
        .error(setErrorInScope.bind($scope));
    });

    app.controller('_ProjectDeployerDeploymentModalController', function($scope, $controller, $timeout, DataikuAPI, StringUtils, PathUtils, PromiseService, ProjectDeployerDeploymentService, DeploymentUtils) {
        $controller('_ProjectDeployerEditDeploymentController', {$scope: $scope});

        let deploymentIds = [];

        $scope.uiState = $scope.uiState || {};
        angular.extend($scope.uiState, {
            deploymentInfo: {},
            selectedFolder: {},
            selectedProject: {}
        });
        $scope.deploymentSettings = {};
        $scope.deployableProjectStatusList = [];


        // --- Project folders

        $scope.canSelect = item => item.canWriteContents;
        $scope.getProjectFolderName = item => item.name;
        $scope.browse = (folderIds) => {
            const destination = PathUtils.makeNLNT(folderIds).split('/').pop();
            const folder = (destination && searchTree($scope.projectFolderHierarchy, destination)) || $scope.projectFolderHierarchy;
            const pathElts = treeToList(folder, folder => folder.parent).map(f => angular.extend({}, f, { toString: () => f.id }));

            return PromiseService.qToHttp($timeout(() => ({
                exists: true,
                directory: true,
                children: folder.children,
                pathElts: pathElts
            }), 0));
        };
        $scope.canEditFolder = function() {
            return $scope.projectFolderHierarchy.id;
        };

        function setProjectFolder() {
            if ($scope.projectFolderHierarchy && $scope.projectFolderHierarchy.id) {
                $scope.uiState.selectedFolder = $scope.projectFolderHierarchy;

                if ($scope.deploymentSettings.projectFolderId) {
                    const folder = searchTree($scope.projectFolderHierarchy, $scope.deploymentSettings.projectFolderId);

                    if (folder) {
                        $scope.uiState.selectedFolder = folder;
                    } else {
                        $scope.deploymentSettings.projectFolderId = ''; // if we couldnt find the ID, reset it
                    }
                }

                $scope.uiState.selectedFolder.pathElts = $scope.uiState.selectedFolder.parent ? treeToList($scope.uiState.selectedFolder, folder => folder.parent).map(f => f.name).join('/') : '/';
            }
        }


        // --- Project/deployment naming ---

        $scope.setDeploymentId = function() {
            if ($scope.deploymentSettings.publishedProjectKey && $scope.deploymentSettings.infraId) {
                $scope.deploymentSettings.id = StringUtils.transmogrify(
                    DeploymentUtils.sanitizeInfraId(`${$scope.deploymentSettings.publishedProjectKey}-on-${$scope.deploymentSettings.infraId}`),
                    deploymentIds,
                    (count, name) => `${name}-${count}`
                );
            }
        };

        $scope.doesDeploymentIdExist = function(deploymentId) {
            return !deploymentIds.includes(deploymentId);
        };

        // default deployment settings
        let automationProjectList = null;
        function setAutomationProjects() {
            if ($scope.deploymentSettings.infraId) {
                resetErrorInScope($scope);
                $scope.publishedProjectKeyExistsOnAutomationNode = false;
                return DataikuAPI.projectdeployer.infras.getProjectKeys($scope.deploymentSettings.infraId).success(infraProjects => {
                    automationProjectList = infraProjects;
                    $scope.setTargetProjectKey();
                });
            }
        }

        $scope.setTargetProjectKey = function() {
            if ($scope.deploymentSettings.publishedProjectKey) {
                const projectList = automationProjectList || ($scope.deploymentSettings.deployedProjectKey ? [$scope.deploymentSettings.deployedProjectKey] : []);

                $scope.deploymentSettings.deployedProjectKey = StringUtils.transmogrify(
                    $scope.deploymentSettings.publishedProjectKey,
                    projectList,
                    (count, name) => `${name}_${count}`
                );

                $scope.publishedProjectKeyExistsOnAutomationNode = $scope.deploymentSettings.deployedProjectKey !== $scope.deploymentSettings.publishedProjectKey;
            }
        };

        $scope.$watch('uiState.selectedProject', function(nv) {
            if (nv && $scope.deploymentSettings) {
                if ($scope.deploymentSettings.publishedProjectKey && $scope.uiState.selectedProject) {
                    $scope.bundles = $scope.uiState.selectedProject.packages;
                }
            }
        });

        $scope.setSelectedProject = function() {
            if ($scope.deployableProjectStatusList) {
                $scope.uiState.selectedProject = $scope.deployableProjectStatusList.find(project => $scope.deploymentSettings && project.projectBasicInfo.id === $scope.deploymentSettings.publishedProjectKey);
            }
        };

        $scope.$watch('projects', function(nv) {
            if (nv) {
                $scope.deployableProjectStatusList = $scope.projects.filter(project => project.packages.length && project.canDeploy);
                $scope.setSelectedProject();
            }
        });

        DataikuAPI.projectdeployer.infras.listLightStatus().success(infras => {
            $scope.deployableInfraStatusList = infras.filter(infra => infra.canDeploy);
        }).error(setErrorInScope.bind($scope));

        $scope.selectedInfraIsMultiNode = function() {
            const infraId = $scope.deploymentSettings.infraId;
            return infraId && $scope.deployableInfraStatusList.find(infra => infra.infraBasicInfo.id === infraId).infraBasicInfo.type === 'MULTI_AUTOMATION_NODE';
        }

        $scope.$watch('deploymentSettings.infraId', function(nv, ov) {
            if (nv) {
                if (ov || !$scope.deploymentSettings.projectFolderId) {
                    $scope.deploymentSettings.projectFolderId = 'ROOT'; // reset
                }

                setAutomationProjects();
                $scope.setFolderHierarchy(setProjectFolder);
                $scope.setDeploymentId();
            }
        });

        $scope.ok = function() {
            ProjectDeployerDeploymentService.openGovernanceStatusNewDeployment($scope.deploymentSettings.publishedProjectKey, $scope.deploymentSettings.infraId, $scope.deploymentSettings.bundleId).then(function() {
                DataikuAPI.projectdeployer.deployments.create($scope.deploymentSettings.id, $scope.deploymentSettings.publishedProjectKey, $scope.deploymentSettings.infraId, $scope.deploymentSettings.bundleId, $scope.deploymentSettings.deployedProjectKey, $scope.deploymentSettings.projectFolderId)
                .success($scope.resolveModal)
                .error(setErrorInScope.bind($scope));
            });
        };

        function setDeployments() {
            DataikuAPI.projectdeployer.deployments.listBasicInfo().success(infras => {
                deploymentIds = infras.deployments.map(d => d.id);
            }).error(setErrorInScope.bind($scope));
        }

        setDeployments();
    });

    app.controller('ProjectDeployerDeploymentCreationModalController', function($scope, $controller, DataikuAPI, ProjectDeployerDeploymentService) {
        $controller('_ProjectDeployerDeploymentModalController', {$scope: $scope});

        $scope.ok = function() {
            return ProjectDeployerDeploymentService.openGovernanceStatusNewDeployment($scope.deploymentSettings.publishedProjectKey, $scope.deploymentSettings.infraId, $scope.deploymentSettings.bundleId).then(function() {
                return DataikuAPI.projectdeployer.deployments.create($scope.deploymentSettings.id, $scope.deploymentSettings.publishedProjectKey, $scope.deploymentSettings.infraId, $scope.deploymentSettings.bundleId, $scope.deploymentSettings.deployedProjectKey, $scope.deploymentSettings.projectFolderId)
                    .success($scope.resolveModal)
                    .error(setErrorInScope.bind($scope));
            });
        };

        $scope.$watch('deploymentSettings.publishedProjectKey', function(nv) {
            if (nv) {
                $scope.setSelectedProject();
                $scope.setTargetProjectKey();
                $scope.setDeploymentId();
            }
        });

        if ($scope.appConfig.licensedFeatures.projectStandardsAllowed) {
            $scope.uiState = $scope.uiState || {};
            $scope.uiState.showBundlePSWarning = false;

            $scope.$watchGroup([
                    'deploymentSettings.publishedProjectKey',
                    'deploymentSettings.bundleId','deploymentSettings.infraId'
                ], function(newValues) {
                    const [publishedProjectKey, bundleId, infraId] = newValues;
                    if (publishedProjectKey && bundleId && infraId) {
                        DataikuAPI.projectdeployer.deployments.getProjectStandardsStatusBundleOnInfra(infraId, publishedProjectKey, bundleId)
                            .success(projectStandardsMessages => {
                                $scope.uiState.showBundlePSWarning = projectStandardsMessages.maxSeverity  === 'WARNING' || projectStandardsMessages.maxSeverity === 'ERROR';
                            })
                            .error(setErrorInScope.bind($scope));
                    } else {
                        $scope.uiState.showBundlePSWarning = false;
                    }
                }
            );
        }
    });

    app.controller('ProjectDeployerDeploymentCopyModalController', function($scope, $controller, DataikuAPI, ProjectDeployerDeploymentService) {
        $controller('_ProjectDeployerDeploymentModalController', {$scope: $scope});

        $scope.ok = function() {
            return ProjectDeployerDeploymentService.openGovernanceStatusDeploymentId($scope.oldDeploymentId, $scope.deploymentSettings.infraId).then(function() {
                return DataikuAPI.projectdeployer.deployments.copy($scope.oldDeploymentId, $scope.deploymentSettings.id, $scope.deploymentSettings.infraId, $scope.deploymentSettings.deployedProjectKey, $scope.deploymentSettings.projectFolderId)
                    .success($scope.resolveModal)
                    .error(setErrorInScope.bind($scope));
            });
        };

        $scope.$watch('deploymentSettings.publishedProjectKey', function(nv) {
            if (nv) {
                $scope.setTargetProjectKey();
                $scope.setDeploymentId();
            }
        });
    });

    app.controller('ProjectDeployerDeploymentController', function(TopNav, $scope, $rootScope, $state, $controller, ActivityIndicator, CreateModalFromTemplate,
                                                                   WT1, DataikuAPI, DeployerUtils, ProjectDeployerDeploymentService, Dialogs, CreateModalFromComponent,
                                                                   projectDeploymentModalDirective, multiNodeInfraApiKeyCreationModalDirective,
                                                                   projectDeploymentProjectStandardsModalDirective) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments');

        $controller('_DeployerDeploymentController', {$scope});

        $scope.localVariables = {
            asJSON: '{}',
            saved: '{}'
        };

        $scope.deleteWarning = 'The deployed project on the Automation node will not be deleted. You will need to manually delete the project on the Automation node.';

        if (!$scope.deploymentSettings) {
            $scope.getDeploymentSettings();
        }

        $scope.getScenarioRuns = function() {
            if ($scope.lightStatus.neverEverDeployed) {
                return;
            }
            const dateFormat = 'YYYY-MM-DD';
            $scope.isLoadingScenarioRuns = true;
            return DataikuAPI.projectdeployer.deployments.scenarioLastRuns($state.params.deploymentId, moment().subtract(14, 'days').format(dateFormat), moment().add(1, 'days').format(dateFormat))
            .success(scenarioRunsByNode => {
                $scope.scenarioRuns = scenarioRunsByNode.map(e => ({nodeId: e.automationNodeId, ...e.scenarioRuns}));
            }).error(() => {
                $scope.scenarioRuns = $scope.scenarioRuns || [];
            }).finally(() => {
                addLastRuns($scope.scenarioRuns);
                $scope.scenarioRows = $scope.scenarioRuns.map(
                    run => run.rows.map(
                        row => ({ nodeId : run.nodeId, columns: run.columns, ...row })
                    )
                ).flat();
                $scope.scenarioRows.sort((row1, row2) => row1.info.name.localeCompare(row2.info.name))

                $scope.hasScenarioRuns = $scope.scenarioRows.length > 0;
                $scope.scenariosHasNodeIds = $scope.scenarioRuns.some(s => !!s.nodeId);
                $scope.isLoadingScenarioRuns = false;
            });
        };

        function addLastRuns(scenarioRuns) {
            for (let scenario of scenarioRuns) {
                (scenario.rows || []).forEach(function(row) {
                    const id = row.uniqueId;
                    if (row.info.lastRunDate && row.info.lastRunOutcome) {
                        row.lastRun = {
                            date: row.info.lastRunDate,
                            dateDisplay: moment(parseInt(row.info.lastRunTimestamp)).format('Y-MM-DD H:m:s (Z)'),
                            outcome: row.info.lastRunOutcome.toLowerCase()
                        };
                    }
                });
            }
        }

        $scope.projectDeploymentIsDirty = function() {
            return $scope.localVariables.asJSON !== $scope.localVariables.saved || $scope.deploymentIsDirty();
        };

        $scope.saveProjectDeployment = function(triggerEvent = true) {
            if (triggerEvent) {
                WT1.event('project-deployer-deployment-save', { deploymentType: 'PROJECT' });
            }
            return ProjectDeployerDeploymentService.openGovernanceStatusDeployment($scope.deploymentSettings).then(function () {
                try {
                    $scope.deploymentSettings.localVariables = JSON.parse($scope.localVariables.asJSON || '{}');
                    return $scope.saveDeployment();
                } catch (err) {
                    ActivityIndicator.error("Invalid format: "+err.message);
                }
            });
        };

        $scope.saveAndUpdateProjectDeployment = function() {
            const savePromise = $scope.saveProjectDeployment(false);
            if (savePromise) {
                return savePromise.then(function() { return $scope.deployProjectDeployment(true, true); });
            }
        };

        $scope.onUpdateFinished = function() {
            $scope.getLightStatus();
            $scope.getScenarioRuns();
        }

        $scope.openProgressModal = function(jobId) {
            CreateModalFromComponent(projectDeploymentModalDirective, {
                deploymentId: $scope.lightStatus.deploymentBasicInfo.id,
                jobId: jobId,
                insufficientPermissionsMessage: $scope.insufficientPermissionsMessage(),
                lastDeploymentAction: $scope.lastDeploymentAction,
                peekingUpdateStarted: $scope.onPeekingUpdateStarted,
                peekingUpdateEnded: $scope.fetchLastDeploymentActionUntilCanceled,
                deployProject: deployProject,
            }, ['modal-wide', 'modal-fixed-height500'])
        }

        function deployProject() {
            DataikuAPI.projectdeployer.deployments.update($scope.lightStatus.deploymentBasicInfo.id).success(function(data) {
                $scope.openProgressModal(data.jobId);
            }).error(setErrorInScope.bind($scope));
        }

        function openProjectStandardsOnDeployment(projectStandardsMessages) {
            return CreateModalFromComponent(projectDeploymentProjectStandardsModalDirective, {
                projectStandardsMessages: projectStandardsMessages
            }, ['modal-medium']);
        };

        function checkProjectStandardsAndDeploy() {
            let projectStandardsPolicy = $scope.lightStatus.infraBasicInfo.projectStandardsPolicy;

            if(projectStandardsPolicy && projectStandardsPolicy.checkPolicy === "WARN") {
                DataikuAPI.projectdeployer.deployments.getProjectStandardsStatusDeploymentId($scope.deploymentSettings.id)
                    .success(projectStandardsMessages => {
                        if(projectStandardsMessages.maxSeverity  === 'WARNING' || projectStandardsMessages.maxSeverity === 'ERROR') {
                            openProjectStandardsOnDeployment(projectStandardsMessages).then(deployProject);
                        } else {
                            deployProject();
                        }
                    })
                    .error(setErrorInScope.bind($scope));
            } else {
                deployProject();
            }
        }

        $scope.deployProjectDeployment = function(skipGovernanceCheck, isSaveAndUpdate = false) {
            let deployAction = $scope.appConfig.licensedFeatures.projectStandardsAllowed ? checkProjectStandardsAndDeploy : deployProject;

            let governCheckFn = () => {
                if (skipGovernanceCheck) {
                    deployAction();
                } else {
                    ProjectDeployerDeploymentService.openGovernanceStatusDeploymentId($scope.lightStatus.deploymentBasicInfo.id).then(deployAction);
                }
                if (!!$scope.lightStatus && $scope.lightStatus.neverEverDeployed) {
                    WT1.event('project-deployer-deployment-deploy', { deploymentType: 'PROJECT' });
                } else if (isSaveAndUpdate) {
                    WT1.event('project-deployer-deployment-save-and-update', { deploymentType: 'PROJECT' });
                } else {
                    WT1.event('project-deployer-deployment-update', { deploymentType: 'PROJECT' });
                }
            };

            if ($scope.lightStatus.neverEverDeployed) {
                governCheckFn();
            } else {
                Dialogs.confirm($scope, " Update deployment",
                "Deploy the bundle " + $scope.deploymentSettings.bundleId + " of project " + $scope.lightStatus.projectBasicInfo.name +
                " on " + $scope.deploymentSettings.infraId).then(function() {
                    governCheckFn();
                });
            }
        };

        $scope.copyDeployment = function() {
            if ($scope.deploymentSettings) {
                return CreateModalFromTemplate('/templates/project-deployer/copy-deployment-modal.html', $rootScope, null, function(modalScope) {
                    modalScope.oldDeploymentId = $scope.deploymentSettings.id;
                    modalScope.deploymentSettings = {
                        publishedProjectKey: $scope.deploymentSettings.publishedProjectKey,
                    };
                }).then(function(newDeployment) {
                    $state.go('projectdeployer.deployments.deployment.status', {deploymentId: newDeployment.id});
                    WT1.event('project-deployer-deployment-copy', { deploymentType: 'PROJECT' });
                });
            }
        };

        $scope.generatePersonalAPIKey = function() {
            CreateModalFromComponent(multiNodeInfraApiKeyCreationModalDirective, {
                infraId: $scope.lightStatus.infraBasicInfo.id,
                isAdmin: $scope.lightStatus.isInfraAdmin,
                fromDeployment: true,
            });
        }

        // run once to get the scenarios runs after load
        const deregister = $scope.$watch("lightStatus", function(nv) {
            if (!nv) return;
            $scope.getScenarioRuns();
            deregister();
        }, false);

        $scope.$watch("lightStatus", function(nv, ov) {
            if ($scope.lightStatus) {
                $scope.isMultiAutomationNodeInfra = $scope.lightStatus.infraBasicInfo.type === "MULTI_AUTOMATION_NODE";
                $scope.isLoadingHeavyStatus = true;
                $scope.getHeavyStatus(false).error(function(a,b,c) {
                    $scope.heavyStatus = {
                        health: "LOADING_FAILED",
                        healthMessages: DeployerUtils.getFailedHeavyStatusLoadMessage(getErrorDetails(a,b,c))
                    };
                }).finally(() => {
                    $scope.isLoadingHeavyStatus = false;
                });
            }
        });

        const allowedTransitions = [
            'projectdeployer.deployments.deployment.status',
            'projectdeployer.deployments.deployment.history',
            'projectdeployer.deployments.deployment.settings'
        ];
        checkChangesBeforeLeaving($scope, $scope.projectDeploymentIsDirty, null, allowedTransitions);
    });

    app.controller('ProjectDeployerDeploymentSettingsController', function($scope, $controller, DataikuAPI, ProjectDeployerDeploymentUtils, CodeMirrorSettingService, AutomationUtils) {
        $controller('_ProjectDeployerEditDeploymentController', {$scope: $scope});

        $scope.codeMirrorSettingService = CodeMirrorSettingService;
        $scope.AutomationUtils = AutomationUtils;

        angular.extend($scope.uiState, {
            selectedBundle: {},
            connectionNames: [],
            settingsPane: 'information'
        });

        $scope.setupDeploymentUI = function() {
            $scope.localVariables.asJSON = JSON.stringify($scope.deploymentSettings.localVariables, null, 2);
            $scope.localVariables.saved = $scope.localVariables.asJSON;
        };

        // Scenarios
        $scope.allScenariosActive = function () {
            const scenarioIds = Object.keys($scope.deploymentSettings.scenariosToActivate);
            return scenarioIds.length && scenarioIds.every(key => $scope.deploymentSettings.scenariosToActivate[key]);
        };

        $scope.toggleScenarios = function () {
            const allActive = $scope.allScenariosActive();

            $scope.uiState.selectedBundle.scenarios.forEach(scenario => {
                $scope.deploymentSettings.scenariosToActivate[scenario.id] = !allActive;
            });
        };

        $scope.getAvailableConnections = function() {
            DataikuAPI.projectdeployer.infras.listConnectionsNames($scope.deploymentSettings.infraId, $scope.deploymentSettings.id).success(connections => {
                $scope.uiState.availableConnections = connections;
            });
        }

        function setProjectFolderPath() {
            if ($scope.deploymentSettings.projectFolderId) {
                const folder = searchTree($scope.projectFolderHierarchy, $scope.deploymentSettings.projectFolderId);

                if (folder && folder.parent) {
                    $scope.projectFolderPath = treeToList(folder, folder => folder.parent).map(f => f.name).join('/');
                }
            }
        }

        function getBundleDetails() {
            if ($scope.deploymentSettings.publishedProjectKey && $scope.deploymentSettings.bundleId) {
                // reset selected bundle in case of error
                $scope.uiState.selectedBundle = { usedCodeEnvs: [], scenarios: [] };
                $scope.uiState.connectionNames = [];
                DataikuAPI.projectdeployer.publishedProjects.getBundleDetailsExtended($scope.deploymentSettings.publishedProjectKey, $scope.deploymentSettings.bundleId).success(bundleDetails => {
                    $scope.uiState.selectedBundle = bundleDetails;
                    $scope.uiState.connectionNames = bundleDetails.usedConnections.map(c => c.name); // for suggestions

                    $scope.deploymentSettings.scenariosToActivate = ProjectDeployerDeploymentUtils.getUpdatedScenarioMap($scope.deploymentSettings.scenariosToActivate, bundleDetails.scenarios);
                });
            }
        }

        function setBundleList() {
            if ($scope.bundles || !$scope.deploymentSettings || !$scope.projects) return;

            const project = $scope.projects.find(project => $scope.deploymentSettings && project.projectBasicInfo.id === $scope.deploymentSettings.publishedProjectKey);

            if (project) {
                $scope.bundles = project.packages;

                // if a deployment's bundle was deleted, still include it in the list of bundles
                const bundleId = $scope.deploymentSettings.bundleId;

                if (!$scope.bundles.some(bundle => bundle.id === bundleId)) {
                    $scope.bundles.unshift({
                        id: bundleId,
                        name: `${bundleId} (deleted)`
                    });
                }
            }
        }

        $scope.$watch('deploymentSettings', function (nv, ov) {
            if (nv) {
                $scope.setupDeploymentUI();
                $scope.setFolderHierarchy(setProjectFolderPath);
                $scope.getAvailableConnections();
            }
        });

        $scope.$watch('deploymentSettings.bundleId', function(nv, ov) {
            if (nv) {
                getBundleDetails();
                setBundleList();
            }
        });

        $scope.$watch('projects', function(nv) {
            if (nv) {
                setBundleList();
            }
        });


        if ($scope.appConfig.licensedFeatures.projectStandardsAllowed) {
            $scope.uiState = $scope.uiState || {};
            $scope.uiState.showBundlePSWarning = false;

            $scope.$watchGroup([
                    'deploymentSettings.publishedProjectKey',
                    'deploymentSettings.bundleId','deploymentSettings.infraId'
                ], function(newValues) {
                    const [publishedProjectKey, bundleId, infraId] = newValues;
                    if (publishedProjectKey && bundleId && infraId) {
                        DataikuAPI.projectdeployer.deployments.getProjectStandardsStatusBundleOnInfra(infraId, publishedProjectKey, bundleId)
                            .success(projectStandardsMessages => {
                                $scope.uiState.showBundlePSWarning = projectStandardsMessages.maxSeverity  === 'WARNING' || projectStandardsMessages.maxSeverity === 'ERROR';
                            })
                            .error(setErrorInScope.bind($scope));
                    } else {
                        $scope.uiState.showBundlePSWarning = false;
                    }
                }
            );
        }
    });

    app.controller('ProjectDeployerDeploymentStatusController', function($scope, TopNav, DataikuAPI, ProjectDeployerDeploymentUtils) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments', null, 'status');

        $scope.uiState = {
            activeTab: 'summary'
        };

        $scope.refresh = function() {
            try {
                $scope.refreshLightAndHeavy()
            } finally {
                $scope.getScenarioRuns()
            }
        }

        $scope.$watch('lightStatus', function() {
            if (!$scope.lightStatus || !$scope.lightStatus.deploymentBasicInfo || !$scope.lightStatus.infraBasicInfo) return;

            const automationProjectKey = ProjectDeployerDeploymentUtils.getAutomationProject($scope.lightStatus.deploymentBasicInfo);

            $scope.automationProjectUrl = null;
            $scope.automationProjectUrls = {};

            if ($scope.isMultiAutomationNodeInfra) {
                for (let automationNode of $scope.lightStatus.infraBasicInfo.automationNodes) {
                    $scope.automationProjectUrls[automationNode.automationNodeId] = ProjectDeployerDeploymentUtils.getAutomationProjectUrl(automationNode.automationNodeExternalUrl, automationProjectKey);
                }
            } else {
                $scope.automationProjectUrl = ProjectDeployerDeploymentUtils.getAutomationProjectUrl($scope.lightStatus.infraBasicInfo.automationNodeExternalUrl, automationProjectKey);
            }

            DataikuAPI.unifiedMonitoring.deployer.deployedProjects.get($scope.lightStatus.deploymentBasicInfo.id).success((monitoring) => {
                if (monitoring) {
                    if (monitoring.disabledInfra) {
                        $scope.monitoring = undefined;
                        if ($scope.uiState.activeTab === 'health') {
                            $scope.uiState.activeTab = 'summary';
                        }
                        $scope.monitoringDisabled = true;
                    }
                    else {
                        $scope.monitoring = monitoring;
                        $scope.monitoringDisabled = false;
                    }
                }
            }).error(setErrorInScope.bind($scope));
        })

        $scope.$watchGroup(['isLoadingHeavyStatus', 'isLoadingScenarioRuns'],
            function ([isLoadingHeavyStatus, isLoadingScenarioRuns]) {
                $scope.isDataLoadingInProgress = isLoadingHeavyStatus || isLoadingScenarioRuns;
            }
        );
    });

    app.component('projectDeploymentUpdateProgress', {
        bindings: {
            deploymentUpdate: '<',
        },
        template: `
            <ul class="raw-unstyled-ul padtop4">
                <li ng-repeat="step in $ctrl.deployment.steps | orderBy:'index'" ng-if="step.status !== $ctrl.STEP_STATUS.NOT_STARTED">
                    <span ng-if="step.status === $ctrl.STEP_STATUS.IN_PROGRESS">{{ step.name | capitalize }}...</span>
                    <span ng-if="step.status !== $ctrl.STEP_STATUS.IN_PROGRESS">Done {{ step.name }}{{ $ctrl.stepStatusDetails(step) }}</span>
                </li>
                <li ng-if="$ctrl.deployment.status === $ctrl.DEPLOY_STATUS.DONE_WITH_WARNINGS"><strong>Project successfully updated (with warnings)</strong></li>
                <li ng-if="$ctrl.deployment.status === $ctrl.DEPLOY_STATUS.DONE"><strong>Project successfully updated</strong></li>
            </ul>

            <deployment-hooks-states hook-execution-status="$ctrl.deployment.deploymentHookExecutionStatus" max-width-px="450"></deployment-hooks-states>

            <div ng-if="$ctrl.deployment.infoMessages.messages.length && $ctrl.deployment.status !== $ctrl.DEPLOY_STATUS.IN_PROGRESS" info-messages-raw-list-with-alert="$ctrl.deployment.infoMessages" class="mtop24 mbot24" />

            <div ng-show="$ctrl.deployment.futureResponse && $ctrl.deployment.futureResponse.log && $ctrl.deployment.futureResponse.log.lines.length > 0">
                <pre smart-log-tail="$ctrl.deployment.futureResponse.log" style="max-height: 400px;"/>
            </div>

            <div ng-show="$ctrl.deployment.error && $ctrl.deployment.error.lines && $ctrl.deployment.error.length > 0">
                <pre smart-log-tail="$ctrl.deployment.error" style="max-height: 400px;"/>
            </div>
        `,
        controller: function(ProjectDeploymentProgress) {
            const $ctrl = this;

            $ctrl.STEP_STATUS = ProjectDeploymentProgress.STEP_STATUS;
            $ctrl.DEPLOY_STATUS = ProjectDeploymentProgress.DEPLOY_STATUS;
            $ctrl.stepStatusDetails = ProjectDeploymentProgress.stepStatusDetails;

            $ctrl.deployment = ProjectDeploymentProgress.initDeployment();

            $ctrl.$onChanges = function(changes) {
                const report = changes?.deploymentUpdate.currentValue?.report;
                if (report) {
                    const hooksStatus = report.deploymentHookExecutionStatus;
                    hooksStatus.hasHooks = true; // TODO This is a by-pass. It will require refactoring of hooks to share responsibility for displaying deployment messages
                    ProjectDeploymentProgress.updateDeploymentSteps(report, hooksStatus, $ctrl.deployment);
                    ProjectDeploymentProgress.updateDeploymentStatus($ctrl.deployment);
                }
            }
        }
    });


    app.controller('ProjectDeployerDeploymentUpdatesController', function($scope, $state, TopNav, DataikuAPI) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments', null, 'last-updates');

        $scope.listDeploymentUpdateHeads = function() {
            return DataikuAPI.projectdeployer.deployments.listDeploymentUpdateHeads($state.params.deploymentId);
        };

        $scope.getDeploymentUpdate = function(startTimestamp) {
            return DataikuAPI.projectdeployer.deployments.getDeploymentUpdate($state.params.deploymentId, startTimestamp);
        };

        $scope.getDeploymentUpdateSettingsDiff = function(startTimestamp) {
            return DataikuAPI.projectdeployer.deployments.getDeploymentUpdateSettingsDiff($state.params.deploymentId, startTimestamp);
        };
    });

    app.controller('ProjectDeployerDeploymentLogsController', function($scope, $state, TopNav, DataikuAPI, Logs) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments', null, 'logs');

        $scope.getLog = DataikuAPI.projectdeployer.deployments.getLog;
        $scope.deploymentId = $state.params.deploymentId;
        $scope.downloadURL = function(deploymentId, logName) {
            return "/dip/api/project-deployer/deployments/stream-log?deploymentId=" + encodeURIComponent(deploymentId) + "&logName=" + encodeURIComponent(logName);
        }
        $scope.downloadAllURL = function(deploymentId) {
            return "/dip/api/project-deployer/deployments/all-logs-zip?deploymentId=" + encodeURIComponent(deploymentId);
        }

        function refreshLogList() {
            DataikuAPI.projectdeployer.deployments.listLogs($scope.deploymentId).success(function(data) {
                $scope.logs = data;
            }).error(setErrorInScope.bind($scope));
        }

        $scope.$watch("lightStatus", function(nv, ov) {
            refreshLogList();
        });
        refreshLogList();
    });

    app.service("ProjectDeployerDeploymentUtils", function() {
        const svc = {};

        svc.getAutomationProject = function(deploymentBasicInfo) {
            return deploymentBasicInfo.deployedProjectKey || deploymentBasicInfo.publishedProjectKey;
        };

        svc.getAutomationProjectUrl = function(baseUrl, projectKey) {
            if (baseUrl.substr(-1) !== '/') {
                baseUrl += '/';
            }

            return `${baseUrl}projects/${projectKey}/`;
        };

        /*
            When switching bundles, check if any of the scenarios
            found in the new bundle already exist in the deployment
            scenarios (from the previous bundle)
            If so, use the same value
        */
        svc.getUpdatedScenarioMap = function(previousBundleScenarios, bundleScenarios) {
            return bundleScenarios.reduce((obj, scenario) => {
                const isExistingScenario = scenario.id in previousBundleScenarios;
                const newScenario = isExistingScenario ? {
                    [scenario.id]: previousBundleScenarios[scenario.id]
                } : {};

                return { ...obj, ...newScenario };
            }, {});
        };

        return svc;
    });

    app.controller('ProjectDeployerDeploymentScenarioRunsController', function($scope, LocalStorage) {
        const LS_FILTER_INPUTS_KEY = "dku.deployer.scenario-runs.filter-inputs";
        const DAYS = 15;

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.scenarioRunsSearchQuery = '';
        $scope.uiState.scenarioRunsFilterInputs = {
            showAll: false,
            sortColumn: "info.name",
            sortDescending: true
        };
        Object.assign($scope.uiState.scenarioRunsFilterInputs, LocalStorage.get(LS_FILTER_INPUTS_KEY));

        $scope.columns = Array.from(Array(DAYS)).map((day, index) => {
            const date = moment().subtract(DAYS - index - 1, 'days');
            return {
                date: date.format('YYYY-MM-DD'),
                dow: date.weekday(),
                dateFormatted: date.format('D/M'),
                dateDay: date.format('ddd'),
                dateTooltip: date.format('MMM. Do')
            }
        });

        $scope.$watch("uiState.scenarioRunsFilterInputs", (nv, ov) => {
            if(!angular.equals(nv, ov)) {
                LocalStorage.set(LS_FILTER_INPUTS_KEY, $scope.uiState.scenarioRunsFilterInputs);
            }
        }, true);

        $scope.filterScenarioRuns = function(scenarioRun) {
            const lcSearchQuery = $scope.uiState.scenarioRunsSearchQuery ? $scope.uiState.scenarioRunsSearchQuery.toLowerCase() : "";
            const minDate = moment().subtract(DAYS - 1, 'days');

            return ($scope.uiState.scenarioRunsFilterInputs.showAll ||
                (scenarioRun.lastRun && scenarioRun.lastRun.date && new Date(scenarioRun.lastRun.date) > minDate && scenarioRun.info.active)) &&
                (scenarioRun.info.name.toLowerCase().includes(lcSearchQuery) || (scenarioRun.nodeId && scenarioRun.nodeId.toLowerCase().includes(lcSearchQuery)));
        }

        $scope.getScenarioLastRunsUrl = function(automationNodeId, scenarioId) {
            if ($scope.isMultiAutomationNodeInfra) {
                return $scope.automationProjectUrls[automationNodeId] + "scenarios/" + scenarioId + "/runs/list/";
            }
            return $scope.automationProjectUrl + "scenarios/" + scenarioId + "/runs/list/";
        };

        $scope.getMonitoringUrl = function(automationNodeId) {
            if ($scope.isMultiAutomationNodeInfra) {
                return $scope.automationProjectUrls[automationNodeId] + "automation";
            }
            return $scope.automationProjectUrl + "automation";
        };

        $scope.hovered = {date : null};

        function getTooltipInfo(row, column) {
            let runs = row.columns.find(c => c.date === column.date).actions[row.uniqueId];
            if (runs) {
                runs = runs.sort((a, b) => b.start - a.start).map(r => {
                return {
                    date: moment(parseInt(r.start)).format('HH:mm:ss'),
                    outcome: r.outcome
                }});
            }
            return {
                dateTooltip: column.dateTooltip,
                runs: runs
            }
        }

        $scope.hover = function(evt, row, column, localScope) {
            $scope.hovered.date = column.date;
            if (column) {
                evt.target.classList.add('mainzone');
                $scope.popoverColumn = getTooltipInfo(row, column);
                localScope.showPopover();
            }
        };

        $scope.unhover = function(evt, column, localScope) {
            $scope.hovered.date = null;
            if (column) {
                evt.target.classList.remove('mainzone');
                $scope.popoverColumn = null;
                localScope.hidePopover();
            }
        };

        $scope.sortColumnClass = function(colName) {
            if (colName === $scope.uiState.scenarioRunsFilterInputs.sortColumn) {
                return $scope.uiState.scenarioRunsFilterInputs.sortDescending ? 'sort-descending' : 'sort-ascending';
            }
            return '';
        }

        $scope.getCellGlobalOutcome = function(row, date, id) {
            const outcomes = row.columns.find(column => column.date === date);
            if (!outcomes || !outcomes.actions[id]) return "";
            if (outcomes.actions[id].some(scenarioOutcome => scenarioOutcome && scenarioOutcome.outcome === "FAILED")) return "failed";
            if (outcomes.actions[id].some(scenarioOutcome => scenarioOutcome && scenarioOutcome.outcome === "WARNING")) return "warning";
            if (outcomes.actions[id].some(scenarioOutcome => scenarioOutcome && scenarioOutcome.outcome === "SUCCESS")) return "success";
            if (outcomes.actions[id].some(scenarioOutcome => !scenarioOutcome)) return "running";
            return "aborted";
        };
    });

    app.controller('ProjectDeployerDeploymentWebappsController', function($scope, $filter, LocalStorage) {
        const LS_FILTER_INPUTS_KEY = "dku.deployer.webapps.filter-inputs";

        $scope.uiState = $scope.uiState || {};
        $scope.uiState.webappsSearchQuery = '';
        $scope.uiState.webappsFilterInputs = {
            showAll: false,
            sortColumn: "name",
            sortDescending: true
        };
        Object.assign($scope.uiState.webappsFilterInputs, LocalStorage.get(LS_FILTER_INPUTS_KEY));

        $scope.webappBackendInfoList = [];

        $scope.$watch("heavyStatus", function() {
            if (!$scope.heavyStatus) return;
            if ($scope.isMultiAutomationNodeInfra) {
                if (!$scope.heavyStatus.heavyStatusPerNode) return;
                const heavyStatusPerNode = $scope.heavyStatus.heavyStatusPerNode;
                $scope.webappBackendInfoList = Object.entries(heavyStatusPerNode)
                    .filter(e => typeof e[1].webappBackendInfoList !== "undefined")
                    .map(e => e[1].webappBackendInfoList.map(webapp => ({nodeId: e[0], ...webapp})))
                    .flat();
            } else {
                if (!$scope.heavyStatus.webappBackendInfoList) return;
                $scope.webappBackendInfoList = $scope.heavyStatus.webappBackendInfoList;
            }
        });

        $scope.getWebappUrl = function(webappInfo) {
            const slugifiedWebappName = $filter('slugify')(webappInfo.name);
            const webappId = `${webappInfo.id}_${slugifiedWebappName}`;
            let baseUrl = $scope.isMultiAutomationNodeInfra ? $scope.automationProjectUrls[webappInfo.nodeId] : $scope.automationProjectUrl;
            return `${baseUrl}webapps/${webappId}/view`;
        };

        $scope.$watch("uiState.webappsFilterInputs", (nv, ov) => {
            if(!angular.equals(nv, ov)) {
                LocalStorage.set(LS_FILTER_INPUTS_KEY, $scope.uiState.webappsFilterInputs);
            }
        }, true);

        $scope.filterWebapps = function(webapp) {
            const query = $scope.uiState.webappsSearchQuery.toLowerCase();
            const name = webapp.name.toLowerCase();
            const nodeId = webapp.nodeId ? webapp.nodeId.toLowerCase() : '';
            return (
                ($scope.uiState.webappsFilterInputs.showAll ? true : webapp.isBackendAutostartOn) &&
                (nodeId.includes(query) || name.includes(query))
            );
        }

        $scope.sortColumnClass = function(colName) {
            if (colName === $scope.uiState.webappsFilterInputs.sortColumn) {
                return $scope.uiState.webappsFilterInputs.sortDescending ? 'sort-descending' : 'sort-ascending';
            }
            return '';
        }
    });

    app.controller('ProjectDeployerDeploymentHistoryController', function(TopNav) {
        TopNav.setNoItem();
        TopNav.setLocation(TopNav.TOP_PROJECT_DEPLOYER, 'deployments', null, 'history');
    });

    app.component('projectInlineDeploymentCard', {
        bindings: {
            lightStatus: '<',
            heavyStatus: '<',
            lastDeploymentAction: '<',
            showLastAction: '<',
            refresh: '<'
        },
        templateUrl: '/templates/project-deployer/inline-deployment-card.html',
        controller: function(DeployerUtils, ProjectDeployerDeploymentUtils, DeployerDeploymentTileService, $state, WT1) {

            const $ctrl = this;
            $ctrl.uiState = {
                openDetails: false
            };

            $ctrl.$onChanges = function(changes) {
                if ((changes.lightStatus || changes.heavyStatus) && $ctrl.lightStatus && $ctrl.heavyStatus) {

                    $ctrl.uiState.infraHref = $state.href("projectdeployer.infras.infra.status", { infraId: $ctrl.lightStatus.infraBasicInfo.id});
                    $ctrl.deploymentStatus = DeployerDeploymentTileService.getDeploymentHealth($ctrl.heavyStatus);
                    const bundleId = $ctrl.lightStatus.deploymentBasicInfo.bundleId;
                    const bundle = $ctrl.lightStatus.packages.find(p => p.id === bundleId);

                    $ctrl.bundleOriginInfo = DeployerUtils.getOriginInfo(bundle.designNodeInfo);
                    $ctrl.isMultiNode = $ctrl.lightStatus.infraBasicInfo.type === 'MULTI_AUTOMATION_NODE';
                    $ctrl.automationInfo = {};
                    $ctrl.automationInfo.automationProjectKey = ProjectDeployerDeploymentUtils.getAutomationProject($ctrl.lightStatus.deploymentBasicInfo);
                    if (!$ctrl.isMultiNode) {
                        $ctrl.automationInfo.automationUrl = ProjectDeployerDeploymentUtils.getAutomationProjectUrl($ctrl.lightStatus.infraBasicInfo.automationNodeExternalUrl, $ctrl.automationInfo.automationProjectKey);
                    } else {
                        $ctrl.automationInfo.automationInfos = [];
                        $ctrl.lightStatus.infraBasicInfo.automationNodes.forEach(automationNode => {
                            $ctrl.automationInfo.automationInfos.push({
                                nodeId: automationNode.automationNodeId,
                                url: ProjectDeployerDeploymentUtils.getAutomationProjectUrl(automationNode.automationNodeExternalUrl, $ctrl.automationInfo.automationProjectKey),
                                projectName: $ctrl.heavyStatus.heavyStatusPerNode[automationNode.automationNodeId].name || $ctrl.automationInfo.automationProjectKey
                            });
                        });
                    }
                }
            }

            $ctrl.toggleDetails = function() {
                $ctrl.uiState.openDetails = !$ctrl.uiState.openDetails;
                WT1.event('project-deployer-deployment-details-toggle', { deploymentType: 'PROJECT', state: $ctrl.uiState.openDetails? 'opened': 'closed' });
            }

            $ctrl.lastUpdateCompleted = function() {
                return $ctrl.lastDeploymentAction?.type === 'UPDATE' && !$ctrl.lastDeploymentAction.inProgress;
            }

            $ctrl.lastActionStatus = function() {
                if ($ctrl.lastDeploymentAction && !$ctrl.lastDeploymentAction.inProgress) {
                    switch ($ctrl.lastDeploymentAction.status) {
                        case 'SUCCESS':
                        case 'INFO':
                            return 'Succeeded';
                        case 'WARNING':
                            return 'Succeeded with warning';
                        case 'ERROR':
                            return 'Failed';
                    }
                }
                return '';
            }
        }
    });

    app.component("projectDeployerMonitoringGovernStatus", {
        bindings: {
            monitoring: '<'
        },
        templateUrl: '/templates/project-deployer/project-deployer-monitoring-govern-status.html',
        controller: function($rootScope, UnifiedMonitoringService) {
            const $ctrl = this;
            $ctrl.uiState = {};

            $ctrl.$onChanges = function() {
                if ($ctrl.monitoring && $ctrl.monitoring.governanceStatus) {
                    $ctrl.uiState.governProjectUrl = `${$rootScope.appConfig.governURL}/artifact/${$ctrl.monitoring.governanceStatus.governArtifactId}`;
                    $ctrl.uiState.governStatusIcon = UnifiedMonitoringService.getStatusIcon($ctrl.monitoring.umGovernanceStatus);
                    $ctrl.uiState.governStatusLabel = UnifiedMonitoringService.getGovernStatusLabel($ctrl.monitoring.governanceStatus.validationStatus);
                    $ctrl.uiState.governPolicyLabel = UnifiedMonitoringService.getGovernPolicyLabel($ctrl.monitoring.governCheckPolicy);
                }
                $ctrl.uiState.hasGovernStatus = $ctrl.monitoring.umGovernanceStatus !== "NO_STATUS";
            }
        }
    });

    app.component("projectDeployerMonitoringDataStatus", {
        bindings: {
            monitoring: '<',
            lightStatus: '<'
        },
        templateUrl: '/templates/project-deployer/project-deployer-monitoring-data-status.html',
        controller: function(UnifiedMonitoringService) {
            const $ctrl = this;
            $ctrl.uiState = {};

            $ctrl.$onChanges = function() {

                $ctrl.uiState.dqStatuses = [];
                if ($ctrl.lightStatus && $ctrl.monitoring) {
                    $ctrl.isMultiNode = $ctrl.lightStatus.infraBasicInfo.type === 'MULTI_AUTOMATION_NODE';
                    // Building dqStatuses as array of statuses which, depending on the infra...

                    // ...being MULTI_AUTOMATION_NODE, are enriched with the nodeId [{nodeId:<string>, ...<monitoring status obj>}, ...]
                    if ($ctrl.isMultiNode) {
                        $ctrl.uiState.dqStatuses = Object.keys($ctrl.monitoring.dataQualityStatuses).map(function (key) { return {nodeId: key, ...$ctrl.monitoring.dataQualityStatuses[key]}; });
                    } else { // ...being a single automation node, is a single element array [{<monitoring status obj>}]
                        $ctrl.uiState.dqStatuses = [{...$ctrl.monitoring.dataQualityStatus}];
                    }

                    $ctrl.uiState.dqStatuses.forEach((dqStatus) => {
                        const emptyCount = dqStatus.emptyCount;
                        const errorCount = dqStatus.errorCount;
                        const okCount = dqStatus.okCount;
                        const warningCount = dqStatus.warningCount;

                        dqStatus.items = [
                            {label: 'OK', counter: okCount, iconCSSClass:'deployer-status-icon dku-icon-checkmark-circle-outline-24', barCSSClass:'deployer-data-status__ok'},
                            {label: 'Warning', counter: warningCount, iconCSSClass:'deployer-status-icon dku-icon-warning-fill-24', barCSSClass:'deployer-data-status__warning'},
                            {label: 'Error', counter: errorCount, iconCSSClass:'deployer-status-icon dku-icon-dismiss-circle-fill-24', barCSSClass:'deployer-data-status__error'},
                            {label: 'No status', counter: emptyCount, iconCSSClass:'deployer-status-icon dku-icon-line-24', barCSSClass:'deployer-data-status__notComputed'}
                        ];
                        dqStatus.numberItems = emptyCount + errorCount + okCount + warningCount;
                    });

                    if ($ctrl.uiState.dqStatuses.length > 0) {
                        $ctrl.uiState.overallDataStatusIcon = UnifiedMonitoringService.getStatusIcon($ctrl.monitoring.umDataQualityStatus);
                        $ctrl.uiState.overallDataStatusLabel = UnifiedMonitoringService.getUMStatusDisplayLabels($ctrl.monitoring.umDataQualityStatus);
                    }

                    $ctrl.uiState.dataQualityUrls = {};
                    if ($ctrl.monitoring.deployedProjectKey) {
                        if (!$ctrl.isMultiNode) {
                            let baseUrl = $ctrl.lightStatus.infraBasicInfo.automationNodeExternalUrl;
                            if (baseUrl) {
                                if (baseUrl.slice(-1) !== '/') {
                                    baseUrl += '/';
                                }
                                $ctrl.uiState.dataQualityUrls[""] = `${baseUrl}projects/${$ctrl.monitoring.deployedProjectKey}/data-quality/current-status`;
                            }
                        } else {
                            for (const node of $ctrl.lightStatus.infraBasicInfo.automationNodes) {
                                let baseUrl = node.automationNodeExternalUrl;
                                if (baseUrl) {
                                    if (baseUrl.slice(-1) !== '/') {
                                        baseUrl += '/';
                                    }
                                    $ctrl.uiState.dataQualityUrls[node.automationNodeId] = `${baseUrl}projects/${$ctrl.monitoring.deployedProjectKey}/data-quality/current-status`;
                                }
                            }
                        }
                    }
                    $ctrl.hasDataStatus = $ctrl.uiState.dqStatuses.length > 0 && $ctrl.uiState.dqStatuses.filter(s=>(s.lastRunRuleDate > -1 && s.hasDataset)).length > 0;
                }
            }
        }
    });

    app.component("projectDeployerMonitoringModelStatus", {
        bindings: {
            monitoring: '<',
            lightStatus: '<'
        },
        templateUrl: '/templates/project-deployer/project-deployer-monitoring-model-status.html',
        controller: function(UnifiedMonitoringService, ProjectDeployerDeploymentUtils) {
            const $ctrl = this;
            $ctrl.uiState = {};
            $ctrl.uiState.um = UnifiedMonitoringService;

            $ctrl.$onChanges = function() {
                if ($ctrl.monitoring && $ctrl.lightStatus) {
                    $ctrl.uiState.overallModelStatusIcon = UnifiedMonitoringService.getStatusIcon($ctrl.monitoring.umProjectModelStatus);
                    $ctrl.uiState.overallModelStatusLabel = UnifiedMonitoringService.getUMStatusDisplayLabels($ctrl.monitoring.umProjectModelStatus);

                    $ctrl.uiState.modelMonitoring = [];

                    const automationProjectKey = ProjectDeployerDeploymentUtils.getAutomationProject($ctrl.lightStatus.deploymentBasicInfo);

                    $ctrl.uiState.modelStatuses = [];

                    if ($ctrl.monitoring.infraType === "MULTI_AUTOMATION_NODE") {
                        $ctrl.automationProjectUrls = [];
                        for (let automationNode of $ctrl.lightStatus.infraBasicInfo.automationNodes) {
                            $ctrl.automationProjectUrls[automationNode.automationNodeId] = ProjectDeployerDeploymentUtils.getAutomationProjectUrl(automationNode.automationNodeExternalUrl, automationProjectKey);
                        }
                        if ($ctrl.monitoring.projectModelStatuses) {
                            $ctrl.uiState.modelStatuses = Object.keys($ctrl.monitoring.projectModelStatuses).map(function (key) { return {nodeid: key, ...$ctrl.monitoring.projectModelStatuses[key]}; });
                        }
                    } else {
                        $ctrl.automationProjectUrl = ProjectDeployerDeploymentUtils.getAutomationProjectUrl($ctrl.lightStatus.infraBasicInfo.automationNodeExternalUrl, automationProjectKey);
                        if ($ctrl.monitoring.projectModelStatus) {
                            $ctrl.uiState.modelStatuses = [{...$ctrl.monitoring.projectModelStatus}];
                        }
                    }

                    $ctrl.uiState.modelStatuses.forEach((modelStatus) => {
                        const autoUrl = $ctrl.automationProjectUrls ? $ctrl.automationProjectUrls[modelStatus.nodeid] : $ctrl.automationProjectUrl;
                        modelStatus.modelMonitoring = [];
                        modelStatus.statuses.forEach((status) => {
                            const smvElements = {
                                smvSmartId: status.smvSmartId,
                                savedModelId:  status.savedModelId,
                                savedModelName: status.savedModelName,
                                smvRowSpan: $ctrl.getNbLinesPerSmv(status),
                                smvUrl: autoUrl + 'savedmodels/' + status.savedModelId + '/p/' + status.smvSmartId + '/tabular-summary'

                            };
                            status.mesMonitoringStatuses.forEach((mes, mesIdx) => {
                                const mesChecks = UnifiedMonitoringService.mesMonitoringStatusToChecks(mes);
                                const mesElements = {
                                    mesName: mes.mesName,
                                    mesId: mes.id,
                                    mesRowSpan: mesChecks.length,
                                    mesUrl: autoUrl + 'modelevaluationstores/' + mes.id + '/evaluations/'
                                };
                                mesChecks.forEach((check, checkIdx) => {
                                    let resElem = {...check};
                                    if (mesIdx === 0 && checkIdx === 0) {
                                        resElem = {...resElem, ...smvElements};
                                    }
                                    if (checkIdx === 0) {
                                        resElem = {...resElem, ...mesElements};
                                    }
                                    modelStatus.modelMonitoring.push(resElem);
                                });
                            });
                        });
                    });
                }
            }

            $ctrl.getNbLinesPerMes = function(status) {
                return (Object.keys(status.errors).length + Object.keys(status.warnings).length + Object.keys(status.successes).length) || 1;
            }

            $ctrl.getNbLinesPerSmv = function(status) {
                return status.mesMonitoringStatuses.reduce((i, s) => $ctrl.getNbLinesPerMes(s) + i, 0) || 1;
            }

            $ctrl.hasModelStatus = function() {
                return $ctrl.uiState.modelStatuses && $ctrl.uiState.modelStatuses.length > 0 && $ctrl.uiState.modelStatuses.filter(s => s.modelMonitoring.length > 0).length > 0;
            }
        }
    });

})();
