(function() {
    'use strict';

    /**
     * <axis-range-subform
     *      axis='x'
     *      chart-type='chart.def.type'
     *      custom-extent='chart.def.xAxisFormatting.customExtent'>
     * </axis-range-subform>
     *
     * @param { String }    axis                        - 'x' or 'y'
     * @param { Object }    customExtent
     * @param { String }    customExtent.editMode       - ('AUTO' or 'MANUAL'); // the mode to use to define extent (auto computed or manually defined)
     * @param { Array }     customExtent.$autoExtent     - [minExtent, maxExtent]; // array of 2 floats: initial min and max values auto-detected (used in AUTO editMode)
     * @param { Array }     customExtent.manualExtent   - [minExtent, maxExtent]; // array of 2 floats or null: user custom min and max values.
     *                                                  If null, min or max are auto computed (used in MANUAL editMode)
     */
    angular.module('dataiku.charts').component('axisRangeSubform', {
        templateUrl: '/static/dataiku/js/simple_report/components/axis-range-subform/axis-range-subform.component.html',
        bindings: {
            axis: '@',
            chartType: '<',
            customExtent: '=',
            includeZero: '=?',
            editModeDisabled: '<'
        },
        controller: function(
            ChartsStaticData,
            ChartAxesUtils,
            ChartFeatures,
            $scope,
            $element
        ) {
            const ctrl = this;
            let minInput, maxInput;
            const NG_INVALID = 'ng-invalid';

            ctrl.extentEditModes = ChartsStaticData.extentEditModes;
            ctrl.getZeroDisabledMessage = getZeroDisabledMessage;
            ctrl.isManualMode = isManualMode;
            ctrl.onEditModeChange = onEditModeChange;
            ctrl.onMaxChange = onMaxChange;
            ctrl.onMinChange = onMinChange;
            ctrl.setExtentMax = setExtentMax;
            ctrl.setExtentMin = setExtentMin;
            ctrl.canChartIncludeZero = canChartIncludeZero;

            //////////

            // On blur, fix invalid or format null input
            function setExtentMin() {
                if (ctrl.customExtent.manualExtent[0] !== null) {
                    const currentMax = ChartAxesUtils.getManualExtentMax(ctrl.customExtent);
                    ctrl.customExtent.manualExtent[0] = validateExtentMin(currentMax, ctrl.customExtent.manualExtent[0]);
                    validateMinInput();
                } else {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            function setExtentMax() {
                if (ctrl.customExtent.manualExtent[1] !== null) {
                    const currentMin = ChartAxesUtils.getManualExtentMin(ctrl.customExtent);
                    ctrl.customExtent.manualExtent[1] = validateExtentMax(currentMin, ctrl.customExtent.manualExtent[1]);
                    validateMaxInput();
                } else {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            // On extent input change, check for invalid values
            function onMinChange(newMin) {
                const currentMax = ChartAxesUtils.getManualExtentMax(ctrl.customExtent);

                if (newMin !== null && parseFloat(newMin) > parseFloat(currentMax)) {
                    invalidateMinInput();
                } else {
                    validateMinInput();
                }
            }

            function onMaxChange(newMax) {
                const currentMin = ChartAxesUtils.getManualExtentMin(ctrl.customExtent);

                if (newMax !== null && parseFloat(newMax) < parseFloat(currentMin)) {
                    invalidateMaxInput();
                } else {
                    validateMaxInput();
                }
            }

            function isManualMode() {
                return ChartAxesUtils.isManualMode(ctrl.customExtent);
            }

            function canChartIncludeZero() {
                return ChartFeatures.canIncludeZero(ctrl.chartType);
            }

            function getZeroDisabledMessage() {
                if (ctrl.isManualMode()) {
                    return 'This option is ignored in manual range edition.';
                }
            }

            function onEditModeChange() {
                if (ctrl.isManualMode()) {
                    $scope.$broadcast('formatManualExtent');
                }
            }

            function validateExtentMin(currentMax, newMin) {
                const min = parseFloat(newMin) > parseFloat(currentMax) ? currentMax : newMin;
                return parseFloat(min);
            }

            function validateExtentMax(currentMin, newMax) {
                const max = parseFloat(newMax) < parseFloat(currentMin) ? currentMin : newMax;
                return parseFloat(max);
            }

            const getMinInput = () => {
                if (!minInput) {
                    minInput = $element.find('input[name=\'min\']');
                }
                return minInput;
            };

            const getMaxInput = () => {
                if (!maxInput) {
                    maxInput = $element.find('input[name=\'max\']');
                }
                return maxInput;
            };

            const validateMinInput = () => getMinInput().removeClass(NG_INVALID);
            const validateMaxInput = () => getMaxInput().removeClass(NG_INVALID);
            const invalidateMinInput = () => getMinInput().addClass(NG_INVALID);
            const invalidateMaxInput = () => getMaxInput().addClass(NG_INVALID);
        }
    });
})();
