(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('LinesUtils', LinesUtils);

    const CLIP_PATH_ID = 'lines-chart-clip-path';


    // (!) This service previously was in static/dataiku/js/simple_report/curves/lines.js
    function LinesUtils(ChartDimension, Fn, ChartAxesUtils, CHART_TYPES, ColorFocusHandler, SVGUtils, ValuesInChartOverlappingStrategy, ValuesInChartPlacementMode) {


        const svc = {

            drawPaths: function(chartDef, chartBase, chartData, facetIndex, lineGs, xDimension, xLabels, xAxis, leftYAxis, rightYAxis, emptyBinsMode, redraw, transition, strokeWidth, lineDashGs) {

                const paths = lineGs.selectAll('path.visible').data(function(d) {
                    return [d];
                });

                paths.enter()
                    .insert('path')
                    .attr('class', 'line visible')
                    .attr('fill', 'none')
                    .attr('stroke-width', strokeWidth)
                    .attr('opacity', chartDef.colorOptions.transparency);

                paths.exit().remove();

                const dashPaths = lineDashGs.selectAll('path.visible').data(function(d) {
                    return [d];
                });

                dashPaths.enter()
                    .insert('path')
                    .attr('class', 'line visible')
                    .attr('fill', 'none')
                    .attr('stroke-dasharray', 12)
                    .attr('stroke-width', strokeWidth);
                dashPaths.exit().remove();
                dashPaths.attr('d', Fn.SELF);

                if (!transition) {
                    paths.attr('d', Fn.SELF)
                        .each(function() {
                            const path = d3.select(this);
                            const wrapper = d3.select(this.parentNode.parentNode);
                            svc.drawPath(path, wrapper, emptyBinsMode, redraw, chartBase, svc.xCoord, svc.yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis);
                        });
                } else {
                    paths.transition().attr('d', Fn.SELF)
                        .each('end', function() {
                            const path = d3.select(this);
                            const wrapper = d3.select(this.parentNode.parentNode);
                            svc.drawPath(path, wrapper, emptyBinsMode, redraw, chartBase, svc.xCoord, svc.yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis);
                        });
                }
            },

            /**
             * - Build a d3 line generator if none provided.
             * - In the wrappers, creates <g>s with class "line".
             * - Bind to them the data computed by the line generator for the given points data.
             */
            configureLines: function(chartDef, chartData, facetIndex, wrappers, lineGenerator, xAxis, leftYAxis, rightYAxis, xDimension, xLabels, emptyBinsMode) {

                if (!lineGenerator) {
                    lineGenerator = d3.svg.line()
                        .x(d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                        .y(d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis))
                        // in DASHED mode, the dashed lines are drawn separately => we must remove missing values from the main line
                        .defined(x => emptyBinsMode === 'ZEROS' || svc.nonZeroCountFilter(x, facetIndex, chartData));
                    // If smoothing, change the interpolation mode (the process of adding new points between existing ones) to cubic interpolation that preserves monotonicity in y.
                    if (chartDef.smoothing) {
                        lineGenerator.interpolate('monotone');
                    }
                }

                const lineGs = wrappers.selectAll('g.line').data(function(d) {
                    d.filteredPointsData = d.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData));
                    const data = (emptyBinsMode === 'ZEROS' || emptyBinsMode == 'DASHED') ? d.pointsData : d.filteredPointsData;
                    return [lineGenerator(data)];
                });

                const lineDashGs = wrappers.selectAll('g.dashedline').data(function(d) {
                    if (emptyBinsMode === 'DASHED') {
                        // null is added after every segment in order to make them disconnected (using defined() below)
                        const data = svc.getEmptySegments(d.pointsData).flatMap(s => [s[0], s[1], null]);
                        const segmentGenerator = d3.svg.line()
                            .x(d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                            .y(d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis))
                            .defined(d => d != null);
                        return [segmentGenerator(data)];
                    }
                    return [];
                });

                lineGs.enter().insert('g', ':first-child').attr('class', 'line');
                lineGs.exit().remove();

                lineDashGs.enter().insert('g', ':first-child').attr('class', 'dashedline');
                lineDashGs.exit().remove();

                return [lineGenerator, lineGs, lineDashGs];
            },

            /**
             * - In the given line wrappers, create <circle> with class "point", a given radius for each points of the lines.
             * - These points will have a color defined by the color scale and an attached tooltip if requested.
             */
            drawPoints: function(chartDef, chartBase, chartData, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, pointsRadius) {

                let points = wrappers.selectAll('circle.point');

                points = points.data(function(d) {
                    return (emptyBinsMode === 'ZEROS') ? d.pointsData : (d.filteredPointsData = d.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData)));
                }, Fn.prop('x'));

                points.enter().append('circle')
                    .attr('class', 'point point--masked')
                    .attr('r', function(d) {
                        return pointsRadius ||
                        (chartDef.valuesInChartDisplayOptions
                            && chartDef.valuesInChartDisplayOptions.displayValues
                            && d.valuesInChartDisplayOptions.displayValues
                            ? Math.round(chartDef.strokeWidth / 2) + 1 : 5);
                    })
                    .attr('fill', function(d) {
                        return chartBase.colorScale(d.colorScaleIndex);
                    })
                    .attr('opacity', function(d) {
                        return chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues && d.valuesInChartDisplayOptions.displayValues ? chartDef.colorOptions.transparency : 0;
                    });

                // Remove potential duplicates
                points.exit().remove();

                points
                    .transition()
                    .duration(200)
                    .ease('easeOutQuad')
                    .attr('cx', d => svc.xCoord(xDimension, xLabels, xAxis)(d))
                    .attr('cy', d => svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis));

                // Remove points that are not linked to others through lines.
                wrappers.selectAll('circle.lonely').remove();

                return points;
            },

            /**
             * - Draw the labels for all displayed lines
             */
            drawLabels: function(chartDef, chartBase, chartData, theme, facetIndex, wrappers, xAxis, xLabels, leftYAxis, rightYAxis, xDimension, emptyBinsMode, labelCollisionDetectionHandler) {
                // to store all nodes across the different lines
                const selectionNodes = [];

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                    const backgroundXMargin = 3;

                    // init the collision detection
                    const mainLabelCollisionDetectionHandler = labelCollisionDetectionHandler || SVGUtils.initLabelCollisionDetection(chartBase);
                    const labelCollisionDetectionHandlersByColorScaleIndex = {};
                    const getCurrentColorLabelCollisionDetectionHandler = (d) => {
                        const colorScaleIndex = d.colorScaleIndex;
                        if (!labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex]) {
                            // we need a separate label collision detection handler for each color scale
                            labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex] = SVGUtils.initLabelCollisionDetection(chartBase);
                        }
                        return labelCollisionDetectionHandlersByColorScaleIndex[colorScaleIndex];
                    };



                    const getPointPosition = d => ({
                        x: d ? svc.xCoord(xDimension, xLabels, xAxis)(d) : null,
                        y: d ? svc.yCoord(d, chartDef, chartData, leftYAxis, rightYAxis) : null
                    });

                    const isEqualPosition = (point1, point2) => {
                        return point1.x === point2.x && point1.y === point2.y;
                    };

                    const checkBoundaries = (position, textElement) => {
                        const textElementWidth = textElement.width;
                        const rectangleHeight = textElement.height;
                        // make sure we are within the chart boundaries
                        const newPosition = {
                            x: position.x,
                            y: position.y
                        };
                        const minVizX = 0;
                        const maxVizX = chartBase.vizWidth;
                        const minVizY = -chartBase.margins.top;
                        const maxVizY = chartBase.vizHeight;
                        let isOutside = false;

                        if (position.x - textElementWidth / 2 < minVizX) {
                            newPosition.x += (minVizX - (position.x - textElementWidth / 2));
                            isOutside = true;
                        } else if (position.x + textElementWidth / 2 > maxVizX) {
                            newPosition.x -= (position.x + textElementWidth / 2 - maxVizX);
                            isOutside = true;
                        }

                        // keep in mind that the position coordinates correspond to the top center of the text element
                        if (position.y < minVizY) {
                            newPosition.y += (minVizY - position.y);
                            isOutside = true;
                        } else if (position.y + rectangleHeight > maxVizY) {
                            newPosition.y -= (position.y + rectangleHeight - maxVizY);
                            isOutside = true;
                        }

                        return {
                            newPosition,
                            isOutside
                        };
                    };


                    const findFinalPosition = (possiblePositions, textElement, onHoverMode, adjustForBoundaries) => {

                        let finalPosition;
                        let finalRectangles;
                        let isOverlap = false;
                        let isOverlapAlt = false;
                        const fittedPositions = [];
                        const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler(textElement) : mainLabelCollisionDetectionHandler;
                        for (let idx = 0; idx < possiblePositions.length; idx += 1) {
                            const position = possiblePositions[idx];
                            if (adjustForBoundaries) {
                                const boundariesCheck = checkBoundaries(position, textElement);
                                if (boundariesCheck.isOutside) {
                                    fittedPositions.push(boundariesCheck.newPosition);
                                    continue;
                                }
                            }

                            const rectangles = SVGUtils.getRectanglesFromPosition(position, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                            if (!labelDetectionHandler.checkOverlaps(rectangles)) {
                                finalPosition = position;
                                finalRectangles = rectangles;
                                break;
                            }
                        }

                        if (!finalPosition) {
                            // no valid position found, maybe some of them crossed the chart boundaries, in that case we check the fitted positions
                            for (let idx = 0; idx < fittedPositions.length; idx += 1) {
                                const position = fittedPositions[idx];
                                const rectangles = SVGUtils.getRectanglesFromPosition(position, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                                if (!labelDetectionHandler.checkOverlaps(rectangles)) {
                                    finalPosition = position;
                                    finalRectangles = rectangles;
                                    break;
                                }
                            }

                            if (!finalPosition) {
                                // still no valid position, just take the first one
                                finalPosition = fittedPositions[0] || possiblePositions[0];
                                finalRectangles = SVGUtils.getRectanglesFromPosition(finalPosition, textElement, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize);
                                if (onHoverMode) {
                                    isOverlapAlt = true;
                                } else {
                                    isOverlap = true;
                                }
                            }
                        }

                        return {
                            labelPosition: finalPosition,
                            labelRectangles: finalRectangles,
                            isOverlap,
                            isOverlapAlt
                        };
                    };

                    const getLabelCoordinates = (previousPoint, currentPoint, nextPoint, currentElement, onHoverMode) => {
                        if (!currentPoint) {
                            return null;
                        }
                        // one thing to keep in mind, the position of the text elem will be at the top center of the text element

                        const defaultSpacing = chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES ? 5 : 12;
                        const distanceFromDataPoint = currentElement.valuesInChartDisplayOptions.spacing ?? defaultSpacing;
                        const placementMode = currentElement.valuesInChartDisplayOptions.placementMode;
                        const adjustBoundaries = !placementMode || placementMode === ValuesInChartPlacementMode.AUTO;
                        const overlappingKey = onHoverMode ? 'isOverlapAlt' : 'isOverlap';
                        const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler(currentElement) : mainLabelCollisionDetectionHandler;

                        const computeCoordinatesFromAngle = (angle, unitVector) => {
                            const tmpRotatedVector = {
                                x: unitVector.x * Math.cos(angle) + unitVector.y * Math.sin(angle),
                                y: -unitVector.x * Math.sin(angle) + unitVector.y * Math.cos(angle)
                            };

                            // compute distance between rectangle center and rectangle boundaries following a particular angle
                            const angleFromXAxis = Math.atan2(tmpRotatedVector.y, tmpRotatedVector.x);
                            const counterclockwiseAngle = Math.PI - angleFromXAxis;
                            const boundaryDistance = Math.min(
                                ((currentElement.width / 2) / Math.abs(Math.cos(counterclockwiseAngle))),
                                ((currentElement.height / 2) / Math.abs(Math.sin(counterclockwiseAngle)))
                            );
                            const distanceToBboxCenter = distanceFromDataPoint + boundaryDistance;

                            const labelCoordinates = {
                                x: currentPoint.x + distanceToBboxCenter * tmpRotatedVector.x,
                                // the y position of the label is at the top center of the text element
                                y: currentPoint.y + distanceToBboxCenter * tmpRotatedVector.y - currentElement.height / 2
                            };

                            return labelCoordinates;
                        };

                        let belowLinePosition = {
                            x: currentPoint.x,
                            y: currentPoint.y + distanceFromDataPoint
                        };
                        let aboveLinePosition = {
                            x: currentPoint.x,
                            y: currentPoint.y - (distanceFromDataPoint + currentElement.height)
                        };

                        let possiblePositions = [belowLinePosition, aboveLinePosition];

                        if (previousPoint && nextPoint) {
                            // standard case, with one data point before and one after

                            // we compute the angle made by the 3 consecutive points
                            const dotProduct = (previousPoint.x - currentPoint.x) * (nextPoint.x - currentPoint.x) + (previousPoint.y - currentPoint.y) * (nextPoint.y - currentPoint.y);
                            const magnitude1 = Math.sqrt(Math.pow(previousPoint.x - currentPoint.x, 2) + Math.pow(previousPoint.y - currentPoint.y, 2));
                            const magnitude2 = Math.sqrt(Math.pow(nextPoint.x - currentPoint.x, 2) + Math.pow(nextPoint.y - currentPoint.y, 2));
                            const angleCosinus = dotProduct / (magnitude1 * magnitude2);

                            // angle is always between 0 and 180 degrees
                            const curveAngle = Math.acos(Math.max(-1, Math.min(1, angleCosinus)));

                            // we have to compute the cross product to get the orientation (clockwise or not), crossProduct < 0 => clockwise orientation
                            const crossProduct = (previousPoint.x - currentPoint.x) * (nextPoint.y - currentPoint.y) - (previousPoint.y - currentPoint.y) * (nextPoint.x - currentPoint.x);

                            // compute the start unit vector
                            const unitVector = {
                                x: (previousPoint.x - currentPoint.x) / magnitude1,
                                y: (previousPoint.y - currentPoint.y) / magnitude1
                            };

                            // preferred position, at the middle of the widest angle formed by the 3 points
                            const rotationAngle = ((2 * Math.PI - curveAngle) / 2) * (crossProduct > 0 ? 1 : -1);
                            // first alternative, at the middle of the smallest angle formed by the 3 points
                            const alternativeAngle1 = (curveAngle / 2) * (crossProduct > 0 ? -1 : 1);
                            // second alternative, at the middle of the 2 preferred positions
                            const alternativeAngle2 = (rotationAngle + alternativeAngle1) / 2;
                            // third alternative, at the opposite of the second alternative
                            const alternativeAngle3 = alternativeAngle2 + Math.PI;

                            const preferredPosition = computeCoordinatesFromAngle(rotationAngle, unitVector);
                            const alternativePosition1 = computeCoordinatesFromAngle(alternativeAngle1, unitVector);
                            const alternativePosition2 = computeCoordinatesFromAngle(alternativeAngle2, unitVector);
                            const alternativePosition3 = computeCoordinatesFromAngle(alternativeAngle3, unitVector);

                            possiblePositions = [preferredPosition, alternativePosition1, alternativePosition2, alternativePosition3];

                            aboveLinePosition = crossProduct <= 0 ? preferredPosition : alternativePosition1;
                            belowLinePosition = crossProduct <= 0 ? alternativePosition1 : preferredPosition;
                        } else if (nextPoint || previousPoint) {
                            let unitVector;
                            if (nextPoint) {
                                // first data point
                                const magnitude = Math.sqrt(Math.pow(nextPoint.x - currentPoint.x, 2) + Math.pow(nextPoint.y - currentPoint.y, 2));

                                // compute the start unit vector
                                unitVector = {
                                    x: (currentPoint.x - nextPoint.x) / magnitude,
                                    y: (currentPoint.y - nextPoint.y) / magnitude
                                };
                            } else {
                                // last data point
                                const magnitude = Math.sqrt(Math.pow(currentPoint.x - previousPoint.x, 2) + Math.pow(currentPoint.y - previousPoint.y, 2));

                                // compute the start unit vector
                                unitVector = {
                                    x: (previousPoint.x - currentPoint.x) / magnitude,
                                    y: (previousPoint.y - currentPoint.y) / magnitude
                                };
                            }

                            const rotationAngle = Math.PI / 2;
                            const alternativeAngle1 = -(Math.PI / 2);
                            // second alternative, at the middle of the 2 preferred positions
                            const alternativeAngle2 = (rotationAngle + alternativeAngle1) / 2;
                            // third alternative, at the opposite of the second alternative
                            const alternativeAngle3 = alternativeAngle2 + Math.PI;

                            belowLinePosition = computeCoordinatesFromAngle(rotationAngle, unitVector);
                            aboveLinePosition = computeCoordinatesFromAngle(alternativeAngle1, unitVector);
                            const alternativePosition2 = computeCoordinatesFromAngle(alternativeAngle2, unitVector);
                            const alternativePosition3 = computeCoordinatesFromAngle(alternativeAngle3, unitVector);

                            possiblePositions = [belowLinePosition, aboveLinePosition, alternativePosition2, alternativePosition3];
                        }

                        if (aboveLinePosition && (placementMode === ValuesInChartPlacementMode.ABOVE)) {
                            possiblePositions = [aboveLinePosition];
                        } else if (belowLinePosition && (placementMode === ValuesInChartPlacementMode.BELOW)) {
                            possiblePositions = [belowLinePosition];
                        }
                        const result = findFinalPosition(possiblePositions, currentElement, onHoverMode, adjustBoundaries);

                        // add the selected position to the for next detections
                        const hideOverlaps = chartDef.valuesInChartDisplayOptions.overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO;
                        const isDisplayed = !!currentElement.valuesInChartDisplayOptions?.displayValues;
                        if ((!hideOverlaps || !result[overlappingKey]) && isDisplayed && result.labelRectangles) {
                            labelDetectionHandler.addBoundingBoxesToQuadTree(result.labelRectangles);
                        }

                        return result;
                    };

                    wrappers.each(function() {
                        const wrapper = d3.select(this);
                        const datum = wrapper.datum();
                        const data = (emptyBinsMode === 'ZEROS') ? datum.pointsData : (datum.filteredPointsData = datum.pointsData.filter(d => svc.nonZeroCountFilter(d, facetIndex, chartData)));
                        const finalData = data.filter(d => d.valuesInChartDisplayOptions && d.valuesInChartDisplayOptions.displayValues && d.valuesInChartDisplayOptions.textFormatting);

                        const getExtraData = (d, idx, onHoverMode) => {
                            const currentPosition = getPointPosition(d);

                            // check if the point is inside charts limits
                            const minVizX = 0;
                            const maxVizX = chartBase.vizWidth;
                            const minVizY = -chartBase.margins.top;
                            const maxVizY = chartBase.vizHeight;

                            if (currentPosition.x < minVizX || currentPosition.x > maxVizX || currentPosition.y < minVizY || currentPosition.y > maxVizY) {
                                return { isInvalid: true };
                            }

                            let previousPosition = null;
                            let nextPosition = null;
                            // find the first predecessor with a different position
                            let prevIndex = idx - 1;
                            while (prevIndex >= 0 && !previousPosition) {
                                const position = getPointPosition(finalData[prevIndex]);
                                if (!isEqualPosition(position, currentPosition)) {
                                    previousPosition = position;
                                }
                                prevIndex -= 1;
                            }
                            // find the first successor with a different position
                            let nextIndex = idx + 1;
                            while (nextIndex < finalData.length && !nextPosition) {
                                const position = getPointPosition(finalData[nextIndex]);
                                if (!isEqualPosition(position, currentPosition)) {
                                    nextPosition = position;
                                }
                                nextIndex += 1;
                            }

                            return getLabelCoordinates(previousPosition, currentPosition, nextPosition, d, onHoverMode);
                        };

                        const labelsDrawContext = {
                            node: wrapper,
                            data: finalData,
                            opacity: chartDef.colorOptions.transparency,
                            textFormatting: chartDef.valuesInChartDisplayOptions.textFormatting,
                            overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                            colorScale: chartBase.colorScale,
                            // to store computed data reused by other function (getLabelXPosition, getBackgroundXPosition...)
                            getExtraData: (d, _, idx, onHoverMode) => {
                                return getExtraData(d, idx, onHoverMode);
                            },
                            getLabelXPosition: (d) => {
                                return d.isInvalid ? 0 : d.labelPosition.x;
                            },
                            getLabelYPosition: (d) => {
                                return d.isInvalid ? 0 : d.labelPosition.y;
                            },
                            getLabelText: (d) => {
                                return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                            },
                            getBackgroundXPosition: (d) => {
                                if (d.isInvalid) {
                                    return 0;
                                }
                                const { x } = d.labelPosition;
                                return x - d.width / 2 - backgroundXMargin;
                            },
                            getBackgroundYPosition: (d) => {
                                if (d.isInvalid) {
                                    return 0;
                                }
                                const { y } = d.labelPosition;
                                return SVGUtils.getSubTextYPosition(d, y);
                            },
                            backgroundXMargin,
                            theme
                        };

                        const labelsSelection = SVGUtils.drawLabels(labelsDrawContext, 'lines');
                        selectionNodes.push(...(labelsSelection?.[0] ?? []));
                    });
                }

                // return the selection to be able to remove them during interaction
                return d3.selectAll(selectionNodes);
            },

            /**
             * - Creates a <g> (group) element with class "wrapper" for each line to be drawn.
             * - Joins the lines data with these wrappers.
             * - Strokes them according to the chart's color scale, set the opacity as per the options, and attach tooltips if requested.
             * - We need to add a key selector (id) to ensures consistent binding between lines data to lines DOM while zooming.
             */
            drawWrappers: function(chartDef, chartBase, linesData, g, isInteractive, redraw, className) {
                let wrappers = g.selectAll('g.' + className);

                if (!redraw) {
                    wrappers = wrappers.data(linesData, d => d.id);
                    wrappers.enter().append('g').attr('class', className)
                        .attr('stroke', function(d) {
                            return chartBase.colorScale(d.colorScaleIndex);
                        });

                    // Remove the exiting selection ie existing DOM elements for which no new data has been found to prevent duplicates.
                    wrappers.exit().remove();
                }

                return wrappers;
            },

            computeTooltipCoords: (datum, facetIndex) => ({ measure: datum.measure, x: datum.xCoord, color: datum.color, facet: facetIndex, colorScaleIndex: datum.colorScaleIndex }),

            getTooltipTargetElementProperties: (domElement, facetIndex) => {
                if (domElement.classList.contains('point')) {
                    const element = d3.select(domElement);
                    element.attr('tooltip-el', true);
                    const datum = d3.select(domElement).datum();
                    if (datum == null) {
                        return null;
                    }
                    return {
                        coords: svc.computeTooltipCoords(datum, facetIndex),
                        measure: datum.measure != null ? datum.measure : 0,
                        showTooltip: true
                    };
                } else if (domElement.classList.contains('line')) {
                    // When pointing a line, we need to access its grand-parent element, which is the line wrapper containing its coordinates.
                    const datum = d3.select(domElement.parentNode.parentNode).datum();
                    if (datum == null) {
                        return null;
                    }
                    return {
                        coords: svc.computeTooltipCoords(datum, facetIndex),
                        measure: datum.measure != null ? datum.measure : 0,
                        showTooltip: false
                    };
                }
                return null;
            },

            computeContextualMenuCoords: (datum, facetIndex) => ({ x: datum.xCoord, color: datum.color, facet: facetIndex }),

            getContextualMenuTargetElementProperties: (domElement) => {
                if (domElement.classList.contains('point')) {
                    const element = d3.select(domElement);
                    element.attr('tooltip-el', true);
                    const datum = d3.select(domElement).datum();
                    return { coords: svc.computeContextualMenuCoords(datum) };
                }
                return null;
            },

            addTooltipAndHighlightAndContextualMenuHandlers: function(chartDef, chartHandler, chartBase, facetIndex, g, wrappers) {
                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrappers, d3, g);

                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);

                const tooltipOptions = { displayColorTooltip: chartDef.type !== CHART_TYPES.MULTI_COLUMNS_LINES };
                chartBase.tooltips.addChartTooltipAndHighlightHandlers(g.node(), (domElement) => svc.getTooltipTargetElementProperties(domElement, facetIndex), colorFocusHandler, tooltipOptions);
                chartBase.contextualMenu.addChartContextualMenuHandler(g.node(), (domElement) => svc.getContextualMenuTargetElementProperties(domElement, facetIndex));
            },

            drawPath: function(path, wrapper, emptyBinsMode, redraw, chartBase, xCoord, yCoord, chartData, chartDef, xDimension, xLabels, xAxis, leftYAxis, rightYAxis) {

                const lineData = wrapper.data()[0];

                // Data points that are not part of a line segment and need to be drawn explicitly
                let lonelyPoints = [];
                if (lineData.filteredPointsData.length === 1) {
                    lonelyPoints = [lineData.filteredPointsData[0]];
                }

                if (emptyBinsMode === 'DASHED' && !redraw) {
                    const emptySegments = svc.getEmptySegments(lineData.pointsData);

                    if (lineData.filteredPointsData.length > 1) {
                        emptySegments.forEach(function(seg, i) {
                            if (i === 0) {
                                if (seg[0].$idx === 0) {
                                    lonelyPoints.push(seg[0]);
                                }
                            } else if (i === emptySegments.length - 1 && seg[1].$idx === lineData.filteredPointsData[lineData.filteredPointsData.length - 1].$idx) {
                                lonelyPoints.push(seg[1]);
                            }
                            if (emptySegments[i + 1] && emptySegments[i][1] === emptySegments[i + 1][0]) {
                                lonelyPoints.push(emptySegments[i][1]);
                            }
                        });
                    }
                }

                const lonelyCircles = wrapper.selectAll('circle.lonely')
                    .data(lonelyPoints, Fn.prop('x'));

                lonelyCircles.remove();

                lonelyCircles.enter().append('circle')
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .attr('class', 'lonely')
                    .attr('fill', function(d) {
                        return chartBase.colorScale(d.colorScaleIndex);
                    })
                    .style('pointer-events', 'none');

                if (emptyBinsMode === 'DASHED') {
                    lonelyCircles.attr('r', 2.5)
                        .attr('cy', chartBase.yAxes[0].scale()(0));
                } else {
                    // If not in dashed mode, lonely circles are lonely normal points
                    lonelyCircles
                        .attr('r', 4)
                        .attr('opacity', chartDef.colorOptions.transparency);
                }

                lonelyCircles.exit().remove();
                lonelyCircles
                    .attr('cx', d => xCoord(xDimension, xLabels, xAxis)(d))
                    .attr('cy', d => yCoord(d, chartDef, chartData, leftYAxis, rightYAxis));
            },

            // Prevent chart to overlap axes
            clipPaths: function(chartBase, chartDef, g, wrappers) {
                const displayedValuesFontSizes = chartDef.genericMeasures.filter(m => m.valuesInChartDisplayOptions && m.valuesInChartDisplayOptions.textFormatting).map(m => m.valuesInChartDisplayOptions.textFormatting.fontSize);
                const extraPadding = displayedValuesFontSizes.length ? Math.max(...displayedValuesFontSizes) + 20 : 10;
                const defs = g.append('defs');
                const clipPathUniqueId = CLIP_PATH_ID + generateUniqueId();

                // Add a bit of margin to handle smoothing mode.
                defs.append('clipPath')
                    .attr('id', clipPathUniqueId)
                    .append('rect')
                    .attr('width', chartBase.vizWidth)
                    .attr('y', -extraPadding)
                    .attr('height', chartBase.vizHeight + extraPadding);

                wrappers.attr('clip-path', 'url(#' + clipPathUniqueId + ')');
                wrappers.style('-webkit-clip-path', 'url(#' + clipPathUniqueId + ')');
            },

            nonZeroCountFilter: function(d, facetIndex, chartData) {
                d.$filtered = chartData.getNonNullCount({ x: d.xCoord, color: d.color, facet: facetIndex }, d.measure) === 0;
                return !d.$filtered;
            },

            xCoord: function(xDimension, xLabels, xAxis) {
                return svc.getXCoord(xDimension, xAxis, xAxis.ordinalScale, xLabels);
            },

            yCoord: function(d, chartDef, chartData, leftYAxis, rightYAxis) {
                let val = chartData.aggr(d.measure).get({ x: d.xCoord, color: d.color });
                const displayAxis = chartDef.genericMeasures[d.measure].displayAxis === 'axis1' ? leftYAxis : rightYAxis;

                if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, displayAxis.id) && val == 0) {
                    val = 1;
                }
                return displayAxis.scale()(val);
            },

            cleanChart: function(g, chartBase) {
                const wrappers = $('g.wrapper');
                chartBase.tooltips.removeChartTooltipHandlers(g.node());
                chartBase.contextualMenu.removeChartContextualMenuHandler(g.node());
                wrappers.off();
            },

            prepareData: function(chartDef, chartData, measureFilter, ignoreLabels = new Set(), useColorDimension = true) {
                const xLabels = chartData.getAxisLabels('x');
                const axisColorLabels = chartData.getAxisLabels('color');
                const colorLabelsLength = (axisColorLabels || [null]).length;
                const colorLabels = useColorDimension && axisColorLabels || [axisColorLabels && axisColorLabels.length ? chartData.getSubtotalLabelIndex('color') : 0];
                const linesData = [];

                colorLabels.forEach(function(colorLabel, colorIndex) {
                    let mIndex = 0;
                    chartDef.genericMeasures.forEach(function(measure, measureIndex) {
                        if (measureFilter && !measureFilter(measure)) {
                            return;
                        }
                        const newColorIndex = useColorDimension ? colorIndex : colorLabel;
                        const colorScaleIndex = (useColorDimension ? newColorIndex : (colorLabelsLength - 1)) + (axisColorLabels ? mIndex : measureIndex);
                        linesData.push({
                            id: _.uniqueId('line_'),
                            color: newColorIndex,
                            measure: measureIndex,
                            colorScaleIndex,
                            valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions,
                            pointsData: xLabels.reduce(function(result, xLabel, xIndex) {
                                if (ignoreLabels.has(xLabel.label)) {
                                    return result;
                                }

                                const measureData = {
                                    x: result.length,
                                    color: newColorIndex,
                                    colorScaleIndex,
                                    measure: measureIndex,
                                    filtered: true,
                                    xCoord: xIndex
                                };

                                const subTextElementsData = SVGUtils.getLabelSubTexts(chartDef, measureIndex, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                    return {
                                        ...subTextElementData,
                                        // add measure data at sub text level also, to retrieve it easily
                                        ...measureData
                                    };
                                });

                                // xCoord is used to get the data in the tensor and x is used for colors
                                result.push({
                                    ...measureData,
                                    textsElements: subTextElementsData,
                                    // add the labels display options to the data point to retrieve it easily
                                    valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                });
                                return result;
                            }, [])
                        });
                        mIndex++;
                    });
                });

                return linesData;
            },

            // Returns the right accessor for the x-coordinate of a label
            getXCoord: function(dimension, xAxis, ordinalXScale, labels) {
                if (ChartDimension.isTimeline(dimension)) {
                    return function(d) {
                        return xAxis.scale()(labels[d.x].tsValue);
                    };
                } else if ((ChartDimension.isGroupedNumerical(dimension) && !ChartDimension.hasOneTickPerBin(dimension)) || ChartDimension.isUngroupedNumerical(dimension)) {
                    return function(d) {
                        return xAxis.scale()(labels[d.x].sortValue);
                    };
                } else {
                    return function(d) {
                        return ordinalXScale(d.x) + (ordinalXScale.rangeBand() / 2);
                    };
                }
            },

            getEmptySegments: function(labels) {
                const emptySegments = [];
                let segment = [];
                let inSegment = false;
                let inLine = false;
                labels.forEach(function(label, i) {
                    label.$idx = i;
                    if (inLine && label.$filtered) {
                        inSegment = true;
                    } else {
                        inLine = true;
                        if (inSegment) {
                            segment[1] = label;
                            emptySegments.push(segment);
                            segment = [label];
                        } else {
                            segment = [label];
                        }
                        inSegment = false;
                    }
                });
                return emptySegments;
            },

            /**
             * Returns true when we must include empty bins to compute measure extent
             * @param {ChartDef} chartDef
             * @return {boolean} true when we must include empty bins to compute measure extent
             */
            hasEmptyBinsToIncludeAsZero: function(chartDef, chartData) {
                // In case of a line/mix chart we must look at the emptyBinsMode option
                return chartDef.genericDimension0.length
                && chartDef.genericDimension0[0].numParams
                && chartDef.genericDimension0[0].numParams.emptyBinsMode === 'ZEROS'
                && chartData.hasEmptyBins;
            }
        };
        return svc;
    }
})();
