#!/bin/bash -e

# Installs or checks the prerequisite packages for Dataiku Data Science Studio
# Supports the following Linux distributions:
# - Debian 11, 12
# - Ubuntu 20.04, 22.04
# - RedHat/CentOS/Oracle Linux/AlmaLinux/Rocky Linux 8.x, 9.x
# - Amazon Linux 2, 2023
# - SLES 15.4 to 15.6
# Supports the following non-Linux distributions:
# - MacOS 12+ (Monterey)
# To be run as root for install, as any user for check

#
# Parse arguments
#
Usage() {
	echo "
Usage: $0 [OPTIONS]
Available options:
	-check : check dependencies only, do not install
	-yes : do not prompt before package installation, assume yes

	-without-java : do not check nor install for Java environment

	-without-python : do not check nor install for Python and Python packages dependencies

	-with-r : include dependencies for R integration

	-with-chrome : include dependencies for dashboard export

	-with-libreoffice : include dependencies for document extraction

	-with-azure : installs the Azure CLI

	-with-docker : installs Docker

	-os DISTRIB VERSION : force OS detection result
" >&2
	exit 1
}

check_only=
yes=
without_java=
without_python=
with_r=
with_chrome=
with_libreoffice=
with_azure=
with_docker=
distrib=
distribVersion=
while [ $# -gt 0 ]; do
	case "$1" in
		"-check" )
			check_only=1
			shift
			;;
		"-yes" )
			yes="-y"
			shift
			;;
		"-without-java" | "-skip_java" )
			without_java=1
			shift
			;;
		"-without-python" | "-skip_python" )
			without_python=1
			shift
			;;
		"-with-r" )
			with_r=1
			shift
			;;
		"-with-chrome" )
			with_chrome=1
			shift
			;;
		"-with-libreoffice" )
			with_libreoffice=1
			shift
			;;
		"-with-azure" )
			with_azure=1
			shift
			;;
		"-with-docker" )
			with_docker=1
			shift
			;;
		"-os" )
			[ $# -ge 3 ] || Usage
			distrib="$2"
			distribVersion="$3"
			shift 3
			;;
		*)
			Usage
			;;
	esac
done

#
# Identify OS distribution and check version
#
notSupported() {
	echo "*** OS distribution not supported *** $distrib $distribVersion" >&2
	exit 1
}

if [ -n "$distrib" ]; then
	echo "* Forcing OS distribution : $distrib $distribVersion"
else
	detectedDistrib=$("$(dirname "$0")"/../_find-distrib.sh)
	read distrib distribVersion <<< "$detectedDistrib"
	echo "+ Detected OS distribution : $distrib $distribVersion"
fi

case "$distrib" in
	debian)
		case "$distribVersion" in
			1[12]*) ;;
			*) notSupported;;
		esac
		arch=$(dpkg --print-architecture)
		if [ "$arch" != "amd64" ]; then
			echo "*** Architecture not supported: '$arch' should be amd64" >&2
			exit 1
		fi
		;;
	ubuntu)
		case "$distribVersion" in
			20.04 | 22.04) ;;
			*) notSupported;;
		esac
		arch=$(dpkg --print-architecture)
		if [ "$arch" != "amd64" ]; then
			echo "*** Architecture not supported: '$arch' should be amd64" >&2
			exit 1
		fi
		;;
	centos | redhat | oraclelinux | almalinux | rocky)
		case "$distribVersion" in
			8* | 9*) ;;
			*) notSupported;;
		esac
		arch=$(rpm -E '%{_arch}')
		if [ "$arch" != "x86_64" ]; then
			echo "*** Architecture not supported: '$arch' should be x86_64" >&2
			exit 1
		fi
		;;
	amazonlinux)
		case "$distribVersion" in
			2) ;;    # Amazon Linux 2
			2023) ;; # Amazon Linux 2023
			*) notSupported;;
		esac
		arch=$(rpm -E '%{_arch}')
		if [ "$arch" != "x86_64" ]; then
			echo "*** Architecture not supported: '$arch' should be x86_64" >&2
			exit 1
		fi
		;;
	suse)
		case "$distribVersion" in
			15.[4-6]) ;;
			*) notSupported;;
		esac
		arch=$(rpm -E '%{_arch}')
		if [ "$arch" != "x86_64" ]; then
			echo "*** Architecture not supported: '$arch' should be x86_64" >&2
			exit 1
		fi
		if [ -n "$yes" ]; then
			yes="-n"
		fi
		;;
	osx)
		major="$(echo "$distribVersion" | cut -d . -f 1)"
		# minor="$(echo "$distribVersion" | cut -d . -f 2)"
		if [ "$major" -ge 12 ]; then
			:
		else
			notSupported
		fi
		;;
	*)
		notSupported
		;;
esac

# Check whether SELinux is installed and not disabled
with_selinux() {
	if selinux=$(/usr/sbin/getenforce 2>/dev/null) && [ "$selinux" != "Disabled" ]; then
		return 0
	else
		return 1
	fi
}

#
# Required packages
# Build list of required packages in PKG:
# - PKG_NAME
# - PKG_NAME>=MIN_VERSION     # Not implemented on Suse
#
case "$distrib" in
	debian | ubuntu )
		PKG="
			acl
			curl
			fontconfig
			fonts-dejavu
			git
			libexpat1
			libfreetype6
			locales
			nginx
			unzip
			zip
			procps
		"
		if [ -z "$without_java" ]; then
			PKG+=" openjdk-17-jre-headless"
		fi
		if [ -z "$without_python" ]; then
			PKG+=" libgomp1"
			if [ "$distrib" = "ubuntu" ]; then
				case "$distribVersion" in
					20.04) PKG+=" python3.9 python3-distutils";;
					22.04) PKG+=" python3.10 python3-distutils";;
				esac
			else    # Debian
				case "$distribVersion" in
					1[12]*) PKG+=" python3 python3-distutils";;
				esac
			fi
		fi
		if with_selinux; then
			PKG+=" policycoreutils policycoreutils-python-utils"
		fi
		if [ -n "$with_r" ]; then
			PKG+=" r-base-dev>=4.1 libicu-dev libcurl4-openssl-dev libssl-dev libxml2-dev"
		fi
		if [ -n "$with_chrome" ]; then
			PKG+=" npm libgtk-3-0 libnss3 libxss1 libasound2 libxtst6 libx11-xcb1 libgbm1"
		fi
		if [ -n "$with_libreoffice" ]; then
			# nogui cannot convert pptx to pdf on ubuntu 22.04, see sc-233028
			if [[ "$distrib" == "ubuntu" && "$distribVersion" == "22.04" ]]; then
				PKG+=" libreoffice-writer libreoffice-impress libreoffice-calc"
			else
				PKG+=" libreoffice-writer-nogui libreoffice-impress-nogui libreoffice-calc-nogui"
			fi
		fi
		if [ -n "$with_azure" ]; then
			PKG+=" azure-cli"
		fi
		if [ -n "$with_docker" ]; then
			PKG+=" jq docker-ce docker-ce-cli containerd.io docker-compose-plugin"
		fi
		;;
	centos | redhat  | oraclelinux | almalinux | rocky)
		PKG="
			acl
			dejavu-sans-fonts
			expat
			freetype
			fontconfig
			git
			glibc-langpack-en
			nginx
			unzip
			zip
			procps-ng
		"
		if [ -z "$without_java" ]; then
			PKG+=" java-17-openjdk-headless"
		fi
		if [ -z "$without_python" ]; then
			case "$distribVersion" in
				8*) PKG+=" python39 libgfortran libgomp";;
				9*) PKG+=" python3 libgfortran libgomp";;
			esac
		fi
		if with_selinux; then
			PKG+=" policycoreutils policycoreutils-python-utils"
		fi
		if [ -n "$with_r" ]; then
			PKG+=" R-core-devel libicu-devel libcurl-devel openssl-devel libxml2-devel"
		fi
		if [ -n "$with_chrome" ]; then
			PKG+=" npm gtk3 libXScrnSaver alsa-lib nss mesa-libgbm libX11-xcb"
		fi
		if [ -n "$with_libreoffice" ]; then
			PKG+=" libreoffice-writer libreoffice-impress libreoffice-calc"
		fi
		if [ -n "$with_azure" ]; then
			PKG+=" azure-cli"
		fi
		if [ -n "$with_docker" ]; then
			PKG+=" jq docker-ce docker-ce-cli containerd.io docker-compose-plugin"
		fi
		;;
	amazonlinux )
		PKG="
			acl
			dejavu-sans-fonts
			expat
			fontconfig
			freetype
			git
			nginx
			unzip
			zip
			procps-ng
		"
		if [ "$distribVersion" = "2023" ]; then
			PKG+=" glibc-langpack-en"
		fi
		if [ -z "$without_java" ]; then
			PKG+=" java-17-amazon-corretto-headless"
		fi
		if [ -z "$without_python" ]; then
			if [ "$distribVersion" = 2 ]; then
				# No available OS package for any supported Python version
				# Requires installation from source
				PKG+=" compat-gcc-48-libgfortran libgomp"
			else
				PKG+=" python3 libgomp"
			fi
		fi
		if with_selinux; then
			if [ "$distribVersion" = 2 ]; then
				PKG+=" policycoreutils policycoreutils-python"
			else
				PKG+=" policycoreutils policycoreutils-python-utils"
			fi
		fi
		if [ -n "$with_r" ]; then
			case "$distribVersion" in
				2)
					# No available OS package for any supported R version
					# Requires installation from source
					PKG+=" libicu-devel libcurl-devel openssl11-devel libxml2-devel"
					;;
				2023)
					PKG+=" R-core-devel>=4 libicu-devel libcurl-devel openssl-devel libxml2-devel"
					;;
			esac
		fi
		if [ -n "$with_chrome" ]; then
			case "$distribVersion" in
				2)
					PKG+=" npm gtk3 libXScrnSaver alsa-lib nss mesa-libgbm libxcb"
					;;
				2023)
					PKG+=" nodejs-npm gtk3 libXScrnSaver alsa-lib nss mesa-libgbm libxcb"
					;;
			esac
		fi
		if [ -n "$with_libreoffice" ]; then
			echo >&2 "*** -with-libreoffice not supported on this distribution: $distrib $distribVersion"
			echo >&2 "*** you need to install LibreOffice manually"
			exit 1
		fi
		if [ -n "$with_azure" ]; then
			echo >&2 "*** -with-azure not supported on this distribution: $distrib $distribVersion"
			exit 1
		fi
		if [ -n "$with_docker" ]; then
			# docker-compose plugin is installed by ad hoc code as it's not available as a package
			PKG+=" jq docker"
		fi
		;;
	suse )
		PKG="
			acl
			dejavu-fonts
			fontconfig
			git-core
			libexpat1
			libfreetype6
			nginx
			unzip
			zip
			gawk
			procps
		"
		if [ -z "$without_java" ]; then
			PKG+=" java-17-openjdk-headless"
		fi
		if [ -z "$without_python" ]; then
			PKG+=" python311 libgomp1"
		fi
		if with_selinux; then
			PKG+=" policycoreutils policycoreutils-python-utils"
		fi
		if [ -n "$with_r" ]; then
			PKG+=" patterns-devel-base-devel_basis gcc-fortran R-core-devel libicu-devel libcurl-devel libopenssl-devel libxml2-devel"
		fi
		if [ -n "$with_chrome" ]; then
			PKG+=" npm-default libgtk-3-0 mozilla-nss libXss1 libasound2 libX11-xcb1 libxkbcommon0 libgbm1"
		fi
		if [ -n "$with_libreoffice" ]; then
			echo >&2 "*** -with-libreoffice not supported on this distribution: $distrib $distribVersion"
			echo >&2 "*** you need to install LibreOffice manually"
			exit 1
		fi
		if [ -n "$with_azure" ]; then
			PKG+=" azure-cli"
		fi
		if [ -n "$with_docker" ]; then
			# docker-compose plugin is installed by ad hoc code as it's not available as a package
			PKG+=" jq docker"
		fi
		;;
esac

#
# Check whether a service is configured to start at boot
#
is_service_enabled() {
	if command -v systemctl >/dev/null; then
		if systemctl list-unit-files "$1.service" | grep -q "^$1\\.service" &&
		   systemctl is-enabled --quiet "$1" 2>/dev/null; then
			echo "enabled"
		fi
	elif command -v chkconfig >/dev/null; then
		if chkconfig "$1"; then
			echo "enabled"
		fi
	else
		# Legacy mode for stripped-down Debian-based systems
		if ls /etc/rc?.d/S*"$1" >/dev/null 2>&1; then
			echo "enabled"
		fi
	fi
}

#
# Disable a boot-time service
#
disable_service() {
	if command -v systemctl >/dev/null; then
		systemctl disable "$1"
	elif command -v chkconfig >/dev/null; then
		chkconfig "$1" off
	else
		update-rc.d "$1" disable
	fi
}

#
# List currently configured APT source URIs
# Only supports the single-line format
#
list_apt_sources() {
	(shopt -s nullglob; awk '
$1 == "deb" {
	for (i = 2; i < NF; i++) {
		if (! index($i, "=")) {
			print $i
			next
		}
	}
}
	' /etc/apt/sources.lis[t] /etc/apt/sources.list.d/*.list /dev/null) |
	sort -u
}

#
# Install required packages
#
install_deps() {
	echo "+ Checking required repositories..."

	case "$distrib" in
		debian )
			apt_update=
			if [ -n "$with_r" ] && ! list_apt_sources | grep -q r-project.org; then
				# Enable CRAN repository as native Debian versions are too old
				echo "+ Adding CRAN repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				gpg --keyserver keyserver.ubuntu.com \
					--recv-key '95C0FAF38DB3CCAD0C080A7BDC78B2DDEABC47B7'
				gpg --armor --yes --output /etc/apt/keyrings/cran_debian_key.asc \
					--export '95C0FAF38DB3CCAD0C080A7BDC78B2DDEABC47B7'
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/cran.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/cran_debian_key.asc] https://cloud.r-project.org/bin/linux/debian $codename-cran40/"
				apt_update=1
			fi
			if [ -n "$with_azure" ] && ! list_apt_sources | grep -q azure-cli; then
				echo "+ Adding azure-cli repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				curl -LSs -o /etc/apt/keyrings/microsoft.asc https://packages.microsoft.com/keys/microsoft.asc
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/azure-cli.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/microsoft.asc] https://packages.microsoft.com/repos/azure-cli $codename main"
				apt_update=1
			fi
			if [ -n "$with_docker" ] && ! list_apt_sources | grep -q download.docker.com; then
				echo "+ Adding docker repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				curl -LSs -o /etc/apt/keyrings/docker.asc https://download.docker.com/linux/debian/gpg
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/docker.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/docker.asc] https://download.docker.com/linux/debian $codename stable"
				apt_update=1
			fi
			if [ -n "$apt_update" ]; then
				apt-get update
			fi
			;;

		ubuntu )
			apt_update=
			if [ -n "$with_r" ] && ! list_apt_sources | grep -q r-project.org; then
				# Enable CRAN repository as native Ubuntu versions are too old
				echo "+ Adding CRAN repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				curl -LSs -o /etc/apt/keyrings/cran_ubuntu_key.asc https://cloud.r-project.org/bin/linux/ubuntu/marutter_pubkey.asc
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/cran.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/cran_ubuntu_key.asc] https://cloud.r-project.org/bin/linux/ubuntu $codename-cran40/"
				apt_update=1
			fi
			if [ -n "$with_azure" ] && ! list_apt_sources | grep -q azure-cli; then
				echo "+ Adding azure-cli repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				curl -LSs -o /etc/apt/keyrings/microsoft.asc https://packages.microsoft.com/keys/microsoft.asc
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/azure-cli.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/microsoft.asc] https://packages.microsoft.com/repos/azure-cli $codename main"
				apt_update=1
			fi
			if [ -n "$with_docker" ] && ! list_apt_sources | grep -q download.docker.com; then
				echo "+ Adding docker repository ..."
				[ -f /usr/bin/dirmngr ] ||
					apt-get $yes install dirmngr
				[ -f /usr/bin/curl ] ||
					apt-get $yes install curl
				install -d /etc/apt/keyrings
				curl -LSs -o /etc/apt/keyrings/docker.asc https://download.docker.com/linux/ubuntu/gpg
				codename=$(source /etc/os-release && echo "$VERSION_CODENAME")
				echo >/etc/apt/sources.list.d/docker.list \
					"deb [arch=amd64 signed-by=/etc/apt/keyrings/docker.asc] https://download.docker.com/linux/ubuntu $codename stable"
				apt_update=1
			fi
			if [ -n "$apt_update" ]; then
				apt-get update
			fi
			;;

		centos | redhat | oraclelinux | almalinux | rocky)
			repolist=$(yum repolist)
			repos=
			# EPEL is required for R
			if [ -n "$with_r" ]; then
				grep -qE '^[!*]?(epel|ol._developer_EPEL)[[:space:]/]' <<< "$repolist" || {
					echo "+ Adding EPEL repository ..."
					if [ "$distrib" = "centos" -o "$distrib" = "almalinux" -o "$distrib" = "rocky" ]; then
						repos+=" epel-release"
					elif [ "$distrib" = "oraclelinux" ]; then
						case "$distribVersion" in
							8*) repos+=" oracle-epel-release-el8";;
							9*) repos+=" oracle-epel-release-el9";;
						esac
					else
						case "$distribVersion" in
							8*) repos+=" https://dl.fedoraproject.org/pub/epel/epel-release-latest-8.noarch.rpm";;
							9*) repos+=" https://dl.fedoraproject.org/pub/epel/epel-release-latest-9.noarch.rpm";;
						esac
					fi
				}
			fi
			if [ "$distrib" = "redhat" ] && grep -q epel-release <<< "$repos"; then
				# On RedHat, EPEL packages may have dependencies in "optional" repo, which is not enabled by default
				case "$distribVersion" in
					8*) echo "* Warning: you may need to enable the RedHat 'CodeReady Linux Builder' repository in case of missing dependencies (codeready-builder-for-rhel-8-*-rpms)";;
					9*) echo "* Warning: you may need to enable the RedHat 'CodeReady Linux Builder' repository in case of missing dependencies (codeready-builder-for-rhel-9-*-rpms)";;
				esac
				echo "* See https://docs.fedoraproject.org/en-US/epel/"
			fi
			if [[ -n "$with_chrome" ]]; then
				# Enable the nodejs 22.x stream if not already installed, as the default is too old
				# We could also parse the current stream version from /etc/dnf/modules.d/nodejs.module and issue a warning if needed
				rpm --quiet -q nodejs || {
					echo "+ Enabling nodejs:22 module stream ..."
					dnf module switch-to $yes nodejs:22
				}
			fi
			if [ -n "$with_azure" ]; then
				case "$distribVersion" in
					8*) rpm --quiet -q packages-microsoft-prod || {
							echo "+ Adding packages-microsoft-com-prod repository ..."
							repos+=" https://packages.microsoft.com/config/rhel/8/packages-microsoft-prod.rpm"
						};;
					9*) rpm --quiet -q packages-microsoft-prod || {
							echo "+ Adding packages-microsoft-com-prod repository ..."
							repos+=" https://packages.microsoft.com/config/rhel/9.0/packages-microsoft-prod.rpm"
						};;
				esac
			fi
			if [ -n "$with_docker" ] && ! grep -qE '^[!*]?docker-ce' <<< "$repolist"; then
				echo "+ Adding docker repository ..."
				rpm --quiet -q yum-utils ||
					yum install $yes yum-utils
				yum-config-manager --add-repo https://download.docker.com/linux/rhel/docker-ce.repo
			fi
			if [ -n "$repos" ]; then
				yum install -q $yes $repos
			fi
			;;

		amazonlinux )
			if [ "$distribVersion" = "2" ]; then
				if ! rpm --quiet -q nginx; then
					echo "+ Enabling nginx1 topic ..."
					amazon-linux-extras enable nginx1
				fi
				if [ -n "$with_chrome" ] && ! rpm --quiet -q epel-release; then
					# Use EPEL for nodejs
					echo "+ Adding EPEL repository ..."
					amazon-linux-extras enable epel
					yum install -q $yes epel-release
				fi
				if [ -n "$with_chrome" ] && ! rpm --quiet -q nodejs; then
					# Version of libuv provided by base OS is too old for EPEL nodejs
					# but would still be selected because of repository priority protections
					# Force installation of EPEL version instead
					echo "+ Installing EPEL version of libuv (nodejs dependency) ..."
					yum install -q $yes libuv --disableplugin=priorities
				fi
				if [ -n "$with_docker" ] && ! yum repolist | grep -q amzn2extra-docker; then
					echo "+ Enabling docker topic ..."
					amazon-linux-extras enable docker
				fi
			fi
			;;

		suse )
			if [ -n "$with_r" ]; then
				zypper repos -u | grep -q "devel:/languages:/R:" || {
					echo "+ Adding devel:languages:R:patched repository for R ..."
					zypper $yes addrepo -n "R patched" obs://devel:languages:R:patched/"$distribVersion" R_patched
					if [ -n "$yes" ]; then
						zypper $yes --gpg-auto-import-keys refresh R_patched
					fi
				}
			fi
			if [ -n "$with_docker" ]; then
				zypper repos -u | grep -q Containers_Module_x86_64 || {
					echo "+ Activating Containers Module ..."
					SUSEConnect -p sle-module-containers/"$distribVersion"/x86_64
				}
			fi
			;;
	esac

	echo "+ Installing required packages..."

	# Check whether nginx is already configured to start on boot
	# as installing the package may unnecessary set it
	nginx_enabled="$(is_service_enabled nginx)"

	case "$distrib" in
		debian | ubuntu )
			pkglist=
			for pkg in $PKG; do
				if [[ "$pkg" == *'>='* ]]; then
					pkg=$(awk -F '>=' '{print $1}' <<<"$pkg")
				fi
				pkglist+=" $pkg"
			done
			apt-get $yes install $pkglist

			# Install required locale if missing
			locale -a | grep -q en_US.utf8 || {
				echo "+ Installing en_US.UTF-8 locale ..."
				case "$distrib" in
					debian)
						sed -i 's/^# *\(en_US.UTF-8\)/\1/' /etc/locale.gen
						locale-gen
						;;
					ubuntu)
						locale-gen en_US.UTF-8
						;;
				esac
			}
			;;

		centos | redhat | amazonlinux | oraclelinux | almalinux | rocky)
			pkglist=
			enablerepo=
			for pkg in $PKG; do
				if [[ "$pkg" == *'>='* ]]; then
					pkg=$(awk -F '>=' '{print $1}' <<<"$pkg")
				fi
				pkglist+=" $pkg"
				# R-core-devel may require enabling additional repositories
				if [ "$pkg" = "R-core-devel" ] && ! rpm --quiet -q R-core-devel; then
					if [[ ("$distrib" == "almalinux" || "$distrib" == "rocky") && "$distribVersion" == 9* ]]; then
						echo "+ Enabling CRB repository for R dependencies..."
						enablerepo="--enablerepo=crb"
					elif [[ ("$distrib" == "centos" || "$distrib" == "almalinux" || "$distrib" == "rocky") && "$distribVersion" == 8* ]]; then
						echo "+ Enabling PowerTools repository for R dependencies..."
						enablerepo="--enablerepo=powertools"
					elif [[ "$distrib" == "redhat" && "$distribVersion" == 9* ]]; then
						echo "**** WARNING ****"
						echo "* Some dependencies for R-core-devel are in the codeready-builder-for-rhel-9 repository"
						echo "* which may not be enabled on some RHEL 9 systems"
						echo "* In case of missing dependencies, make sure to enable this repository"
						echo "* using 'subscription-manager'"
						echo "*"
						echo "* You can also install this package separately with:"
						echo "*    sudo yum install R-core-devel --enablerepo=CODEREADY_REPO"
						echo "* where CODEREADY_REPO is the name of this repository on your server"
						echo "* which you can retrieve with:"
						echo "*     sudo yum repolist --all | grep -i codeready"
						echo "*****************"
					elif [[ "$distrib" == "redhat" && "$distribVersion" == 8* ]]; then
						echo "**** WARNING ****"
						echo "* Some dependencies for R-core-devel are in the codeready-builder-for-rhel-8 repository"
						echo "* which may not be enabled on some RHEL 8 systems"
						echo "* In case of missing dependencies, make sure to enable this repository"
						echo "* using 'subscription-manager'"
						echo "*"
						echo "* You can also install this package separately with:"
						echo "*    sudo yum install R-core-devel --enablerepo=CODEREADY_REPO"
						echo "* where CODEREADY_REPO is the name of this repository on your server"
						echo "* which you can retrieve with:"
						echo "*     sudo yum repolist --all | grep -i codeready"
						echo "*****************"
					elif [[ "$distrib" == "oraclelinux" && "$distribVersion" == 9* ]]; then
						echo "+ Enabling CodeReady Builder repository for R dependencies..."
						enablerepo="--enablerepo=ol9_codeready_builder"
					elif [[ "$distrib" == "oraclelinux" && "$distribVersion" == 8* ]]; then
						echo "+ Enabling CodeReady Builder repository for R dependencies..."
						enablerepo="--enablerepo=ol8_codeready_builder"
					fi
				fi
				# openssl11-devel conflicts with openssl-devel on Amazon Linux 2
				if [ "$pkg" = "openssl11-devel" -a "$distrib" = "amazonlinux" -a "$distribVersion" = 2 ] &&
					rpm --quiet -q openssl-devel; then
					echo "* The 'openssl-devel' package is currently installed, and would conflict with the required 'openssl11-devel' dependency"
					echo "* Removing it"
					yum remove $yes openssl-devel
				fi
			done
			yum $yes install $enablerepo $pkglist

			# Amazon Linux 2 does not provide any supported version of Python
			if [ -z "$without_python" -a "$distrib" = "amazonlinux" -a "$distribVersion" = 2 ]; then
				if [ ! -x /usr/local/bin/python3.9 ]; then
					echo "*"
					echo "* DSS requires Python version 3.9, 3.10 or 3.11, none of which is natively available on Amazon Linux 2"
					if [ -z "$yes" ]; then
						read -p "* Do you want to install a locally-compiled version of Python 3.9 in /usr/local/bin ? [Y/n] "
						case "$REPLY" in
							[Yy] | "") "$(dirname "$0")"/al2-install-python39.sh;;
							*) echo "* Make sure to provide a supported version of Python to the DSS installer script";;
						esac
					else
						echo "* Installing a locally-compiled version of Python 3.9 in /usr/local/bin"
						"$(dirname "$0")"/al2-install-python39.sh
					fi
				fi
			fi

			# Amazon Linux 2 does not provide any supported version of R
			if [ -n "$with_r" -a "$distrib" = "amazonlinux" -a "$distribVersion" = 2 ]; then
				if [ ! -x /usr/local/bin/R ]; then
					echo "*"
					echo "* DSS requires R 4 version 4.1 or later, which is not natively available on Amazon Linux 2"
					if [ -z "$yes" ]; then
						read -p "* Do you want to install a locally-compiled version of R 4.5 in /usr/local/bin ? [Y/n] "
						case "$REPLY" in
							[Yy] | "") "$(dirname "$0")"/al2-install-r45.sh;;
							*) echo "* Make sure to install a supported version of R";;
						esac
					else
						echo "* Installing a locally-compiled version of R 4.5 in /usr/local/bin"
						"$(dirname "$0")"/al2-install-r45.sh
					fi
				fi
			fi
			;;

		suse )
			pkglist=
			for pkg in $PKG; do
				pkglist+=" $pkg"
			done
			zypper $yes install $pkglist
			;;
	esac

	if [ -z "$nginx_enabled" -a -n "$(is_service_enabled nginx)" ]; then
		echo "+ Disabling nginx service"
		disable_service nginx
	fi

	if [ -n "$with_docker" ]; then
		if [[ "$distrib" == "amazonlinux" || "$distrib" == "suse" ]]; then
			# Install docker-compose from github as it is not available as a package
			[ -f /usr/local/lib/docker/cli-plugins/docker-compose ] || {
				echo "+ Installing docker compose plugin from github ..."
				mkdir -p /usr/local/lib/docker/cli-plugins
				curl -LsS https://github.com/docker/compose/releases/download/v2.20.3/docker-compose-linux-x86_64 \
					-o /usr/local/lib/docker/cli-plugins/docker-compose
				chmod +x /usr/local/lib/docker/cli-plugins/docker-compose
			}
		fi
		echo "+ Enabling and starting docker service"
		systemctl enable --now docker.service ||
			echo >&2 "* Warning : could not enable / start docker service"
	fi
}


#
# Check package minimum version
# checkMin{Deb,Rpm}Version VERSION MIN_VERSION
#
checkMinDebVersion() {
	dpkg --compare-versions "$1" ge "$2"
}

checkMinRpmVersion() {
	case "$distrib" in
		centos | redhat | oraclelinux | almalinux | rocky)
			systemPython=/usr/libexec/platform-python
			;;
		amazonlinux)
			if [ "$distribVersion" = "2" ]; then
				systemPython=/usr/bin/python
			else
				systemPython=/usr/bin/python3
			fi
			;;
		*)
			echo >&2 "*** checkMinRpmVersion : not yet implemented"
			exit 1
			;;
	esac
	"$systemPython" -c '
import sys, rpm
if rpm.labelCompare(("0", sys.argv[1], None), ("0", sys.argv[2], None)) < 0:
	sys.exit(1)
' "$1" "$2"
}

#
# Check required packages
#
check_deps() {
	err=0

	echo "+ Checking required packages..."
	case "$distrib" in
		debian | ubuntu )
		for pkg in $PKG; do
			case "$pkg" in
				*'>='*)
					minver=$(awk -F '>=' '{print $2}' <<<"$pkg")
					pkg=$(awk -F '>=' '{print $1}' <<<"$pkg")
					;;
				*)
					minver=
					;;
			esac
			if [ "$(dpkg-query -W -f='${Status}\n' $pkg:amd64 2>/dev/null)" = "install ok installed" ]; then
				arch=amd64
			elif [ "$(dpkg-query -W -f='${Status}\n' $pkg:all 2>/dev/null)" = "install ok installed" ]; then
				arch=all
			else
				echo "*** Error: package $pkg not found" >&2
				err=1
				minver=
			fi
			if [ -n "$minver" ]; then
				pkgver=$(dpkg-query -W -f='${Version}\n' "$pkg:$arch" 2>/dev/null)
				checkMinDebVersion "$pkgver" "$minver" || {
					echo "*** Error: package $pkg has version $pkgver should be >= $minver" >&2
					err=1
				}
			fi
		done
		;;

		centos | redhat | amazonlinux | oraclelinux | almalinux | rocky)
		for pkg in $PKG; do
			case "$pkg" in
				*'>='*)
					minver=$(awk -F '>=' '{print $2}' <<<"$pkg")
					pkg=$(awk -F '>=' '{print $1}' <<<"$pkg")
					;;
				*)
					minver=
					;;
			esac
			rpm --quiet -q "$pkg" || {
				echo "*** Error: package $pkg not found" >&2
				err=1
				minver=
			}
			if [ -n "$minver" ]; then
				pkgver=$(rpm -q --queryformat "%{VERSION}\n" "$pkg")
				checkMinRpmVersion "$pkgver" "$minver" || {
					echo "*** Error: package $pkg has version $pkgver should be >= $minver" >&2
					err=1
				}
			fi
		done
		;;

		suse )
		for pkg in $PKG; do
			rpm -q "$pkg" >&/dev/null || {
				echo "*** Error: package $pkg not found" >&2
				err=1
			}
		done
		;;
	esac

	if [ -n "$with_docker" ]; then
		if [[ "$distrib" == "amazonlinux" || "$distrib" == "suse" ]]; then
			# docker-compose is installed manually on these platforms
			[ -f /usr/local/lib/docker/cli-plugins/docker-compose ] || {
				echo "*** Error: docker-compose plugin not found" >&2
				err=1
			}
		fi
	fi

	return $err
}

#
# Main entry point
#
if [ -n "$check_only" ]; then
	check_deps
else
	install_deps
fi
