/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.datasets.fs;

import com.dataiku.dip.datasets.fs.GCSclient;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dss.shadelib.com.google.api.client.http.GenericUrl;
import com.dataiku.dss.shadelib.com.google.api.client.http.HttpResponse;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Random;

public class GCSOutputStream
extends OutputStream {
    public static final int DEFAULT_PART_SIZE = 0x400000;
    private static final int MAX_ATTEMPTS = 10;
    private static final int MAXIMUM_BACK_OFF_TIMEOUT = 32000;
    private final GCSclient.GCSBlobFile blobFile;
    private final long fileSize;
    private final int partSize;
    private final boolean waitBetweenRetries;
    private final byte[] buffer;
    private boolean initialized;
    private boolean closed;
    private String resumableSessionUri;
    private long gcsUploadIndex;
    private int bufferCount;
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.gcs.client.upload");

    public GCSOutputStream(GCSclient.GCSBlobFile blobFile, long fileSize) {
        this(blobFile, fileSize, 0x400000, true);
    }

    public GCSOutputStream(GCSclient.GCSBlobFile blobFile, long fileSize, int partSize) {
        this(blobFile, fileSize, partSize, true);
    }

    public GCSOutputStream(GCSclient.GCSBlobFile blobFile, long fileSize, int partSize, boolean waitBetweenRetries) {
        Preconditions.checkArgument((partSize > 0 && partSize % 262144 == 0 ? 1 : 0) != 0, (Object)"partSize must be a multiple of 256 KB");
        this.blobFile = (GCSclient.GCSBlobFile)Preconditions.checkNotNull((Object)blobFile, (Object)"blobFile cannot be null");
        this.fileSize = fileSize;
        this.partSize = partSize;
        this.waitBetweenRetries = waitBetweenRetries;
        this.buffer = new byte[partSize];
    }

    @Override
    public synchronized void write(int arg) throws IOException {
        this.initialize();
        this.buffer[this.bufferCount++] = (byte)arg;
        if (this.bufferCount >= this.partSize) {
            this.flushToGCS(false);
        }
    }

    @Override
    public synchronized void write(byte[] b, int off, int len) throws IOException {
        this.initialize();
        if (b == null) {
            throw new NullPointerException();
        }
        if (off < 0 || off > b.length || len < 0 || off + len > b.length || off + len < 0) {
            throw new IndexOutOfBoundsException();
        }
        int bytesToWrite = len;
        int currentOffset = off;
        while (bytesToWrite > 0) {
            int bytesCopied = Math.min(bytesToWrite, this.partSize - this.bufferCount);
            System.arraycopy(b, currentOffset, this.buffer, this.bufferCount, bytesCopied);
            this.bufferCount += bytesCopied;
            if (this.bufferCount >= this.partSize) {
                this.flushToGCS(false);
            }
            bytesToWrite -= bytesCopied;
            currentOffset += bytesCopied;
        }
    }

    @Override
    public synchronized void close() throws IOException {
        if (!this.closed) {
            this.initialize();
            this.flushToGCS(true);
            this.closed = true;
        }
    }

    private void initialize() throws IOException {
        if (this.closed) {
            throw new IOException("Cannot write into an GCSOutputStream that has been closed.");
        }
        if (!this.initialized) {
            this.sendInitiationRequest();
            this.initialized = true;
        }
    }

    private void sendInitiationRequest() throws IOException {
        String body = "{'name': '" + this.blobFile.name + "'}";
        HashMap<String, Object> headers = new HashMap<String, Object>();
        headers.put("Content-Length", Collections.singletonList(Long.valueOf(body.length())));
        headers.put("Accept-Encoding", Collections.singletonList("gzip"));
        HashMap<String, String> queryParameter = new HashMap<String, String>();
        queryParameter.put("bucket", this.blobFile.bucket);
        queryParameter.put("uploadType", "resumable");
        queryParameter.put("name", this.blobFile.name);
        GCSclient.InitUploadResponse response = this.blobFile.getClient().initUpload(body, queryParameter, headers);
        if (response.statusCode != 200) {
            throw new IOException(String.format("Fail to initialize upload to '%s', response code = %d", this.blobFile.name, response.statusCode));
        }
        this.resumableSessionUri = response.resumableSessionUri;
    }

    private void flushToGCS(boolean isLast) throws IOException {
        try {
            if (this.bufferCount > 0 || isLast && this.gcsUploadIndex == 0L) {
                this.uploadToGcs(isLast);
                this.bufferCount = 0;
            }
        }
        catch (IOException e) {
            this.cancelGCSUpload();
            throw e;
        }
        catch (Exception e) {
            this.cancelGCSUpload();
            throw new IOException(String.format("An unexpected error occurred while uploading [bucket=%s, path=%s] to GCS.", this.blobFile.bucket, this.blobFile.name), e);
        }
    }

    private void uploadToGcs(boolean isLastChunk) throws IOException {
        long rangeBegin = this.gcsUploadIndex;
        long rangeEnd = rangeBegin + (long)this.bufferCount - 1L;
        if (!isLastChunk && this.fileSize > 0L && rangeBegin + (long)this.bufferCount >= this.fileSize) {
            isLastChunk = true;
        }
        String contentRange = this.getContentRange(rangeBegin, rangeEnd, isLastChunk);
        Map<String, Object> uploadHeaders = this.buildUploadHeaders(this.bufferCount, contentRange);
        int retryCount = 0;
        boolean chunkUploaded = false;
        logger.infoV("Uploading chunk %s into [bucket=%s, name=%s] with content-length=%d, content-range=%s", new Object[]{contentRange, this.blobFile.bucket, this.blobFile.name, this.bufferCount, contentRange});
        GCSclient.UploadChunkResponse response = this.blobFile.getClient().uploadChunk(new GenericUrl(this.resumableSessionUri), this.buffer, 0, this.bufferCount, uploadHeaders);
        while (retryCount < 10 && !chunkUploaded) {
            switch (response.statusCode) {
                case 200: 
                case 201: {
                    if (!isLastChunk) {
                        throw new IOException("Upload is not complete. Received 200/201 response for a chunk that is not the last one.");
                    }
                    chunkUploaded = true;
                    this.gcsUploadIndex = rangeEnd + 1L;
                    logger.infoV("Upload successful for [bucket=%s, name=%s]", new Object[]{this.blobFile.bucket, this.blobFile.name});
                    break;
                }
                case 308: {
                    if (response.range != null && (Long)response.range.getMaximum() == rangeEnd) {
                        if (isLastChunk) {
                            throw new IOException("Upload is not complete. Received 308 Resume Incomplete response for last chunk.");
                        }
                        chunkUploaded = true;
                        this.gcsUploadIndex = rangeEnd + 1L;
                        retryCount = 0;
                        break;
                    }
                    logger.infoV("Retry last chunk upload: received only partial range: %s but was expecting [%d..%d]", new Object[]{response.range, rangeBegin, rangeEnd});
                    this.waitBeforeNextAttempt(++retryCount, rangeBegin, rangeEnd);
                    response = this.blobFile.getClient().uploadChunk(new GenericUrl(this.resumableSessionUri), this.buffer, 0, this.bufferCount, uploadHeaders);
                    break;
                }
                case 500: 
                case 502: 
                case 503: 
                case 504: {
                    ++retryCount;
                    logger.infoV("Received %d response when uploading chunk. Resuming upload", new Object[]{response.statusCode});
                    response = this.blobFile.getClient().resumeUpload(new GenericUrl(this.resumableSessionUri), this.buildUploadHeaders(0, "*/*"));
                    break;
                }
                default: {
                    throw new IOException(String.format("Unexpected status code %d - %s received while uploading into bucket=%s, name=%s", response.statusCode, response.statusMessage, this.blobFile.bucket, this.blobFile.name));
                }
            }
            this.waitBeforeNextAttempt(retryCount, rangeBegin, rangeEnd);
        }
        if (retryCount >= 10) {
            throw new IOException(String.format("Unable to upload chunk [%d..%d] after %d retries", rangeBegin, rangeEnd, retryCount));
        }
    }

    private void waitBeforeNextAttempt(int retryCount, long rangeBegin, long rangeEnd) throws IOException {
        if (this.waitBetweenRetries && retryCount > 0 && retryCount < 10) {
            try {
                long timeOut = GCSOutputStream.exponentialBackOffTimeOut(retryCount);
                logger.infoV("Waiting for %d ms before retrying", new Object[]{timeOut});
                Thread.sleep(timeOut);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException(String.format("Unable to upload chunk [%d..%d] after %d retries. Interrupted while waiting before trying again.", rangeBegin, rangeEnd, retryCount));
            }
        }
    }

    private Map<String, Object> buildUploadHeaders(int contentLength, String contentRange) {
        HashMap<String, Object> result = new HashMap<String, Object>();
        if (this.fileSize > 0L) {
            result.put("X-Upload-Content-Length", this.fileSize);
        }
        result.put("X-Upload-Content-Type", "application/octet-stream");
        result.put("Content-Type", ImmutableList.of((Object)"application/octet-stream"));
        result.put("Content-Length", ImmutableList.of((Object)String.valueOf(contentLength)));
        if (contentRange != null) {
            result.put("Content-Range", ImmutableList.of((Object)("bytes " + contentRange)));
        }
        return result;
    }

    private String getContentRange(long rangeBegin, long rangeEnd, boolean isLast) {
        String contentRange = null;
        if (this.fileSize > 0L) {
            contentRange = String.format("%d-%d/%d", rangeBegin, Math.max(0L, rangeEnd), this.fileSize);
        } else if (isLast) {
            if (this.bufferCount > 0) {
                contentRange = String.format("%d-%d/%d", rangeBegin, Math.max(0L, rangeEnd), rangeBegin + (long)this.bufferCount);
            }
        } else {
            contentRange = String.format("%d-%d/*", rangeBegin, Math.max(0L, rangeEnd));
        }
        return contentRange;
    }

    private void cancelGCSUpload() {
        logger.info((Object)String.format("Cancelling upload [bucket=%s, path=%s]", this.blobFile.bucket, this.blobFile.name));
        try {
            HttpResponse response = this.blobFile.getClient().cancelUpload(new GenericUrl(this.resumableSessionUri));
            if (response != null && !response.isSuccessStatusCode()) {
                logger.debug((Object)String.format("Unable to cancel upload. Details: %s", response.parseAsString()));
            }
        }
        catch (Exception e) {
            logger.debug((Object)"Unable to cancel upload", (Throwable)e);
        }
        this.closed = true;
    }

    private static long exponentialBackOffTimeOut(int n) {
        int random = new Random().nextInt(1000);
        int seconds = n <= 0 ? 0 : 250 * (2 << Math.min(10, n));
        logger.infoV("n=%d => seconds=%d, random=%d", new Object[]{n, seconds, random});
        return Math.min(seconds + random, 32000);
    }
}

