(function(){
'use strict';

const app = angular.module('dataiku.flow.graph');


/* We get a first SVG flow generated by graphviz in the backend,
 * frontend post-processing is done here
 */


app.service('ProjectFlowGraphStyling', function($filter, FlowGraph, CachedAPICalls, LoggerProvider, TimingService, DataikuAPI, $rootScope, $q, objectTypeFromNodeFlowType, $state, $compile, FlowZoneMoveService) {

const SIZE = 100;

const logger = LoggerProvider.getLogger('flow');

let flowIconset;
 CachedAPICalls.flowIcons.success(function(data) {
    flowIconset = data;
});

const formatters = {
    'RECIPE': restyleRunnableNode("recipe"),
    'LABELING_TASK': restyleRunnableNode("labeling_task"),
    'LOCAL_DATASET': restyleDatasetNode(true),
    'FOREIGN_DATASET': restyleDatasetNode(false),
    'LOCAL_SAVEDMODEL': restyleModelNode(true),
    'FOREIGN_SAVEDMODEL': restyleModelNode(false),
    'LOCAL_MODELEVALUATIONSTORE': restyleEvaluationStoreNode(true),
    'FOREIGN_MODELEVALUATIONSTORE': restyleEvaluationStoreNode(false),
    'LOCAL_RETRIEVABLE_KNOWLEDGE': restyleRetrievableKnowledgeNode(true),
    'FOREIGN_RETRIEVABLE_KNOWLEDGE': restyleRetrievableKnowledgeNode(false),
    'LOCAL_MANAGED_FOLDER': restyleFolderNode(true),
    'FOREIGN_MANAGED_FOLDER': restyleFolderNode(false),
    'LOCAL_STREAMING_ENDPOINT': restyleStreamingEndpointNode(true),
    'FOREIGN_STREAMING_ENDPOINT': restyleStreamingEndpointNode(false)
};

this.restyleGraph = TimingService.wrapInTimePrinter("ProjectFlowGraphStyling::restyleGraph", function(svg, $scope) {
    const zones = svg.find('.zone_cluster');
    if (zones) {
        for (let i = 0; i < zones.length; i++) {
            const zone = zones[i];
            try {
                restyleZone(svg, zone, $scope);
            } catch (e) {
                logger.error("Failed to restyle flow zone: ", e);
            }
        }
    }
    svg.find(".zone_label_remove").remove();
    svg.find('text').remove();
    svg.find('title').remove();

    const nodes = $('.node:not(.zone)', svg);
    if (nodes) {
        for (let i = 0; i<nodes.length; i++) {
            const g = nodes[i];
            try {
                restyleNode(g);
            } catch (e) {
                logger.error("Failed to restyle flow node: ", e);
            }
        }
    }
    svg.find('.node.zone>svg g polygon').remove();
});

function drawRepeatedObjectSticker(svg, coords) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx,
        cy: coords.cy,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX,
        y: coords.indicatorY,
        width: 20,
        height: 20,
        class: 'flow-repeated-object__indicator',
    }, $(`<span size="32" class="dku-icon-arrow-repeat-16">`)));
}

function drawInformationSticker(svg, coords, offset = { x: 0, y: 0 }) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx - offset.x,
        cy: coords.cy - offset.y,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX - offset.x,
        y: coords.indicatorY - offset.y,
        width: 20,
        height: 20,
        class: 'flow-information-sticker__indicator',
    }, $(`<span size="32" class="icon-info-sign">`)));
}

function drawFeatureGroupSticker(svg, coords) {
    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-information-sticker__background',
        cx: coords.cx,
        cy: coords.cy,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX,
        y: coords.indicatorY,
        width: 20,
        height: 20,
        class: 'flow-feature-group__indicator',
    }, $(`<span size="32" class="icon-dku-label-feature-store">`)));
}

function drawDiscussionSticker(svg, discussionCount, has_unread_discussions, coords,
        offset = { x: 0, y: 0 }, indicatorOffset = { x: 0, y: 0 }) {

    svg.appendChild(makeSVG('ellipse', {
        class: 'flow-discussions-sticker__background',
        cx: coords.cx - offset.x,
        cy: coords.cy - offset.y,
        rx: coords.rx,
        ry: coords.ry
    }));

    svg.appendChild(makeForeignObject({
        x: coords.indicatorX - offset.x - indicatorOffset.x,
        y: coords.indicatorY - offset.y - indicatorOffset.y,
        width: 20,
        height: 20,
        class: 'flow-discussions-sticker__indicator' + ((has_unread_discussions) ? ' flow-discussions-sticker__indicator--unread' : ''),
    }, $(`<span size="32" class="icon-dku-discussions">`)));

    svg.appendChild(makeForeignObject({
        x: coords.contentX - offset.x,
        y: coords.contentY - offset.y,
        width: 40,
        height: 20,
        class: 'flow-discussions-sticker__content',
    }, $(`<span>` + (discussionCount > 9 ? '9+' : discussionCount) + `</span>`)));
}

function hasUnreadDiscussion(flowNode) {
    return (($rootScope.discussionsUnreadStatus || {}).unreadFullIds || []).find(discuId => (flowNode.discussionsFullIds || []).includes(discuId));
}

function restyleNode(g) {
    const element = $(g);
    const nodeType = element.attr('data-type');
    if (formatters[nodeType]) {
        formatters[nodeType](element, g);
    }
}

function restyleZone(svg, zone_cluster, $scope) {
    const jZone_cluster = $(zone_cluster);
    const cluster_polygon = jZone_cluster.find('>polygon');
    const text = jZone_cluster.find(">text");
    const id = zone_cluster.id.split("_").splice(2).join('');
    const sanitizedId = sanitize(id);
    const zone = svg.find(`.zone[id='zone_${id}']`);
    const zone_polygon = zone.find('>polygon');
    // Calculate difference between zone top and cluster top (Gives us the height for the text)

    if (zone_polygon && cluster_polygon) {
        const zone_coords = polygonToRectData(zone_polygon);
        const cluster_coords = polygonToRectData(cluster_polygon);
        const zoneNode = FlowGraph.node(`zone_${id}`);
        const collapseIcon = zoneNode.customData.isCollapsed ? "icon-resize-full" : "icon-resize-small";
        let height = (zone_coords.y - cluster_coords.y) * 2;
        if (!height > 0) {
            logger.warn("Calculated height is not a valid number, default to 44");
            height = 44;
        }
        const foreignObject = makeForeignObject({
            x: cluster_coords.x,
            y: cluster_coords.y,
            width: zone_coords.width,
            height,
            class: 'zone_header'
        }, $(`<div><p>${sanitize(text.text())}</p><span ng-if="zonesManualPositioning && canMoveZones" class="move-zone-shortcut">{{altKey}} + Click to drag</span><button ng-click="buildZone('${sanitizedId}')" class="btn btn--secondary" style="position: static" ><i class="icon-play" /> <span translate="PROJECT.FLOW.GRAPH.ZONE.HEADER.BUILD">Build</span></button><i ng-click="toggleZoneCollapse([{id:'${sanitizedId}'}])" class="${collapseIcon} cursor-pointer" id="collapse-button-zone-${sanitizedId}"/><i ng-click="zoomOnZone('${sanitizedId}')" class="icon-DKU_expand cursor-pointer"/></div>`));
        const color = d3.rgb(zoneNode.customData.color);
        const zoneTitleColor = (color.r*0.299 + color.g*0.587 + color.b*0.114) >= 128 ? "#000" : "#FFF";
        foreignObject.style = `background-color: ${color}; color: ${zoneTitleColor}; border-bottom: none;`;
        zone_cluster.appendChild(makeSVG('g', {
            class: 'tool-simple-zone',
            transform: `translate(${cluster_coords.x + cluster_coords.width}, ${cluster_coords.y + cluster_coords.height})`,
            'data-height': 0
        }));
        zone_cluster.appendChild(foreignObject);
        $compile(foreignObject)($scope);
        zone_polygon.remove();
    }

    $(zone_cluster)[0].setAttribute("data-zone-title", sanitize(text[0].textContent));
    $(zone_cluster)[0].setAttribute("data-id", "zone_" + sanitizedId);

    if ($scope.zonesManualPositioning) {
        FlowZoneMoveService.setZoneEdges(svg, `zone_${sanitizedId}`, $scope.nodesGraph);
    }
}

function restyleDatasetNode(local) {
    // Note that differentiation local/foreign is made with CSS
    return function (element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const dataset = FlowGraph.node(nodeId);

        const dotPolygon = element.find("polygon");
        if (dotPolygon.length) {
            const coords = polygonToRectData(dotPolygon);

            let clazz = 'newG';
            if (dataset.neverBuilt) {
                clazz += ' never-built-computable';
            }
            const newG = makeSVG('g', {class: clazz, transform: `translate(${coords.x} ${coords.y})`});
            d3.select(newG).classed("bzicon", true);

            const othersZones = FlowGraph.nodeSharedBetweenZones(dataset);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const margin = isExported && othersZones.size > 0 ? 4 : 0;
            const counterBoxMargin = (isExported || isImported) && othersZones.size > 0 ? 1 : 0;

            if (dataset.partitioned) {
                newG.appendChild(makeSVG('rect', {
                    x: -10,
                    y: -10,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-rectangle partitioning-indicator' + (isImported ? ' dataset-zone-imported' : '')
                }));

                newG.appendChild(makeSVG('rect', {
                    x: -5,
                    y: -5,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-rectangle partitioning-indicator' + (isImported ? ' dataset-zone-imported' : '')
                }));
            }


            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill dataset-zone-exported'
                }));
            }
            newG.appendChild(makeSVG('rect', {
                x: margin,
                y: margin,
                width: coords.width - (margin*2),
                height: coords.height - (margin*2),
                class: 'fill dataset-rectangle main-dataset-rectangle' + (isImported ? ' dataset-zone-imported' : '')
            }));

            // Grey box containing count records
            newG.appendChild(makeSVG('rect', {
                x: counterBoxMargin,
                y: 61,
                width: coords.width - (counterBoxMargin * 2),
                height: coords.height - counterBoxMargin - 61,
                class: 'nodecounter__wrapper'
            }));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': '72'
            }));

            // Display of count records value
            newG.appendChild(makeForeignObject({
                x: -30,
                y: 56,
                width: coords.width + 61,
                height: 48,
                class: 'nodecounter__text'
            }, $('<div><span></span></div>')));

            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.07,
                width: coords.width + 60,
                height: 45,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(dataset.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            const icon = $filter("toModernIcon")($filter("datasetTypeToIcon")(dataset.datasetType, 48), 48);
            const iconElement = $(`<div class="flow-tile faic jcc h100 w100"><i class="${icon}" /></div>`);

            newG.appendChild(makeForeignObject({
                x: 0,
                y: 0,
                width: 72,
                height: 72,
                class: 'nodeicon' + (isImported ? 'dataset-imported' : '')
            }, iconElement));

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: 72,
                height: 72
            }));

            newG.appendChild(makeForeignObject({
                x: 56,
                y: -15,
                width: 34,
                height: 34,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Repeated object sticker
            if (dataset.veLoopDatasetRef) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57,
                };

                drawRepeatedObjectSticker(newG, coords);
                offset.y += 18;
            }

            // Information sticker
            // To display when the dataset has a short description
            if (angular.isDefined(dataset.shortDesc) && dataset.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57
                };

                drawInformationSticker(newG, coords, offset);
                offset.y += 18;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && dataset.discussionsFullIds && dataset.discussionsFullIds.length > 0) {
                let unread = hasUnreadDiscussion(dataset);
                let coords = {
                    cx: 0,
                    cy: 67.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: -10,
                    indicatorY: 57.5,
                    contentX: -20,
                    contentY: 57.5,
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, dataset.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});
            }

            // Feature group sticker
            // To display when the dataset is a feature group
            if (angular.isDefined(dataset.featureGroup) && dataset.featureGroup) {
                let coords = {
                    cx: 70,
                    cy: 67,
                    rx: 11,
                    ry: 11,
                    indicatorX: 60,
                    indicatorY: 58
                };

                drawFeatureGroupSticker(newG, coords);
            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleModelNode(local) {
    const isLLMGenericSm = (savedModel) => savedModel.savedModelType && ['LLM_GENERIC', 'PLUGIN_AGENT', 'PYTHON_AGENT', 'TOOLS_USING_AGENT', "RETRIEVAL_AUGMENTED_LLM"].includes(savedModel.savedModelType);
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const sm = FlowGraph.node(nodeId);
        if (isLLMGenericSm(sm)) {
            element.addClass("fine-tuned-sm");
        }


        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y-coords.height})`});

            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 0.707,
                height: coords.height * 0.707,
                transform: `translate(${coords.width/2}) rotate(45)`,
                opacity: 0,
                class: 'fill'
            }));

            const othersZones = FlowGraph.nodeSharedBetweenZones(sm);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const savedModelClass = isLLMGenericSm(sm) ? ' llm-model' : '';

            if (sm.partitioned) {
                /* Partition boxes */
                for (let offset of [-10, -5, 0]) {
                    newG.appendChild(makeSVG('rect', {
                        x: offset,
                        y: offset,
                        width: coords.width * 1.41421,
                        height: coords.height * 1.41421,
                        transform: `translate(${coords.width}) rotate(45)`,
                        class: 'fill node__rectangle--partitioned partitioning-indicator'
                    }));
                }

                /* White background for the rest of the icon */
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'node__rectangle--blank'
                }));
            }

            if (isExported || isImported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'fill main-model-rectangle model-zone-' + (isExported ? 'exported' : 'imported') + savedModelClass
                }));
            }

            let iconText;
            if (sm.externalSavedModelType) {
                const externalSmTypeToIcon = {
                    "sagemaker": "model_amazon_sagemaker",
                    "azure-ml": "model_azureml",
                    "vertex-ai": "model_google_vertex",
                    "databricks": "model_databricks",
                    "mlflow": "model_mlflow"
                }
                iconText = flowIconset.icons[externalSmTypeToIcon[sm.externalSavedModelType]];
            }
            if (iconText === undefined) {
                let iconKey = 'clustering';
                if (sm.taskType === 'PREDICTION') {
                    switch (sm.predictionType) {
                        case 'TIMESERIES_FORECAST':
                            iconKey = 'timeseries';
                            break;
                        case 'DEEP_HUB_IMAGE_CLASSIFICATION':
                        case 'DEEP_HUB_IMAGE_OBJECT_DETECTION':
                            iconKey = 'computer_vision';
                            break;
                        case 'CAUSAL_REGRESSION':
                        case 'CAUSAL_BINARY_CLASSIFICATION':
                            iconKey = 'causal';
                            break;
                        default:
                            iconKey = 'regression';
                            if (sm.backendType === 'KERAS') {
                                iconKey = 'deep_learning';
                            }
                    }
                } else if (sm.savedModelType === 'PYTHON_AGENT') {
                    iconKey = 'ai_agent_code';
                } else if (sm.savedModelType === 'PLUGIN_AGENT') {
                    iconKey = 'ai_agent_plugin';
                } else if (sm.savedModelType === 'TOOLS_USING_AGENT') {
                    iconKey = 'ai_agent_visual';
                } else if (sm.savedModelType === 'RETRIEVAL_AUGMENTED_LLM') {
                    iconKey = 'llm_augmented';
                } else if (isLLMGenericSm(sm)) {
                    iconKey = 'fine_tuning';
                }

                iconText = flowIconset.icons['model_' + iconKey];
            }
            const iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("path,rect");
            const newG2 = makeSVG("g");
            iconElt[0].forEach(i => newG2.appendChild(i));
            d3.select(newG2).attr("transform", " scale(0.707, 0.707) translate(1, 1)")
            let klass = "bzicon sm-icon";
            klass += isImported ? ' model-zone-imported': isExported ? ' model-zone-exported' : '';
            klass += savedModelClass;
            d3.select(newG2).classed(klass, true);
            newG.appendChild(newG2);

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            if (!sm.description) {
                sm.description = "Saved model...";
            }

            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                opacity: 0,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 2.07,
                width: coords.width*2+60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(sm.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the sm has a short description
            if (angular.isDefined(sm.shortDesc) && sm.shortDesc.length > 0) {
                let coords = {
                    cx: 27,
                    cy: 63,
                    rx: 10,
                    ry: 10,
                    indicatorX: 17,
                    indicatorY: 53
                };

                drawInformationSticker(newG, coords);
                offset.x += 12;
                offset.y += 12;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && sm.discussionsFullIds && sm.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(sm);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, sm.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleEvaluationStoreNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const mes = FlowGraph.node(nodeId);

        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG bzicon', transform: `translate(${coords.x} ${coords.y-coords.height})`});

            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width * 0.707,
                height: coords.height * 0.707,
                transform: `translate(${coords.width/2}) rotate(45)`,
                opacity: 0,
                class: 'fill'
            }));

            const othersZones = FlowGraph.nodeSharedBetweenZones(mes);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);

            if (mes.partitioned) {
                /* Partition boxes */
                for (let offset of [-10, -5, 0]) {
                    newG.appendChild(makeSVG('rect', {
                        x: offset,
                        y: offset,
                        width: coords.width * 1.41421,
                        height: coords.height * 1.41421,
                        transform: `translate(${coords.width}) rotate(45)`,
                        class: 'fill node__rectangle--partitioned partitioning-indicator'
                    }));
                }

                /* White background for the rest of the icon */
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`
                }));
            } else {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`
                }));
            }

            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width * 1.41421,
                    height: coords.height * 1.41421,
                    transform: `translate(${coords.width}) rotate(45)`,
                    class: 'fill evaluation-store-zone-exported'
                }));
            }

            const icon = 'dku-icon-model-evaluation-store-48';
            newG.appendChild(makeForeignObject({
                x: 12,
                y: 12,
                width: 48,
                height: 48,
                class: 'nodeicon' + (isImported ? ' evaluation-store-zone-imported': isExported ? ' evaluation-store-zone-exported' : '')
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            if (!mes.description) {
                mes.description = "Model evaluation store...";
            }

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width * 1.41421,
                height: coords.height * 1.41421,
                transform: `translate(${coords.width})  rotate(45)`,
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 2.07,
                width: coords.width*2+60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(mes.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the mes has a short description
            if (angular.isDefined(mes.shortDesc) && mes.shortDesc.length > 0) {
                let coords = {
                    cx: 27,
                    cy: 63,
                    rx: 10,
                    ry: 10,
                    indicatorX: 17,
                    indicatorY: 53
                };

                drawInformationSticker(newG, coords);
                offset.x += 12;
                offset.y += 12;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && mes.discussionsFullIds && mes.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(mes);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, mes.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}


function restyleRetrievableKnowledgeNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const rk = FlowGraph.node(nodeId);

        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);

            let clazz = 'newG bzicon retrievable-knowledge'
            if (rk.neverBuilt) {
                clazz += ' never-built-computable';
            }
            const newG = makeSVG('g', {class: clazz, transform: `translate(${coords.x} ${coords.y})`});

            const othersZones = FlowGraph.nodeSharedBetweenZones(rk);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);
            const margin = isExported && othersZones.size > 0 ? 4 : 0;

            if (isExported) {
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width,
                    height: coords.height,
                    class: 'fill retrievable-knowledge-zone-exported'
                }));
            }
            newG.appendChild(makeSVG('rect', {
                x: margin,
                y: margin,
                width: coords.width - (margin*2),
                height: coords.height - (margin*2),
                class: 'fill retrievable-knowledge-rectangle main-retrievable-knowledge-rectangle' + (isImported ? ' retrievable-knowledge-zone-imported' : '')
            }));

            const icon = 'dku-icon-cards-stack-48';
            newG.appendChild(makeForeignObject({
                x: 12,
                y: 13,
                width: 48,
                height: 48,
                class: 'nodeicon' + (isImported ? ' retrievable-knowledge-zone-imported': isExported ? ' retrievable-knowledge-zone-exported' : '')
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': coords.height
            }));

            if (!rk.description) {
                rk.description = "Knowledge Bank...";
            }

            newG.appendChild(makeSVG('rect', {
                class: 'selection-outline',
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.07,
                width: coords.width + 60,
                height: 45,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(rk.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 56,
                y: -15,
                width: 34,
                height: 34,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the rk has a short description
            if (angular.isDefined(rk.shortDesc) && rk.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 67,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 57
                };

                drawInformationSticker(newG, coords);
                offset.y += 20;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && rk.discussionsFullIds && rk.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(rk);
                let coords = {
                    cx: 24,
                    cy: 64.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: 14,
                    indicatorY: 55,
                    contentX: 4,
                    contentY: 55
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, rk.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleFolderNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const nodeZoneId = element.attr('data-zone-id');
        const folder = FlowGraph.node(nodeId);

        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            const coords = polygonToRectData(dotPolygon);

            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y})`});

            const othersZones = FlowGraph.nodeSharedBetweenZones(folder);
            const isExported = othersZones && !othersZones.has(nodeZoneId);
            const isImported = othersZones && othersZones.has(nodeZoneId);

            element.find("text").remove();

            const iconText = flowIconset.icons["folder"];
            let iconElt = d3.select($.parseXML(iconText)).select("svg").select("g")
            iconElt.attr("transform", " scale(0.57, 0.57) ");
            iconElt.classed("bzicon" + (isImported ? " folder-zone-imported" : ''), true);

            newG.appendChild(iconElt[0][0]);

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width}, 0)`,
                'data-height': coords.height
            }));

            if (!folder.description) {
                folder.description = "Managed folder";
            }
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.01,
                width: coords.width + 60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sanitize(folder.description.replace(/([-_.])/g, '$1\u200b')) + '</span></div>')));

            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                opacity: 0
            }));

            newG.appendChild(makeForeignObject({
                x: 48,
                y: -13,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Information sticker
            // To display when the folder has a short description
            if (angular.isDefined(folder.shortDesc) && folder.shortDesc.length > 0) {
                let coords = {
                    cx: 1,
                    cy: 55,
                    rx: 10,
                    ry: 10,
                    indicatorX: -9,
                    indicatorY: 45
                };

                drawInformationSticker(newG, coords);
                offset.y += 13;
            }

            // Discussion sticker
            // To display when the dataset has an unread discussion
            if (local && folder.discussionsFullIds && folder.discussionsFullIds.length > 0) {

                let unread = hasUnreadDiscussion(folder);
                let coords = {
                    cx: 0,
                    cy: 51.5,
                    rx: 10.5,
                    ry: 9,
                    indicatorX: -10,
                    indicatorY: 42,
                    contentX: -20,
                    contentY: 42
                };
                const isFirefox = navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
                drawDiscussionSticker(newG, folder.discussionsFullIds.length, unread, coords, offset, {x: 0, y: isFirefox ? -2 : 0});

            }

            $(dotPolygon).replaceWith(newG);
        }
    };
}

function restyleStreamingEndpointNode(local) {
    return function(element) {
        const nodeId = element.attr('data-id');
        const sm = FlowGraph.node(nodeId);

        const streamingEndpoint = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');
        if (dotPolygon.length) {
            element.find("text").remove();
            const coords = polygonToRectData(dotPolygon);
            const newG = makeSVG('g', {class: 'newG bzicon', transform: `translate(${coords.x} ${coords.y})`});

            newG.appendChild(makeSVG('path', {
                d: "M0,0 L48,0 L64,29 L48,58 L0,58 Z",
                opacity: 1,
                class: 'fill'
            }));

            const icon = $filter("toModernIcon")($filter("datasetTypeToIcon")(streamingEndpoint.streamingEndpointType, 48), 48);
            newG.appendChild(makeForeignObject({
                x: 5,
                y: 5,
                width: 48,
                height: 48,
                class: 'nodeicon'
            }, $(`<div class="flow-tile" style="text-align: center"><i class="${icon}" /></div>`)));

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${coords.width + 26}, 24)`,
                'data-height': coords.height
            }));

            newG.appendChild(makeSVG('path', {
                class: 'selection-outline',
                d: "M-1,-1 L48.5,-1 L65,29 L48.5,59 L-1,59 Z"
            }));

                // x: -coords.width*0.2071,
            newG.appendChild(makeForeignObject({
                x: -30,
                y: coords.height * 1.2,
                width: coords.width + 60,
                height: 42,
                class: 'nodelabel-wrapper'
            }, $('<div><span>' + sm.name.replace(/([-_.])/g, '$1\u200b') + '</span></div>')));

            newG.appendChild(makeForeignObject({
                x: 45,
                y: -10,
                width: 32,
                height: 32,
                class: 'node-totem'
            }, $(`<span size="32">`)));

            $(dotPolygon).replaceWith(newG);
        }
    }
}

function restyleRunnableNode(runnableType) {
    return function(element, g) {
        if (!(["recipe", "labeling_task"].includes(runnableType))) {return}
        const isRecipe = runnableType === 'recipe';
        const nodeId = element.attr('data-id');
        const runnable = FlowGraph.node(nodeId);

        const iconScale = 0.52

        const dotEllipse = element.find("ellipse");
        if (dotEllipse.length) {
            const coords = circleToRectData(dotEllipse);
            const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y}) scale(${iconScale}, ${iconScale})`});

            const recipeType = isRecipe ? runnable.recipeType : "labeling_task";
            const iconText = flowIconset.icons[$filter("recipeFlowIcon")(recipeType)];
            d3.select(newG).classed("bzicon recipeicon-" + recipeType, true);

            if (isRecipe && (runnable.recipeType.startsWith("CustomCode_") || runnable.recipeType.startsWith("App_"))) {
                const colorClass = $filter("recipeTypeToColorClass")(runnable.recipeType);
                d3.select(newG).classed("universe-fill " + colorClass, true);
            }

            // WARNING: this conflicts with graphviz layout: recipe names and other nodes can overlap.
            // newG.appendChild(makeForeignObject({
            //     x: -30,
            //     y: coords.height * 1.07,
            //     width: coords.width + 60,
            //     height: 45,
            //     class: 'nodelabel-wrapper',
            //     transform: `scale(${1/iconScale}, ${1/iconScale})`
            // }, $('<div><span>' + recipe.description.replace(/([-_])/g, '$1\u200b') + '</span></div>')));

            d3.select(g).attr("data-recipe-type", isRecipe ? runnable.recipeType : "Labeling");

            let iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("g");
            // Some icons don't have a g ...
            if (iconElt.length === 0 || iconElt[0].length === 0) {
                iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("path");
            }
            try {
                $(dotEllipse).replaceWith(newG);
                iconElt[0].forEach(function(x){
                    newG.appendChild(x);
                });
                // Invisible rect to capture mouse events, else
                // the holes in the icon don't capture the mouse
                newG.appendChild(makeSVG('rect', {
                    x: 0,
                    y: 0,
                    width: coords.width / iconScale,
                    height: coords.height / iconScale,
                    opacity: 0
                }));
            } catch (e) {
                logger.error("Failed patch recipe icon", e)
            }

            if (isRecipe && (runnable.recipeType.startsWith("CustomCode_") || runnable.recipeType.startsWith("App_"))){
                const icon = $filter("toModernIcon")($filter("recipeTypeToIcon")(runnable.recipeType, 48), 48);
                newG.appendChild(makeForeignObject({
                    x: 0,
                    y: 0,
                    width: SIZE,
                    height: SIZE,
                    class: 'nodeicon'
                }, $('<div class="recipe-custom-code-object"><i class="' + icon + '"></i></div>')));
            }

            newG.appendChild(makeSVG('g', {
                class: 'tool-simple-zone',
                transform: `translate(${SIZE}, 0) scale(${1/iconScale}, ${1/iconScale})`,
                'data-height': SIZE * iconScale
            }));

            newG.appendChild(makeSVG('circle', {
                class: 'selection-outline',
                r: 50*iconScale,
                cx: 50*iconScale,
                cy: 50*iconScale,
                transform: `scale(${1/iconScale}, ${1/iconScale})`,
            }));

            newG.appendChild(makeForeignObject({
                x: 30,
                y: -17,
                width: 32,
                height: 32,
                class: 'node-totem',
                transform: `scale(${1/iconScale}, ${1/iconScale})`
            }, $(`<span size="32">`)));

            let offset = {x: 0, y: 0};

            // Repeated object sticker
            if (runnable.veLoopDatasetRef) {
                let coords = {
                    cx: 14,
                    cy: 86,
                    rx: 10 / iconScale,
                    ry: 10 / iconScale,
                    indicatorX: -1.5,
                    indicatorY: 76,
                };

                drawRepeatedObjectSticker(newG, coords);
                offset.x += 11;
                offset.y += 34;
            }

            // Information sticker
            // To display when the recipe has a short description
            if (angular.isDefined(runnable.shortDesc) && runnable.shortDesc.length > 0) {
                let coords = {
                    cx: 14,
                    cy: 86,
                    rx: 10 / iconScale,
                    ry: 10 / iconScale,
                    indicatorX: -0.5,
                    indicatorY: 69,
                };

                drawInformationSticker(newG, coords, offset);
                if (offset.x || offset.y) {
                    // We've drawn the repeated object sticker too
                    offset.x -= 10;
                    offset.y += 36.5;
                } else {
                    offset.x += 11;
                    offset.y += 34;
                }
            }

            // Discussion sticker
            // To display when the recipe has an unread discussion
            if (runnable.discussionsFullIds.length > 0) {
                let unread = hasUnreadDiscussion(runnable);
                let coords = {
                    cx: 7.5,
                    cy: 90.5,
                    rx: 10.5 / iconScale,
                    ry: 9 / iconScale,
                    indicatorX: -10,
                    indicatorY: 75,
                    contentX: -12,
                    contentY: 80,
                };
                drawDiscussionSticker(newG, runnable.discussionsFullIds.length, unread, coords, offset);
            }
        }
    }
}
});

app.service('ProjectFlowGraphLayout', function(FlowGraph, LoggerProvider) {

    var logger = LoggerProvider.getLogger('flow');

    // This function takes isolated ("draft") computables
    // and move them to a container to put them next to the flow
    // Note that things are called datasets for historical reasons but they might be other things
    this.relayout = function(globalSvg) {
        const zones = $(globalSvg).find("svg");
        if (zones && zones.length > 0) {
            for (let i = 0; i < zones.length; i++) {
                this.relayoutSVG($(zones[i]), true);
            }
        } else {
            this.relayoutSVG(globalSvg, false);
        }
    };

    this.relayoutSVG = (svg, zones) => {
        const usedDatasets = makeSVG('g', { class: 'usedDatasets' });
        const inputDatasets = makeSVG('g', { class: 'inputDatasets' });
        const draftDatasets = makeSVG('g', { class: 'draftDatasets' });
        svg.find('g[class=graph]').append(usedDatasets);
        $(usedDatasets).append(inputDatasets);
        svg.find('g[class=graph]').append(draftDatasets);

        svg.find('g[data-type]').each(function (index, boxElement) {
            const nodeId = $(boxElement).data("id");
            const node = FlowGraph.node(nodeId);
            if(!node) {
                logger.warn('Graph node does not exist: ', nodeId)
                return;
            }

            if(!node.predecessors.length && !node.successors.length) {
                draftDatasets.appendChild(boxElement);
            } else if(!node.predecessors.length) {
                inputDatasets.appendChild(boxElement);
            } else {
                usedDatasets.appendChild(boxElement);
            }
        });

        // Relayout
        const hasChildNodes = usedDatasets.childNodes.length + inputDatasets.childNodes.length > 1;
        let datasetsPerColumns = Math.max(inputDatasets.childNodes.length, Math.floor(Math.sqrt(draftDatasets.childNodes.length)), 1);
        let gridWidth = 180;
        let gridHeight = 180;
        let columnHeight = gridHeight * datasetsPerColumns;
        if(inputDatasets.childNodes.length && !zones) {
            datasetsPerColumns = inputDatasets.childNodes.length;
            gridWidth = inputDatasets.childNodes[0].getBBox().width + 50;
            columnHeight = inputDatasets.getBBox().height;
            gridHeight = columnHeight / datasetsPerColumns;
        }
        let nbFullRows = Math.floor(draftDatasets.childNodes.length / datasetsPerColumns) || 1;
        for(let index = 0; index < draftDatasets.childNodes.length; index++) {
            // full row
            let offset = 0;
            if (nbFullRows > 0 && index >= nbFullRows * datasetsPerColumns) {
                offset = (columnHeight - gridHeight * (draftDatasets.childNodes.length % datasetsPerColumns))/2;
            }
            let dx = Math.floor(index / datasetsPerColumns) * gridWidth + 20;
            let dy = offset + index % datasetsPerColumns * gridHeight + 50;
            $(draftDatasets).children().eq(index).find('g').first().attr('transform', `translate(${dx}  ${dy})`);
        }

        //  move draftDatasets out of the way
        setTimeout(function() {
            let usedDatasetsBB = usedDatasets.getBBox();
            let draftDatasetsBB = draftDatasets.getBBox();
            let translateX = usedDatasetsBB.x - draftDatasetsBB.x - draftDatasetsBB.width - 200;
            let translateY = usedDatasetsBB.y - draftDatasetsBB.y;
            if (!zones) {
                draftDatasets.setAttribute('transform', 'translate(' + translateX + ' ' + translateY + ')');
            } else {
                let translateGraphY = hasChildNodes && draftDatasets.childNodes.length ? -usedDatasetsBB.y + 32 : hasChildNodes ? -usedDatasetsBB.y : -draftDatasetsBB.y;
                svg.find('g[class=graph]')[0].setAttribute('transform', 'translate(32,' + translateGraphY + ')');
            }
        });
    }
});


app.service('InterProjectGraphLayout', function(FlowGraph) {
    // Similar to project flow, this function takes isolated ("standalone") projects
    // and move them to a container to put them below the graph of connected (by exposed elements) projects
    this.relayout = function(svg) {
        let connectedProjects = makeSVG('g', { class: 'connectedProjects' });
        let inputProjects = makeSVG('g', { class: 'inputProjects' });
        let standAloneProjects = makeSVG('g', { class: 'standAloneProjects' });
        let projectFolders = makeSVG('g', { class: 'projectFolders' });

        svg.find('g[class=graph]').append(projectFolders);
        svg.find('g[class=graph]').append(connectedProjects);
        $(connectedProjects).append(inputProjects);
        svg.find('g[class=graph]').append(standAloneProjects);

        svg.find('g[data-type]').each(function (index, boxElement) {
            let node = FlowGraph.node($(boxElement).data("id"));
            // if its a draft dataset, move it to another elt
            if (node.nodeType == "PROJECT_FOLDER") {
                projectFolders.appendChild(boxElement);
            } else if (!node.predecessors.length && !node.successors.length) {
                standAloneProjects.appendChild(boxElement);
            } else if (!node.predecessors.length) {
                inputProjects.appendChild(boxElement);
            } else {
                connectedProjects.appendChild(boxElement);
            }
        });

        // Relayout
        let minProjectsPerRow = 3;
        let maxProjecsPerRow = 10;

        let projectsPerRows = Math.min(Math.max(minProjectsPerRow, Math.floor(Math.sqrt(standAloneProjects.childNodes.length))), maxProjecsPerRow);
        let cellMargin = 20
        let cellWidth = 150 + cellMargin;
        let cellHeight = 100 + cellMargin;
        let pfCellHeight = 40 + cellMargin;
        let rowWidth = cellWidth * projectsPerRows;
        if (inputProjects.childNodes.length) {
            cellWidth = inputProjects.childNodes[0].getBBox().width + 50;
            cellHeight = inputProjects.childNodes[0].getBBox().height + cellMargin;
            projectsPerRows = Math.min(Math.max(minProjectsPerRow, Math.floor(connectedProjects.getBBox().width / cellWidth)), maxProjecsPerRow);
        }
        if (projectFolders.childNodes.length) {
            pfCellHeight = projectFolders.childNodes[0].getBBox().height + cellMargin;
        }
        for(let index = 0; index < standAloneProjects.childNodes.length; index++) {
            let tr = (index % projectsPerRows) * cellWidth +' '+ Math.floor(index / projectsPerRows) * cellHeight;
            $(standAloneProjects).children().eq(index).find('g').first()
                .attr('transform', 'translate('+ tr +')');
        }
        for (let index = 0; index < projectFolders.childNodes.length; index++) {
            let tr = (index % projectsPerRows) * cellWidth + ' ' + Math.floor(index / projectsPerRows) * pfCellHeight;
            $(projectFolders).children().eq(index).find('g').first()
                .attr('transform', 'translate(' + tr + ')');
        }
        //  move standAloneProjects out of the way
        let connectedProjectsBB = connectedProjects.getBBox();
        let standAloneProjectsBB = standAloneProjects.getBBox();
        let sapTranslateX = connectedProjectsBB.x;
        let sapTranslateY = connectedProjectsBB.y + connectedProjectsBB.height + 100;
        standAloneProjects.setAttribute('transform', 'translate(' + sapTranslateX + ' ' + sapTranslateY + ')');
        // move projectFolders out of the way (upside)
        let projectFoldersBB = projectFolders.getBBox();
        let pfTranslateX = connectedProjectsBB.x;
        let pfTranslateY = connectedProjectsBB.y - (projectFoldersBB.height + 100);
        projectFolders.setAttribute('transform', 'translate(' + pfTranslateX + ' ' + pfTranslateY + ')');
    };
});


app.service('InterProjectGraphStyling', function($filter, ImageUrl, FlowGraph, CachedAPICalls, LoggerProvider) {

    const SIZE = 100;

    const logger = LoggerProvider.getLogger('projectsGraph');

    const formatters = {
        'PROJECT_FOLDER': restyleNodeForProjectFolder,
        'PROJECT': restyleNodeForProject,
        'BUNDLE_EO': restyleNodeForExposedObject,
    };
    let flowIconset;
    CachedAPICalls.flowIcons.success(function(data) {
        flowIconset = data;
    });

    this.restyleGraph = function(svg, graph) {
        svg.find('title').remove();
        svg.find('g[data-type]').each(function (index, g) {
            try {
                restyleNode(g);
            } catch (e) {
                logger.error("Failed to restyle flow node: ", e);
            }
        });
    };

    function restyleNode(g) {
        const element = $(g);
        const nodeType = element.attr('data-type');

        if (formatters[nodeType]) {
            formatters[nodeType](element, g);
        }
    };

    function createExposedObjectSvg(nodeIcons, index, x, y, d) {
        const g = makeSVG('g', {
            class: nodeIcons[index].type,
            transform: 'translate(' + x + ' ' + y + ') scale(' + d/SIZE +',' + d/SIZE +')'
        });
        nodeIcons[index].elt[0].forEach(function(x){
            g.appendChild(x);
        });
        addCountToExposedObjectSvg(g, nodeIcons, index, x, y, d)
        return g
    }

    function addCountToExposedObjectSvg(g, nodeIcons, index, x, y, d) {
        const r = SIZE / 5;
        const cx = SIZE/2 + SIZE/(2 * Math.sqrt(2));
        const cy = SIZE/2 + SIZE/(2 * Math.sqrt(2));

        const circle = makeSVG('circle', {
            r: r,
            cx: cx,
            cy: cy,
            class: 'count-circle'
        });

        const text = makeSVG('text', {
            x: cx,
            y: cy,
            class: "count-text"
        });
        text.textContent = nodeIcons[index].nbElements;

        g.appendChild(circle);
        g.appendChild(text);
    }

    // several exposed objects (of different type) will result in one nodes containing several icons
    function drawExposedObjectNodeIcons(newG, nodeIcons, coords) {
        switch (nodeIcons.length) {
            case 1: {
                const g0 = createExposedObjectSvg(nodeIcons, 0, 1, 1, coords.width - 2);
                newG.appendChild(g0);
                break;
            }
            case 2: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1 + coords.height/4;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1 + coords.width/2;
                const y1 = 1 + coords.height/4;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                break;
            }
            case 3: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1 + coords.height/4;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1 + coords.width/2;
                const y1 = 1;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/2;
                const y2 = 1 + coords.height/2;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                break;
            }
            case 4: {
                const diameter = (coords.width - 2)/2;

                const x0 = 1;
                const y0 = 1;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1;
                const y1 = 1 + coords.height/2;;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/2;
                const y2 = 1;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                const x3 = 1 + coords.width/2;
                const y3 = 1 + coords.height/2;
                const g3 = createExposedObjectSvg(nodeIcons, 3, x3, y3, diameter);
                newG.appendChild(g3);

                break;
            }
            case 5: {
                const diameter = (coords.width - 2)/3;

                const x0 = 1 + coords.width/3;
                const y0 = 1 ;
                const g0 = createExposedObjectSvg(nodeIcons, 0, x0, y0, diameter);
                newG.appendChild(g0);

                const x1 = 1;
                const y1 = 1 + coords.height/3;
                const g1 = createExposedObjectSvg(nodeIcons, 1, x1, y1, diameter);
                newG.appendChild(g1);

                const x2 = 1 + coords.width/3;
                const y2 = 1 + coords.height/3;
                const g2 = createExposedObjectSvg(nodeIcons, 2, x2, y2, diameter);
                newG.appendChild(g2);

                const x3 = 1 + coords.width * 2/3;
                const y3 = 1 + coords.height/3;
                const g3 = createExposedObjectSvg(nodeIcons, 3, x3, y3, diameter);
                newG.appendChild(g3);

                const x4 = 1 + coords.width/3;
                const y4 = 1 + coords.height * 2/3;
                const g4 = createExposedObjectSvg(nodeIcons, 4, x4, y4, diameter);
                newG.appendChild(g4);
            }
        }
    }

    function restyleNodeForProjectFolder(element) {
        const nodeId = element.attr('data-id');
        const projectFolder = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');

        //replacing polygon by g of same size
        const coords = polygonToRectData(dotPolygon);
        const newG = makeSVG('g', {class: 'newG', transform: 'translate(' + coords.x + ' ' + coords.y + ')'});
        $(dotPolygon).replaceWith(newG);
        element.find("text").remove();

        // fill
        newG.appendChild(makeSVG('rect', {
            x: 0,
            y: 0,
            width: coords.width,
            height: coords.height,
            class: 'fill'
        }));

        // html-content
        newG.appendChild(makeForeignObject(
            {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                class: 'project-folder-meta nodelabel-wrapper'
            },
            $(
                '<div>' +
                    '<p class="single-line"><i class="icon-folder-close"></i>' + sanitize(projectFolder.description.replace(/([-_.])/g, '$1\u200b')) + '</p>' +
                '</div>'
            )
        ));
    }

    function restyleNodeForProject(element) {
        const nodeId = element.attr('data-id');
        const project = FlowGraph.node(nodeId);
        const dotPolygon = element.find('polygon');

        //replacing polygon by g of same size
        const coords = polygonToRectData(dotPolygon);
        const newG = makeSVG('g', {class: 'newG',
            transform: 'translate(' + coords.x + ' ' + coords.y + ')'
        });
        if (project.isArchived) {
            d3.select(newG).classed("archived", true);
        }
        if (project.isForbidden) {
            d3.select(newG).classed("forbidden", true);
        }
        if (project.isNotInFolder) {
            d3.select(newG).classed("not-in-folder", true);
        }
        $(dotPolygon).replaceWith(newG);
        element.find("text").remove();

        //fill
        newG.appendChild(makeSVG('rect', {
            x: 0,
            y: 0,
            width: coords.width,
            height: coords.height,
            class: 'fill'
        }));

        // html-content
        newG.appendChild(makeForeignObject(
            {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                class: 'project-meta nodelabel-wrapper'
            },
            $(
                '<div>' +
                    '<i class="img-area">' + (project.isForbidden ? '' : ('<img src="' + ImageUrl(project.name, project.projectKey, project.projectKey, "PROJECT", project.objectImgHash, "80x200", project.imgColor, project.imgPattern, project.showInitials) + '" />')) + '</i>' +
                    '<p>' + sanitize(project.description.replace(/([-_.])/g, '$1\u200b')) + '</p>' +
                '</div>'
            )
        ));

        if (project.projectAppType === 'APP_TEMPLATE') {
            const appIconSize = 25;
            newG.appendChild(makeForeignObject(
                {
                    x: 0,
                    y: coords.height - appIconSize,
                    width: appIconSize,
                    height: appIconSize,
                    class: 'project-meta'
                },
                $(`
                    <div class="app-template-overlay app-template-overlay--graph" title="${project.isAppAsRecipe ? 'Application-as-recipe' : 'Visual application'} template">
                        <i class="${project.isAppAsRecipe ? 'icon-dku-application-as-recipe' : 'icon-project-app'}"></i>
                    </div>
                `)
            ));
        }

        // forbidden-project (we can still see it because we can read exposed object from it)
        if (project.isForbidden) {
            newG.appendChild(makeForeignObject({
                x: 113,
                y: 64,
                width: 30,
                height: 30,
                class: 'forbidden-icon'
            }, $('<div><i class="icon-lock" /></div>')));

        }

        // forbidden-project (we can still see it because we can read exposed object from it)
        if (project.isNotInFolder) {
            newG.appendChild(makeForeignObject({
                x: 35,
                y: 63,
                width: 30,
                height: 30,
                class: 'not-in-folder-icon'
            }, $('<div><span class="icon-stack"><i class="icon-folder-close"></i><i class="icon-ban-circle icon-stack-base"></i></span></div>')));

        }
    }

    function restyleNodeForExposedObject(element) {
        const icons = [];
        function addIcon (type, nbElements) {
            const icon = {};
            const iconText = flowIconset.icons[type];
            let iconElt = d3.select($.parseXML(iconText)).select("svg").selectAll("circle, path, rect");
            icon.elt = iconElt;
            icon.type = type;
            icon.nbElements = nbElements;
            icons.push(icon);
        }

        function addIcons(bundle) {
            if (bundle.exposedDatasets.length > 0) {
                addIcon("eo_datasets", bundle.exposedDatasets.length);
            }
            if (bundle.exposedFolders.length > 0) {
                addIcon("eo_folders", bundle.exposedFolders.length);
            }
            if (bundle.exposedModels.length > 0) {
                addIcon("eo_models", bundle.exposedModels.length);
            }
            if (bundle.exposedNotebooks.length > 0) {
                addIcon("eo_notebooks", bundle.exposedNotebooks.length);
            }
            if (bundle.exposedWebApps.length > 0) {
                addIcon("eo_webapps", bundle.exposedWebApps.length);
            }
            if (bundle.exposedReports.length > 0) {
                // TODO there is no Flow icon named "eo_reports.svg", so will likely not work. But reports are not shown on Flow, so...
                addIcon("eo_reports", bundle.exposedReports.length);
            }
            if (bundle.errorMessages.length > 0) {
                // TODO there is no Flow icon named "error.svg", so will likely not work. But reports are not shown on Flow, so...
                addIcon("error", bundle.errorMessages.length);
            }
        }

        const nodeId = element.attr('data-id');
        const bundle = FlowGraph.node(nodeId);

        const dotEllipse = element.find("ellipse");

        const coords = circleToRectData(dotEllipse);
        const newG = makeSVG('g', {class: 'newG', transform: `translate(${coords.x} ${coords.y})` });
        d3.select(newG).classed("bzicon", true);

        addIcons(bundle);
        if (icons.length == 0) {
            throw new Error("No icons for type", bundle);
        }
        drawExposedObjectNodeIcons(newG, icons, coords);

        try {
            $(dotEllipse).replaceWith(newG);
            // Invisible rect to capture mouse events, else
            // the holes in the icon don't capture the mouse
            newG.appendChild(makeSVG('rect', {
                x: 0,
                y: 0,
                width: coords.width,
                height: coords.height,
                opacity: 0
            }));
        } catch (e) { /* Nothing for now */ }

        const text = element.find('text:not(.count-text)');
        text.remove();
    }
});


app.service('FlowGraphFiltering', function() {
    function getBBoxAccountingForTranslation(svgElt) {
        if (!svgElt.hasAttribute("transform")) {
            return svgElt.getBBox();
        }

        const parent = svgElt.parentNode;
        const tmpGElt = makeSVG('g');
        parent.insertBefore(tmpGElt, svgElt);
        tmpGElt.appendChild(svgElt)
        const bbox = tmpGElt.getBBox();
        parent.insertBefore(svgElt, tmpGElt);
        tmpGElt.remove();
        return bbox;
    }

    //TODO @flow move to another service
    this.getBBoxFromSelector = function (globalSvg, selector) {
        const svgs = $(selector).closest('svg'); // Retrieve correct svg (Mostly in case of zones)
        let gTopLeft, gBottomRight;
        svgs.each(function() {
            const svg = $(this);
            let topLeft, bottomRight;
            function addItemToBBox(refBBox) {
                return function() {
                    let bbox = getBBoxAccountingForTranslation(this);
                    if (refBBox && svg.is(globalSvg)) {
                        // "draftDatasets" (unconnected objects) have been translated,
                        // We need to compensate for that (Only when no zones are present)
                        bbox = {
                            x: bbox.x + refBBox.x,
                            y: bbox.y + refBBox.y,
                            width: bbox.width,
                            height: bbox.height,
                        };
                    }

                    if (topLeft === undefined) {
                        topLeft = svg[0].createSVGPoint();
                        topLeft.x = bbox.x;
                        topLeft.y = bbox.y;

                        bottomRight = svg[0].createSVGPoint();
                        bottomRight.x = bbox.x + bbox.width;
                        bottomRight.y = bbox.y + bbox.height;
                    } else {
                        topLeft.x = Math.min(topLeft.x, bbox.x);
                        topLeft.y = Math.min(topLeft.y, bbox.y);

                        bottomRight.x = Math.max(bottomRight.x, bbox.x + bbox.width);
                        bottomRight.y = Math.max(bottomRight.y, bbox.y + bbox.height);
                    }
                }
            }
            const graphBBox = svg.find('g.graph')[0].getBBox();
            const isZone = $(selector).parentsUntil(svg, '.usedDatasets').length === 0 && $(selector).parentsUntil(svg, '.draftDatasets').length === 0;
            const matrix = isZone ? svg[0].getTransformToElement(globalSvg[0]) : $('.usedDatasets', svg)[0].getTransformToElement(globalSvg[0]);

            if (isZone) {
                svg.find(selector).each(addItemToBBox());
            } else {
                $('.usedDatasets', svg).find(selector).each(addItemToBBox());
                $('.draftDatasets', svg).find(selector).each(addItemToBBox(graphBBox));
            }

            if (topLeft === undefined) {
                //console.info("Cannot compute bounding box around empty set of items");
                return undefined;
            }

            topLeft = topLeft.matrixTransform(matrix);
            bottomRight = bottomRight.matrixTransform(matrix);

            if (gTopLeft === undefined) {
                gTopLeft = { x: topLeft.x, y: topLeft.y };
                gBottomRight = { x: bottomRight.x, y: bottomRight.y };
            } else {
                gTopLeft.x = Math.min(gTopLeft.x, topLeft.x);
                gTopLeft.y = Math.min(gTopLeft.y, topLeft.y);

                gBottomRight.x = Math.max(gBottomRight.x, bottomRight.x);
                gBottomRight.y = Math.max(gBottomRight.y, bottomRight.y);
            }
        });

        if (gTopLeft === undefined) {
            return undefined;
        }

        return {
            x: gTopLeft.x,
            y: gTopLeft.y,
            width: gBottomRight.x - gTopLeft.x,
            height: gBottomRight.y - gTopLeft.y
        };
    };

    this.fadeOut = function(svg, filter) {
        // Fade out nodes that need to
        if (filter && filter.doFading) {
            let d3node = d3.select(svg[0]);
            d3node.selectAll('.node').classed('filter-faded', true);
            d3node.selectAll('.edge').classed('filter-faded', true);

            $.each(filter.nonFadedNodes, function(idx, nodeId) {
                let elt = svg.find(' [data-id="' + nodeId + '"]')[0];
                if (elt == null) {
                    // maybe a saved model
                    let savedmodel_nodeId = 'savedmodel' + nodeId.substring('dataset'.length);
                    elt = svg.find(' [data-id="' + savedmodel_nodeId + '"]')[0];
                    if (elt == null) {
                        // maybe a model evaluation store
                        let modelevaluationstore_nodeId = 'modelevaluationstore' + nodeId.substring('dataset'.length);
                        elt = svg.find(' [data-id="' + modelevaluationstore_nodeId + '"]')[0];
                        if (elt == null) {
                            // or managed folder
                            let managedfolder_nodeId = 'managedfolder' + nodeId.substring('dataset'.length);
                            elt = svg.find(' [data-id="' + managedfolder_nodeId + '"]')[0];
                            if (elt == null) {
                                return;
                            }
                        }
                    }
                }
                d3.select(elt).classed('filter-faded', false);
            });
            $.each(filter.nonFadedEdges, function(idx, toNodeId) {
                svg.find(' [data-to="' + toNodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let savedmodel_nodeId = 'savedmodel' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + savedmodel_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let modelevaluationstore_nodeId = 'modelevaluationstore' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + modelevaluationstore_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
                let managedfolder_nodeId = 'managedfolder' + toNodeId.substring('dataset'.length);
                svg.find(' [data-to="' + managedfolder_nodeId + '"]').each(function () {
                    d3.select(this).classed('filter-faded', false);
                });
            });
        }
    };
});


function polygonToRectData(polygon) { // polygon is an svg element
    const points = $(polygon).attr('points').split(' ');
    // points = [top right, top left, bottom left, bottom right, top right]
    return {
        x: parseFloat(points[1].split(',')[0]),
        y: parseFloat(points[1].split(',')[1]),
        width: parseFloat(points[0].split(',')[0], 10) - parseFloat(points[1].split(',')[0], 10),
        height: parseFloat(points[2].split(',')[1], 10) - parseFloat(points[1].split(',')[1], 10)
    };
}

function circleToRectData(ellipse) {// ellipse is an svg element
    const el = $(ellipse);
    return {
        x: el.attr('cx') - el.attr("rx"),
        y: el.attr('cy') - el.attr("ry"),
        width: el.attr('rx') * 2,
        height: el.attr('ry') * 2
    };
}

function makeForeignObject(attrs, jq) {
    const el = makeSVG('foreignObject', attrs)
    $(el).append(jq);
    return el;
}

})();
