(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * ChartTypeChangeUtils service
     * Provides utils methods to compute chart type changes by retrieving information on given data/measures/dimensions.
     */
    app.factory('ChartTypeChangeUtils', function(CHART_TYPES) {
        const service = {
            isUsableAsUA: (measure) => {
                return measure.column != null && measure.type !== 'CUSTOM';
            },

            isUsableAsMeasure: (ua) => {
                return ua.type === 'NUMERICAL';
            },

            measureToUA: (measure) => {
                return {
                    column: measure.column,
                    type: measure.type,
                    chartDefAttributeName: measure.chartDefAttributeName
                };
            },

            uaToMeasure: (ua) => {
                return {
                    column: ua.column,
                    type: ua.type,
                    function: 'AVG',
                    chartDefAttributeName: ua.chartDefAttributeName
                };
            },

            uaDimToDim: (ua) => {
                return {
                    column: ua.column,
                    type: ua.type,
                    chartDefAttributeName: ua.chartDefAttributeName
                };
            },

            uaPairToDim: (uaPair) => {
                const ret = [];
                if (uaPair.uaXDimension.length) {
                    ret.push({
                        column: uaPair.uaXDimension[0].column,
                        type: uaPair.uaXDimension[0].type,
                        chartDefAttributeName: 'uaXDimension'
                    });
                }
                if (uaPair.uaYDimension.length) {
                    ret.push({
                        column: uaPair.uaYDimension[0].column,
                        type: uaPair.uaYDimension[0].type,
                        chartDefAttributeName: 'uaYDimension'
                    });
                }
                return ret;
            },



            /**
             * Return an array of all "keys" objects and empty them in "object"
             * @param {Record<string, unknown>} object
             * @param {string[]} keys
             * @returns {unknown[]}
             */
            takeAllAndEmpty: (object, keys) => {
                if (!object) {
                    return [];
                }
                let response = [];
                for (const key of keys) {
                    if (!object[key]) {
                        continue;
                    }
                    // Addition of the chartDefAttributeName in each object allowing to retrieve these data more efficiently after (
                    response = [
                        ...response,
                        ...object[key].map(v => ({ ...v, chartDefAttributeName: key }))
                    ];
                    object[key] = [];
                }

                return response;
            },

            getChartDefMeasuresProperties: () => {
                return [
                    'genericMeasures',
                    'colorMeasure',
                    'sizeMeasure',
                    'xMeasure',
                    'yMeasure',
                    'tooltipMeasures'
                ];
            },

            getChartDefDimensionsProperties: () => {
                return [
                    'genericDimension0',
                    'boxplotBreakdownDim',
                    'genericDimension1',
                    'xDimension',
                    'yDimension',
                    'groupDimension',
                    'animationDimension',
                    'facetDimension'
                ];
            },

            takeAllMeasures: (chartDef) => {
                return service.takeAllAndEmpty(chartDef, service.getChartDefMeasuresProperties());
            },

            takeAllUA: (chartDef) => {
                return service.takeAllAndEmpty(chartDef, [
                    'uaSize',
                    'uaColor',
                    'uaTooltip',
                    'boxplotValue'
                ]);
            },

            takeAllMeasuresWithUA: (chartDef) => {
                return service.takeAllMeasures(chartDef)
                    .concat(service.takeAllUA(chartDef).filter(service.isUsableAsMeasure).map(service.uaToMeasure));
            },

            takeAllUAWithMeasures: (chartDef) => {
                return service.takeAllUA(chartDef)
                    .concat(service.takeAllMeasures(chartDef).filter(service.isUsableAsUA).map(service.measureToUA));
            },

            /**
             * Find and "steal" all existing dimensions from the chart.
             */
            takeAllExistingDimensions:(chartDef) => {
                // Order of props are important, else the logic after will reverse
                const dimensions = service.takeAllAndEmpty(chartDef, service.getChartDefDimensionsProperties());

                return [
                    ...dimensions,
                    ...service.takeAllAndEmpty(chartDef, [
                        'uaXDimension',
                        'uaYDimension'
                    ]).map(service.uaDimToDim),
                    ...[].concat(...service.takeAllAndEmpty(chartDef, [
                        'uaDimensionPair'
                    ]).map(service.uaPairToDim))
                ];
            },

            takeAllInX: (chartDef) => {
                switch (chartDef.type) {
                    case CHART_TYPES.STACKED_BARS:
                        return [chartDef.genericMeasures, chartDef.xMeasure, chartDef.uaXDimension, chartDef.xDimension];
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return chartDef.uaDimensionPair.reduce((acc, pair) => {
                            if (pair.uaXDimension && pair.uaXDimension.length) {
                                acc.push(pair.uaXDimension);
                            }
                            return acc;
                        }, []);
                    default:
                        return [chartDef.genericDimension0, chartDef.xMeasure, chartDef.uaXDimension, chartDef.xDimension];
                }
            },

            takeAllInY: (chartDef) => {
                switch(chartDef.type) {
                    case CHART_TYPES.BOXPLOTS:
                        return [chartDef.boxplotValue];
                    case CHART_TYPES.STACKED_BARS:
                        return [chartDef.genericDimension0, chartDef.yMeasure, chartDef.uaYDimension, chartDef.yDimension];
                    case CHART_TYPES.SCATTER_MULTIPLE_PAIRS:
                        return chartDef.uaDimensionPair.reduce((acc, pair) => {
                            if (pair.uaYDimension && pair.uaYDimension.length) {
                                acc.push(pair.uaYDimension);
                            }
                            return acc;
                        }, []);
                    default:
                        return [chartDef.genericMeasures, chartDef.yMeasure, chartDef.uaYDimension, chartDef.yDimension];
                }
            }
        };

        return service;
    });
})();
