(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedChartUtils', StackedChartUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/stacked-columns.js
     */
    function StackedChartUtils(ChartAxesUtils, ChartsStaticData, SVGUtils) {
        return {
            prepareData: function(chartDef, chartData, axisDim) {
                const colorLabels = chartData.getAxisLabels('color') || [null],
                    facetLabels = chartData.getAxisLabels('facet') || [null],
                    animationLabels = chartData.getAxisLabels('animation') || [null],
                    xLabels = chartData.getAxisLabels(axisDim || 'x'),
                    // only one axis at a time can display in logscale so that logic is enough
                    hasLogScale = chartDef.xAxisFormatting.isLogScale || ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, ChartsStaticData.LEFT_AXIS_ID);

                const animationData = { frames: [], maxTotal: 0, minTotal: 0 };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [], maxTotal: 0, minTotal: 0 };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { stacks: [], maxTotal: 0, minTotal: 0 };
                        xLabels.forEach(function(xLabel, x) {
                            chartData.fixAxis(axisDim || 'x', x);

                            let positiveTotal = hasLogScale ? 1 : 0;
                            let negativeTotal = 0;

                            let count = 0;
                            const stackData = [];

                            const totalSubTextElementsData = SVGUtils.getLabelSubTexts(chartDef, undefined, chartDef.stackedColumnsOptions ? chartDef.stackedColumnsOptions.totalsInChartDisplayOptions : chartDef.valuesInChartDisplayOptions, true).map(function(subTextElementData) {
                                const res = {
                                    ...subTextElementData,
                                    aggregationTotal: positiveTotal + negativeTotal
                                };
                                res[axisDim || 'x'] = x;
                                return res;
                            });

                            colorLabels.forEach(function(colorLabel, c) {
                                chartData.fixAxis('color', c);
                                totalSubTextElementsData.forEach(function(subTextElementData) {
                                    if (subTextElementData.aggregationIndex !== undefined) {
                                        // compute total for custom aggregations
                                        const d = chartData.aggr(subTextElementData.aggregationIndex).get();
                                        subTextElementData.aggregationTotal += d;
                                    }
                                });

                                chartDef.genericMeasures.forEach(function(measure, m) {
                                    const d = chartData.aggr(m).get();

                                    if (chartDef.variant == 'stacked_100' && d < 0) {
                                        throw new ChartIAE('Cannot represent negative values on a 100% Stacked chart. Please use another chart.');
                                    }

                                    let base, top;
                                    if (d >= 0) {
                                        base = positiveTotal;
                                        top = positiveTotal + d;
                                        positiveTotal = top;
                                    } else {
                                        base = negativeTotal;
                                        top = negativeTotal + d;
                                        negativeTotal = top;
                                    }

                                    const measureData = {
                                        color: c,
                                        measure: m,
                                        facet: f,
                                        animation: a,
                                        count: chartData.getNonNullCount({}, m),
                                        base,
                                        value: d,
                                        top
                                    };
                                    measureData[axisDim || 'x'] = x;

                                    const measureSubTextElementsData = SVGUtils.getLabelSubTexts(chartDef, m, measure.valuesInChartDisplayOptions, false).map(function(subTextElementData) {
                                        return {
                                            ...subTextElementData,
                                            // add measure data at sub text level also, to retrieve it easily
                                            ...measureData
                                        };
                                    });

                                    const point = {
                                        ...measureData,
                                        textsElements: measureSubTextElementsData,
                                        valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions
                                    };

                                    stackData.push(point);

                                    count += chartData.getNonNullCount({}, m);
                                });
                            });


                            if (chartDef.variant == 'stacked_100' && positiveTotal > 0) {
                                // Do a second pass and divide by total
                                let totalPercent = 0;
                                stackData.forEach(function(point, p) {
                                    const update = {
                                        value: point.value / positiveTotal,
                                        base: totalPercent,
                                        top: point.value / positiveTotal + totalPercent
                                    };
                                    Object.assign(point, update);
                                    // update data on sub texts as well
                                    point.textsElements.forEach(function(subTextElementData) {
                                        Object.assign(subTextElementData, update);
                                    });

                                    totalPercent += point.value;
                                });

                                positiveTotal = 1;
                            }

                            const totalData = { data: stackData, total: positiveTotal + negativeTotal, positiveTotal, negativeTotal, count,
                                spacing: chartDef.stackedColumnsOptions?.totalsInChartDisplayOptions?.spacing
                            };
                            facetData.stacks.push({ ...totalData, textsElements: totalSubTextElementsData.map(function(subTextElementData) {
                                return { ...subTextElementData, ...totalData };
                            }) });
                            facetData.maxTotal = Math.max(facetData.maxTotal, positiveTotal);
                            facetData.minTotal = Math.min(facetData.minTotal, negativeTotal);
                        });

                        frameData.maxTotal = Math.max(frameData.maxTotal, facetData.maxTotal);
                        frameData.minTotal = Math.min(frameData.minTotal, facetData.minTotal);
                        frameData.facets.push(facetData);
                    });

                    animationData.maxTotal = Math.max(animationData.maxTotal, frameData.maxTotal);
                    animationData.minTotal = Math.min(animationData.minTotal, frameData.minTotal);
                    animationData.frames.push(frameData);
                });

                return animationData;
            }
        };
    }

})();
