# encoding: utf-8
"""
Main entry point for containerized execution.
This will retrieve the definition of the stuff to be executed and execute it,
forking the process (in the image's code environment if necessary).
"""

import dataiku
from dataiku.base import remoterun
from dataiku.container.file_exchange import FileExchanger
from dataiku.core import debugging, intercom
from dataiku.base.utils import safe_unicode_str, tar_extractall
import sys, os, json, logging, tarfile, threading, traceback, subprocess, re, time, socket, errno
import os.path as osp
from dataiku.base.remoterun import safe_os_environ_update
import requests
import shutil
import tempfile
from dataiku.container.code_studio_server_thread import CommandServerThread

from dataiku.container.compute_resource_usage import ProcessResourceUsageMonitor

OPT_DIR = '/opt/dataiku'
HOME_DIR = '/home/dataiku'
# list of env vars that runner.py will potentially modify, and that are passed from runner.py to the subprocess
MODIFIED_VARS = ["PYTHONPATH", "DKU_CURRENT_PROJECT_KEY", "DKU_SOURCE_LIB_R_PATH", "R_LIBS", "DKU_CODE_STUDIO_BROWSER_PATH"]
TAR_BUFFER_SIZE = 10 * 16 * 1024

def use_syspath_for_subprocess_pythonpath(chunks):
    """
    Use the current Python interpreter sys.path to rebuild a PYTHONPATH relevant for subprocess execution,
        usually after having called load_libs() to add its path

    Calling `read_dku_env_and_set()` will overwrite the current process environment variables with the ones from
        "remote-run-env-def.json". That will also overwrite the current `PYTHONPATH`.
        Some subprocesses need to access the PYTHONPATH when loading libs, running other processes
        (e.g: pytorch via deephub), ...
    """
    filtered_sys_path = [p for p in sys.path if p.startswith(HOME_DIR) or p.startswith(OPT_DIR)]
    # PBE: patch for mixed python version support
    # THIS SHOULD BE REWORKED
    filtered_sys_path = [p for p in filtered_sys_path if not p.startswith(sys.prefix) and not p.startswith(sys.exec_prefix)]
    filtered_sys_path.extend(chunks)
    os.environ['PYTHONPATH'] = ':'.join(filtered_sys_path)
    logging.info("running on python {}".format(sys.version))
    logging.info("now using PYTHONPATH: {}".format(os.environ['PYTHONPATH']))


def setup_log():
    logging.basicConfig(level=logging.DEBUG, format='[%(asctime)s] [%(process)s/%(threadName)s] [%(levelname)s] [%(name)s] %(message)s')
    debugging.install_handler()

def setup_os_environ():
    with open("remote-run-env-def.json", 'r') as fd:
        dku_exec_env = json.load(fd)
    dku_vars = dku_exec_env.get("env", {})
    # remove the vars that we tampered with
    for k in MODIFIED_VARS:
        if k in dku_vars:
            del dku_vars[k]
    safe_os_environ_update(dku_vars)

def _atomic_stream_extractall(rawresponse, dest):
    dest = os.path.abspath(dest)
    parent_dir = os.path.dirname(dest)
    if not os.path.isdir(parent_dir):
        os.makedirs(parent_dir)
    tmp_dir = tempfile.mkdtemp(dir=parent_dir)

    with tarfile.open(fileobj=rawresponse, mode='r|', bufsize=TAR_BUFFER_SIZE) as tar:
        tar_extractall(tar, tmp_dir)

    if os.path.isdir(dest):
        shutil.rmtree(dest, ignore_errors=True)
    elif os.path.exists(dest):
        os.remove(dest)

    os.renames(tmp_dir, dest)

def fetch_libs(execution_id, scope):
    logging.info("Fetching %s libraries" % scope)
    data = {'executionId': execution_id, 'fileKind': '%s_LIB' % scope.upper(), 'compress': False}
    resp = intercom.jek_or_backend_post_call('containers/get-file', stream=True, data=data)

    if resp.status_code == 200:
        _atomic_stream_extractall(resp.raw, '%s/lib/%s' % (HOME_DIR, scope))
    elif resp.status_code != 404:
        raise Exception("Error fetching %s libraries, HTTP status %d: %s" % (scope, resp.status_code, resp.text))
    else:
        logging.warning("Could not fetch %s libraries, not found" % scope)


def fetch_plugin(execution_id, plugin_id):
    logging.info("Fetching plugin %s" % plugin_id)
    data = {'executionId': execution_id, 'fileKind': 'PLUGIN', 'pluginId': plugin_id, 'compress': False}
    resp = intercom.jek_or_backend_post_call('containers/get-file', stream=True, data=data)

    if resp.status_code == 200:
        _atomic_stream_extractall(resp.raw, '%s/plugin' % HOME_DIR)
    elif resp.status_code != 404:
        raise Exception("Error fetching %s plugin, HTTP status %d: %s" % (plugin_id, resp.status_code, resp.text))
    else:
        logging.warning("Could not fetch %s plugin, not found" % plugin_id)


def fetch_dir(execution_id, path, dest=None, file_kind='CONTEXT_DIR'):
    FileExchanger(execution_id, file_kind).fetch_dir(path, dest)


def read_version():
    with open("%s/dss-version.json" % OPT_DIR, 'r') as fd:
        return json.load(fd)

# for sub processes
def read_execution():
    with open("%s/execution.json" % HOME_DIR, 'r') as fd:
        return json.load(fd)

def load_libs(with_instance_libs=True, with_project_libs=True, with_plugin_libs=True):
    if with_instance_libs:
        lib_dir = "/lib/instance"
        if os.path.isdir(HOME_DIR + lib_dir):
            sys.path.append(HOME_DIR + lib_dir)

    if with_plugin_libs:
        lib_dir = "/plugin/python-lib"
        if os.path.isdir(HOME_DIR + lib_dir):
            sys.path.append(HOME_DIR + lib_dir)

    if with_project_libs:
        with open(osp.join(HOME_DIR, "lib", "project", "project-lib-paths.json")) as f:
            project_libs_paths = json.load(f)
        for lib_subpath in project_libs_paths["pythonPath"]:
            full_path = osp.join(HOME_DIR, "lib", "project", lib_subpath)
            logging.debug("Adding %s to Pythonpath", full_path)
            sys.path.append(full_path)
        # in case of an empty project_libs_paths["pythonPath"], we suspect it's a project
        # where the autocompleteProjectLibrariesContent_NT() never ran (= you never looked
        # at the libraries section). In order to still handle the project libraries in
        # code studios, we make a bet that the python folder will be created soon
        if len(project_libs_paths["pythonPath"]) == 0 and "DKU_CURRENT_PROJECT_KEY" in os.environ:
            logging.info("Expecting <project>/lib/python to be added at some point")
            full_path = osp.join(HOME_DIR, "lib", "project", "project-python-libs", os.environ.get("DKU_CURRENT_PROJECT_KEY"), "python")
            logging.debug("Adding %s to Pythonpath", full_path)
            sys.path.append(full_path)


def send_container_done(execution_id):
    """
    For executions that may send heavy results (e.g. trained model files), manually send to the backend that the
    container execution has completed.
    """
    try:
        intercom.jek_or_backend_void_call('containers/put-file',
                                          params={
                                              'executionId': execution_id,
                                              'fileKind': 'EXECUTION_DIR',
                                              'path': 'container_done.txt'
                                          },
                                          data="1")
    except Exception as e:
        logging.error("Could not send result: %s" % e)


def set_env_for_r_libs():
    chunks = []

    with open(osp.join(HOME_DIR, "lib", "project", "project-lib-paths.json")) as f:
        project_libs_paths = json.load(f)
    for lib_subpath in project_libs_paths["rsrcPath"]:
        full_path = osp.join(HOME_DIR, "lib", "project", lib_subpath)
        logging.debug("Adding %s to R source path", full_path)
        chunks.append(full_path)

    for lib_dir in ["/lib/instance", "/plugin/R-lib"]:
        if os.path.isdir(HOME_DIR + lib_dir):
            chunks.append(HOME_DIR + lib_dir)

    os.environ["DKU_SOURCE_LIB_R_PATH"] = ":".join(chunks)


def remove_code_env_resources_env_vars_from_dku_exec_env(exec_env):
    """
    Remove environment variables from the code env resources_env.json if it exists, from the dku_exec_env,
    as they are already set in the Dockerfile
    """
    resources_env_vars_path = osp.join(OPT_DIR, "code-env", "resources_env.json")

    if not osp.exists(resources_env_vars_path):
        return

    with open(resources_env_vars_path, 'r') as f:
        resources_env_vars = json.load(f).get("variables", {})

    env = exec_env.get("env", {})
    for resources_env_var, _ in resources_env_vars.items():
        if resources_env_var in env:
            del env[resources_env_var]

def run_subprocess(execution_id, command, path = 'error.json',
                   send_error_json = True, fail_if_subprocess_failed = False):
    # do exactly what subprocess.call() would do, but get the PID. Also make it python2-friendly
    p = subprocess.Popen(command, shell=True, stdin=subprocess.PIPE, env=dict(os.environ, LC_ALL='en_US.utf8', DKU_CONTAINER_EXEC='1'))
    try:
        logging.info("Running subprocess in PID %s" % p.pid)
        # spawn a thread to monitor the usage of the subprocess
        monitor = ProcessResourceUsageMonitor(command, p.pid)
        monitor.daemon = True
        monitor.start()
        try:
            error_code = p.wait()
        except:
            p.kill()
            p.wait()
            raise
        finally:
            monitor.stop()
            monitor.join()
    finally:
        # close everything, like Popen would do (in python3)
        # when you arrive here, p.wait() has already been done
        if p.stdout:
            p.stdout.close()
        if p.stderr:
            p.stderr.close()
        # Flushing a BufferedWriter may raise an error
        if p.stdin:
            p.stdin.close()


    return handle_subprocess_return_value(execution_id, error_code, path, send_error_json, fail_if_subprocess_failed)

def run_popen(command, **extra_env):
    env = dict(os.environ, LC_ALL='en_US.utf8', DKU_CONTAINER_EXEC='1')
    env.update(extra_env)
    p = subprocess.Popen(command, shell=True, env=env)
    logging.info("Running subprocess in PID %s" % p.pid)
    # spawn a thread to monitor the usage of the subprocess
    # but unlike run_subprocess we can't wait on its proper termination
    monitor = ProcessResourceUsageMonitor(command, p.pid)
    monitor.daemon = True
    monitor.start()
    return p

def handle_subprocess_return_value(execution_id, error_code, path = 'error.json',
                   send_error_json = True, fail_if_subprocess_failed = False):
    if os.WIFEXITED(error_code):
        status = "exited with status"
        error_code = os.WEXITSTATUS(error_code)
    elif os.WIFSTOPPED(error_code):
        status = "stopped by signal"
        error_code = os.WSTOPSIG(error_code)
    elif os.WIFSIGNALED(error_code):
        status = "terminated by signal"
        error_code = os.WTERMSIG(error_code)
        if error_code == 9:
            # Supposedly didn't get the opportunity to send a structured error
            send_error_json = True
    else:
        status = "finished with code"
    status = "Containerized process %s %d" % (status, error_code)
    if error_code == 0:
        logging.info(status)
    else:
        logging.error(status)

    if send_error_json:
        if error_code != 0 and not os.path.isfile('error.json'):
            # Subprocess has not created an error file but I am supposed to send one, so craft it
            message = "Containerized process execution failed, return code %d" % error_code
            if error_code == 9:
                message = message + " (killed - maybe out of memory?)"
            with open("error.json", 'w') as fd: # 'w' OK because we use json.dump
                json.dump({
                    "errorType" : safe_unicode_str("SubProcessFailed"),
                    "message" : safe_unicode_str(message)
                }, fd)
        if os.path.isfile('error.json'):
            logging.info("Sending error.json to backend/JEK")
            try:
                logging.info("Clear remote DSS session before sending error.json")
                intercom.clear_remote_dss()
                with open('error.json', 'rb') as f:
                    intercom.jek_or_backend_void_call('containers/put-file',
                          params={'executionId': execution_id, 'fileKind': 'EXECUTION_DIR', 'path': path},
                          data=f)
            except Exception as e:
                logging.error("Could not send error to backend: %s" % e)

    if error_code != 0 and fail_if_subprocess_failed:
        logging.error("Failing container because subprocess failed (code %s)" % error_code)
        sys.exit(error_code)

def run_metrics_server():
    def serve_metrics():
        command = "/opt/dataiku/bin/python -m dataiku.webapps.metrics"
        while True:
            error_code = subprocess.call(command, shell=True, env=dict(os.environ, LC_ALL='en_US.utf8', DKU_CONTAINER_EXEC='1'))
            logging.info("Metrics server returned %s" % error_code)

    t = threading.Thread(target=serve_metrics)
    t.daemon = True
    t.start()

def build_backend_url():
    backend_host = remoterun.get_env_var("DKU_BACKEND_HOST", "127.0.0.1")
    try:
        import socket
        backend_host = socket.gethostbyname(backend_host)
    except Exception as e:
        logging.warn('Failed to resolve backend host %s : %s' % (backend_host, str(e)))
    backend_port = remoterun.get_env_var("DKU_BACKEND_PORT")
    backend_protocol = remoterun.get_env_var("DKU_BACKEND_PROTOCOL")
    backend_url = '%s://%s:%s' % (backend_protocol, backend_host, backend_port)

    return backend_url

def build_nginx_conf(servers):
    template = """
# This file is automatically generated.
  
error_log stderr;
pid "nginx.pid";
daemon off;
working_directory "nginx";

events {
    worker_connections 1000;
}

http {
    gzip on;
    gzip_types text/javascript application/json text/css image/svg+xml;
    client_max_body_size 0;
    large_client_header_buffers 8 64k;
    server_tokens off;

    types {
        text/html       html htm shtml;
        text/css        css;
        text/javascript js;
        text/css        less;
        audio/mpeg      mp3;
        image/svg+xml   svg;
        application/pdf pdf;
    }

    # define the format so to never expose remote_user which may contains API keys
    log_format anonymous '$remote_addr - - [$time_local] ' '"$request" $status $body_bytes_sent ' '"$http_referer" "$http_user_agent"';
    
    access_log "nginx/access.log" anonymous;
    client_body_temp_path "nginx";
    proxy_temp_path "nginx";
    # Define these even if we don't use them to avoid permission issues
    fastcgi_temp_path "nginx";
    scgi_temp_path "nginx";
    uwsgi_temp_path "nginx";

    proxy_http_version 1.1;
    proxy_next_upstream off; # Don't retry
    proxy_read_timeout 3600; # We have long queries
    proxy_set_header Host $http_host;
    proxy_set_header Upgrade $http_upgrade;
    proxy_set_header Connection "upgrade";
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;

    %s
}

""" % servers
    return template

def start_nginx_for_webapp(ping_path, backend_check_access, redirect_url, hide_access_token, access_token_cookie_name, project_key, web_app_id, web_app_type, outside_port, inside_port):
    backend_url = build_backend_url()
    nginx_conf = os.path.join(HOME_DIR, 'nginx.conf')

    ping_location = """
        location = %s {
          # the ping path is not authenticated (should be a safe item, like css)
          proxy_pass http://127.0.0.1:%s%s;
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          error_page 502 @error502_location;
        }
""" % (ping_path, inside_port, ping_path)

    if hide_access_token:
        access_token_purge = """
          set $new_cookie $http_cookie;
          if ($http_cookie ~ "^(.*)%s\\s*=[^;]+;(.*)$") { # when cookie is not last
            set $new_cookie $1$2;
          }
          if ($http_cookie ~ "^(.*)%s\\s*=[^;]+\\s*$") { # when cookie is last
            set $new_cookie $1;
          }
          proxy_set_header Cookie $new_cookie;
""" % (access_token_cookie_name, access_token_cookie_name)
    else:
        access_token_purge = ''

    if backend_check_access:
        # 401 handling depends on whether we were given a redirect url
        if redirect_url is not None and len(redirect_url) > 0:
            error_401_location = """
        location ^~ /redirect-login/ {
          resolver 127.0.0.1;
          proxy_pass %s;
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          proxy_set_header X-DKU-projectKey %s;
          proxy_set_header X-DKU-webAppId %s;
          proxy_set_header X-DKU-webAppType %s;
          error_page 502 @error502_location;
        }

        location @error401_location {
          return 302 redirect-login/login;
        }
""" % (redirect_url, project_key, web_app_id, web_app_type)
        else:
            error_401_location = """
        location @error401_location {
          root /opt/dataiku/web;
          try_files /webapp-error-401.html /webapp-error-401.html;
          sub_filter "http://localhost:%s" $auth_studiourl;
          sub_filter_once on;
        }
""" % (outside_port)

        # 403 handling is fixed
        error_403_location = """
        location @error403_location {
          root /opt/dataiku/web;
          try_files /webapp-error-403.html /webapp-error-403.html;
          sub_filter "http://localhost:%s" $auth_studiourl;
          sub_filter "ERROR_MESSAGE_PLACEHOLDER" $auth_accessforbiddenreason;
          sub_filter_once on;
        }
""" % (outside_port)

        # the callback for authenticating
        auth_location = """
        location = /auth_on_dss {
          resolver 127.0.0.1;
          proxy_pass %s/dip/api/webapps/check-access/%s/%s/;
          proxy_pass_request_body off;
          proxy_set_header Content-Length "";
          proxy_set_header Content-Type "";
          proxy_set_header Host "%s";
          proxy_set_header Sec-WebSocket-Protocol "";
          proxy_set_header X-Original-URI $request_uri;
        }
""" % (backend_url, project_key, web_app_id, remoterun.get_env_var("DKU_BACKEND_HOST", "127.0.0.1"))

        # the webapp itself
        main_location = """
        location ^~ / {
          %s
          proxy_pass http://127.0.0.1:%s/;
          absolute_redirect off;                     # remove eventual absolute redirects ...
          proxy_redirect ~^https?\\://[^/]+(.*)$ $1;  # ... also if webapps do an internal redirect
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          auth_request /auth_on_dss;
          auth_request_set $auth_studiourl $upstream_http_x_dku_studiourl;
          auth_request_set $auth_accessforbiddenreason $upstream_http_x_dku_accessforbiddenreason;
          error_page 502 @error502_location;
          error_page 401 @error401_location;
          error_page 403 @error403_location;
          proxy_http_version 1.1; 
          proxy_set_header Upgrade $http_upgrade; 
          proxy_set_header Connection "upgrade"; 
        }
""" % (access_token_purge, inside_port)
    else:
        auth_location = ''
        error_401_location = ''
        error_403_location = ''

        main_location = """
        location ^~ / {
          %s
          proxy_pass http://127.0.0.1:%s/;
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          error_page 502 @error502_location;
          proxy_http_version 1.1; 
          proxy_set_header Upgrade $http_upgrade; 
          proxy_set_header Connection "upgrade"; 
        }
""" % (access_token_purge, inside_port)

    server_template = """server {
        listen %s;

        root ".";

%s
        
%s
        
%s

        location @error502_location {
          root /opt/dataiku/web;
          try_files /webapp-error-502.html /webapp-error-502.html;
        }
        
%s

%s
    }
""" % (outside_port, auth_location, error_401_location, error_403_location, main_location, ping_location)

    # make the final nginx.conf
    template = build_nginx_conf(server_template)
    with open(nginx_conf, 'wb') as f:
        f.write(template.encode('utf8'))
    subprocess.Popen(['nginx', '-p', HOME_DIR, '-c', nginx_conf])

def setup_webapp_exposition(security_info, project_key, web_app_id, web_app_type, ping_path):
    if security_info.get('backendCheckAccess', False) or security_info.get('hideAccessToken', False):
        web_app_port = 10001
        web_app_host = "127.0.0.1" # since it's not exposed by docker or kubernetes, a 0.0.0.0 would be secure too
        # run a nginx to proxy
        start_nginx_for_webapp(ping_path, security_info.get('backendCheckAccess', False), security_info.get('redirectUrl', None), security_info.get('hideAccessToken', False), security_info.get('accessTokenCookieName', None), project_key, web_app_id, web_app_type, 10000, 10001)
    else:
        web_app_port = 10000
        web_app_host = "0.0.0.0"
    return web_app_host, web_app_port

def setup_code_studio_webapp_exposition(security_info, project_key, web_app_id,  ping_path, outside_port, inside_port):
    if security_info.get('backendCheckAccess', False) or security_info.get('hideAccessToken', False):
        # run a nginx to proxy
        start_nginx_for_webapp(ping_path, security_info.get('backendCheckAccess', False), security_info.get('redirectUrl', None), security_info.get('hideAccessToken', False), security_info.get('accessTokenCookieName', None), project_key, web_app_id, 'CODE_STUDIO_AS_WEBAPP', outside_port, inside_port)


def start_nginx_for_code_studio(ping_path, project_key, code_studio_id, ports):
    backend_url = build_backend_url()
    nginx_conf = os.path.join(HOME_DIR, 'nginx.conf')

    # the callback for authenticating
    auth_location = """
        location = /auth_on_dss {
          resolver 127.0.0.1;
          proxy_pass %s/dip/api/code-studio-objects/check-access/%s/%s/;
          proxy_pass_request_body off;
          proxy_set_header Content-Length "";
          proxy_set_header Content-Type "";
          proxy_set_header Host "%s";
          proxy_set_header Sec-WebSocket-Protocol "";
          proxy_set_header X-Original-URI $request_uri;
        }
""" % (backend_url, project_key, code_studio_id, remoterun.get_env_var("DKU_BACKEND_HOST", "127.0.0.1"))

    # the code studio itself
    servers = []
    for port in ports:
        inside_port = port["insidePort"]
        outside_port = port["outsidePort"]

        if '${port}' in ping_path:
            effective_ping_path = ping_path.replace("${port}", str(inside_port))
        else:
            effective_ping_path = ping_path

        ping_location = """
        location = %s {
          # the ping path is not authenticated (should be a safe item, like css)
          proxy_pass http://127.0.0.1:%s%s;
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          error_page 502 @error502_location;
        }
""" % (effective_ping_path, inside_port, effective_ping_path)

        error_401_location = """
        location @error401_location {
          root /opt/dataiku/web;
          try_files /code-studio-error-401.html /code-studio-error-401.html;
          sub_filter "http://localhost:%s" $auth_studiourl;
          sub_filter_once on;
        }
""" % (outside_port)

        # 403 handling is fixed
        error_403_location = """
        location @error403_location {
          root /opt/dataiku/web;
          try_files /code-studio-error-403.html /code-studio-error-403.html;
          sub_filter "http://localhost:%s" $auth_studiourl;
          sub_filter "ERROR_MESSAGE_PLACEHOLDER" $auth_accessforbiddenreason;
          sub_filter_once on;
        }
""" % (outside_port)

        # default 502
        error_502_location = """
        location @error502_location {
          root /opt/dataiku/web;
          try_files /code-studio-error-502.html /code-studio-error-502.html;
        }
"""

        main_location = """
        location ^~ / {
          proxy_pass http://127.0.0.1:%s/;
          absolute_redirect off;                     # remove eventual absolute redirects ...
          proxy_redirect ~^https?\\://[^/]+(.*)$ $1;  # ... also if webapps do an internal redirect
          proxy_next_upstream off; # Don't retry
          proxy_read_timeout 3600; # We have long queries
          auth_request /auth_on_dss;
          auth_request_set $auth_studiourl $upstream_http_x_dku_studiourl;
          auth_request_set $auth_accessforbiddenreason $upstream_http_x_dku_accessforbiddenreason;
          error_page 502 @error502_location;
          error_page 401 @error401_location;
          error_page 403 @error403_location;
          proxy_http_version 1.1; 
          proxy_set_header Upgrade $http_upgrade; 
          proxy_set_header Connection "upgrade"; 
        }
""" % (inside_port)

        server_template = """
    server {
        listen %s;

        root ".";

%s
        
%s
        
%s

%s
        
%s

%s
    }
""" % (outside_port, auth_location, error_401_location, error_403_location, error_502_location, main_location, ping_location)

        servers.append(server_template)

    # make the overall nginx.conf file, with one server per exposed port
    template = build_nginx_conf("\n\n".join(servers))
    with open(nginx_conf, 'wb') as f:
        f.write(template.encode('utf8'))
    subprocess.Popen(['nginx', '-p', HOME_DIR, '-c', nginx_conf])


def setup_vars_for_python_env(dku_vars, update_full_environ=True):
    # dku_vars == read_dku_env_and_set(no_fail=False, force=True)
    if update_full_environ:
        safe_os_environ_update(dku_vars)
    else:
        # for recipes and webapps, we have a risk of passing large variables for the custom config
        # so we're stricter
        picked = {}
        # MODIFIED_VARS contains the list of variables that runner.py might use, no more
        for k in MODIFIED_VARS:
            if k in dku_vars:
                picked[k] = dku_vars[k]
        safe_os_environ_update(picked)


def setup_python_env(execution_id, dku_vars, update_full_environ=True, with_instance_libs=True, with_project_libs=True, with_plugin_libs=True):
    if with_instance_libs:
        fetch_libs(execution_id, 'instance')
    if with_project_libs:
        fetch_libs(execution_id, 'project')
    setup_vars_for_python_env(dku_vars, update_full_environ)
    load_libs(with_instance_libs, with_project_libs, with_plugin_libs)
    use_syspath_for_subprocess_pythonpath(dku_vars.get('python', {}).get('pythonPathChunks', []))


def setup_r_env(execution_id, dku_vars, update_environ=False):
    fetch_libs(execution_id, 'instance')
    fetch_libs(execution_id, 'project')
    if update_environ:
        safe_os_environ_update(dku_vars)
    set_env_for_r_libs()


def setup_python_or_r_env(execution_id, dku_vars):
    fetch_libs(execution_id, 'instance')
    fetch_libs(execution_id, 'project')

    safe_os_environ_update(dku_vars)
    set_env_for_r_libs()
    load_libs()

    use_syspath_for_subprocess_pythonpath(dku_vars.get('python', {}).get('pythonPathChunks', []))


def replace_subpath_in_chunk(chunk, project_key, replacements):
    lib_subpaths_to_try = ['project-python-libs/%s/' % project_key, 'project-r-src/%s/' % project_key]
    for lib_subpath in lib_subpaths_to_try:
        lib_path = osp.join(HOME_DIR, "lib", "project", lib_subpath)
        if chunk.startswith(lib_path):
            # this is a python or R lib chunk from the desired project
            path_inside_lib_project = chunk[len(lib_path):] # by construction, this won't be starting with /
            if len(path_inside_lib_project) > 0 and not path_inside_lib_project.endswith('/'):
                path_inside_lib_project = path_inside_lib_project + '/' # so that it can be compared to the replacement keys
            # by construction the keys of replacements have a trailing slash and no leading slash
            for k in replacements:
                if path_inside_lib_project.startswith(k):
                    return replacements[k] + path_inside_lib_project[len(k):]
    return chunk

def replace_subpath_in_chunks(libs, project_key, replacements):
    chunks = libs.split(':')
    chunks = [replace_subpath_in_chunk(chunk, project_key, replacements) for chunk in chunks]
    return ':'.join(chunks)


def restart_runner_process():
    logging.info("Restarting runner process")
    cmd = [sys.executable, "-m", "dataiku.container.runner"] + sys.argv[1:]
    try:
        with subprocess.Popen(
            cmd, stdin=sys.stdin, stdout=sys.stdout, stderr=sys.stderr
        ) as process:
            sys.exit(process.wait())
    except Exception as e:
        logging.exception("Failed to restart process")
        sys.exit(1)


def fetch_latest_dss_code(execution):
    if execution.get("resyncPythonCode", False):
        if os.environ.get("DKU_CONTAINER_EXEC_SKIP_PYTHON_RESYNC") == "1":
            logging.info("Running with patched DSS code")
        else:
            os.environ["DKU_CONTAINER_EXEC_SKIP_PYTHON_RESYNC"] = "1"
            logging.info("Fetching latest DSS code...")
            fetch_libs(execution["id"], "dss")
            os.environ["PYTHONPATH"] = "{0}/lib/dss/:{1}".format(
                HOME_DIR, os.environ.get("PYTHONPATH", "")
            )
            logging.info("New PYTHONPATH={}".format(os.environ["PYTHONPATH"]))
            logging.info("DSS code fetched")
            restart_runner_process()
    else:
        logging.info("Running with DSS code from image")


def main():
    setup_log()
    os.chdir(HOME_DIR)
    dss_version = read_version().get('product_version')
    logging.info("Fetching job definition")
    execution_id = sys.argv[1]
    try:
        execution = intercom.jek_or_backend_json_call('containers/get-execution',
                          data={ 'executionId' : execution_id, 'version': dss_version })
    except Exception as e:
        logging.error("Could not reach DSS: %s" % e)
        # improve logging and error reporting by trying to see if the host we're trying to reach makes sense
        h = remoterun.get_env_var("DKU_BACKEND_HOST", "127.0.0.1")
        try:
            resolved = socket.gethostbyname(h)
            logging.info("Backend host %s resolved to %s" % (h, resolved))
        except Exception as e2:
            logging.error("UnknownHostException %s : %s" % (h, e2))

        os._exit(1) # don't pretend everything is fine
    if dss_version != execution.get('dssVersion'):
        logging.warn("Container image was build with version %s, but execution was sent from DSS version %s"
                     % (dss_version, execution.get('dssVersion')))
    # segregate the remote-run-env-def.json (might contain stuff we don't want logged)
    # also the R-exec-wrapper expects it
    dku_exec_env = execution.get('envResource', {'env':{}, 'python':{}, 'r':{}})
    # Currently used for NLP resources, see build-images.py
    dku_exec_env["env"]["DKU_RESOURCES_DIR"] = osp.join(OPT_DIR, "resources")
    execution['envResource'] = None
    dumpable_execution = json.loads(json.dumps(execution))
    dumpable_execution["payload"] = None
    dumpable_execution["definition"] = None
    logging.info("got exec: " + json.dumps(dumpable_execution))
    with open("execution.json", 'w') as fd: # 'w' OK , we use json.dump
        json.dump(execution, fd)

    fetch_latest_dss_code(execution)

    # add the lib folders on the path
    if execution['type'] != 'RECIPE_R':
        python_env = dku_exec_env.get('python', {})
        python_env['pythonPathChunks'] = ['%s/lib/%s' % (HOME_DIR, scope) for scope in ['instance', 'project']]
        dku_exec_env['python'] = python_env
    else:
        r_env = dku_exec_env.get('r', {})
        r_env['rPathChunks'] = ['%s/lib%s' % (HOME_DIR, scope) for scope in ['instance', 'project']]
        dku_exec_env['r'] = r_env

    remove_code_env_resources_env_vars_from_dku_exec_env(dku_exec_env)

    with open("remote-run-env-def.json", 'w') as fd: # 'w' OK, we use json.dump
        json.dump(dku_exec_env, fd)
    dku_vars = dku_exec_env.get("env", {})

    if os.path.isfile("%s/code-env/bin/python" % OPT_DIR):
        python_bin = os.path.abspath("%s/code-env/bin/python" % OPT_DIR)
    else:
        python_bin = os.path.abspath("%s/bin/python" % OPT_DIR)
    if os.path.isfile("%s/code-env/bin/R" % OPT_DIR):
        r_bin = os.path.abspath("%s/code-env/bin/R" % OPT_DIR)
    else:
        r_bin = "%s/R/bin/R" % OPT_DIR
    if execution['type'] == 'TEST_PING':
        logging.info("Try to ping backend")
        payload = json.loads(execution['payload'])
        ping_url = "%s://%s:%s/future/container-test-ping" % (os.getenv("DKU_SERVER_PROTOCOL"), os.getenv("DKU_SERVER_HOST"), str(payload['futurePort']))
        try:
            verify = intercom.get_verify_cert(False)
            r = requests.get(ping_url, {"testId": payload['testId']},
                             headers={"X-DKU-IPythonSharedSecret": payload['futureSharedSecret']},
                             verify=verify)
            logging.info("Container successfully pinged the Future Kernel")
            return
        except Exception as e:
            logging.error("Could not reach Future Kernel: %s" % e)
            os._exit(1)  # don't pretend everything is fine

    if execution['type'] == 'RECIPE_PYTHON':
        logging.info("Executing Python recipe")

        definition = json.loads(execution['definition'])
        if definition['recipeType'].startswith('CustomCode_') and "DKU_CUSTOM_RECIPE_PLUGIN_ID" in dku_vars:
            logging.info("Getting plugin %s" % dku_vars["DKU_CUSTOM_RECIPE_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_CUSTOM_RECIPE_PLUGIN_ID"])

        setup_python_env(execution_id, dku_vars, False)
        logging.info("Running user code")
        if definition['recipeType'] == 'python' or definition['recipeType'].startswith('CustomCode_'):
            with open('dku_code.py', 'wb') as fd:
                fd.write(execution['payload'].encode('utf8'))
            run_subprocess(execution_id, '%s -m dataiku.container.exec_py_recipe' % python_bin)
        else:
            raise Exception("Unsupported recipe type: %s" % definition['recipeType'])

    elif execution['type'] == 'DOCTOR_PYTHON':
        logging.info("Training doctor model")
        setup_python_env(execution_id, dku_vars)

        definition = json.loads(execution['definition'])
        logging.info("Running doctor server")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_doctor_server' % python_bin,
                       send_error_json=False, fail_if_subprocess_failed=True)
        send_container_done(execution_id)

    elif execution['type'] in {'RECIPE_PREDICTION_TRAIN_PYTHON', 'RECIPE_CLUSTERING_TRAIN_PYTHON', 'RECIPE_PREDICTION_TRAIN_DEEPHUB',
                               'RECIPE_PREDICTION_TRAIN_TIMESERIES', 'RECIPE_PREDICTION_TRAIN_CAUSAL'}:
        logging.info("Training doctor model")
        setup_python_env(execution_id, dku_vars)

        logging.info("Running doctor main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_train_recipe' % python_bin)
    elif execution['type'] == 'RECIPE_FINE_TUNING_LLM':
        logging.info('Fine-tuning LLM base model')
        setup_python_env(execution_id, dku_vars)
        run_subprocess(execution_id, '%s -m dataiku.container.exec_fine_tuning_recipe' % python_bin)

    elif execution['type'] in {'RECIPE_PREDICTION_SCORE_PYTHON', 'RECIPE_PREDICTION_SCORE_KERAS', 'RECIPE_CLUSTERING_SCORE_PYTHON',
                               'RECIPE_PREDICTION_SCORE_DEEPHUB', 'RECIPE_PREDICTION_SCORE_MLFLOW', 'RECIPE_PREDICTION_SCORE_TIMESERIES',
                               'RECIPE_PREDICTION_SCORE_CAUSAL'}:
        logging.info("Scoring dataset")
        setup_python_env(execution_id, dku_vars)

        logging.info("Running scoring recipe main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_score_recipe' % python_bin)

    elif execution['type'] in {'RECIPE_PREDICTION_EVAL_PYTHON', 'RECIPE_PREDICTION_EVAL_KERAS', 'RECIPE_PREDICTION_EVAL_MLFLOW',
                               'RECIPE_PREDICTION_EVAL_TIMESERIES', 'RECIPE_PREDICTION_EVAL_CAUSAL'}:
        logging.info("Evaluating model")
        setup_python_env(execution_id, dku_vars)

        logging.info("Running evaluation recipe main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_eval_recipe' % python_bin)

    elif execution['type'] == 'RECIPE_PREDICTION_STANDALONE_EVAL_PYTHON':
        logging.info("Evaluating external model")

        logging.info("Running standalone evaluation recipe main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_standalone_eval_recipe' % python_bin)

    elif execution['type'] == 'RECIPE_LLM_EVALUATION_PYTHON':
        logging.info("Evaluating llm model")

        logging.info("Running llm evaluation recipe main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_llm_eval_recipe' % python_bin)

    elif execution['type'] == 'RECIPE_CLUSTERING_CLUSTER_PYTHON':
        logging.info("Clustering dataset")

        setup_python_env(execution_id, dku_vars)
        logging.info("Running doctor main")
        run_subprocess(execution_id, '%s -m dataiku.container.exec_cluster_recipe' % python_bin)

    elif execution['type'] == 'RECIPE_R':
        logging.info("Executing R recipe")

        setup_r_env(execution_id, dku_vars, update_environ=False)

        logging.info("Running user code")
        definition = json.loads(execution['definition'])
        if definition['recipeType'].startswith('CustomCode_') and "DKU_CUSTOM_RECIPE_PLUGIN_ID" in dku_vars:
            logging.info("Getting plugin %s" % dku_vars["DKU_CUSTOM_RECIPE_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_CUSTOM_RECIPE_PLUGIN_ID"])

        if definition['recipeType'] == 'r' or definition['recipeType'].startswith('CustomCode_'):
            with open('code.R', 'wb') as fd:
                fd.write(execution['payload'].encode('utf8'))
            run_subprocess(execution_id, '/bin/sh -c "EXECUTION_ID=%s %s --quiet --no-save --args code.R < %s/R/R-exec-wrapper.R"' % (execution_id, r_bin, OPT_DIR))
        else:
            raise Exception("Unsupported recipe type: %s" % definition['recipeType'])

    elif execution['type'] == 'SIMPLE_PYTHON':
        definition = json.loads(execution['definition'])

        dku_plugin_id = remoterun.get_env_var("DKU_PLUGIN_ID", '')
        if len(dku_plugin_id) > 0:
            logging.info("Getting plugin %s" % dku_plugin_id)
            fetch_plugin(execution_id, dku_plugin_id)

        setup_python_env(
            execution_id,
            dku_vars,
            update_full_environ=True,
            with_instance_libs=definition["load_instance_libs"],
            with_project_libs=definition["load_project_libs"],
            with_plugin_libs=True
        )

        logging.info("Executing kernel entry point: %s" % definition['module'])
        run_subprocess(execution_id, '%s -m %s %s %s --server-cert=%s' % (python_bin, definition['module'], definition['port'], definition['secret'], definition.get('server_cert', 'NA')),
                       send_error_json=False)

    elif execution['type'] == 'ML_INTERACTIVE_SCORING':
        logging.info("Executing ML_INTERACTIVE_SCORING kernel")
        setup_python_env(execution_id, dku_vars)

        definition = json.loads(execution['definition'])
        run_subprocess(execution_id,
                       '%s -m %s %s %s --server-cert=%s' % (python_bin, definition['module'],
                                           definition['port'], definition['secret'], definition.get('server_cert', 'NA')),
                       send_error_json=False, fail_if_subprocess_failed=True)
        send_container_done(execution_id)

    elif execution['type'] == 'NOTEBOOK_PYTHON' or execution['type'] == 'NOTEBOOK_R':
        logging.info("Executing notebook")

        if execution['type'] == 'NOTEBOOK_PYTHON':
            setup_python_env(execution_id, dku_vars)
        else:
            setup_r_env(execution_id, dku_vars, update_environ=True)


        definition = json.loads(execution['definition'])

        # fixup env vars that pass the libraries to the subprocess
        if execution['type'] == 'NOTEBOOK_PYTHON':
            old_libs_var = os.environ.get('PYTHONPATH', '')
            if old_libs_var is not None and len(old_libs_var) > 0:
                chunks = old_libs_var.split(':')
            else:
                chunks = []
            for lib_dir in ["/lib/project", "/lib/instance", "/plugin/python-lib"]:
                if os.path.isdir(HOME_DIR + lib_dir):
                    chunks.append(HOME_DIR + lib_dir)
            os.environ['PYTHONPATH'] = ':'.join(chunks)
        elif execution['type'] == 'NOTEBOOK_R':
            old_libs_var = os.environ.get('R_LIBS', '')
            if old_libs_var is not None and len(old_libs_var) > 0:
                chunks = old_libs_var.split(':')
            else:
                chunks = []
            for lib_dir in ["/lib/instance"]:
                if os.path.isdir(HOME_DIR + lib_dir):
                    chunks.append(HOME_DIR + lib_dir)
            os.environ['R_LIBS'] = ':'.join(chunks)

        logging.info("Setup forwarding")
        from dataiku.notebook.kernel_side_forwarder import KernelSideForwarder

        forward = KernelSideForwarder(definition)
        try:
            local_connection_file_name = forward.initialize()
        except:
            logging.exception("Failed to setup forwarder")
            # no 'failure' per se can happen
            os._exit(0)

        logging.info("Start the ioloop")
        from tornado import ioloop
        # run ioloop in separate thread
        ioloop_instance = ioloop.IOLoop.instance() # get it here, because running ioloop.IOLoop.instance() gives another instance...
        def threaded_loop():
            logging.info("starting IOLoop")
            try:
                ioloop_instance.start()
            except:
                logging.info("IOLoop failure")
                traceback.print_exc()
                os._exit(1)
        t = threading.Thread(target=threaded_loop)
        t.daemon = True
        t.start()

        if execution['type'] == 'NOTEBOOK_PYTHON':
            try:
                popen = run_popen('%s -m ipykernel_launcher -f %s' % (python_bin, local_connection_file_name))
                forward.set_subprocess(popen)
                error_code = popen.wait()
                handle_subprocess_return_value(execution_id, error_code)
            finally:
                # no 'failure' per se can happen
                os._exit(0)
        else:
            try:
                popen = run_popen('%s --slave -e "IRkernel::main()" --args %s' % (r_bin, local_connection_file_name))
                forward.set_subprocess(popen)
                error_code = popen.wait()
                handle_subprocess_return_value(execution_id, error_code)
            finally:
                # no 'failure' per se can happen
                os._exit(0)

    elif execution['type'] == 'WEBAPP_STD':
        logging.info("Executing standard webapp backend")

        if len(dku_vars.get("DKU_PLUGIN_ID", '')) > 0:
            logging.info("Getting plugin %s" % dku_vars["DKU_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_PLUGIN_ID"])

        definition = json.loads(execution['definition'])
        deps_base = definition.get('htmlDepsBase', None)
        if deps_base is not None and len(deps_base) > 0:
            # also fetch the zip of the packaged html dependencies
            fetch_dir(execution_id, deps_base, deps_base, file_kind="EXECUTION_DIR")

        with open('command.json', 'w') as f: # 'w' OK we use json.dump
            json.dump(definition, f)

        web_app_host, web_app_port = setup_webapp_exposition(definition.get('securityInfo', {}), definition['projectKey'], definition['webAppId'], 'STANDARD', '/__ping')

        start_code = """from dataiku.webapps.backend import serve
serve("./command.json", "%s", %s)
""" % (web_app_host, web_app_port)
        with open('start_webapp.py', 'wb') as fd:
            fd.write(start_code.encode('utf8'))

        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'liveliness.marker'), 'w') as f: # 'w' OK we write a string
            f.write('alive')

        setup_python_env(execution_id, dku_vars, False)

        run_metrics_server()
        logging.info("Running webapp")
        try:
            run_subprocess(execution_id, '%s -m dataiku.container.exec_py_webapp' % python_bin)
        finally:
            logging.info("Webapp subprocess complete, exiting runner")
            # no 'failure' per se can happen
            os._exit(0)

    elif execution['type'] == 'WEBAPP_BOKEH':
        logging.info("Executing bokeh webapp backend")

        if len(dku_vars.get("DKU_PLUGIN_ID", '')) > 0:
            logging.info("Getting plugin %s" % dku_vars["DKU_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_PLUGIN_ID"])

        definition = json.loads(execution['definition'])
        if not os.path.exists('backend'):
            os.mkdir('backend')
        with open(os.path.join('backend', 'main.py'), 'wb') as fd:
            fd.write(definition['code'].encode('utf8'))

        web_app_host, web_app_port = setup_webapp_exposition(definition.get('securityInfo', {}), definition['projectKey'], definition['webAppId'], 'BOKEH', '/static/js/bokeh.min.js')

        start_code = """from dataiku.webapps.run_bokeh import main
import sys
sys.argv = [sys.argv[0], './backend', '%s', '%s', '%s']
main()
""" % (definition.get("nbProcesses", 1), web_app_host, web_app_port)
        with open('start_webapp.py', 'wb') as fd:
            fd.write(start_code.encode('utf8'))

        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'liveliness.marker'), 'w') as f: # 'w' OK, we write a string
            f.write('alive')

        setup_python_env(execution_id, dku_vars, False)

        run_metrics_server()
        logging.info("Running webapp")
        try:
            run_subprocess(execution_id, '%s -m dataiku.container.exec_py_webapp' % python_bin)
        finally:
            # no 'failure' per se can happen
            os._exit(0)

    elif execution['type'] == 'WEBAPP_SHINY':
        logging.info("Executing shiny webapp backend")
        setup_r_env(execution_id, dku_vars, update_environ=False)

        if len(dku_vars.get("DKU_PLUGIN_ID", '')) > 0:
            logging.info("Getting plugin %s" % dku_vars["DKU_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_PLUGIN_ID"])

        definition = json.loads(execution['definition'])
        with open('server.R', 'wb') as fd:
            fd.write(definition['server'].encode('utf8'))
        with open('ui.R', 'wb') as fd:
            fd.write(definition['ui'].encode('utf8'))

        web_app_host, web_app_port = setup_webapp_exposition(definition.get('securityInfo', {}), definition['projectKey'], definition['webAppId'], 'SHINY', '/shared/shiny.js')

        start_code = """shiny::runApp('%s',port=%s,host="%s")
""" % (os.getcwd(), web_app_port, web_app_host)
        with open('start_webapp.R', 'wb') as fd:
            fd.write(start_code.encode('utf8'))

        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'liveliness.marker'), 'w') as f: # 'w' OK we write a string
            f.write('alive')

        run_metrics_server()
        logging.info("Running webapp")
        try:
            run_subprocess(execution_id, '/bin/sh -c "EXECUTION_ID=%s %s --quiet --no-save --args start_webapp.R < %s/R/R-exec-wrapper.R"' % (execution_id, r_bin, OPT_DIR))
        finally:
            # no 'failure' per se can happen
            os._exit(0)

    elif execution['type'] == 'WEBAPP_DASH':
        logging.info("Executing dash webapp backend")

        if len(dku_vars.get("DKU_PLUGIN_ID", '')) > 0:
            logging.info("Getting plugin %s" % dku_vars["DKU_PLUGIN_ID"])
            fetch_plugin(execution_id, dku_vars["DKU_PLUGIN_ID"])

        definition = json.loads(execution['definition'])
        if not os.path.exists('backend'):
            os.mkdir('backend')
        with open(os.path.join('backend', 'main.py'), 'wb') as fd:
            fd.write(definition['code'].encode('utf8'))

        web_app_host, web_app_port = setup_webapp_exposition(definition.get('securityInfo', {}), definition['projectKey'], definition['webAppId'], 'DASH', '/__ping')

        start_code = """from dataiku.webapps.run_dash import main
import sys
sys.argv = [sys.argv[0], './backend', '%s', '%s', '%s', '%s']
main()
""" % (definition.get("nbProcesses", 1), str(definition.get("serveLocally", True)).lower(), web_app_host, web_app_port)
        with open('start_webapp.py', 'wb') as fd:
            fd.write(start_code.encode('utf8'))

        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'liveliness.marker'), 'w') as f:  # 'w' OK, we write a string
            f.write('alive')

        setup_python_env(execution_id, dku_vars, False)

        run_metrics_server()
        logging.info("Running webapp")
        try:
            run_subprocess(execution_id, '%s -m dataiku.container.exec_py_webapp' % python_bin)
        finally:
            # no 'failure' per se can happen
            os._exit(0)

    elif execution['type'] == 'RECIPE_CPYTHON':
        logging.info("Executing Continuous Python recipe")

        definition = json.loads(execution['definition'])

        setup_python_env(execution_id, dku_vars)

        logging.info("Running user code")
        code_mode = definition.get("codeMode", "FREE_FORM")
        if code_mode == "FREE_FORM":
            with open('dku_code.py', 'wb') as fd:
                fd.write(execution['payload'].encode('utf8'))
        elif  code_mode == "FUNCTION":
            function = intercom.jek_or_backend_json_call('cpython/start-one', data={ 'functionId' : definition['functionId'], 'version': dss_version })
            logging.info("Starting function %s" % json.dumps(function))

            start_code = """from dataiku.continuous.server import main
import sys
sys.argv = [sys.argv[0], '%s', '%s', '--server-cert=%s']
main()
""" % (function["linkPort"], function["linkSecret"], function.get("linkServerCert", 'NA'))
            with open('dku_code.py', 'wb') as fd:
                fd.write(start_code.encode('utf8'))
        else:
            raise Exception("Unsupported recipe code mode: %s" % code_mode)

        run_subprocess(execution_id, '%s -m dataiku.container.exec_continuous_py_recipe' % python_bin, fail_if_subprocess_failed=True)
    elif execution['type'] == 'CODE_STUDIO':
        # We import here because on Windows this module doesn't exist,
        # and this file is imported when we do `import dataiku`
        import resource
        logging.info("Executing Code Studio")
        logging.info("  Disabling core dumps")
        resource.setrlimit(resource.RLIMIT_CORE, (0, 0))

        definition = json.loads(execution['definition'])

        env_variable_exposition_path = 'DKU_CODE_STUDIO_BROWSER_PATH'
        exposition_path = definition['baseUrl']
        logging.info("Setting %s to %s" % (env_variable_exposition_path, exposition_path))
        os.environ[env_variable_exposition_path] = exposition_path

        # inform that the pod is alive (technically it's more a startup probe that a
        # liveliness one, but startup is k8s>=1.20). It's useful because at least if
        # runner.py is failing before arriving here (ex: network connectivity issues
        # to DSS, or api ticket invalid because DSS backend restarted) then k8s will
        # notice the crashloop. The real startup probe marker is below, so when startup
        # is done, the kubelet will see the pod is alive
        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'liveliness.marker'), 'w') as f:  # 'w' OK, we write a string
            f.write('alive')

        synced_zones = definition.get('syncedZones', [])
        logging.info("Starting Code Studio file server with zones " + json.dumps(synced_zones, indent=4))
        logging.info("Excluded from sync " + json.dumps(definition.get('excludedFromSync', []), indent=4))
        command_server = CommandServerThread(
            execution_id,
            definition['projectKey'],
            definition['codeStudioId'],
            synced_zones,
            definition.get('excludedFromSync'),
            definition.get("fileAdjustmentScript"),
            definition.get("reattachOnRestart"))
        # hook for tests: delay controlled by a variable
        startup_delay = dku_vars.get('DKU_TEST_CSIO_STARTUP_DELAY_S')
        if startup_delay is not None and len(startup_delay) > 0:
            logging.info("Delaying startup for %s seconds..." % startup_delay)
            try:
                time.sleep(int(startup_delay))
                logging.info("Done waiting")
            except:
                logging.warning("Unable to delay startup for : %s" % startup_delay)
        # initial pull of files
        for zone in synced_zones:
            try:
                command_server.handle_command('push_bundle_to_code_studio', '-1', {'zone':zone['id']})
            except Exception as e:
                logging.error("Cannot retrieve %s : %s" % (zone['id'], str(e)))
        # inform that the pod is synced
        with open(os.path.join(os.environ.get('TMPDIR', '/tmp'), 'initial-sync.marker'), 'w') as f:  # 'w' OK, we write a string
            f.write('done')
        # then really start processing UI-sent commands
        if definition.get("handleCommands", True):
            command_server.start_polling_commands()

        # prepare the env variables, notably put the projects lib bundles in the
        # pythonpath and r_libs.
        setup_python_or_r_env(execution_id, dku_vars)
        # fiddle with the PYTHONPATH and DKU_SOURCE_LIB_R_PATH in order to put
        # the paths of the codeStudio-fetched locations first
        replacements = command_server.project_key_path_replacements()
        project_key = definition['projectKey']
        with open(osp.join(HOME_DIR, "lib", "project", "project-lib-paths.json")) as f:
            project_libs_paths = json.load(f)

        extra_env = {}
        for k in ['PYTHONPATH', 'DKU_SOURCE_LIB_R_PATH']:
            libs = os.environ[k]
            logging.info("Replace subpath in chunks of '%s' from %s with %s" % (libs, json.dumps(project_libs_paths), json.dumps(replacements)))
            extra_env[k] = replace_subpath_in_chunks(libs, project_key, replacements)

        # setup a nginx in front of the main exposed port
        def start_nginx():
            ping_path = definition.get("readinessProbeUrl", "")
            # if the ping path is bogus, it just means we shouldn't try to use readiness probes
            if ping_path is None or len(ping_path) == 0:
                ping_path = "/__ping"  # bogus value, but at least it won't break the nginx conf
            # the ping_path could still contain the ${baseUrl} part (or even ${baseUrlPortXXXX}, if launched by a webapp with a tcp probe),
            # so replace it here (you can't know its value beforehand since it can contain the pod name)
            expanded_path = os.path.expandvars(exposition_path)
            ping_path = ping_path.replace("${baseUrl}", expanded_path)
            if definition.get('webAppId') is not None and len(definition.get("ports", [])) == 1:
                ping_path = re.sub(r'\$\{baseUrlPort\d{1,5}\}', expanded_path, ping_path)
                forwarded_port = definition['ports'][0]
                setup_code_studio_webapp_exposition(definition.get('securityInfo', {}), definition['projectKey'], definition['webAppId'], ping_path, forwarded_port['outsidePort'], forwarded_port['insidePort'])
            elif definition.get('webAppId') is None and len(definition.get("ports", [])) > 0:
                ping_path = re.sub(r'\$\{baseUrlPort\d{1,5}\}', expanded_path + '/${port}', ping_path)  # keep in sync with baseUrlLNTPerPort in CodeStudioRuntime.java
                start_nginx_for_code_studio(ping_path, definition['projectKey'], definition['codeStudioId'], definition['ports'])

        # launch nginx only after the webapp is started, since the nginx will respond to an eventual TCP probe in lieu of the webapp...
        def wait_and_start_nginx(probe_port):
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            try:
                logging.info("Nginx launch waiting for app start on port %s" % probe_port)
                s.settimeout(1)
                st = time.time()
                while time.time() - st < 60:
                    try:
                        s.connect(('127.0.0.1', probe_port))  # Watching if the probed webapp is up
                        logging.info("Nginx launch waiting: App started")
                        break
                    except socket.error as sock_err:  # Waiting for the probed webapp to be up
                        if sock_err.errno == errno.ECONNREFUSED or sock_err.errno == errno.ECONNABORTED or sock_err.errno == errno.ECONNRESET:
                            logging.info("Nginx launch waiting: App not started, waited %fs" % (time.time() - st))
                            time.sleep(1)
                        else:
                            raise sock_err
                    except socket.timeout:
                        logging.info("Nginx launch waiting: Timeout connecting to app, the app might be stuck, waited %ds" % (time.time() - st))
                        time.sleep(1)
            finally:
                s.close()
            start_nginx()  # start anyway, even if probe did not work, just in case

        probe_port = definition.get('tcpProbePort', 0)
        if probe_port:
            watch_start = threading.Thread(target=wait_and_start_nginx, args=(probe_port,))
            watch_start.start()
        else:
            start_nginx()

        logging.info("Running Code Studio entrypoints: %s" % definition['entrypoints'])
        sub_processes = []
        try:
            for i, entrypoint in enumerate(definition['entrypoints']):
                logging.info("Running Code Studio entrypoint %s: %s" % (i, entrypoint))
                sub_processes.append(run_popen(entrypoint, **extra_env))
            exit_codes = [p.wait() for p in sub_processes]
        finally:
            # no 'failure' per se can happen
            os._exit(0)

    else:
        logging.error("Unknown execution type: '%s'" % execution['type'])
        sys.exit(1)  # not supposed to happen


if __name__ == "__main__":
    main()
