from typing import List
from dataiku.eda.types import Literal

from dataiku.eda.computations.computation import Computation
from dataiku.eda.computations.context import Context
from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.types import MultiComputationModel, MultiComputationResultModel


class MultiComputation(Computation):
    def __init__(self, computations: List[Computation]):
        self.computations = computations

    @staticmethod
    def get_type() -> Literal["multi"]:
        return "multi"

    @staticmethod
    def build(params: MultiComputationModel) -> 'MultiComputation':
        return MultiComputation([Computation.build(computation) for computation in params['computations']])

    @staticmethod
    def _require_result_checking() -> bool:
        return False

    def describe(self) -> str:
        return "Multi"

    def apply(self, idf: ImmutableDataFrame, ctx: Context) -> MultiComputationResultModel:
        results = []
        for idx, computation in enumerate(self.computations):
            with ctx.sub(str(idx), brackets=True) as sub:
                results.append(computation.apply_safe(idf, sub))

        return {
            "type": MultiComputation.get_type(),
            "results": results  # Results are already checked when apply_safe() is called on the inner computations
        }
