from pathlib import Path
from typing import Optional, Any

from mcp.types import Root
from pydantic import FileUrl

from dataiku.base.remoterun import is_running_remotely

# custom properties set in the settings tab
CLIENT_INIT_TIMEOUT_PROPERTY="dku.mcp.client.init.timeout"
CLIENT_INIT_TIMEOUT_DEFAULT=5*60  # 5 minutes

CLIENT_REQUEST_TIMEOUT_PROPERTY="dku.mcp.client.request.timeout"
CLIENT_REQUEST_TIMEOUT_DEFAULT=60  # 1 minute

CLIENT_CWD_PROPERTY="dku.mcp.client.cwd"  # default depends on the executable

CLIENT_ROOTS_PROPERTY_PREFIX="dku.mcp.client.roots."
CLIENT_ROOTS_DEFAULT=[] if not is_running_remotely() else None  # default to no local FS access when running locally and all access inside containers

WATCHDOG_INTERVAL_PROPERTY="dku.mcp.watchdog.interval"
WATCHDOG_INTERVAL_DEFAULT=10  # 10 seconds


class MCPClientConfig:
    """ Wrap an agent tool config to ease access to fastmcp client settings
    """
    def __init__(self, config: dict):
        self.config = config

    def _get_dku_property(self, key: str) -> Optional[Any]:
        return self.config.get("dkuProperties", {}).get(key)

    def _get_dku_property_str(self, key: str, default=None) -> Optional[str]:
        value = self._get_dku_property(key)
        if value is None:
            return default
        return str(value)

    def _get_dku_property_int(self, key: str, default=None) -> Optional[int]:
        value = self._get_dku_property(key)
        if value is None:
            return default
        return int(value)

    @property
    def cwd(self):
        return self._get_dku_property_str(CLIENT_CWD_PROPERTY, None)

    @property
    def init_timeout(self):
        return self._get_dku_property_int(CLIENT_INIT_TIMEOUT_PROPERTY, CLIENT_INIT_TIMEOUT_DEFAULT)

    @property
    def request_timeout(self):
        return self._get_dku_property_int(CLIENT_REQUEST_TIMEOUT_PROPERTY, CLIENT_REQUEST_TIMEOUT_DEFAULT)

    @property
    def roots(self) -> Optional[list[dict]]:
        """ Roots limit the server to a list of paths. Paths must be local and absolute
        See https://modelcontextprotocol.io/specification/2025-06-18/client/roots#root
        """
        roots = []
        for key, value in self.config.get("dkuProperties", {}).items():
            if key.startswith(CLIENT_ROOTS_PROPERTY_PREFIX):
                name = key.replace(CLIENT_ROOTS_PROPERTY_PREFIX, "")
                if not value.startswith('file://'):
                    p = Path(value)
                    if p.is_absolute():
                        value = p.as_uri()
                    # else will raise an exception in Pydantic with details
                uri = FileUrl(value)
                roots.append(Root(name=name, uri=uri))
        if not roots:
            return CLIENT_ROOTS_DEFAULT
        return roots

    @property
    def watchdog_interval(self):
        return self._get_dku_property_int(WATCHDOG_INTERVAL_PROPERTY, WATCHDOG_INTERVAL_DEFAULT)
