#
# nginx configuration file generation
#
import os, subprocess, six
import base


template = '''
# This file is automatically generated. Manual edits will be lost upon DSS updates.

# Hostname will be written in the prefix
Hostname    "collectd"
BaseDir     "{{DIP_HOME}}/run"
PIDFile     "{{DIP_HOME}}/run/collectd.pid"

PluginDir   "{{DKUINSTALLDIR}}/tools/collectd/lib/collectd"
TypesDB     "{{DKUINSTALLDIR}}/tools/collectd/share/collectd/types.db"

AutoLoadPlugin true
CollectInternalStats false
Interval {{COLLECTD_INTERVAL}}

<Plugin logfile>
    LogLevel info
    File "{{DIP_HOME}}/run/collectd.log"
    Timestamp true
    PrintSeverity false
</Plugin>

<Plugin cpu>
    ReportByCpu true
    ReportByState true
    ValuesPercentage true
    ReportNumCpu true
    ReportGuestState true
    SubtractGuestState true
</Plugin>

<Plugin "df">
    Device "{{DATADIR_DEVICE}}"
    ValuesAbsolute true
    ValuesPercentage true
    ReportByDevice true
</Plugin>

<Plugin disk>
    Disk "/^{{DATADIR_DISK}}?$/"
</Plugin>

<Plugin interface>
    Interface "lo"
    IgnoreSelected true
    ReportInactive false
</Plugin>

<Plugin load>
    ReportRelative true
</Plugin>

<Plugin memory>
    ValuesAbsolute true
    ValuesPercentage true
</Plugin>

<Plugin processes>
    CollectFileDescriptor true
    CollectContextSwitch true
    CollectMemoryMaps true
    <ProcessMatch "dss_backend" ".*java -Ddku\.backend .*-cp {{ESCAPED_DIP_HOME}}/bin:.*">
	CollectFileDescriptor false
	CollectContextSwitch true
    </ProcessMatch>
    <ProcessMatch "dss_fek" ".*java -Ddku\.fek .*-cp {{ESCAPED_DIP_HOME}}/bin:.*">
	CollectFileDescriptor false
	CollectContextSwitch true
    </ProcessMatch>
    <ProcessMatch "dss_jek" ".*java -Ddku\.jek .*-cp {{ESCAPED_DIP_HOME}}/bin:.*">
	CollectFileDescriptor false
	CollectContextSwitch true
    </ProcessMatch>
    <ProcessMatch "dss_cak" ".*java -Ddku\.cak .*-cp {{ESCAPED_DIP_HOME}}/bin:.*">
	CollectFileDescriptor false
	CollectContextSwitch true
    </ProcessMatch>
    <ProcessMatch "govern_server" ".*java -Ddku\.governserver .*-cp {{ESCAPED_DIP_HOME}}/bin:.*">
    CollectFileDescriptor false
    CollectContextSwitch true
    </ProcessMatch>
</Plugin>

<Plugin write_graphite>
  <Node "{{CARBON_HOSTNAME}}">
    Host "{{CARBON_HOSTNAME}}"
    Port "{{CARBON_PORT}}"
    Protocol "tcp"
    ReconnectInterval 0
    LogSendErrors true
    Prefix "{{COLLECTD_PREFIX}}."
    StoreRates true
    AlwaysAppendDS false
    EscapeCharacter "_"
    SeparateInstances true
    PreserveSeparator true
    DropDuplicateFields true
  </Node>
</Plugin>

'''

tmpl_vars = [
    "DKUINSTALLDIR", "DIP_HOME"
]

#
# Generates the collectd configuration corresponding to dipHome
# Assumes the environment variables in env-default have been sourced
#
def generate_collectd_config(dipHome):
    config = dipHome.get_install_config()
    carbon_server, carbon_port = config.getOption("collectd","graphite_server").split(":")
    collectd_interval = config.getIntOption("collectd","interval",300)
    collectd_prefix = config.getOption("collectd","prefix")
    result = template.replace("{{CARBON_HOSTNAME}}", carbon_server)
    result = result.replace("{{CARBON_PORT}}", carbon_port)
    result = result.replace("{{COLLECTD_PREFIX}}", collectd_prefix)
    result = result.replace("{{COLLECTD_INTERVAL}}", str(collectd_interval))
    result = result.replace("{{ESCAPED_DIP_HOME}}", os.environ["DIP_HOME"].replace(".","\."))

    # Get dataidr device
    df_stdout, df_sterr = subprocess.Popen(
            ["df",os.environ["DIP_HOME"]], stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False).communicate()
    datadir_device = six.ensure_str(df_stdout).strip().split("\n")[-1].split(" ")[0]
    datadir_disk = datadir_device.split("/")[-1]
    result = result.replace("{{DATADIR_DEVICE}}", datadir_device)
    result = result.replace("{{DATADIR_DISK}}", datadir_disk)

    for var in tmpl_vars:
        result = result.replace("{{%s}}" % var, os.environ[var])
    return result

#
# Prints the collectd configuration on standard output
#
if __name__ == "__main__":
    print(generate_collectd_config(base.DipHome(os.environ["DIP_HOME"])))
