(function() {
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/core.js
     */
    app.controller('DashboardCoreController', function($scope, $element, $rootScope, $timeout, $stateParams, DataikuAPI, TopNav, $filter, $state, $controller, CreateModalFromTemplate, WT1, DashboardUtils, DashboardFilters, DSSVisualizationThemeUtils, translate) {
        $controller('DashboardsCommonController', { $scope: $scope });
        $scope.uiState = { tooltipReload: 'Reload tiles', displayNotification: false };

        DataikuAPI.dashboards.getSummary($stateParams.projectKey, $stateParams.dashboardId, $stateParams.exportId).success(function(data) {
            $scope.ownerDisplayName = data.ownerDisplayName;

            $scope.dashboard = data.object;
            $scope.dashboard.pages.forEach(page => {
                /*
                 * client id or cid is a unique identifier automatically assigned
                 * when they're first created.
                 * Client ids are handy when the model has not yet been saved to the server,
                 * and does not yet have its eventual true id, but already needs to be visible in the UI.
                 */
                page.cid = _.uniqueId();
            });

            $scope.interest = data.interest;
            $scope.timeline = data.timeline;
            $scope.canEdit = $scope.canEditDashboard($scope.dashboard);
            if ($scope.canEdit) {
                $scope.origDashboard = angular.copy(data.object);
            }

            TopNav.setItem(TopNav.ITEM_DASHBOARD, $stateParams.dashboardId, $scope.dashboard);

            $scope.$watch('dashboard.theme', function(theme) {
                if (theme && theme.generalFormatting && theme.generalFormatting.fontFamily) {
                    $element.css('--visualization-font-family', theme.generalFormatting.fontFamily);
                    $element.css('--visualization-font-color', theme.generalFormatting.fontColor);
                }
            });

            $scope.$watch('dashboard.name', function(nv) {
                if (!nv) {
                    return;
                }
                TopNav.setPageTitle(nv + ' - Dashboard');
                $state.go($state.current, { dashboardName: $filter('slugify')(nv), separator: '_' }, { location: 'replace', inherit: true, notify: false, reload: false });
            });
        }).error(setErrorInScope.bind($scope));

        $scope.isDirty = function() {
            return !angular.equals($scope.dashboard, $scope.origDashboard);
        };

        $scope.revertChanges = function() {
            $scope.dashboard = angular.copy($scope.origDashboard);
        };

        $scope.toggleDashboardListed = function(dashboard) {
            if (dashboard.listed) {
                DataikuAPI.dashboards.getMissingReaderAuthorizations(dashboard.projectKey, [dashboard.id]).success(function(data) {
                    if (data.length) {
                        CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', $scope, null, function(newScope) {
                            newScope.initForDashboardsWithAuths([dashboard], true, data);
                        }).then(function() {
                            $scope.makeDashboardListed(dashboard);
                        });
                    } else {
                        $scope.makeDashboardListed(dashboard);
                    }
                });
            } else {
                $scope.makeDashboardListed(dashboard);
            }
        };

        $scope.requestFullScreen = function() {
            $scope.$broadcast('toggleFullScreen');
        };

        $scope.reloadAllTiles = function() {
            $scope.uiState.displayNotification = false;
            $scope.uiState.tooltipReload = '';
            $scope.$broadcast('dashboard-reload-tiles');
        };

        $scope.displayNotification = function() {
            $scope.uiState.displayNotification = true;
            $scope.uiState.tooltipReload = 'Update available, click to reload tiles';
        };

        $scope.$on('dashboards-reload-needed', () => {
            if (!$scope.dashboard) {
                return;
            }
            if ($scope.dashboard.reloadWhenEventReceived) {
                $scope.reloadAllTiles();
            } else {
                $scope.displayNotification();
            }
        });

        $scope.toggleFilterPanel = function() {
            $timeout(() => DashboardUtils.toggleFilterPanel($scope.dashboard, getCurrentPage()));
        };

        $scope.toggleAllFilterPanels = (toggle) => {
            $scope.dashboard.pages.forEach(page => $scope.toggleFilterPanel(page, toggle));
        };

        $scope.isFilterPanelOpened = function() {
            const currentPage = getCurrentPage();
            if (currentPage) {
                return currentPage.showFilterPanel;
            }
        };

        let crossFilterNotificationTimeoutId;
        const unregisterDashboardFiltersChangedListener = $rootScope.$on('dashboardFiltersChanged', (_, { currentFilters, previousFilters }) => {

            $scope.filtersCount = currentFilters ? currentFilters.length : 0;

            const isInitialisation = !currentFilters.length && !previousFilters.length;

            if (isInitialisation || $scope.isFilterPanelOpened()) {
                return;
            }

            $timeout.cancel(crossFilterNotificationTimeoutId);

            if (currentFilters && previousFilters) {
                if (currentFilters.length === previousFilters.length) {
                    $scope.filterChangedNotificationLabel = translate('DASHBOARD.FILTER.PANEL.FILTERS.FILTER_UPDATED', 'Filter updated');
                } else if (currentFilters.length > previousFilters.length) {
                    $scope.filterChangedNotificationLabel = translate('DASHBOARD.FILTER.PANEL.FILTERS.FILTER_ADDED', 'Filter added');
                } else {
                    $scope.filterChangedNotificationLabel = translate('DASHBOARD.FILTER.PANEL.FILTERS.FILTER_REMOVED', 'Filter removed');
                }

                crossFilterNotificationTimeoutId = $timeout(() => {
                    $scope.filterChangedNotificationLabel = null;
                }, 5000);
            }
        });

        $scope.shouldDisplayShowHideFiltersButton = (editable) => {
            const currentPage = getCurrentPage();
            if (currentPage) {
                const canFilter = DashboardFilters.canFilter(editable, currentPage.filters, currentPage.enableCrossFilters);
                const isFiltersPanelDocked = DashboardFilters.isFiltersPanelDocked(currentPage.filtersParams.panelPosition);
                return canFilter && isFiltersPanelDocked;
            }
        };

        $scope.$on('$destroy', () => {
            unregisterDashboardFiltersChangedListener();
            DSSVisualizationThemeUtils.hideThemeAppliedSnackbar();
            DashboardFilters.resetAllDashboardPagesFilterPanelStates();
        });

        function getCurrentPage() {
            return DashboardUtils.getCurrentPage($scope.dashboard, $state.params.pageId);
        }

        function getVisiblePageId(viewMode) {
            let pageId = '';
            const currentPage = $scope.dashboard.pages.find(page => page.id === $state.params.pageId);
            const firstVisiblePage = $scope.dashboard.pages.find(page => page.show);
            if (viewMode === 'view' && currentPage && !currentPage.show) {
                if (firstVisiblePage) {
                    pageId = firstVisiblePage.id;
                }
            } else if ($state.params.pageId) {
                pageId = $state.params.pageId;
            } else if ($scope.dashboard.pages && $scope.dashboard.pages.length) {
                pageId = $scope.dashboard.pages[0].id;
            }
            return pageId;
        }

        function getVisiblePageIndex(pageId) {
            if (pageId) {
                return $scope.dashboard.pages.findIndex(page => page.id === pageId);
            }
            return '';
        }

        function switchViewMode(viewMode) {
            if (viewMode !== 'view' && viewMode !== 'edit') {
                return;
            }
            WT1.event(`dashboard-to-${viewMode}-mode-switch`);
            DSSVisualizationThemeUtils.hideThemeAppliedSnackbar();
            DashboardFilters.resetAllDashboardPagesFilterPanelStates();
            const pageId = getVisiblePageId(viewMode);
            $scope.uiState.currentPageIdx = getVisiblePageIndex(pageId);
            $state.go(`projects.project.dashboards.dashboard.${ viewMode }`, { pageId });
        }

        $scope.switchToView = () => switchViewMode('view');
        $scope.switchToEdit = () => switchViewMode('edit');
    });
})();
