(function() {
'use strict';

const app = angular.module('dataiku.datasets');

app.controller("StreamOrientedDatasetControllerFragment", function($scope, $rootScope, $stateParams, MonoFuture, $q,
                Assert, DataikuAPI, WT1, DatasetsService, Dialogs, DatasetUtils, LoggerProvider,
                FutureProgressModal, CreateModalFromTemplate, PluginConfigUtils) {

    var Logger = LoggerProvider.getLogger('datasets.stream');

    $scope.dataset.formatParams = $scope.dataset.formatParams || {};

    function applyCustomFormatDefaults() {
        if ($scope.dataset.formatType && 
            ($scope.dataset.formatType.startsWith("jformat") || $scope.dataset.formatType.startsWith("format"))) {
            $scope.dataset.formatParams.config = $scope.dataset.formatParams.config || {};
            if ($scope.formats && $scope.formats[$scope.dataset.formatType]) {
                $scope.formats[$scope.dataset.formatType].params.some(function(elt) {
                    if (elt.type === "autoconfig") {
                        PluginConfigUtils.setDefaultValues(elt.params, $scope.dataset.formatParams.config);
                        return true;
                    }
                    return false;
                });
            }
        }
    }

    function filterAvailableFormats(loadedPlugins, currentFormatType) {
        const visibilityFilter = PluginConfigUtils.shouldComponentBeVisible(loadedPlugins, currentFormatType, format => format.name);

        return Object.values($scope.formats)
            .filter(visibilityFilter)
            .reduce((acc, item) => {
                acc[item.name] = item;
                return acc;
            }, {});
    }

    $scope.availableFormats = filterAvailableFormats($rootScope.appConfig.loadedPlugins, $scope.dataset.formatType);

    $scope.detectScheme = function() {
        return DataikuAPI.datasets.detectFilePartitioning($scope.dataset);
    };
    $scope.testScheme = function() {
        return DataikuAPI.datasets.testFilePartitioning($scope.dataset);
    };

    $scope.setSchemaUserModified = function() {
        $scope.schemaJustModified = true;
        $scope.dataset.schema.userModified = true;
    };

    $scope.inferStorageTypesFromData = function(){
        Dialogs.confirm($scope,
            "Infer storage types from data",
            "This only takes into account a very small sample of data, and could lead to invalid data. "+
            "For safer typing of data, use a prepare recipe.").then(function(){
                $scope.preview(true);
        });
    };

    function waitTestDetectFuture(mfWrapped){
        var deferred = $q.defer();
        mfWrapped.success(function(data) {
            $scope.testingFuture = null;
            deferred.resolve(data.result);
        }).update(function(data){
            $scope.testingFuture = data;
        }).error(function (data, status, headers) {
            $scope.testingFuture = null;
            $scope.detectionResults = null;
            $scope.doingPreviewOrDetection = false;
            setErrorInScope.bind($scope)(data, status, headers);
        });
        return deferred.promise;
   }

    $scope.detectFormat = function () {
        //$scope.detectionResults = null;
        $scope.doingPreviewOrDetection = true;
        WT1.event("dataset-detectformat-start", {datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed})

        waitTestDetectFuture(MonoFuture($scope).wrap(DataikuAPI.datasets.detect_format)($stateParams.projectKey, $scope.dataset))
            .then(function(dRes){
            Logger.info('Got detection result', dRes, $scope.dataset.params);
            $scope.detectionResults = dRes;
            $scope.doingPreviewOrDetection = false;
            if (!$scope.dataset.name && !$scope.uiState.new_dataset_name_manually_edited) {
                $scope.uiState.new_dataset_name = $scope.detectionResults.suggestedName;
            }
            if ($scope.detectionResults.connectionOK) {
                if ($scope.detectionResults.empty) {
                    $scope.updateTableFromPreviewResult();
                } else {
                    if ($scope.detectionResults.format) {
                        /* Load the detected format params in the dataset */
                        $scope.dataset.formatType = $scope.detectionResults.format.type;
                        $scope.dataset.formatParams = $scope.detectionResults.format.params || {};

                        applyCustomFormatDefaults();
        
                        $scope.updateTableFromPreviewResult();
                    } else {
                        //  $scope.detectionResults.format = { table : null,
                        //     errorMessage : 'Failed to detect a format, please manually configure',
                        //     metadata : {}
                        // };
                    }
                }
                $scope.availableFormats = filterAvailableFormats($rootScope.appConfig.loadedPlugins, $scope.dataset.formatType);
            }
            getDigestTime($scope, function(time) {
                var d = $.isEmptyObject($scope.dataset.formatParams) ? "failed" : JSON.stringify($scope.dataset.formatParams);
                WT1.event("dataset-detectformat-done",
                    {datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed, digestTime : time,
                     detectedType : $scope.dataset.formatType, detectedParams :  d});
            });
        });//.error(setErrorInScope.bind($scope));
    };

    $scope.$watch("dataset.formatType", function() {
    	if (!$scope.dataset.formatType) return;
        /* get plugin desc of format if needed*/
        if ($scope.dataset.formatType.startsWith("jformat")) {
            $scope.loadedDesc = $rootScope.appConfig.customJavaFormats.filter(function(x){
                return x.formatterType == $scope.dataset.formatType;
            })[0];
            $scope.pluginId = $scope.loadedDesc ? $scope.loadedDesc.ownerPluginId : null;
        }
        if ($scope.dataset.formatType.startsWith("format")) {
            $scope.loadedDesc = $rootScope.appConfig.customPythonFormats.filter(function(x){
                return x.formatterType == $scope.dataset.formatType;
            })[0];
            $scope.pluginId = $scope.loadedDesc ? $scope.loadedDesc.ownerPluginId : null;
        }

        applyCustomFormatDefaults();

    });

    $scope.preview = function (inferStorageTypes) {
        $scope.doingPreviewOrDetection = true;
        if (inferStorageTypes == null) inferStorageTypes = false;

        WT1.event("dataset-preview-start", {datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed})
        waitTestDetectFuture(MonoFuture($scope).wrap(DataikuAPI.datasets.preview)($stateParams.projectKey, $scope.dataset, inferStorageTypes))
        .then(function (data) {
            Logger.info('Got preview result', data, $scope.dataset.params);
            $scope.clearParamChangeInfo();

            $scope.detectionResults = data;
            $scope.doingPreviewOrDetection = false;
            if (!$scope.firstPreviewDone) {
                if (data.connectionOK) {
                    //$scope.goToPreview();
                }
                $scope.firstPreviewDone = true;
            }
            //$scope.testOrDetectPartitioningIfNeeded();
            $scope.updateTableFromPreviewResult();
            getDigestTime($scope, function(time) {
                WT1.event("dataset-preview-done", {
                    datasetType : $scope.dataset.type,
                    datasetManaged : $scope.dataset.managed,
                    digestTime : time,
                    formatType : $scope.dataset.formatType,
                });
            });
        })
    };

    /* Manual change of format type : redo a detection limited to this type*/
    $scope.onFormatTypeChanged = function () {
        // eslint-disable-next-line no-undef
        clear($scope.dataset.formatParams);

        WT1.event("dataset-detectformat-onetype-start",{
                    datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed, formatType : $scope.dataset.formatType});

        $scope.availableFormats = filterAvailableFormats($rootScope.appConfig.loadedPlugins, $scope.dataset.formatType);

        $scope.doingPreviewOrDetection = true;
        waitTestDetectFuture(MonoFuture($scope).wrap(DataikuAPI.datasets.detectOneFormat)($stateParams.projectKey, $scope.dataset, $scope.dataset.formatType))
            .then(function(data) {
            $scope.doingPreviewOrDetection = false;
            $scope.detectionResults = data;
            if (!$scope.detectionResults.format) {
                /* Could not detect anything for this format ... */
                // eslint-disable-next-line no-undef
                clear($scope.dataset.formatParams);
                $scope.detectionResults.format = { table : null,
                    errorMessage : 'Failed to detect suitable parameters for this format, please manually configure',
                    metadata : {}
                };
            } else {
                var fmt = $scope.detectionResults.format;
                Assert.trueish(fmt.type == $scope.dataset.formatType, 'detected format is not the current dataset format');
                // Don't change formatParams, clear and refill
                if ($scope.dataset.formatParams == null) $scope.dataset.formatParams = {}
                // eslint-disable-next-line no-undef
                mapCopyContent($scope.dataset.formatParams, fmt.params);
                
                applyCustomFormatDefaults();
            }
            $scope.updateTableFromPreviewResult();

            WT1.event("dataset-detectformat-onetype-done",{
                datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed, formatType : $scope.dataset.formatType});
        });
    };

    /* Change of format params : handle a few odd cases and then update preview */
    $scope.onFormatParamsChanged = function () {
        if ($scope.dataset.formatParams.style == 'excel' && $scope.dataset.formatParams.quoteChar != '"') {
            $scope.dataset.formatParams.quoteChar = '"';
        }

        if ($scope.dataset.formatType === 'excel') {
            //The backend really needs this to be false specificially if sheetsToColumn was never set
            if (_.isNil($scope.dataset.formatParams.sheetsToColumnOld)) {
                $scope.dataset.formatParams.sheetsToColumnOld = false;
            }
        }

        /*  User changed format params, so stop changing them automatically if core params change */
        $scope.redetectFormatOnCoreParamsChanged = false;
        $scope.preview();
    };

    $scope.clearParamChangeInfo = function() {
        if ($scope.dataset.formatType === 'excel') {
            if ($scope.dataset.formatParams) {
                $scope.dataset.formatParams.sheetsToColumnOld = Boolean($scope.dataset.formatParams.sheetsToColumn);
            }
        }
    };

    $scope.updateTableFromPreviewResult = function () {
        Assert.inScope($scope, 'detectionResults');
        var dRes = $scope.detectionResults;
        if (!dRes.connectionOK) {
            dRes.format = { table : null, errorMessage : "Connection failed: " + dRes.connectionErrorMessage };
            return;
        }
        $scope.consistency = { empty : dRes.empty };
        $scope.consistency.kind = DatasetUtils.getKindForConsistency($scope.dataset);
        if (dRes.empty) {
            if ($scope.dataset.formatType == null) {
                $scope.dataset.formatType = 'csv';
                $scope.dataset.formatParams = {
                    'separator': '\t',
                    'charset': 'utf8'
                };
            }
        } else {
            Assert.trueish(dRes.format, 'no format detected');
            $scope.consistency.result = dRes.format.schemaDetection;
            if (dRes.format.schemaDetection) {
                $scope.dataset.schema = dRes.format.schemaDetection.newSchema;
            }
            $scope.schemaJustModified = false;
        }
    };

    $scope.clearDataset = function(){
        DatasetsService.clear($scope, $scope.dataset.projectKey, $scope.dataset.name).then(function(){
            $scope.preview();
        });
    };

    $scope.overwriteSchema = function(newSchema) {
        WT1.event("dataset-discard-schema-changed",{
                    datasetType : $scope.dataset.type, datasetManaged : $scope.dataset.managed});
        $scope.dataset.schema =angular.copy(newSchema);
        $scope.schemaJustModified = false;
        $scope.consistency = null;
        // Trigger a new preview
        $scope.preview();
    };

    $scope.discardConsistencyError= function(){
        $scope.consistency = null;
    };

    $scope.schemaIsUserModified = function () {
        return $scope.dataset.schema != null && $scope.dataset.schema.userModified;
    };

    $scope.testSchemaConsistencyOnAllFiles = function(){
        Dialogs.confirm($scope, "Test schema consistency on all files",
            "This operation can be very slow if your dataset has many files").then(function(){
                DataikuAPI.datasets.testSchemaConsistencyOnAllFiles($scope.dataset).success(function(data){
                    FutureProgressModal.show($scope, data, "Checking schema consistency").then(function(result){
                        CreateModalFromTemplate("/templates/datasets/modals/all-files-schema-consistency-modal.html", $scope, null, function(newScope){
                            newScope.result = result;
                        })
                    });
                }).error(setErrorInScope.bind($scope));
        });
    }

    $scope.dataset.formatParams = $scope.dataset.formatParams || {};

});


app.controller("TwitterStreamDatasetController", function($scope, $stateParams, Assert, DataikuAPI, $timeout, WT1) {
    $scope.onLoadComplete = function () {
        Assert.inScope($scope, 'dataset');
        Assert.trueish($scope.dataset.type == "Twitter", 'not a Twitter dataset');

        if (!$scope.dataset.params.keywords) {
            $scope.dataset.params.keywords = [];
        }

        if (!$scope.dataset.formatType) {
            $scope.dataset.formatType = "csv";
        }

        if ($scope.dataset.formatParams == null || Object.keys($scope.dataset.formatParams).length === 0) {
            $scope.dataset.formatParams = {
                "quoteChar": "\"",
                "escapeChar": "\\",
                "style": "unix",
                "charset": "utf8",
                "arrayMapFormat": "json",
                "parseHeaderRow": "false",
                "separator": "\t"
            };
        }
        $scope.dataset.partitioning = {
            "filePathPattern": "%Y/%M/%D/%H/.*",
            "ignoreNonMatchingFile": false,
            "considerMissingRequestedPartitionsAsEmpty": false,
            "dimensions": [
            {
              "name": "date",
              "type": "time",
              "params": {
                "period": "HOUR"
              }
            }
            ]
        };

        if(!$scope.dataset.schema){
            $scope.dataset.schema = {userModified : false};
        }

        if(!$scope.dataset.schema.columns){
            $scope.dataset.schema.columns = [];
        }

        $scope.toggleField = function(field){
            if($scope.containsColumn(field)){ // remove
                $scope.dataset.schema.columns = $scope.dataset.schema.columns.filter(
                    function(e){ return e.name != field; });
            } else {
                $scope.dataset.schema.columns.push({
                    "name": field,
                    "type": "string",
                    "maxLength": 1000
                });
            }
        }

        $scope.containsColumn = function(name){
            for(var c in $scope.dataset.schema.columns){
                if($scope.dataset.schema.columns[c]["name"] == name){
                    return true;
                }
            }
            return false;
        }

        if($scope.dataset.schema.columns.length === 0){
            $scope.dataset.schema.columns.push({
                "name": "id_str",
                "type": "string",
                "maxLength": 1000
            });
            $scope.dataset.schema.columns.push({
                "name": "created_at",
                "type": "string",
                "maxLength": 1000
            });
            $scope.dataset.schema.columns.push({
                "name": "user.screen_name",
                "type": "string",
                "maxLength": 1000
            });
            $scope.dataset.schema.columns.push({
                "name": "text",
                "type": "string",
                "maxLength": 1000
            });
        }

        $scope.keywordItems = $scope.dataset.params.keywords.map(function(f) { return { keyword: f }; });
    }

    $scope.keywordsChanged = function(newKeywords = []) {
        [].splice.apply($scope.dataset.params.keywords,
            [0, $scope.dataset.params.keywords.length].concat(newKeywords.map(function(fi) { return fi.keyword; })));
    };

    $scope.isReady = function(){
        return !angular.isUndefined($scope.dataset.params.keywords) &&
                ($scope.dataset.params.keywords.length !== 0) && ($scope.connectionTwitter) && $scope.dataset.params.path && $scope.dataset.params.path!='/';
    };

    $scope.$watch("uiState.new_dataset_name", function(nv, ov) {
        if (nv && nv.length &&
            (($scope.dataset.params.path == "/") || !$scope.dataset.params.path || (ov && $scope.dataset.params.path == ('/'+$stateParams.projectKey + '.' + ov)))) {
            $scope.dataset.params.path =  "/" + $stateParams.projectKey + '.' + nv;
        }
    }, true);

    $scope.isRunning = false;
    $scope.hasData = false;

    $scope.saveHooks.push(function() {
        if($scope.dataset && $scope.dataset.params) {
            var path = $scope.dataset.params.path;
            return path && path!='/';
        }
        return false;
    });

    // isRunning is true if there's a twitter capture of the current dataset
    // hasData is true if at least a tweet has been written

    if (angular.isDefined($scope.dataset.name)) {
        DataikuAPI.datasets.getTwitterStatus($stateParams.projectKey, $scope.dataset.name).success(function(res) {
            $scope.isRunning = res.isStarted;
            $scope.hasData = res.hasData;
        });
    }

    if (angular.isUndefined($scope.dataset.params.path)) {
        $scope.dataset.params.path = '/';
    }

    // If there's an active twitter connection, we retrieve it...
    if (angular.isUndefined($scope.connectionTwitter)) {
        DataikuAPI.connections.getTwitterConfig().success(function(data) {
            $scope.connectionTwitter = data.connection;
        }).error(function(){
            setErrorInScope.bind($scope);
        });
    }

    // ... Otherwise, we see if there's a twitter connection
    if(!$scope.connectionTwitter){
        DataikuAPI.connections.getNames('Twitter').success(function (data) {
            $scope.connectionsTwitter = data;
            if ($scope.connectionsTwitter.length > 0) {
                $scope.connectionTwitterSelection = $scope.connectionsTwitter[0];
            }
            $scope.hasConnectionsTwitter = $scope.connectionsTwitter.length > 0;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.toggleStreaming = function(start){
        DataikuAPI.datasets.controlTwitterStreaming($stateParams.projectKey, $scope.dataset.name, start).success(function() {
            $scope.isRunning = start;
        }).error(function() {
            setErrorInScope.bind($scope);
            DataikuAPI.datasets.getTwitterStatus($stateParams.projectKey, $scope.dataset.name).success(function(res) {
                $scope.isRunning = res.isStarted == "true" ? true : false;
            });
        });
    };

    DataikuAPI.connections.getTypeAndNames('Filesystem').success(function (data) {
        $scope.connections = data.map(connection => connection['name']);
        $scope.connectionsWithDescriptions = data;
        if ($scope.connections.length > 0 && ($scope.dataset.params.connection == null || $scope.dataset.params.connection.length === 0)) {
            for(var i = 0; i < $scope.connections.length; i++){
                if($scope.connections[i] == "filesystem_managed"){
                    $scope.dataset.params.connection = $scope.connections[i];
                } else {
                    $scope.dataset.params.connection = $scope.connections[0];
                }
            }
        }
    }).error(setErrorInScope.bind($scope));
});


/** For the moment, we have one controller for managed fs-like and external ... */
app.controller("ManagedFSLikeDatasetController", function($scope, $controller) {
    $controller("ExternalStreamOrientedDatasetController", {$scope:$scope});
});


app.controller("ExternalStreamOrientedDatasetController", function($scope, $controller, $stateParams, DataikuAPI, $timeout, WT1, CreateModalFromTemplate) {
    $controller("StreamOrientedDatasetControllerFragment", {$scope:$scope});

    
    $scope.listItemsEnabled = false;

    // Call this from a child controller if you want to manage your file list view in fs-files-listing-v2.html/list-items-fattable.html or anything else managed with filteredMultiSelectRows
    // Ensures listItems is available is scope - which is the file paths with the selections managed by list-items-fattable.html, which expects this in scope
    // Child controllers should of course not CHANGE the listItems reference (they can edit the list but not point to a new list) - it will cause some tricky bugs!
    $scope.enableListItems = function() {
        $scope.listItemsEnabled = true;
    }

    // We often need to make sure the list order is reapplied in response to a user selection
    $scope.refreshItemListOrderAndSelection = function() {
        if ($scope.listItemsEnabled && $scope.listItems) {
            $scope.$broadcast('refresh-sort-and-selection');
        }
    }

    $scope.onLoadComplete = function () {
        $scope.uiState.activeTab = $scope.dataset.type.startsWith('Sample_') ? 'preview': 'connection';
        $scope.uiState.autoTestOnFileSelection = true;
        $scope.uiState.listFilesSelectedOnly = false;
        $scope.clearParamChangeInfo();

        if ($stateParams.datasetName) {
            /* If, when we arrive, the dataset does not have a format yet, then each time the core params
             * change, we'll redo a detection. That allows us to better react to the core params actually
             * targeting different files during the process
             */
            if (angular.isDefined($scope.dataset.formatParams)) {
                $scope.redetectFormatOnCoreParamsChanged = false;
                $scope.uiState.autoTestOnFileSelection = false; // No need to auto-test as it will be done for each file in bulk selection and the user can manually do that once for all of them
            } else {
                $scope.redetectFormatOnCoreParamsChanged = true;
            }
            /* Fire initial preview-detection */
            //$scope.preview();
        }
    };

    $scope.getFilesListingHeight = function(filesListing) {
        return Math.min(28*filesListing.paths.length, 235);
    }

    $scope.getAlertClassForResultsOnConnectionTab = function(dRes){
        if (!dRes) return "";
        if (!dRes.connectionOK) return "alert-error";
        if (dRes.empty) return "alert-warning";
        if (!dRes.format || !dRes.format.ok) return "alert-error";
        if (dRes.format && dRes.format.ok && dRes.format.schemaDetection && dRes.format.schemaDetection.warningLevel == 'WARN') return "alert-warning";
        return "alert-success";
    }
    $scope.getAlertClassForResultsOnPreviewTab = function(dRes) {
        if (!dRes) return "";
        if (!dRes.connectionOK) return "alert-error";
        if (dRes.empty) return "alert-warning";
        if (!dRes.format || !dRes.format.ok) return "alert-error";
        return "ng-hide"; // Don't display if OK
    }

    $scope.startUpdateFromHive = function() {
        CreateModalFromTemplate("/templates/datasets/fragments/update-from-hive-modal.html", $scope, "UpdateDatasetFromHiveController");
    };

    $scope.detectOrPreview = function(){
        var shouldDetect = ($scope.dataset.formatParams == null || $scope.dataset.formatParams.length === 0 || angular.isUndefined($scope.dataset.formatType) || $scope.redetectFormatOnCoreParamsChanged);

        if (shouldDetect) {
            $scope.detectFormat();
        } else {
            $scope.preview();
        }
    }

    /* Params changed : trigger smart detection-or-preview */
    $scope.onCoreParamsChanged = function(){
        if ($scope.uiState.autoTestOnFileSelection) $scope.detectOrPreview();
    };

    /* Manual force format redetection: drop current data and trigger detection */
    $scope.forceFormatRedetection = function () {
        $scope.dataset.formatType = null;
        // eslint-disable-next-line no-undef
        clear($scope.dataset.formatParams); // formatParams is watched, never reassign it.
        $scope.detectFormat();
    };

     $scope.listFiles = function(providerType){
        DataikuAPI.fsproviders.listFiles(providerType, $scope.dataset.params, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {},
            $scope.dataset.params.filesSelectionRules, $scope.uiState.listFilesSelectedOnly).success(function(data){
                    $scope.uiState.filesListing = data;
                    if ($scope.listItemsEnabled ) {
                        $scope.$applyAsync($scope.populateListItems);
                    }
            }).error(setErrorInScope.bind($scope));
    }

    $scope.populateListItems = function() {
            $scope.listItems = $scope.uiState.filesListing.paths;
            for (let item of $scope.listItems) {
                item.$selected = item.selected;
            }
    }

    $scope.addExplicitSelect = function(path) {
        path.selected = true;
        $scope.dataset.params.filesSelectionRules.explicitFiles.push(path.path);
        $scope.dataset.params.filesSelectionRules.explicitFiles = [].concat($scope.dataset.params.filesSelectionRules.explicitFiles); // touch the array to force a refresh of the listForm
        
        // Adjust selected files/size. 
        // For the FilesInFolderController this is redundant as applySelection rules is called afterwards, but for others it is needed
        $scope.uiState.filesListing.selectedFiles ++;
        $scope.uiState.filesListing.selectedSize += path.size;
    };

    /* itemsToRemove: ruleIndex -> item[] */
    $scope.removeExplicitSelects = function(itemsToRemove) {
        let newExplicitFiles = [];
        const explicitFiles = $scope.dataset.params.filesSelectionRules.explicitFiles;
        for (let i = 0; i < explicitFiles.length; i++) {
            if (i in itemsToRemove) {
                for (let item of itemsToRemove[i]) {
                    item.selected = false;
                    $scope.uiState.filesListing.selectedFiles--;
                    $scope.uiState.filesListing.selectedSize -= item.size;
                }
            } else {
                newExplicitFiles.push(explicitFiles[i]);
            }
        }
        $scope.dataset.params.filesSelectionRules.explicitFiles = newExplicitFiles;
    };

    $scope.addIncludeRule = function(path) {
        path.selected = true;
        var p = path.path;
        if (p.startsWith('/')) p = p.substring(1);
        $scope.dataset.params.filesSelectionRules.includeRules.push({expr:p,mode:'GLOB',matchingMode:'FULL_PATH'});
        $scope.dataset.params.filesSelectionRules.includeRules = [].concat($scope.dataset.params.filesSelectionRules.includeRules); // touch the array to force a refresh of the listForm

        // Adjust selected files/size. 
        // For the FilesInFolderController this is redundant as applySelection rules is called afterwards, but for others it is needed
        $scope.uiState.filesListing.selectedFiles++;
        $scope.uiState.filesListing.selectedSize+= path.size;
    }

    $scope.addExcludeRule = function(path) {
        path.selected = false;
        var p = path.path;
        if (p.startsWith('/')) p = p.substring(1);
        $scope.dataset.params.filesSelectionRules.excludeRules.push({expr:p,mode:'GLOB',matchingMode:'FULL_PATH'});
        $scope.dataset.params.filesSelectionRules.excludeRules = [].concat($scope.dataset.params.filesSelectionRules.excludeRules); // touch the array to force a refresh of the listForm

        // Adjust selected files/size.
        // For the FilesInFolderController this is redundant as applySelection rules is called afterwards, but for others it is needed
        $scope.uiState.filesListing.selectedFiles--;
        $scope.uiState.filesListing.selectedSize-= path.size;
    };

    $scope.redrawFatTable = function() {
        $timeout(function() {
            $scope.$broadcast("redrawFatTable");
        });
    };

    function updateCreationWarning(){
        let warnings = [];
        $scope.uiState.creationWarning = null;
        if (!$scope.uiState.new_dataset_name) return;

        if (["HDFS", "Filesystem", "SCP", "SFTP", "FTP", "S3", "GCS", "Azure", "SharePointOnline"].indexOf($scope.dataset.type) >= 0) {
            if (!$scope.dataset.params || !$scope.dataset.params.path || "/" == $scope.dataset.params.path || "" == $scope.dataset.params.path) {
                warnings.push("Dataset at root of connection. This is atypical. Do you want to create a managed dataset?");
            }
        }

        if ($scope.dataset.type != 'Inline') {
            if (!$scope.dataset.formatType || !$scope.dataset.formatParams) {
                warnings.push("No format configured, dataset won't be usable");
            }
            if (!$scope.dataset.schema || !$scope.dataset.schema.columns || !$scope.dataset.schema.columns.length) {
                warnings.push("No schema set, dataset won't be usable");
            }
        }
        if (warnings.length > 0) {
            $scope.uiState.creationWarning = warnings.join("; ");
        }
    }


    $scope.$watch("dataset", updateCreationWarning, true);
    $scope.$watch("uiState.new_dataset_name", updateCreationWarning);

});

app.controller("UpdateDatasetFromHiveController", function($scope, DataikuAPI, $stateParams) {
    var handleHiveImportability = function(data) {
        var messages = data.importability.messages;
        $scope.hiveDataset = data.importability.dataset;
        $scope.hiveSyncOutcome = messages.error ? 'FAILED' : (messages.warning ? 'WARNING' : 'SUCCESS');
        $scope.hiveSyncMessage = '';
        messages.messages.forEach(function(message) {
            $scope.hiveSyncMessage = $scope.hiveSyncMessage + "\n" + (message.details || message.message);
        });
        if ($scope.hiveDataset == null && $scope.hiveSyncOutcome == 'SUCCESS') {
            // mmh. should not be null.
            $scope.hiveSyncOutcome = 'FAILED';
            $scope.hiveSyncMessage = $scope.hiveSyncMessage || "No dataset could be built from table";
        } else {
            $scope.connectionIsSubdirSynchronized = data.connectionIsSubdirSynchronized;
            $scope.schemaIncompatibilities = data.schemaIncompatibilities;
            $scope.connectionIncompatibility = data.connectionIncompatibility;
            $scope.pathIncompatibility = data.pathIncompatibility;
            $scope.partitioningIncompatibility = data.partitioningIncompatibility;
        }
    };

    $scope.hasIncompatibilities = function() {
        return $scope.connectionIsSubdirSynchronized || ($scope.schemaIncompatibilities && $scope.schemaIncompatibilities.length) || $scope.connectionIncompatibility || $scope.pathIncompatibility || $scope.partitioningIncompatibility;
    };

    $scope.checkHiveImportability = function(importability) {
        $scope.hiveCheckInProgress = true;
        DataikuAPI.datasets.checkHiveSync($stateParams.projectKey, $stateParams.datasetName).success(function (data) {
            $scope.hiveCheckInProgress = false;
            handleHiveImportability(data);
        }).error(function(a,b,c) {
            $scope.hiveCheckInProgress = false;
            setErrorInScope.bind($scope)(a,b,c);
        });
    };


    $scope.checkHiveImportability();
});

app.service("FSProviderUtils", function(){

})

app.directive('fsFilesSelection', ($timeout, FilePatternUtils) => {
    return {
        restrict: 'A',
        templateUrl: '/templates/datasets/fragments/fs-files-selection.html',
        scope: {
            params : '=',
            // this flag is here so we can clear hard-coded styles in the template one area (connection type) at a time, then eventually remove them
            clearStyleHacks : '@?'
        },
        link: function($scope) {

            $scope.validateRule = function(rule) {
                // trigger a revalidation in the expression by appending a space (simple impl, room for improvement)
                // (timeout is used as otherwise it the changes is applied before the rule mode is changed)
                $timeout(() => {
                    if (rule && rule.expr) {
                        rule.expr = rule.expr + " "; 
                    }
                });
            };

            $scope.isRegExpValid = function(expr) {
                if (!expr) return true;
                try {
                    new RegExp(expr.trim());
                } catch (e) {
                    return false;
                }
                return true;
            };
        
            $scope.isGlobValid = function(globExpr) {
                if (!globExpr) {
                    return true;
                }

                // test we can process with fullGlobRegExp - that is the superset of patterns
                // This is not a strict test but the backend seems to allow anything anyway, with similar code
                try {
                    FilePatternUtils.fullPathGlobRegExp(globExpr);
                } catch(e) {
                    return false;
                }
                
                return true;
            };

            $scope.filesSelectionRulesModes = [['ALL', 'All'],
                                               ['EXPLICIT_SELECT_FILES', 'Explicitly select files'],
                                               ['RULES_ALL_BUT_EXCLUDED', 'All but excluded'],
                                               ['RULES_INCLUDED_ONLY', 'Only included']];
            $scope.filesSelectionRulesModesDesc = ['No filtering',
                                                   'Select individual files from the directory at Path',
                                                   'Any file from the directory at Path that match a rule is ignored ',
                                                   'Only files from the directory at Path that match a rule are taken'];
            if (!$scope.params.filesSelectionRules) {
                $scope.params.filesSelectionRules = {
                    mode: "ALL", includeRules:[], excludeRules:[], explicitFiles:[]
                };
            }
            $scope.prepareNewRule = function(rule) {
                if (!rule.mode) rule.mode = "GLOB";
                if (!rule.matchingMode) rule.matchingMode = "FULL_PATH";
            }
        }
    };
});

app.directive('fsProviderBucketSelector', function(DataikuAPI, $stateParams, WT1, Debounce) {
    return {
        restrict: 'A',
        templateUrl: '/templates/datasets/fs-provider-bucket-selector.html',
        scope: {
            providerType : '=',
            projectKey : '=',
            config : '=',
            contextVars : '=',
            bucketLabel : '=',
            bucketProperty : '='
        },
        link: function($scope, element, attrs) {
            $scope.selectorUiState = {
                mode:'CUSTOM',
                fetchingBucketList: false,
                couldListBuckets: null,
                bucketsListError: null,
                buckets: null
            };

            var fetchConnectionMetadata = function() {
                if ($scope.config == null || $scope.config.connection == null) return;
                $scope.selectorUiState.bucketsListError = null;
                DataikuAPI.fsproviders.testConnection($scope.providerType, $scope.config, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {}, false)
                    .success(function (data) {
                        $scope.connMeta = data;
                    })
                    .error(function(data, status, headers) {
                        setErrorInScope.bind($scope)(data, status, headers);
                    });
            };
            const fetchConnectionMetadataWithReinitialization = function() {
                $scope.selectorUiState.buckets = null;
                fetchConnectionMetadata();
            };
            fetchConnectionMetadata();
            $scope.$watch("config.connection", fetchConnectionMetadataWithReinitialization);
            $scope.$watch('config.' + $scope.bucketProperty, Debounce().withDelay(200, 1000).withScope($scope).wrap(fetchConnectionMetadata));

            $scope.fetchBucketList = function() {
                $scope.selectorUiState.fetchingBucketList = true;
                DataikuAPI.fsproviders.testConnection($scope.providerType, $scope.config, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {}, true)
                .success(function (data) {
                    $scope.connMeta = data;
                    if (data.buckets) {
                        $scope.selectorUiState.buckets = data.buckets.split(",");
                    } else {
                        $scope.selectorUiState.buckets = null;
                    }
                    $scope.selectorUiState.couldListBuckets = data.couldListBuckets;
                    $scope.selectorUiState.bucketsListError = data.bucketsListError;
                    $scope.selectorUiState.mode = 'SELECT';
                })
                .error(setErrorInScope.bind($scope))
                .finally(function() {$scope.selectorUiState.fetchingBucketList = false;});
            };
        }
    };
});

app.directive('fsProviderSharepointDriveSelector', function(DataikuAPI, $stateParams, WT1, Debounce, Logger, Dialogs) {
    return {
        restrict: 'A',
        templateUrl: '/templates/datasets/fs-provider-sharepoint-drive-selector.html',
        scope: {
            providerType : '=',
            projectKey : '=',
            config : '=',
            contextVars : '=',
            bucketLabel : '=',
            bucketProperty : '='
        },
        link: function($scope, element, attrs) {
            $scope.selectorUiState = {
                fetchingSitesAndDrives: false,
                sharePointError: null,
                sharePointWarning: null,
                sites: null,
                drives: null,
                siteSearch: null
            };
            // Flag dirtiness of the site search input
            // It will reset after a site fetch. It is meant to track user input and allows to distinguish between
            // a user input search and a user click site switch. Not a perfect solution but works for now since
            // the watcher on `config.siteId` does not know if the change is due to a user click or a user input
            $scope.userEditedSiteSearch = false;

            var fetchConnectionMetadata = function() {
                if ($scope.config == null || $scope.config.connection == null) return;
                $scope.selectorUiState.sharePointError = null;
                const params = {...$scope.config};
                if (params.siteId) {
                    params.siteSearch = undefined;
                }
                DataikuAPI.fsproviders.testConnection($scope.providerType, params, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {}, false)
                    .success(function (data) {
                        $scope.connMeta = data;
                    })
                    .error(function(data, status, headers) {
                        setErrorInScope.bind($scope)(data, status, headers);
                    });
            };
            const fetchConnectionMetadataWithReinitialization = function() {
                $scope.selectorUiState.sites = null;
                $scope.selectorUiState.drives = null;
                fetchConnectionMetadata();
                if ($scope.config && $scope.config.connection) {
                    if ($scope.config.siteSearch != null && $scope.selectorUiState.siteSearch==null) {
                        $scope.selectorUiState.siteSearch = $scope.config.siteSearch;
                    }
                    $scope.fetchSitesAndDrives().then(() =>{
                        // The first fetch might not include drives since the siteID was not available
                        if ($scope.config.drive && $scope.selectorUiState.drives == null) {
                            $scope.fetchSitesAndDrives();
                        }
                    });
                }
            };

            /**
             * Update the site and drive ID in case the config params only has site name and drive name
             * Typically in case of fetching settings in a managed folder with sharepoint connection
             */
            $scope.updateSiteAndDriveId = function updateSiteAndDriveId() {
                if ($scope.config == null) return;
                if ($scope.selectorUiState.sites == null) {
                    Logger.warn('Sharepoint site list is not defined, cannot update site ID');
                    return;
                }
                if ($scope.selectorUiState.drives == null) {
                    Logger.warn('Sharepoint drive list is not defined, cannot update drive ID');
                    return;
                }

                // Convert the site name to site ID and immediately clear the site name so that only the site ID is used in the backend
                if ($scope.config.siteId == null && $scope.config.site != null) {
                    const site = $scope.selectorUiState.sites.find(site => site.name === $scope.config.site);
                    if (site != null) {
                        $scope.config.siteId = site.id;
                        $scope.config.site = null;
                    } else {
                        Logger.error('Sharepoint site ' + $scope.config.site + ' does not exist');
                    }
                }
                
                if ($scope.config.driveId == null && $scope.config.drive != null) {
                    const drive = $scope.selectorUiState.drives.find(drive => drive.name === $scope.config.drive);
                    if (drive != null) {
                        $scope.config.driveId = drive.id;
                        $scope.config.drive = null;
                    } else {
                        Logger.error('Sharepoint drive ' + $scope.config.drive + ' does not exist');
                    }
                }
            }

            $scope.fetchSitesAndDrives = function() {
                $scope.selectorUiState.fetchingSitesAndDrives = true;
                Logger.info('Fetching sites and drives');
                if ($scope.selectorUiState.siteSearch != null) {
                    $scope.config.siteSearch = $scope.selectorUiState.siteSearch + "*";
                } else {
                    $scope.config.siteSearch = $scope.selectorUiState.siteSearch;
                }

                return DataikuAPI.fsproviders.testConnection($scope.providerType, $scope.config, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {}, true)
                .success(function (data) {
                    const MAX_DISPLAYED_SITES = 1000;
                    $scope.connMeta = data;
                    $scope.selectorUiState.sharePointWarning = null;
                    if (data.sites) {
                        const sites = JSON.parse(data.sites);

                        if (sites.length > MAX_DISPLAYED_SITES) {
                            $scope.selectorUiState.sharePointWarning = "Your search returned "
                                + sites.length + " sites. Only "
                                + MAX_DISPLAYED_SITES + " can be displayed. Please refine your search.";
                            sites.length = MAX_DISPLAYED_SITES;
                            $scope.selectorUiState.sites = sites;
                        } else {
                            $scope.selectorUiState.sites = sites;
                        }
                        // If the search does not return any site, we will use the previous siteId since it is for sure valid
                        // If the search only has one result
                        if (sites.length === 1) {
                            // Select the first site by default.
                            // If there is only one site that matches, the user must have
                            // input the exact site name.
                            $scope.config.siteId = sites[0].id;
                        } else if (sites.length > 1 && $scope.selectorUiState.siteSearch) {
                            // Greedily scan and match every site name to the user input.
                            // In case none match, the selected site will be the first one from the list.
                            for (const site of sites) {
                                if (site.name === $scope.selectorUiState.siteSearch) {
                                    $scope.config.siteId = site.id;
                                    break;
                                }
                            }
                        }
                    } else {
                        $scope.selectorUiState.sites = null;
                    }
                    if (data.drives) {
                        const drives = JSON.parse(data.drives);
                        $scope.selectorUiState.drives = drives;
                    } else {
                        $scope.selectorUiState.drives = null;
                    }
                    if (data.sharePointError) {
                        $scope.selectorUiState.sharePointError = data.sharePointError.split(": Error code")[0];
                    }
                    $scope.updateSiteAndDriveId();
                })
                .error(setErrorInScope.bind($scope))
                .finally(function() {
                    $scope.selectorUiState.fetchingSitesAndDrives = false;
                    $scope.config.siteSearch = null;
                });
            };

            $scope.searchAllSites = function() {
                $scope.selectorUiState.siteSearch = "";
                $scope.config.siteSearch = "";
                return $scope.fetchSitesAndDrives();
            };
            $scope.searchSite = function(){
                $scope.selectorUiState.sharePointError = null;
                if(!("siteSearch" in $scope.selectorUiState)  ||
                        $scope.selectorUiState.siteSearch != null &&
                        $scope.selectorUiState.siteSearch != '*' &&
                        $scope.selectorUiState.siteSearch != '') {
                    $scope.fetchSitesAndDrives();
                } else {
                    Dialogs.confirmUnsafeHTML($scope, "Really list all sites?",
                        "<p>This will list <strong>all sites</strong> of your tenant.</p>"+
                        "<p> On large entreprise SharePoints, this is likely "+
                        "to be long and to cause your browser to become unresponsive.</p>").then($scope.searchAllSites.bind())
                };
            }
            $scope.editSiteSearch = function() {
                // Set the input field to the site selected in the drive config
                var siteSearch = null;
                for (var site of $scope.selectorUiState.sites) {
                    if (site.id == $scope.config.siteId) {
                        siteSearch = site.name;
                        break;
                    };
                }
                $scope.selectorUiState.sites = null;
                $scope.selectorUiState.siteSearch = siteSearch;
                $scope.userEditedSiteSearch = true;
                $scope.selectorUiState.sharePointError = null;
            };

            // WATCHERS

            $scope.$watch('config.siteId', function(nv,ov) {
                // Every time the site id changes, we need to fetch again to update the list of associated drives
                if (nv !== ov) {
                    if (!$scope.userEditedSiteSearch) {
                        // If the user has not edited the site search, the change is due to a user select from the list
                        // We reset it because the existing search is not relevant anymore
                        $scope.selectorUiState.siteSearch = null;
                    }
                    $scope.fetchSitesAndDrives();
                    $scope.userEditedSiteSearch = false;
                }
            });

            $scope.$watch('config.connection', function(nv,ov) {
                if (nv !== ov && $scope.selectorUiState.fetchingSitesAndDrives == false) {
                    $scope.selectorUiState.siteSearch = $scope.config.site;
                    $scope.fetchSitesAndDrives();
                }
            });
            $scope.$watch("config.connection", fetchConnectionMetadataWithReinitialization);
            $scope.$watch('config.site', Debounce().withDelay(200, 1000).withScope($scope).wrap(fetchConnectionMetadata));
            // ON INIT
            fetchConnectionMetadata();
            if (($scope.config.connection !== null)
                && ($scope.selectorUiState.drives == null)
                && ($scope.selectorUiState.sites == null)
                && ($scope.selectorUiState.fetchingSitesAndDrives == false)
            ) {
                if (!$scope.config.siteId) {
                    $scope.selectorUiState.siteSearch = $scope.config.site;
                }
                $scope.fetchSitesAndDrives();
            }
        }
    };
});


app.controller("UploadWizardController",
/**
 * @typedef {Object} UploadWizardControllerScope
 * @property {number} totalNewFiles
 * @property {string} projectKey
 * @property {(drop: (uploadedFiles: FileList) => void, files: Array<UploadedFile>) => void} uploadWizardDrop
 * @property {(deleteFile: deleteFile, file: UploadedFile, event: Event) => void} uploadWizardDelete
 * @property {() => void} onCoreParamsChanged
 * @property {() => void} disableUnsavedChangesWarning
 * @property {() => void} startCreatingMultipleDatasets
 * @property {() => void} startCreatingSingleDataset
 * @property {() => boolean} isLastDataset
 * @property {() => void} createDatasetAndGoNext
 * @property {() => void} skipDatasetCreation
 * @property {() => void} onLoadComplete
 * @property {() => string} getAlertClassForDetectionResults
 * @property {() => void} detectOrPreview
 * @property {() => void} forceFormatRedetection
 * @property {() => boolean} isFixConfigPrimaryAction
 */
/**
 * 
 * @param {UploadWizardControllerScope} $scope 
 * @param {RootScope} $rootScope 
 * @param {Controller} $controller 
 * @param {StateParams} $stateParams 
 * @param {DataikuAPI} DataikuAPI 
 * @param {WT1} WT1 
 * @param {StateUtils} StateUtils 
 * @param {ActivityIndicator} ActivityIndicator 
 * @watch dataset
 * @watch uiState.new_dataset_name
 * @watch dataset.savedFiles
 * @watch $rootScope.totalNewFiles
 */
function($scope, $rootScope, $controller, $stateParams, DataikuAPI, WT1, StateUtils, ActivityIndicator) {
    $controller("StreamOrientedDatasetControllerFragment", {$scope:$scope});
    $scope.totalNewFiles = 0;

    $scope.uploadWizardDrop = function(drop, files){
        $scope.totalNewFiles = files.length;
        return drop(files);
    }

    $scope.uploadWizardDelete = function(deleteFile, file, $event){
        $scope.totalNewFiles = 0;
        deleteFile(file, $event);
    }
    $scope.projectKey = $stateParams.projectKey;

    $scope.onCoreParamsChanged = function() {
        // Nothing to do in this case
    };

    $scope.disableUnsavedChangesWarning = function() {
        $scope.origDataset = null;
    }

    function startCreatingDatasets(multipleDatasets) {
        return function(files){
            $scope.files = files;
            WT1.event("upload-files-dataset-create", {nFiles: $scope.files.length, nDatasets: multipleDatasets ? $scope.files.length : 1});
            $scope.uiState.multipleDatasets = multipleDatasets;
            $scope.uiState.multipleFilesPerDataset = $scope.files.length > 1 && !multipleDatasets;
            $scope.uiState.activeTab = "setupDataset";
            $scope.uiState.showDatasetNameZone = true;
            firstDataset();
        }
    }
    $scope.startCreatingMultipleDatasets = startCreatingDatasets(true);
    $scope.startCreatingSingleDataset = startCreatingDatasets(false);

    function firstDataset() {
        $scope.uiState.fileIndex = 0;
        if ($scope.uiState.multipleDatasets) {
            $scope.dataset.params.previewFile = $scope.files[$scope.uiState.fileIndex].path;
            $scope.dataset.params.sheetName = $scope.files[$scope.uiState.fileIndex].sheetName;
            setTitle(`Dataset creation: 1 of ${$scope.files.length}`);
        } else if ($scope.files.length > 1) {
            // Check if there is only one distinct file with multiple sheets in scope.files
            // (for excel each sheet in the same file is represented an item in scope.files)
            if ($scope.files.length === $scope.files.filter(f => f.path === $scope.files[0].path).length) {
                setTitle(`Single dataset with ${$scope.files.length} sheets`);
            } else {
                setTitle(`Single dataset with ${$scope.files.length} files`);
            }
            // Use the sheet names of the first file (since in the case of multiple Excel files with different sheet names,
            // the format param form only displays those in the first sheet)
            let sheetNames = $scope.files.filter(f => f.sheetName && f.path === $scope.files[0].path).map(f => f.sheetName).join("\n");
            if (!sheetNames) {
                sheetNames = "__DKU_ALL__"; // Give a hint to the backend to use all sheets for the preview/import in case we found no matching sheet
            }
            $scope.dataset.params.sheetName = sheetNames;
        }
        $scope.forceFormatRedetection();
    }

    function nextDataset() {
        $scope.uiState.fileIndex++;
        $scope.uiState.new_dataset_name = "";
        $scope.dataset.schema.userModified = false;
        $scope.dataset.name = null;
        $scope.dataset.params.previewFile = $scope.files[$scope.uiState.fileIndex].path;
        $scope.dataset.params.sheetName = $scope.files[$scope.uiState.fileIndex].sheetName;
        setTitle(`Dataset creation: ${$scope.uiState.fileIndex + 1} of ${$scope.files.length}`);
        $scope.detectionResults = null;
        $scope.forceFormatRedetection();
    }

    function setTitle(title) {
        $scope.uiState.newSettingsTitle = title;
    }

    $scope.isLastDataset = function() {
        if (!$scope.uiState.multipleDatasets) {
            return true;
        }
        return $scope.uiState.fileIndex === $scope.files.length - 1;
    }

    $scope.createDatasetAndGoNext = function() {
        $scope.dataset.name = $scope.uiState.new_dataset_name;
        if ($scope.uiState.multipleDatasets) {
            $scope.dataset.params.files = [$scope.files[$scope.uiState.fileIndex].path];
        }

        return DataikuAPI.datasets.create($stateParams.projectKey, $scope.dataset, $stateParams.zoneId).success(function() {
            $rootScope.$broadcast(dkuEvents.datasetChanged);
            ActivityIndicator.success(`Dataset "${$scope.dataset.name}" created!`);
            if ($scope.files && $scope.files.length > 1 && $scope.isLastDataset() && $scope.uiState.multipleDatasets) {
                // Last dataset of multiple files redirects to the flow and files are not merged into a single dataset
                $scope.disableUnsavedChangesWarning();
                DataikuAPI.datasets.upload.deleteUploadBox($scope.dataset.params.uploadBoxId).success(function() {
                    StateUtils.go.project($stateParams.projectKey); // It is called project but it redirect to the flow
                }).error(setErrorInScope.bind($scope));
            } else if ($scope.isLastDataset()) {
                // If there is no multiple files, go to dataset
                $scope.disableUnsavedChangesWarning();
                StateUtils.go.dataset($scope.uiState.new_dataset_name);
            } else {
                nextDataset();
            }
        }).error(setErrorInScope.bind($scope));
    }

    $scope.skipDatasetCreation = function(){
        ActivityIndicator.info(`Dataset "${$scope.uiState.new_dataset_name}" skipped.`);
        if($scope.isLastDataset()){
            $scope.disableUnsavedChangesWarning();
            DataikuAPI.datasets.upload.deleteUploadBox($scope.dataset.params.uploadBoxId).success(function() {
                    StateUtils.go.project($stateParams.projectKey); // It is called project but it redirect to the flow
                }).error(setErrorInScope.bind($scope));
        } else {
            nextDataset();
        }
    }

    $scope.onLoadComplete = function () {
        //nothing
    };

    $scope.getAlertClassForDetectionResults = function(dRes) {
        if (!dRes) return "";
        if (!dRes.connectionOK) return "alert-error";
        if (dRes.empty) return "alert-warning";
        if (!dRes.format || !dRes.format.ok) return "alert-error";
        if (dRes.format && dRes.format.ok && dRes.format.schemaDetection && dRes.format.schemaDetection.warningLevel == 'WARN') return "alert-warning";
        return "ng-hide"; // Don't display if OK
    };

    $scope.detectOrPreview = function(){
        let shouldDetect = ($scope.dataset.formatParams == null || $scope.dataset.formatParams.length === 0 || angular.isUndefined($scope.dataset.formatType) || $scope.redetectFormatOnCoreParamsChanged);
        if (shouldDetect) {
            $scope.detectFormat();
        } else {
            $scope.preview();
        }
    };

    /* Manual force format redetection: drop current data and trigger detection */
    $scope.forceFormatRedetection = function () {
        $scope.dataset.formatType = null;
        clear($scope.dataset.formatParams); // formatParams is watched, never reassign it.
        $scope.detectFormat();
    };

    function updateCreationWarning() {
        let warnings = [];
        $scope.uiState.creationWarning = null;
        if (!$scope.dataset.formatType || !$scope.dataset.formatParams) {
            warnings.push("No format configured, dataset won't be usable");
        }
        if (!$scope.dataset.schema || !$scope.dataset.schema.columns || !$scope.dataset.schema.columns.length) {
            warnings.push("No schema set, dataset won't be usable");
        }
        if (warnings.length > 0) {
            $scope.uiState.creationWarning = warnings.join("; ");
        }
    }

    /**
     * If there is one file loaded detect format to display the dataset preview.
     * If there is more than one file loaded clear detectionResult
     */
    function onSavedFilesChange(){
        if($scope.dataset && $scope.dataset.savedFiles && $scope.dataset.savedFiles.length === 1 && $scope.totalNewFiles <= 1 ){
            $scope.dataset.params.sheetName = $scope.dataset.savedFiles[0].sheetName;
            $scope.detectFormat();
        } else {
            $scope.detectionResults = null;
        }
    }

    // used on the template to switch primary action from create dataset to configure
    // simplistic but match display of warnings or errors
    $scope.isFixConfigPrimaryAction = function () {
      if ($scope.detectionResults) {
        if ($scope.detectionResults.fileForDetectionPath && !$scope.detectionResults.format) {
          return true;
        } else if ($scope.detectionResults.format &&  $scope.detectionResults.format.type && !$scope.detectionResults.format.ok) {
          return true;
        } else if (
          $scope.detectionResults.format &&
          $scope.detectionResults.format.ok &&
          $scope.detectionResults.format.schemaDetection.warningLevel === "WARN"
        ) {
          return true;
        }
      }
      return false;
    };

    /** WATCHERS */
    $scope.$watch("dataset", updateCreationWarning, true);
    $scope.$watch("uiState.new_dataset_name", updateCreationWarning);
    // this is used to check the number of new dropped files from the flow
    $scope.$watch(
      function () {
        return $rootScope.totalNewFiles;
      },
      function () {
        $scope.totalNewFiles = $rootScope.totalNewFiles;
      }
    );
    $scope.$watchCollection("dataset.savedFiles", onSavedFilesChange);
});

app.directive('fsProviderSettings', function(DataikuAPI, $stateParams, WT1) {
    return {
        restrict: 'A',
        replace: true,
        templateUrl: '/templates/datasets/fs-provider-settings.html',
        scope: {
            connectionHasMetadata : '=',
            providerType : '=',
            config : '=',
            projectKey : '=',
            contextVars : '=',
            defaultPath : '=',
            pathChangeNeedsConfirm : '=',
            onChange : '&',
            checkAllowManagedFolders: '<',
        },
        link: function($scope, element, attrs) {
            if ($scope.config.$resetConnection) {
                delete $scope.config.$resetConnection;
                $scope.config.connection = null;
            }

            var initConnectionFieldIfEmpty = function() {
                if ($scope.config == null) return;
                if ($scope.connections == null) return;
                if ($scope.connections.length > 0 && ($scope.config.connection == null || $scope.config.connection.length === 0)) {
                    $scope.config.connection = $scope.connections[0];
                }
            };
            var initPathFieldIfEmpty = function() {
                if ($scope.config == null) return;
                if (angular.isUndefined($scope.config.path)) {
                    $scope.config.path = $scope.defaultPath != null ? $scope.defaultPath : '/';
                }
            };
            $scope.$watch("config", function() {
                if ($scope.config == null) return;
                initPathFieldIfEmpty();
                initConnectionFieldIfEmpty();
            }, true);
            $scope.$watch("providerType", function(nv, ov) {
            	if (nv && ov && nv != ov && $scope.config) {
            		// clear the connection to avoid testing an provider with a connection of the wrong type
        			$scope.config.connection = null;
            	}
                if ($scope.providerType == null) return;
                if ($scope.providerType == 'S3') {
                    $scope.connectionType = 'EC2';
                } else if ($scope.providerType == 'SFTP' || $scope.providerType == 'SCP') {
                    $scope.connectionType = 'SSH';
                } else {
                    $scope.connectionType = $scope.providerType;
                }
                DataikuAPI.connections.getTypeAndNames($scope.connectionType, $scope.checkAllowManagedFolders).success(function (data) {
                    $scope.connections = data.map(connection => connection['name']);
                    $scope.connectionsWithDescriptions = data;
                    initConnectionFieldIfEmpty();
                }).error(setErrorInScope.bind($scope));
            });

            $scope.browse = function (path, isDirectory) {
                if (path == null)
                    path = $scope.defaultPath != null ? $scope.defaultPath : '/';
                var configAnchoredAtRoot = angular.copy($scope.config);
                // We discard stuff to have a shorter serialized version of our dataset
                configAnchoredAtRoot.path = $scope.defaultPath != null ? $scope.defaultPath : '/';

                WT1.event("fsprovider-fs-browse", {providerType : $scope.providerType, path : path});

                // Ugly workaround. Angular 1.2 unwraps promises (don't understand why)
                // Except if the promise object has a $$v.
                // See https://github.com/angular/angular.js/commit/3a65822023119b71deab5e298c7ef2de204caa13
                // and https://github.com/angular-ui/bootstrap/issues/949
                var promise = DataikuAPI.fsproviders.browse($scope.providerType, configAnchoredAtRoot, $scope.projectKey || $stateParams.projectKey, $scope.contextVars || {}, path, isDirectory);
                promise.$$v = promise;
                return promise;
            };


        }
    };
});


app.controller("_FSLikeDatasetControllerFragment", function($scope) {
    if (angular.isUndefined($scope.dataset.params.variablesExpansionLoopConfig)) {
        $scope.dataset.params.variablesExpansionLoopConfig = {
            mode: "CREATE_VARIABLE_FOR_EACH_COLUMN"
        }
    }
});


app.controller("FilesystemDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("AzureDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("SharePointOnlineDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("GCSDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("HDFSDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
    if (angular.isUndefined($scope.dataset.params.metastoreSynchronizationEnabled)) {
        $scope.dataset.params.metastoreSynchronizationEnabled = false;
    }
});

app.controller("S3DatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("DatabricksVolumeDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("FTPDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});
app.controller("SFTPDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("SCPDatasetController", function($scope, $controller) {
    $controller("_FSLikeDatasetControllerFragment", {$scope:$scope});
});

app.controller("HTTPDatasetController", function($scope, Dialogs, $timeout) {
    if (!$scope.dataset.params || !$scope.dataset.params.sources) {
        $scope.dataset.params = {
            sources: [],
            consider404AsEmpty: true,
            useGlobalProxy: true,
            previewPartition: '',
            partitions: []
        };
    }

    $scope.uiState ={
        urls: ($scope.dataset.params.sources || []).map(function(source) { return source.url; }).join('\n'),
        partitioned: $scope.dataset.partitioning.dimensions ? ($scope.dataset.partitioning.dimensions.length > 0) : false,
        previewPartition: $scope.dataset.params.previewPartition && $scope.dataset.params.previewPartition.length > 0
            ? partitionId2Obj($scope.dataset.params.previewPartition) : {},
        partitionList: $scope.dataset.params.partitions && $scope.dataset.params.partitions.length > 0
            ? $scope.dataset.params.partitions.map(partitionId2Obj) : []
    };
    $scope.hasTimeDimension = function() {
        return $scope.dataset.partitioning.dimensions.some(function(d) { return d.type === 'time'; });
    };
    $scope.$watch('uiState.urls', function(nv) {
        $scope.dataset.params.sources = nv.split(/(?:\r?\n)+/).map(function(url) {
            return { url: url.trim() };
        }).filter(function(_) { return _.url.length > 0; });
    });
    $scope.$watch('uiState.previewPartition', function(nv) {
        if (nv == null) return;
        $scope.dataset.params.previewPartition = partitionObj2Id(nv);
    }, true);
    $scope.$watch('uiState.partitionList', function(nv) {
        $scope.dataset.params.partitions = nv.map(partitionObj2Id);
    }, true);
    $scope.$watch('dataset.partitioning.dimensions', function(nv, ov) {
        if (nv == null) return;
        if (nv.length != ov.length) {
            var oldNames = ov.map(function(d) {return d.name;});
            var newNames = nv.map(function(d) {return d.name;});
            var toAdd = newNames.filter(function(name) {return oldNames.indexOf(name) < 0;});
            var toDel = oldNames.filter(function(name) {return newNames.indexOf(name) < 0;});
            let cleanPartition = function(obj) {
                toAdd.forEach(function(name) {obj[name] = '';});
                toDel.forEach(function(name) {delete obj[name];});
            };
            cleanPartition($scope.uiState.previewPartition);
            $scope.uiState.partitionList.forEach(cleanPartition);
        } else {
            // rename fields in UI partition list
            let cleanPartition = function(obj) {
                for (var i = 0; i < nv.length; i++) {
                    if (nv[i].name !== ov[i].name) {
                        obj[nv[i].name] = obj[ov[i].name];
                        delete obj[ov[i].name];
                    }
                }
            };
            cleanPartition($scope.uiState.previewPartition);
            $scope.uiState.partitionList.forEach(cleanPartition);
        }
    }, true);
    $scope.addPartitionToList = function() {
        $scope.uiState.partitionList.push(partitionId2Obj(''));
        $timeout(function() {
            $('table.table.table-partition-values tr').slice(-1).find('input[type="text"]').first().focus();
        });
    };
    $scope.removePartitionFromList = function(i) {
        if (i >= 0 && i < $scope.uiState.partitionList.length) {
            $scope.uiState.partitionList.splice(i, 1);
        }
    };
    $scope.partitionListPrompt = function() {
        Dialogs.prompt($scope, "Partitions list",
            "Edit partitions list, one partition ID per line, dimensions separated by |",
            $scope.dataset.params.partitions.join('\n'),
            {
                type: 'textarea',
                placeholder: 'dim1_value1|dim2_value1\ndim1_value2|dim2_value2'
            }
        ).then(function(nv) {
            $scope.uiState.partitionList = nv.split(/(?:\r?\n)+/)
                .map(function(_) { return _.trim(); })
                .filter(function(_) { return !!_; })
                .map(partitionId2Obj);
        });
    };

    function partitionId2Obj(id) {
        var parts = (id || '').split('|');
        return $scope.dataset.partitioning.dimensions.reduce(function(obj, dim, i) {
            obj[dim.name] = parts[i] || '';
            return obj;
        }, {});
    }
    function partitionObj2Id(obj) {
        return $scope.dataset.partitioning.dimensions
            .map(function(dim, i) { return obj[dim.name]; })
            .join('|');
    }

    $scope.getDimensionVariable = function(d) {
        if (d.type == 'time') {
            var patterns = ['%Y'];
            if (d.params.period == 'MONTH') {
                patterns.push('%M');
            } else if (d.params.period == 'DAY') {
                patterns.push('%M');
                patterns.push('%D');
            } else if (d.params.period == 'HOUR') {
                patterns.push('%M');
                patterns.push('%D');
                patterns.push('%H');
            }
            return patterns.join(", ");
        } else {
            return '%{' + d.name + '}';
        }
    };
});

app.controller("PluginFSProviderDatasetController", function($scope, $rootScope, $controller, $stateParams, DataikuAPI, $state, TopNav, CreateModalFromTemplate, PluginConfigUtils) {
    // nothing to add to base behavior
});

app.controller("FilesystemProviderController", function() {
	// nothing to add to base behavior
});

app.controller("AzureProviderController", function($scope) {
    $scope.uiState = {
        enterManually : false,
        selectedContainer : $scope.config.container
    };

    $scope.$watch('connMeta', function () {
        if ($scope.connMeta) {
            $scope.containers = null;
            if ($scope.connMeta.containers) {
                $scope.containers = $scope.connMeta.containers.split(',');
                $scope.uiState.selectedContainer = $scope.containers.filter(function(b) {return b == $scope.config.container;})[0];
            }
        }
    });
});

app.controller("SharePointOnlineProviderController", function($scope) {
});

app.controller("GCSProviderController", function($scope) {
    $scope.uiState = {
        enterManually : false,
        selectedBucket : $scope.config.bucket
    };

    $scope.$watch('connMeta', function () {
        if ($scope.connMeta) {
            $scope.buckets = null;
            if ($scope.connMeta.buckets) {
                $scope.buckets = $scope.connMeta.buckets.split(',');
                $scope.uiState.selectedBucket = $scope.buckets.filter(function(b) {return b == $scope.config.bucket;})[0];
            }
        }
    });
});

app.controller("HDFSProviderController", function() {
	// nothing to add to base behavior
});

app.controller("S3ProviderController", function($scope) {
    $scope.uiState = {
        enterManually : false,
        selectedBucket : $scope.config.bucket
    };

    $scope.$watch('connMeta', function () {
        if ($scope.connMeta) {
            $scope.buckets = null;
            if ($scope.connMeta.buckets) {
                $scope.buckets = $scope.connMeta.buckets.split(',');
                $scope.uiState.selectedBucket = $scope.buckets.filter(function(b) {return b == $scope.config.bucket;})[0];
            }
        }
    });
});

app.controller("DatabricksVolumeProviderController", function($scope) {
    $scope.uiState = {
        enterManually : false,
        selectedVolume : $scope.config.volume
    };

    $scope.$watch('connMeta', function () {
        if ($scope.connMeta) {
            $scope.volumes = null;
            if ($scope.connMeta.volumes) {
                $scope.volumes = $scope.connMeta.volumes.split(',');
                $scope.uiState.selectedVolume = $scope.buckets.filter(function(b) {return b == $scope.config.bucket;})[0];
            }
        }
    });
});


app.controller("FTPProviderController", function($scope) {
    if (angular.isUndefined($scope.config.timeout)) {
        $scope.config.timeout = 30000;
    }
});

app.controller("SFTPProviderController", function($scope) {
    if (angular.isUndefined($scope.config.timeout)) {
        $scope.config.timeout = 10000;
    }
});

app.controller("SCPProviderController", function($scope) {
    if (angular.isUndefined($scope.config.timeout)) {
        $scope.config.timeout = 10000;
    }
});

app.controller("PluginFSProviderController", function($scope, $rootScope, $controller, $state, $stateParams, Assert, DataikuAPI, TopNav, CreateModalFromTemplate, PluginConfigUtils) {
    $scope.config.config = $scope.config.config || {};
    $scope.loadedDesc = $rootScope.appConfig.customFSProviders.filter(function(x){
        return x.fsProviderType == $scope.providerType;
    })[0];

    Assert.inScope($scope, 'loadedDesc');

    $scope.desc = $scope.loadedDesc.desc;

    // put default values in place
    PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.config.config);

    $scope.pluginDesc = $rootScope.appConfig.loadedPlugins.filter(function(x){
        return x.id == $scope.loadedDesc.ownerPluginId;
    })[0];
});


app.controller("BaseUploadedFilesController",
    /**
     * @typedef {Object} BaseUploadedFilesControllerScope
     * @property {Array<UploadedFile>} files files registered in DSS (post processed)
     * @property {Array<File>} uploadedFiles raw files uploaded by the user
     * @property {number} downloadingFiles
     * @property {(uploadedFiles: Array<File>) => void} drop
     * @property {deleteFile} deleteFile Delete file and return the remaining files in the current scope
     */
    /**
     * @typedef {Object} UploadedFile
     * @property {number} progress
     * @property {string} path
     * @property {number} length
     */

    /**
     * BaseUploadedFilesController
     * @param {BaseUploadedFilesControllerScope} $scope 
     * @param {DataikuAPI} DataikuAPI 
     * @param {WT1} WT1 
     * @param {StateParams} $stateParams 
     * @param {ActivityIndicator} ActivityIndicator 
     * @param {Q} $q 
     */

    /**
     *
     * @callback deleteFile
     * @param {UploadedFile} file file to be deleted from the current scope
     * @param {Event} event
     * @returns {Promise<Array<UploadedFile>>} remaining files (currently in scope)
     */
function($scope, DataikuAPI, WT1, $stateParams, ActivityIndicator, $q) {
    // Customized data about the uploaded files
    $scope.files = [];
    // Raw files
    $scope.uploadedFiles = [];

    $scope.downloadingFiles = 0;

    function countNumberOfNonExcelFiles(files) {
        // With Excel files, `files` can contain multiple times the same file with different sheet names, so we count them separately as "sheets"
        return files.filter(f => !f.sheetName).length;
    }

    function countNumberOfSheets(files) {
        // Return the number of sheets found in the various Excel files that have been uploaded
        return files.filter(f => f.sheetName).length;
    }

    $scope.drop = function (uploadedFiles) {
        function handleUploadedFile(uploadedFile) {
            /** @type {UploadedFile} */
            var file = {
                    progress: 0,
                    path: uploadedFile.name,
                    length: uploadedFile.size
                };
            $scope.files.push(file);
            $scope.uploadedFiles.push(uploadedFile);
            $scope.downloadingFiles++;
            DataikuAPI.datasets.upload.addFileToDataset($stateParams.projectKey, uploadedFile, $scope.dataset, function (e) {
                // progress bar
                if (e.lengthComputable) {
                    $scope.$apply(function () {
                        file.progress = Math.round(e.loaded * 100 / e.total);
                    });
                }
            }).then(function (data) {
                //success
                var index = $scope.files.indexOf(file);
                try {
                    data = JSON.parse(data);
                    if (data.wasArchive) {
                        ActivityIndicator.success("Extracted "  + data.files.length + " files from Zip archive");
                    }
                    // replace stub file object by result of upload
                    $scope.files = $scope.files.slice(0, index).concat(data.files).concat($scope.files.slice(index + 1));
                    $scope.files.sort(function (a, b) {
                        return a.path < b.path;
                    });
                } catch(e){
                    // a lot can go wrong
                    $scope.files = $scope.files.slice(0, index).concat($scope.files.slice(index + 1));
                }
                $scope.numberOfFiles = countNumberOfNonExcelFiles($scope.files);
                $scope.numberOfSheets = countNumberOfSheets($scope.files);
                $scope.downloadingFiles--;
            }, function(payload){
                // delete faulty file
                $scope.files.splice($scope.files.indexOf(file), 1);

                try {
                    setErrorInScope.bind($scope)(JSON.parse(payload.response), payload.status, function(h){return payload.getResponseHeader(h)});
                } catch(e) {
                    // The payload.response is not JSON
                    setErrorInScope.bind($scope)({$customMessage: true, message: (payload.response || "Unknown error").substring(0, 20000), httpCode: payload.status}, payload.status);
                }

                $scope.numberOfFiles = countNumberOfNonExcelFiles($scope.files);
                $scope.numberOfSheets = countNumberOfSheets($scope.files);
                $scope.downloadingFiles--;
            });
        }

        function handleFiles(uploadedFiles) {
            // if its a brand new dataset, instantiate an uploadbox first
            // upload files with progress bar

            for (var i = uploadedFiles.length - 1; i >= 0; i--) {
                handleUploadedFile(uploadedFiles[i]);
            }
        }
        if ($scope.dataset.name == null && !$scope.dataset.params.uploadBoxId) {
            DataikuAPI.datasets.upload.createUploadBox().success(function (data) {
                $scope.dataset.params.uploadBoxId = data.id;
                handleFiles(uploadedFiles);
            }).error(setErrorInScope.bind($scope));
        } else {
            handleFiles(uploadedFiles);
        }
    };

    $scope.deleteFile = function (file, e) {
        // Can't use Promise instead of $q because the AngularJS framework will not be aware about the changes
        // It won't update the binded UI and we will have to call $scope.$apply() manually
        return $q((resolve, reject) => {
                WT1.event("dataset-upload-remove-file");
                e.preventDefault();
                e.stopPropagation();

                // Only actually remove the file is there are no more files referencing this file (with Excel sheets, we can have
                // the same file referenced multiple times with different sheet names)
                let occurrences = $scope.files.filter((f) => f.path === file.path).length;
                if (occurrences > 1) {
                    $scope.files.splice($scope.files.indexOf(file), 1);
                    $scope.numberOfFiles = countNumberOfNonExcelFiles($scope.files);
                    $scope.numberOfSheets = countNumberOfSheets($scope.files);
                    resolve($scope.files);
                } else {
                    DataikuAPI.datasets.upload.removeFile($stateParams.projectKey, $scope.dataset, file.path)
                        .success(function() {
                            $scope.files.splice($scope.files.indexOf(file), 1);
                            $scope.uploadedFiles.splice($scope.uploadedFiles.findIndex((uploadedFile) => uploadedFile.name === file.path), 1);
                            $scope.numberOfFiles = countNumberOfNonExcelFiles($scope.files);
                            $scope.numberOfSheets = countNumberOfSheets($scope.files);
                            resolve($scope.files);
                        })
                        .error(function(data, status, headers, config, statusText, xhrStatus) {
                            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                            reject(data);
                        });
                }
            }
        )
    };
});


app.controller("UploadedFilesController", function($scope, $controller, DataikuAPI, $stateParams, $rootScope) {
    $controller("BaseUploadedFilesController", {$scope: $scope});

    // Cannot save if there is no file
    $scope.saveHooks.push(function() {
        return $scope.files && $scope.files.length > 0;
    });

    // Cannot save if downloading
    $scope.saveHooks.push(function() {
        return $scope.downloadingFiles == 0;
    });

    function watchFilesList() {
        // list of uploaded files (with finished upload)
        $scope.$watch(function () {
            return $.grep($scope.files, function (f) {
                return angular.isUndefined(f.progress);
            });
        }, function (nv, ov) {
            if (nv !== ov) {
                $scope.onCoreParamsChanged();
                $scope.dataset.savedFiles = nv;
            }
        }, true);
    };

    // init files
    if ($scope.dataset.name != null) {
        DataikuAPI.datasets.upload.listFiles($stateParams.projectKey, $scope.dataset.name).success(function (data) {
            $scope.files = data;
            watchFilesList();
        }).error(setErrorInScope.bind($scope));
    } else {
        DataikuAPI.datasets.listManagedUploadableConnections($stateParams.projectKey).success(function(data) {
            $scope.uploadableConnections = data.connections;
            $scope.dataset.params.$uploadConnection = $scope.uploadableConnections[0];
        }).error(setErrorInScope.bind($scope));
        // restore files from saved, see #6840
        if ($scope.dataset.savedFiles && $scope.dataset.savedFiles.length) {
            [].push.apply($scope.files, $scope.dataset.savedFiles);
        }
        watchFilesList();
        $scope.$watch("dataset.params.$uploadConnection", function() {
            if (!$scope.dataset.params.$uploadConnection) return;
            $scope.dataset.params.uploadConnection = $scope.dataset.params.$uploadConnection.name;
            $scope.dataset.params.uploadFSProviderType = $scope.dataset.params.$uploadConnection.fsProviderTypes[0];
        });
    }
    // if some files have been dropped onto the Flow, they have been put in the rootScope. They should be uploaded and then removed from the root scope.
    if ($rootScope.uploadedFiles) {
        const uploadedFiles = $rootScope.uploadedFiles;
        $rootScope.totalNewFiles = uploadedFiles.length;
        delete $rootScope.uploadedFiles;
        $scope.drop(uploadedFiles);
    }
});

app.controller("FilesInFolderController", function($scope, $controller, DataikuAPI, $stateParams, SmartId, LoggerProvider, FilePatternUtils, Throttle) {
    $scope.fileChoice = {};

    const logger = LoggerProvider.getLogger('datasets.FilesInFolderController');


    // put the prefill in place if there is one
    if ($stateParams.fromOdbSmartId && $scope.dataset && $scope.dataset.params && !$scope.dataset.params.folderSmartId) {
        $scope.dataset.params.folderSmartId = $stateParams.fromOdbSmartId;
        if ($stateParams.fromOdbItemPath && $stateParams.fromOdbItemDirectory == "false") {
            $scope.dataset.params.filesSelectionRules = {mode: "EXPLICIT_SELECT_FILES", includeRules: [], excludeRules: [], explicitFiles: [$stateParams.fromOdbItemPath]}
        } else if ($stateParams.fromOdbItemPath && $stateParams.fromOdbItemDirectory == "true") {
            var globbed = $stateParams.fromOdbItemPath + "/**/*";
            if (globbed.startsWith('/')) globbed = globbed.substring(1)
            $scope.dataset.params.filesSelectionRules = {mode: "RULES_INCLUDED_ONLY", includeRules: [{mode: "GLOB", matchingMode: "FULL_PATH", expr: globbed}], excludeRules: [], explicitFiles: []}
        }
        $scope.dataset.projectKey = $scope.dataset.projectKey || $stateParams.projectKey; // otherwise the folderSmartId is a bit irrelevant
    }

    var refreshManagedFolder = function() {
        if ($scope.managedfolders) {
            $scope.managedfolder = $scope.managedfolders.filter(function(f) {return f.smartId == $scope.dataset.params.folderSmartId;})[0];
        } else {
            $scope.managedfolder = null;
        }
    };

    DataikuAPI.managedfolder.listWithAccessible($stateParams.projectKey).success(function(data) {
        data.forEach(function(ds) {
            ds.foreign = (ds.projectKey != $stateParams.projectKey);
            ds.smartId = SmartId.create(ds.id, ds.projectKey);
        });
        $scope.managedfolders = data;
        refreshManagedFolder();
    }).error(setErrorInScope.bind($scope));

    $scope.usePartitioningFromFolder = function() {
        if (!$scope.managedfolder) return;
        $scope.dataset.partitioning = $scope.managedfolder.partitioning ? angular.copy($scope.managedfolder.partitioning) : {dimensions: []};
    };

    $scope.$watch("dataset.params.folderSmartId", function(oldValue, newValue) {   
        if (oldValue !== newValue) {
            $scope.uiState.filesListing = null;
            refreshManagedFolder();
        }
    });

    if ($stateParams.prefillParams) {
        var prefillParams = JSON.parse($stateParams.prefillParams);
        if (prefillParams.folderSmartId && !$scope.dataset.params.folderSmartId) {
            $scope.dataset.params.folderSmartId = prefillParams.folderSmartId;
        }
        if (prefillParams.itemPathPattern && !$scope.dataset.params.itemPathPattern) {
            $scope.dataset.params.itemPathPattern = prefillParams.itemPathPattern;
        }
        $scope.dataset.projectKey = $scope.dataset.projectKey || $stateParams.projectKey; // otherwise the folderSmartId is a bit irrelevant
    }

    //*** fs-files-listing-v2.html integration START ***
    // This section has what is needed to talk to the UI in fs-files-listing-v2.html (which extends list-items-fattable.html )
    // It would be similar for other file selection controllers

    // We need to have listItems in scope as fs-files-listing-v2.html/list-items-fattable.html expects this
    $scope.enableListItems();

    // we need to set the attribute select-add-or-toggle in list-items-fattable.html
    // this means clicking in the list outside the checkbox behaves well (otherwise it unselects what was selected before)
    $scope.activateSelectAddOrToggle  =  function() {
        return true;
    };

    // Configure search and filtering of files list
    // In most cases we want to keep the selected items on the top so we first sort by selected in descending order
    // Note `$selected` is the flag managed by list-items-fattable.html and dip.js/filteredMultiSelectRows reflecting the users choice
    // While `selected` is the state the backend returns and reset by applySelectionRules()
    const SORT_SELECTED_AT_BOTTOM = "$selected";
    const SORT_SELECTED_AT_TOP = "!$selected"; 
    $scope.sortBy = [
        { value: [SORT_SELECTED_AT_TOP, '-lastModified', 'path'], label: 'Last modified'},
        { value: [SORT_SELECTED_AT_TOP,'path'], label: 'Path' },
        { value: [SORT_SELECTED_AT_TOP,'size', 'path'], label: 'File Size' },
        { value: SORT_SELECTED_AT_TOP, label: 'Selected'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: ''
        },
        filterParams: {
            userQueryTargets: ['path'],
            propertyRules: {}
        },
        orderQuery: $scope.sortBy[0].value, // -> 'Last modified' desc selection (with selected at top)
        orderReversed: false,
    }, $scope.selection || {});

    // To keep selected items on top (or bottom *), when a user switches the mode or sort order we need to reset the first ordering component (order on itemselection state)
    // But we don't do that when the user actually choses to order by the current selections (in that case, $scope.selection.orderQuery will be a single string, not an array)
    var keepRelevantItemsOnTop = function () {        
        if (!($scope.selection && $scope.selection.orderQuery)) return;

        // Array implies not a simple order by `$selected`
        if (Array.isArray($scope.selection.orderQuery)) {
            if ($scope.dataset.params.filesSelectionRules.mode === "RULES_ALL_BUT_EXCLUDED") {
                // * for RULES_ALL_BUT_EXCLUDED we want the UNselected items on top!
                $scope.selection.orderQuery[0] = $scope.selection.orderReversed ? SORT_SELECTED_AT_TOP : SORT_SELECTED_AT_BOTTOM;
            } else {
                $scope.selection.orderQuery[0] = $scope.selection.orderReversed ? SORT_SELECTED_AT_BOTTOM : SORT_SELECTED_AT_TOP;
            }
        }
    }
    
    $scope.$watch("dataset.params.filesSelectionRules.mode", function(newMode, oldMode) {
        if ($scope.uiState.filesListing && newMode !== oldMode) {
            keepRelevantItemsOnTop();
            $scope.applySelectionRules().then($scope.refreshItemListOrderAndSelection);
        }
    });

    $scope.$watch("[selection.orderReversed, selection.orderQuery]" , function(newValue, oldValue) {
        if (newValue !== oldValue) {
            keepRelevantItemsOnTop();
        }
    });

    $scope.$watchCollection("selection.selectedObjects", () => 
    {
        if (!($scope.uiState.filesListing && $scope.dataset.params.filesSelectionRules && $scope.selection)) return;
        // A change in selection of in the list by the user (checkboxes or click on row)
        // `selection.selectedObjects`is managed/provided by the filtered-multi-select-rows directive
        $scope.updateRulesForSelection();
    });

    const rulesWatch = function() {
        if ($scope.uiState.filesListing) {
            $scope.applySelectionRules().then($scope.refreshItemListOrderAndSelection);
        }
    };

    const explicitFilesWatch = function() {
        if ($scope.uiState.filesListing) {
            $scope.applySelectionRules().then(function() {
                // We don't want the list to reorder on selection for this one unless they have chosen selection ordering
                // I.e. explicit files works differently to the other modes, except when sort by Selection is on
                if ($scope.selection.orderQuery === SORT_SELECTED_AT_TOP || $scope.selection.orderQuery === SORT_SELECTED_AT_BOTTOM) {
                    $scope.refreshItemListOrderAndSelection();
                } else {
                    // We just need to make sure the selection is updated as otherwise we won't pick up things later
                    // e.g. when the item we just added manually is deleted
                    $scope.$broadcast('refresh-selection');
                }
            });
        }
    }

    $scope.$watch("dataset.params.filesSelectionRules.includeRules", rulesWatch, true);
    
    $scope.$watch("dataset.params.filesSelectionRules.excludeRules", rulesWatch, true);

    //For explicit files do the debounce here
    $scope.$watch("dataset.params.filesSelectionRules.explicitFiles", Throttle().withDelay(400).wrap(explicitFilesWatch), true);

    $scope.fileListCheckboxDisabled = function(item) {
       return !$scope.isItemSelectable(item);
    };

    $scope.isItemSelectable = function(item) {
        switch ($scope.dataset.params.filesSelectionRules.mode) {
            case "ALL": 
                return false;
            case "EXPLICIT_SELECT_FILES":
                return true;
            case "RULES_INCLUDED_ONLY":
                 //once you select you cannot deselect (you can only remove rules via the left hand panel)
                return !item.$selected;
            case  "RULES_ALL_BUT_EXCLUDED":
                // For this case, once you deselect you cannot select (you can only remove rules via the left hand panel)
                return item.$selected;
            default:
                return false;
        }
    };

    function expandSelectionRules() {
        return DataikuAPI.fsproviders.expandSelectionRules(
                $scope.dataset.type, $scope.dataset.params, $stateParams.projectKey)
            .error(setErrorInScope.bind($scope));
    }

    // Cache expanded selection rules in the promise, as the fulfilled promise will
    // always return the same thing. We only call the backend again when it will
    // change, in applySelectionRules()
    let expandedSelectionRulesPromise;

    function getExpandedSelectionRulesPromise() {
        if (!expandedSelectionRulesPromise) {
            expandedSelectionRulesPromise = expandSelectionRules();
        }
        return expandedSelectionRulesPromise;
    }

    // When we detect a change made by the user in the RHS file list, we update the rules accordingly
    $scope.updateRulesForSelection = function() {
        if (
            !$scope.dataset.type ||
            !$scope.dataset.params.filesSelectionRules ||
            !$scope.dataset.params.filesSelectionRules.mode
        ) {
            return;
        }

        switch ($scope.dataset.params.filesSelectionRules.mode) {
            case "EXPLICIT_SELECT_FILES":
                // Handle both additions and deletions
                getExpandedSelectionRulesPromise().then(function(resp) {
                    if (
                        !$scope.selection ||
                        !$scope.selection.allObjects ||
                        !$scope.selection.selectedObjects
                    ) {
                        return;
                    }
                    const fsRules = resp.data;
                    const filePathPredicate = rulesToFilePathPredicate(fsRules);
                    const itemsToRemove = {};
                    for (let item of $scope.selection.allObjects) {
                        const matchesObj = filePathPredicate(item.path);
                        if (!item.$selected && matchesObj.matches) {
                            for (let ruleIndex of matchesObj.ruleIndices) {
                                if (!(ruleIndex in itemsToRemove)) {
                                    itemsToRemove[ruleIndex] = [];
                                }
                                itemsToRemove[ruleIndex].push(item);
                            }
                        }
                    }
                    $scope.removeExplicitSelects(itemsToRemove);
                    for (let item of $scope.selection.selectedObjects) {
                        if (!filePathPredicate(item.path).matches) {
                            $scope.addExplicitSelect(item);
                        }
                    }
                });
                break;
            case "RULES_INCLUDED_ONLY":
                // Just handle addition of rules here
                // We just look at items selected in the file list (for whom item.$selected, managed by the list UI, is always true)
                // Here item.selected indicates if the current filesSelectionRules already imply this item will be included
                // If not, this means it has just been selected in the file list, so we add a rule to cover it
                if ($scope.selection && $scope.selection.selectedObjects) {
                    for (let item of $scope.selection.selectedObjects) {
                        if (!item.selected) {
                            $scope.addIncludeRule(item);
                        }
                    }
                }
                break;
            case "RULES_ALL_BUT_EXCLUDED":
                // Just handle addition of exclude rules via deselection of files in list 
                // item.selected indicates if the current filesSelectionRules already consider this item selected, meaning not excluded
                // If it is not yet excluded, and it has just been unselected in the file list, add a rule to exclude it
                if ($scope.listItems) {
                    for (let item of $scope.listItems) {
                        if (!item.$selected && item.selected) {
                            $scope.addExcludeRule(item);
                        }
                    }
                }
                break;
            default:
                // Do nothing - cannot change selection in ALL mode
        }        
    }

    // Front-end file selection rules handling 
    // set item.selected and item.$selected based on the file selection rules
    // Emulates com.dataiku.dip.fs.FileSelectionRule and FileSelectionRules in the backend
    $scope.applySelectionRules = function() {
        if (!$scope.dataset.type || !$scope.dataset.params) {
            return Promise.resolve();
        }

        // Refresh the expanded selection rules
        expandedSelectionRulesPromise = expandSelectionRules();
        return expandedSelectionRulesPromise.then(function(resp) {
            if (!$scope.listItems) {
                return;
            }
            const fsRules = resp.data;
            if (fsRules.mode === "RULES_INCLUDED_ONLY") {
                addRegexpsForRules(fsRules.includeRules);
            } else if (fsRules.mode === "RULES_ALL_BUT_EXCLUDED") {
                addRegexpsForRules(fsRules.excludeRules);
            }

            const filePathPredicate = rulesToFilePathPredicate(fsRules);

            $scope.uiState.filesListing.selectedFiles = 0;
            $scope.uiState.filesListing.selectedSize = 0;

            for (let item of $scope.listItems) {
                let fileSelected = filePathPredicate(item.path).matches;
                item.selected = fileSelected;
                item.$selected = fileSelected;

                if (fileSelected) {
                    $scope.uiState.filesListing.selectedFiles++;
                    $scope.uiState.filesListing.selectedSize += item.size;
                }
            }
        });
    };

    /**
     * Takes the $scope.dataset.params.filesSelectionRules object (see FileSelectionRules.java in backend) 
     * and converts to a predicate which will return true or false for a given file path
     * (there may be a lot of files, so processing the rules once should perform better)
     * 
     * @param {FileSelectionRules} fsRules - see FilesSelectionRule.java - object with a mode and either excludeRules, includeRules (FileSelectionRule[])
     *   or explicitFiles (string[])
     * @returns predicate function that given a filepath, returns an object with two properties:
     *   - matches: boolean for whether it matches
     *   - ruleIndices: indices of the rules which are the reason for the outcome (if
     *     applicable)
     */
    function rulesToFilePathPredicate(fsRules) {
        //When we need to actually look at the path, we need to first normalize it and also may need to extract the file
        //So here's some shared logic among the predicates
        function wrapWithPathNormalisation(pathBasedPredicate, needFileName = true) {
            return function(path) {
                let normalizedPath = FilePatternUtils.normalizeSlashes(path);

                //Empty paths aren't matched by anything (and shouldn't really happen)
                if (!normalizedPath) {
                    return { matches: false };
                }
                return pathBasedPredicate(normalizedPath, needFileName ? FilePatternUtils.extractFileNameFromPath(normalizedPath) : null);
            }
        }
        
        function alwaysTrue(){
            return { matches: true };
        }

        function alwaysFalse(){
            return { matches: false };
        }

        switch (fsRules.mode) {
            case "ALL":
                return alwaysTrue;
            case "EXPLICIT_SELECT_FILES": {
                if (!fsRules.explicitFiles || fsRules.explicitFiles.length === 0) {
                    return alwaysFalse;
                }
                const normalizedEFPaths = fsRules.explicitFiles.map(FilePatternUtils.normalizeSlashes);
                return wrapWithPathNormalisation((normalizedPath) => {
                    const matchingRuleIndices = normalizedEFPaths.keys().filter(
                            (i) => normalizedEFPaths[i] && normalizedPath === normalizedEFPaths[i]).toArray();
                    return { matches: matchingRuleIndices.length > 0, ruleIndices: matchingRuleIndices };
                }, false);
            }
            case "RULES_INCLUDED_ONLY":
                return wrapWithPathNormalisation((normalizedPath, fileName) => {
                    const matchingRuleIndices = fsRules.includeRules.keys().filter((i) => {
                        const rule = fsRules.includeRules[i];
                        if (rule.$alwaysMatch) {
                            return true;
                        }
                        const pathOrName = rule.matchingMode === "FILENAME" ? fileName : normalizedPath;
                        return rule.$regexp && rule.$regexp.test(pathOrName);
                    }).toArray();
                    return { matches: matchingRuleIndices.length > 0, ruleIndices: matchingRuleIndices };
                });
            case "RULES_ALL_BUT_EXCLUDED":
                return wrapWithPathNormalisation((normalizedPath, fileName) => {
                    const matchingRuleIndices = fsRules.excludeRules.keys().filter((i) => {
                        const rule = fsRules.excludeRules[i];
                        if (rule.$alwaysMatch) {
                            return true;
                        }
                        const pathOrName = rule.matchingMode === "FILENAME" ? fileName : normalizedPath;
                        return rule.$regexp && rule.$regexp.test(pathOrName);
                    }).toArray();
                    return { matches: matchingRuleIndices.length === 0, ruleIndices: matchingRuleIndices };
                });
            default:
               return alwaysTrue;
        }
    }

    /**
     * Takes an array of file selection rules and adds $regExp (RegExp object) to each for the compiled pattern 
     * or sets rule.$alwaysMatch to true if appropriate.
     * 
     * @param {FileSelectionRule[]} rules - array of rules each with mode (REGEXP/GLOB), matchingMode (FULL_PATH/FILENAME) and expr (the pattern)
     */
    function addRegexpsForRules(rules) {
        for (let rule of rules) {
            if (rule.mode==='REGEXP') {
                const trimmedExpr = rule.expr ? rule.expr.trim() : "";
                //This is what the backend does 
                rule.$alwaysMatch = trimmedExpr === "";
                
                try {
                    // g flag not needed (and we'd would have to reset the lastIndex every time)
                    rule.$regexp = trimmedExpr ? new RegExp("^" + trimmedExpr + "$", 'i') : null;
                } catch(e) {
                    //The user should have been warned it is not a good regexp
                    rule.$regexp = null;
                }
                
            } else if (rule.mode==='GLOB') {
                const trimmedExpr = rule.expr ? rule.expr.trim() : "";
                //This is what the backend does 
                rule.$alwaysMatch = rule.matchingMode === "FULL_PATH" && (trimmedExpr === "" || trimmedExpr === "**" || trimmedExpr === "**/*");

                try {
                    if (rule.matchingMode === "FILENAME") {
                        rule.$regexp = FilePatternUtils.fileNameGlobRegExp(trimmedExpr);
                    } else {
                        rule.$regexp = FilePatternUtils.fullPathGlobRegExp(trimmedExpr);
                    }
                } catch(e) {
                    logger.warn("Glob regex could not be compiled: " + trimmedExpr);
                }
            }
        }
    }

    //*** fs-files-listing-v2.html integration END ***

    $scope.testAndGetSchema = function() {
        $scope.detectOrPreview();
    }

});

app.controller("FormatExcelController", function($scope, Debounce) {
    $scope.data = {
        cellSelectionModes: [
            {id: 1, name: 'By Lines'},
            {id: 2, name: 'By Range'}
        ]
    }
    $scope.uiState = {
        // Default to lines selection unless cellRange has already been populated
        // (edge case note - the user can choose "By range" then still save with an empty string for a range, but it is not actually saved
        // and we treat it as 'By Lines' mode, and we revive it as that - there is no separate empty string case here)
        cellSelectionMode: $scope.dataset.formatParams.cellRange ?  $scope.data.cellSelectionModes[1] : $scope.data.cellSelectionModes[0] 
    };
    let lastDatasetFormatParams = angular.copy($scope.dataset.formatParams);
    $scope.onCellSelectionModeUpdate = function() {
        // Backup the params before switching the mode
        const temp = angular.copy($scope.dataset.formatParams);
        $scope.dataset.formatParams = lastDatasetFormatParams;
        lastDatasetFormatParams = temp;
        // Ignore cell range if user is choosing lines selection and vice versa
        if ($scope.uiState.cellSelectionMode.id === 1) {
            // User is now choosing lines selection
            $scope.dataset.formatParams.cellRange = null;
        } else {
            $scope.dataset.formatParams.skipRowsAfterHeader = null;
            $scope.dataset.formatParams.skipRowsBeforeHeader = null;
        }
    }

    $scope.onExcelFormatParamsChanged = function () {
        if ($scope.uiState.cellSelectionMode.id === 2 &&  // import by cell range
            !$scope.cellRangeValidator($scope.dataset.formatParams.cellRange)
        ) {
            return;
        }
        $scope.onFormatParamsChanged();
    }

    /**
     * First hand validation of the cell range. A second validation should be performed on the backend side
     * @param value {string} The value to validate
     */
    $scope.cellRangeValidator = function(value) {
        if (!value) return true;
        const ranges = value.split(',');
        const regex = /^([^!]+!)?([A-Za-z]+)?([1-9][0-9]*)?(:([A-Za-z]+)?([1-9][0-9]*)?)?$/;
        for (const range of ranges) {
            if (!regex.test(range.trim())) {
                return false;
            }
        }
        return true;
    }

    // Does a bit of magic so that we don't trigger a preview as soon as users type in custom properties
    // Why: because, there is no onBlur event on the editable list component
    // How: Trigger a preview only when a valid property has been entered (with a key *and* a value) and
    //      after user has not modified the editable list for 1.5 seconds
    let lastCustomProperties = nonEmptyCustomProperties($scope.dataset.formatParams.customProperties);
    function nonEmptyCustomProperties(customProperties) {
        return !customProperties ? [] : customProperties.filter(item => {
            return item.key && item.value;
        });
    }
    $scope.onExcelCustomPropertiesChanged = Debounce().withScope($scope).withDelay(1500, 1500).wrap(function() {
        if (angular.equals(lastCustomProperties, nonEmptyCustomProperties($scope.dataset.formatParams.customProperties))) {
            return;
        }
        lastCustomProperties = nonEmptyCustomProperties($scope.dataset.formatParams.customProperties);
        $scope.onExcelFormatParamsChanged();
    });
});
}());
