/* global describe, beforeEach, module, inject, it, expect */

describe("auto feature generation service", function() {
    beforeEach(() => {
        module('dataiku.mock');
        module('dataiku.aiSqlGeneration');
        module('dataiku.recipes');
        module('dataiku.mock');
        module( 'dataiku.autoFeatureGeneration', function($provide) {
            $provide.value('RecipeUtils', {});
            $provide.value('DatasetUtils', {});
            $provide.value('NumberFormatter', {});
            $provide.value('Assert', {});
            $provide.value('Logger', {});
            $provide.value('$filter', {});
        });
    });

    const DEFAULT_RELATIONSHIP = {
        table1Index: 1,
        hasDateColumns: {},
        timeIndexColumn2: null,
        timeWindow: null
    }

    const DATASET_NAMES = {
        primary: "primary",
        enrichment: "enrichment"
    }


    it("disable save button in the modal to add an enrichment dataset for non time aware primary datasets", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const relationship = angular.copy(DEFAULT_RELATIONSHIP);
            relationship.hasDateColumns[DATASET_NAMES.primary] = false;
            const emptyCutoff = AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.EMPTY.name;

            let isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, emptyCutoff);
            expect(isRelationshipValid).toBe(false);

            const noCutoff = AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DEFAULT.name;
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, noCutoff);
            expect(isRelationshipValid).toBe(false);

            relationship.dataset2 = DATASET_NAMES.enrichment;
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, noCutoff);
            expect(isRelationshipValid).toBe(true);
        });
    });

    it("disable save button in the modal to add an enrichment dataset for time aware primary datasets", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const relationship = angular.copy(DEFAULT_RELATIONSHIP);
            relationship.hasDateColumns[DATASET_NAMES.primary] = true;
            const columnCutoff = AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name;

            let isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(false);

            relationship.timeIndexColumn = "cutoff_time";
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(false);
        });
    });

    it("disable save button in the modal to add an enrichment dataset for non time aware enrichment datasets", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const relationship = angular.copy(DEFAULT_RELATIONSHIP);
            relationship.hasDateColumns[DATASET_NAMES.primary] = true;
            relationship.hasDateColumns[DATASET_NAMES.enrichment] = false;
            const columnCutoff = AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name;

            relationship.dataset2 = DATASET_NAMES.enrichment;
            relationship.hasDateColumns[DATASET_NAMES.enrichment] = false;
            let isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(false);

            relationship.timeIndexMode = AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.name;
            relationship.timeIndexColumn = "cutoff";
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(true);
        });
    });

    it("disable save button in the modal to add an enrichment dataset for time aware enrichment datasets", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const relationship = angular.copy(DEFAULT_RELATIONSHIP);
            relationship.hasDateColumns[DATASET_NAMES.primary] = true;
            relationship.hasDateColumns[DATASET_NAMES.enrichment] = true;
            const columnCutoff = AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name;
            relationship.timeIndexColumn = "cutoff";

            relationship.timeIndexMode = AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.EMPTY.name;
            relationship.dataset2 = DATASET_NAMES.enrichment;
            let isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(false);

            relationship.timeIndexMode = AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DEFAULT.name;
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(true);

            relationship.timeIndexMode = AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DATE_COLUMN.name;
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(false);

            relationship.timeIndexColumn2 = AutoFeatureGenerationRecipeService.TIME_INDEX_MODE.DATE_COLUMN.name;
            isRelationshipValid = AutoFeatureGenerationRecipeService.isNewAfgRelationshipValid(relationship, false, columnCutoff);
            expect(isRelationshipValid).toBe(true);
        });
    });

    function createNewRecipe(inputs) {
        return  {inputs : {main : {items : inputs}} };
    }

    it("remove a dataset with two inputs", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const virtualInputs = [{index: 0, originLabel: "myFicusCollecttion"}, {index: 1, originLabel: "mybonzaiCollection"}];
            const relationships = [{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 0
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 1
                        },
                        "type": "EQ"
                    }
                ]
            }];

            const recipe = createNewRecipe([{"ref": "ficus", "deps": []},{"ref": "bonzai", "deps": []}]);
            const params = {virtualInputs, relationships}
            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([1], 1, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual([{"ref": "ficus", "deps": []}]);
            expect(params.virtualInputs).toEqual([{index: 0, originLabel: "myFicusCollecttion"}]);
            expect(params.relationships).toEqual([]);
        });
    });

    it("remove the primary dataset with two inputs", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const virtualInputs = [{
                index: 0,
                originLabel: "myFicusCollecttion",
                timeIndexColumn: "ficusBirthday"
            }, {index: 1, originLabel: "mybonzaiCollection", timeIndexColumn: "bonzainBirthday"}];

            const relationships = [{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 0
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 1
                        },
                        "type": "EQ"
                    }
                ]
            }];

            const cutoffTime = {mode: AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name};
            const params = {virtualInputs, relationships, cutoffTime}

            const recipe = createNewRecipe([{"ref": "ficus", "deps": []},{"ref": "bonzai", "deps": []}]);
            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([0], 0, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual([{"ref": "bonzai", "deps": []}]);
            expect(params.virtualInputs).toEqual([{index: 0, originLabel: "mybonzaiCollection", timeIndexColumn: null, timeWindows: []}]);
            expect(params.relationships).toEqual([]);
        });
    });


    it("remove the primary dataset with three inputs", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const virtualInputs = [{
                index: 0,
                originLabel: "myFicusCollecttion",
                timeIndexColumn: "ficusBirthday"
            }, {index: 1, originLabel: "mybonzaiCollection", timeIndexColumn: "bonzainBirthday"}, {index: 2, originLabel: "mycypreeCollection"}];
            const relationships = [{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 0
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 1
                        },
                        "type": "EQ"
                    }
                ]
            }, {
                "table2": 2,
                "table1": 1,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 1
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 2
                        },
                        "type": "EQ"
                    }
                ]
            }];
            const cutoffTime = {mode: AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name};
            const params = {virtualInputs, relationships, cutoffTime}

            const recipe = createNewRecipe([{"ref": "ficus", "deps": []},{"ref": "bonzai", "deps": []}, {"ref": "cypree", "deps": []}])

            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([0,1,2], 0, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual([]);
            expect(params.virtualInputs).toEqual([]);
            expect(params.relationships).toEqual([]);
        });
    });

    it("remove a duplicated dataset", function () {
        inject(function (AutoFeatureGenerationRecipeService) {
            const virtualInputs = [{
                index: 0,
                originLabel: "myFicusCollecttion",
                timeIndexColumn: "ficusBirthday"
            }, {index: 1, originLabel: "mybonzaiCollection", timeIndexColumn: "bonzainBirthday"}, {
                index: 0,
                originLabel: "mycypreeCollection"
            }];
            const relationships = [{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 0
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 1
                        },
                        "type": "EQ"
                    }
                ]
            }, {
                "table2": 2,
                "table1": 1,
                "conditionsMode": "AND",
                "type": "MANY_TO_ONE",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 1
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 2
                        },
                        "type": "EQ"
                    }
                ]
            }];

            const cutoffTime = {mode: AutoFeatureGenerationRecipeService.CUTOFF_TIME_MODE.DATE_COLUMN.name};
            const params = {virtualInputs, relationships, cutoffTime}
            const recipe = createNewRecipe([{"ref": "ficus", "deps": []},{"ref": "bonzai", "deps": []}]);
            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([2], 2, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual([{"ref": "ficus", "deps": []}, {"ref": "bonzai", "deps": []}]);
            expect(params.virtualInputs).toEqual([{
                index: 0,
                originLabel: "myFicusCollecttion",
                timeIndexColumn: "ficusBirthday"
            }, {index: 1, originLabel: "mybonzaiCollection", timeIndexColumn: "bonzainBirthday"}]);
            expect(params.relationships).toEqual([{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 0
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 1
                        },
                        "type": "EQ"
                    }
                ]
            }]);
        });
    });

    function getParameterSeveralBranches() {
        const virtualInputs = [{
            index: 0,
            originLabel: "myFicusCollecttion",
            timeIndexColumn: "ficusBirthday"
        }, {index: 1, originLabel: "mybonzaiCollection", timeIndexColumn: "bonzainBirthday"}, {index: 2, originLabel: "mycypreeCollection"}, {index: 3, originLabel: "myoakCollection"}];
        const relationships = [{
            "table2": 1,
            "table1": 0,
            "conditionsMode": "AND",
            "type": "ONE_TO_MANY",
            "on": [
                {
                    "column1": {
                        "name": "user_id",
                        "table": 0
                    },
                    "column2": {
                        "name": "user_id",
                        "table": 1
                    },
                    "type": "EQ"
                }
            ]
        }, {
            "table2": 2,
            "table1": 1,
            "conditionsMode": "AND",
            "type": "MANY_TO_ONE",
            "on": [
                {
                    "column1": {
                        "name": "user_id",
                        "table": 1
                    },
                    "column2": {
                        "name": "user_id",
                        "table": 2
                    },
                    "type": "EQ"
                }
            ]
        },
            {
                "table2": 3,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_ONE",
                "on": [
                    {
                        "column1": {
                            "name": "user_id",
                            "table": 1
                        },
                        "column2": {
                            "name": "user_id",
                            "table": 2
                        },
                        "type": "EQ"
                    }
                ]
            }
        ];
        const cutoffTime = {mode: "COLUMN"};
        return {virtualInputs, relationships, cutoffTime}
    }

    function getRecipeSeveralBranches() {
        const items = [{"ref": "ficus", "deps": []},{"ref": "bonzai", "deps": []}, {"ref": "cypree", "deps": []}, {"ref": "oak", "deps": []}];
        return createNewRecipe(items);
    }

    it("remove a dataset with several branches", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const recipe = getRecipeSeveralBranches();
            const params = getParameterSeveralBranches();
            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([1,2], 1, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual([{"ref": "ficus", "deps": []},{"ref": "oak", "deps": []}]);
            expect(params.virtualInputs).toEqual([{index:0,originLabel:"myFicusCollecttion",timeIndexColumn:"ficusBirthday"}, {index: 1, originLabel: "myoakCollection"}]);
            expect(params.relationships).toEqual([{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_ONE",
                "on": [{
                    "column1": {"name": "user_id", "table": 1},
                    "column2": {"name": "user_id", "table": 0},
                    "type": "EQ"
                }]
            }])
        });
    });

    it("remove a dataset with several branches and replaced datasets", function() {
        inject(function(AutoFeatureGenerationRecipeService) {
            const recipe = getRecipeSeveralBranches();
            const params = getParameterSeveralBranches();

            params.virtualInputs[1].index = 3;
            params.virtualInputs[3].index = 1;
            const columnsTabData = AutoFeatureGenerationRecipeService.getDefaultColumnsTabData(recipe);

            AutoFeatureGenerationRecipeService.removeDatasets([3], 3, params, recipe, columnsTabData);

            expect(recipe.inputs.main.items).toEqual( [{"ref": "ficus", "deps": []}, {"ref": "cypree", "deps": []}, {"ref": "oak", "deps": []}]);
            expect(params.virtualInputs).toEqual([{
                "index": 0,
                "originLabel": "myFicusCollecttion",
                "timeIndexColumn": "ficusBirthday"
            }, {"index": 2, "originLabel": "mybonzaiCollection", "timeIndexColumn": "bonzainBirthday"}, {
                "index": 1,
                "originLabel": "mycypreeCollection"
            }]);
            expect(params.relationships).toEqual([{
                "table2": 1,
                "table1": 0,
                "conditionsMode": "AND",
                "type": "ONE_TO_MANY",
                "on": [{
                    "column1": {"name": "user_id", "table": 0},
                    "column2": {"name": "user_id", "table": 1},
                    "type": "EQ"
                }]
            }, {
                "table2": 2,
                "table1": 1,
                "conditionsMode": "AND",
                "type": "MANY_TO_ONE",
                "on": [{
                    "column1": {"name": "user_id", "table": 1},
                    "column2": {"name": "user_id", "table": 2},
                    "type": "EQ"
                }]
            }])
        });
    });
});