import argparse
import shlex
import json
from dataiku.core import intercom

def call_llm_mesh(path, override_llm=None, json_body=None):
    body = None
    if json_body:
        body = json.dumps(json_body)
    params = {
        "userLlmId": override_llm
    }
    return intercom.backend_json_call("code-assistant" + path, body, params=params)

class BaseIPythonGPTCommand:
    def __init__(self, command_name, context):
        self.command_name = command_name
        self.context = context

    def _customize_parser(self, parser):
        return parser

    def _execute(self, args, request):
        raise NotImplementedError

    def build_parser(self):
        parser = argparse.ArgumentParser(prog="%%%%%s" % self.command_name)
        return self._customize_parser(parser)

    def parse_args(self, line):
        parser = self.build_parser()
        # raise Exception("Parsing: %s" % shlex.split(line))
        return parser.parse_args(shlex.split(line))

    def execute(self, line, cell=None):
        try:
            if cell is None:
                # Used as a line magic, don't parse arguments
                # But with an exception, if arguments are just "--help", so that "%aiask --help" can work
                if line == "--help":
                    args = self.parse_args(line)
                else:
                    args = self.parse_args("")
            else:
                # Used as a cell magic, the header line is the arguments
                args = self.parse_args(line)
        except SystemExit: #NOSONAR
            return

        request = cell == None and line or cell

        results = self._execute(args, request)
        return results
        
    def format_exception(self, e):
        message = str(e).replace('None: b', '')
        if 'AI Code Assistant: ' in message:
            return'%s.\nCheck your settings.' % message
        else:
            return'External error querying language model: %s.' % message

class ChatCommand(BaseIPythonGPTCommand):
    def __init__(self, command_name, context):
        super(ChatCommand, self).__init__(command_name, context)
        self.split_non_code_output = False

    def _customize_parser(self, parser):
        parser.add_argument(
            "-c", "--continue",
            dest="continue_conversation",
            help="Continue the previous conversation (only takes %%aiask and %%aiwrite into account)",
            action="store_true",
        )
        parser.add_argument(
            "-np", "--no-previous",
            dest="no_previous",
            help="Do not pass the last run cell as context",
            action="store_true",
        )
        parser.add_argument(
            "-df", "--dataframes",
            dest="dataframes",
            help="Tell Code Assistant about which Pandas Dataframes exist in the current notebook",
            action="store_true"
        )
        parser.add_argument(
            "--override-llm",
            help="Override LLM id. Advanced usage"
        )
        parser.add_argument(
            "--temperature",
            help="LLM temperature Advanced usage",
            type=float,
        )
        parser.add_argument(
            "--max-tokens",
            help="LLM max tokens. Advanced usage",
            type=int,
        )
        return parser

    def _execute(self, args, request):
        chat_message_history = self.context["chat_message_history"]
        if not args.continue_conversation:
            chat_message_history = []

        system_message = {"role": "system", "content": """
You are a Python data science coding assistant. Your goal is to write Python code
to solve the user's problem. Please generate clear code with comments"""}

        user_messages = []

        if not args.no_previous:
            # add last-run cell as context for the query - filter out magic commands
            user_ns = self.context["shell"].user_ns
            cell_context = None
            if not user_ns["_i"].startswith("%"):
                cell_context = user_ns["_i"]
            elif not user_ns["_ii"].startswith("%"):
                cell_context = user_ns["_ii"]
            elif not user_ns["_iii"].startswith("%"):
                cell_context = user_ns["_iii"]
            else:
                # 'In' and '_ih' contain the history of executed code, and not the actual input. Since '%%chat' is
                # translated to 'get_ipython().run_cell_magic("chat")', it's easier to filter using the dynamic '_i<n>' vars
                # Moreover, _i, _ii and _iii already contain the last 3 inputs, so no need to parse these again
                # Parsing in reverse order since _i<n> is the nth execution, and we want the last one
                for i in range(len(user_ns["In"]) - 3, 0, -1):
                    c = user_ns.get("_i" + str(i), None)
                    if c is not None and not c.startswith("%"):
                        cell_context = c
                        break
    
            if cell_context is not None:
                content = "The user is currently editing a Jupyter noteboook. For information, this is code of the last cell that the user has executed:\n"
                content += cell_context
                user_messages.append({"role": "user", "content": content})
        

        if args.dataframes:
            user_ns = self.context["shell"].user_ns
            user_ns_hidden = self.context["shell"].user_ns_hidden
            nonmatching = object()  # This can never be in user_ns
            out = [ i for i in user_ns
                    if not i.startswith('_') \
                    and (user_ns[i] is not user_ns_hidden.get(i, nonmatching)) ]

            dfs = []
            try:
                for k in out:
                    v = self.context["shell"].user_ns[k]
                    if (str(type(v)) == "<class 'pandas.core.frame.DataFrame'>"):
                        dfs.append((k, v.columns))

                if len(dfs) > 0:
                    content = "The following dataframes exist. Here are the variable names and columns for each:\n"
                    for df in dfs:
                        content += " * `%s`: columns: %s\n" % (df[0], ",".join(df[1]))
                    content +="\n\n"
                    user_messages.append({"role": "user", "content": content})
            except Exception:
                pass

        user_messages.append({"role": "user", "content": request})

        messages = [system_message] + chat_message_history + user_messages +[{"role": "system",
            "content": "If you write Python code, write in in markdown, making sure to put the ```python marker to have it highlighted. Thanks!"
            }]

        try:
            json_body = {
                "query": {
                    "messages": messages
                },
                "settings": {}
            }
            if args.temperature:
                json_body["settings"]["temperature"] = args.temperature
            if args.max_tokens:
                json_body["settings"]["maxOutputTokens"] = args.max_tokens

            resp = call_llm_mesh("/chat-completion", override_llm=args.override_llm, json_body=json_body)

            # handling error if query has failed
            if not resp['ok']:
                return resp.get('errorMessage', "An unknown error has occurred, please contact your administrator for more information.")

            # handling response
            chat_response = resp["text"]
            chat_message_history.extend(user_messages)
            chat_message_history.append({"role": "assistant", "content": chat_response})
            self.context["chat_message_history"] = chat_message_history

            if self.split_non_code_output:
                if chat_response.find("```python\n") >= 0:
                    new_response = ""
                    in_markdown = False
                    after_markdown = False

                    for line in chat_response.splitlines():
                        if not in_markdown and line.startswith("```python"):
                            in_markdown=True
                        elif in_markdown and line.startswith("```"):
                            in_markdown=False
                            after_markdown = True
                        else:
                            if in_markdown:
                                new_response += line + "\n"
                            elif after_markdown:
                                new_response += "# " + line + "\n"
                            else:
                                new_response += "# " + line + "\n"
                    
                    chat_response = new_response

                chat_response = "# This code was automatically generated using AI Code Assistant. Please review for accuracy and security before running\n" + chat_response

                return chat_response
            else:
                chat_response = chat_response + "\n\n*This response was automatically generated using AI Code Assistant. Please review for accuracy and security before running*"
                return chat_response

        except Exception as e:
            return self.format_exception(e)

class AnalyzeCommand(BaseIPythonGPTCommand):
    def __init__(self, command_name, context):
        super(AnalyzeCommand, self).__init__(command_name, context)

    def _customize_parser(self, parser):
        parser.add_argument(
            "-t", "--terse",
            help="Make the explanation more terse",
            dest="verbosity", action="store_const", const="terse")
        parser.add_argument(
            "-v", "--verbose",
            help="Make the explanation more verbose",
            dest="verbosity", action="store_const", const="verbose")
        parser.add_argument(
            "--override-llm",
            help="Override LLM id. Advanced usage"
        )
        parser.add_argument(
            "--temperature",
            help="LLM temperature Advanced usage",
            type=float,
        )
        parser.add_argument(
            "--max-tokens",
            help="LLM max tokens. Advanced usage",
            type=int,
        )
        return parser

    def _execute(self, args, request):
        system_message = {"role": "system", "content": """
You are a Python data science coding assistant. Your goal is to explain Python code.
Assume that the user has basic understanding of Python, and avoid explaining trivial things
such as imports"""}

        content = "I'm editing a python Jupyter notebook. Explain this cell content:\n\n"
        content += request
        messages = [system_message, {"role": "user", "content": content}]

        if args.verbosity == "terse":
            messages.append({"role": "system", "content": "Please keep the explanation short"})
        elif args.verbosity == "verbose":
            messages.append({"role": "system", "content": "Please make the explanation very verbose"})
        else:
            messages.append({"role": "system", "content": "Focus on the important parts"})

        try:
            json_body = {
                "query": {
                    "messages": messages
                },
                "settings": {}
            }
            if args.temperature:
                json_body["settings"]["temperature"] = args.temperature
            if args.max_tokens:
                json_body["settings"]["maxOutputTokens"] = args.max_tokens

            resp = call_llm_mesh("/chat-completion", override_llm=args.override_llm, json_body=json_body)

            chat_response = resp["text"]
            chat_response += "\n\n*This explanation was automatically generated using AI Code Assistant. Please review for accuracy*"
            return chat_response
        except Exception as e:
            return self.format_exception(e)