import random

from albumentations.core.transforms_interface import DualTransform
from albumentations.augmentations.crops import functional as F


class DkuRandomCrop(DualTransform):
    """
    Crop a random part of the input.

    Args:
        min_kept_ratio (float, int): minimum kept ratio after the cropping
        max_kept_ratio (float, int): maximum kept ratio after the cropping
        preserve_aspect_ratio (bool): whether the cropping should preserve the aspect ratio of the original image

    Targets:
        image, bboxes

    Decided to do our own implementation of the random crop because we did not find a Crop within the Albumentations
    library that accepted ratio as params.
    Implementation inspired from albumentations.RandomSizedBBoxSafeCrop
    """
    def __init__(self, min_kept_ratio, max_kept_ratio, preserve_aspect_ratio=True, always_apply=False, p=1.0):
        super(DkuRandomCrop, self).__init__(always_apply, p)
        self._check_ratio(min_kept_ratio, "min_kept_ratio")
        self._check_ratio(max_kept_ratio, "max_kept_ratio")
        self.min_kept_ratio = min_kept_ratio
        self.max_kept_ratio = max_kept_ratio
        self.preserve_aspect_ratio = preserve_aspect_ratio

    @staticmethod
    def _check_ratio(ratio, name):
        if not isinstance(ratio, (float, int)):
            raise TypeError("{} should be a number, it is {} instead".format(name, type(ratio)))
        if ratio > 1 or ratio < 0:
            raise ValueError("{} should be between 0 and 1, actually is {}".format(name, ratio))

    def apply(self, img, crop_height=0, crop_width=0, h_start=0, w_start=0, **params):
        return F.random_crop(img, crop_height, crop_width, h_start, w_start)

    def apply_to_bbox(self, bbox, crop_height=0, crop_width=0, h_start=0, w_start=0, rows=0, cols=0, **params):
        return F.bbox_random_crop(bbox, crop_height, crop_width, h_start, w_start, rows, cols)

    @property
    def targets_as_params(self):
        return ["image"]

    def get_transform_init_args_names(self):
        return "min_kept_ratio", "max_kept_ratio", "preserve_aspect_ratio"

    def _get_random_ratio_in_interval(self):
        return self.min_kept_ratio + random.random() * (self.max_kept_ratio - self.min_kept_ratio)

    def get_params_dependent_on_targets(self, params):
        img_h, img_w = params["image"].shape[:2]

        if self.preserve_aspect_ratio:
            ratio = self._get_random_ratio_in_interval()
            h_ratio = w_ratio = ratio
        else:
            h_ratio = self._get_random_ratio_in_interval()
            w_ratio = self._get_random_ratio_in_interval()

        return {
            "crop_height": int(img_h * h_ratio),
            "crop_width": int(img_w * w_ratio),
            "h_start": random.random(),
            "w_start": random.random()
        }
