CREATE FUNCTION artifacts_at_timestamp(epoc_at TEXT, artifact_ids TEXT[])
RETURNS TABLE (
    artifact_id TEXT,
    artifact_blueprint_id TEXT,
    artifact_version_id TEXT,
    artifact_json_without_relationships JSONB,
    artifact_json_with_relationships JSONB,
    signoffs JSONB,
    user_mapping_user_login TEXT,
    group_mapping_group_name TEXT,
    global_api_key_mapping_api_key_id TEXT,
    blueprint_json JSONB,
    blueprint_version_json JSONB,
    blueprint_version_trace_json JSONB
)
LANGUAGE plpgsql AS
    $func$
BEGIN
    RETURN QUERY
WITH
artifacts AS (
    SELECT *
    FROM artifacts_history
    WHERE _revision_id IN (
        -- With DISTINCT ON, You tell PostgreSQL to return a single row for each distinct group defined by the ON clause. 
        -- Which row in that group is returned is specified with the ORDER BY clause.
        SELECT DISTINCT ON (h.artifact_id) h._revision_id
        FROM artifacts_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.artifact_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
blueprints AS (
    SELECT *
    FROM blueprints_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.blueprint_id) h._revision_id
        FROM blueprints_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.blueprint_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
blueprint_versions AS (
    SELECT *
    FROM blueprint_versions_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.blueprint_version_blueprint_id, h.blueprint_version_version_id) h._revision_id
        FROM blueprint_versions_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.blueprint_version_blueprint_id, h.blueprint_version_version_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
blueprint_version_traces AS (
    SELECT *
    FROM blueprint_version_traces_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.blueprint_version_trace_blueprint_id, h.blueprint_version_trace_version_id) h._revision_id
        FROM blueprint_version_traces_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.blueprint_version_trace_blueprint_id, h.blueprint_version_trace_version_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
user_mappings AS (
    SELECT *
    FROM user_mappings_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.user_mapping_user_login, h.user_mapping_artifact_id) h._revision_id
        FROM user_mappings_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.user_mapping_user_login, h.user_mapping_artifact_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
group_mappings AS (
    SELECT *
    FROM group_mappings_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.group_mapping_group_name, h.group_mapping_artifact_id) h._revision_id
        FROM group_mappings_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.group_mapping_group_name, h.group_mapping_artifact_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
global_api_key_mappings AS (
    SELECT *
    FROM global_api_key_mappings_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.global_api_key_mapping_api_key_id, h.global_api_key_mapping_artifact_id) h._revision_id
        FROM global_api_key_mappings_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.global_api_key_mapping_api_key_id, h.global_api_key_mapping_artifact_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
signoffs AS (
    SELECT *
    FROM signoffs_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.signoff_artifact_id, h.signoff_step_id) h._revision_id
        FROM signoffs_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.signoff_artifact_id, h.signoff_step_id, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
reference_relationships AS (
    SELECT *
    FROM reference_relationships_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.reference_relationship_artifact_id_from, h.reference_relationship_json_data) h._revision_id
        FROM reference_relationships_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.reference_relationship_artifact_id_from, h.reference_relationship_json_data, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
uploaded_file_relationships AS (
    SELECT *
    FROM uploaded_file_relationships_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.uploaded_file_relationship_json_data) h._revision_id
        FROM uploaded_file_relationships_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.uploaded_file_relationship_json_data, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
),
time_series_relationships AS (
    SELECT *
    FROM time_series_relationships_history
    WHERE _revision_id IN (
        SELECT DISTINCT ON (h.time_series_relationship_json_data) h._revision_id
        FROM time_series_relationships_history h
        WHERE h._revision_timestamp <= epoc_to_timestamp(epoc_at)
        ORDER BY h.time_series_relationship_json_data, h._revision_timestamp DESC, h._revision_id DESC
    )
    AND _revision_json ->> 'operation' != 'DELETE'
)

-- SAME VIEWS AND FINAL REQUEST AS NORMAL TABLES
SELECT
    artifacts.artifact_id,
    artifacts.artifact_blueprint_id,
    artifacts.artifact_version_id,
    artifacts.artifact_json AS artifact_json_without_relationships,
    artifact_json_with_relationships.artifact_json_with_relationships AS artifact_json_with_relationships,
    signoffs.signoffs AS signoffs,
    user_mappings.user_mapping_user_login AS user_mapping_user_login,
    group_mappings.group_mapping_group_name AS group_mapping_group_name,
    global_api_key_mappings.global_api_key_mapping_api_key_id AS global_api_key_mapping_api_key_id,
    blueprints.blueprint_json,
    blueprint_versions.blueprint_version_json,
    blueprint_version_traces.blueprint_version_trace_json
FROM
    artifacts artifacts
    JOIN blueprints
        ON artifacts.artifact_blueprint_id = blueprints.blueprint_id
    JOIN blueprint_versions
        ON artifacts.artifact_blueprint_id = blueprint_versions.blueprint_version_blueprint_id AND artifacts.artifact_version_id = blueprint_versions.blueprint_version_version_id
    JOIN blueprint_version_traces
        ON artifacts.artifact_blueprint_id = blueprint_version_traces.blueprint_version_trace_blueprint_id AND artifacts.artifact_version_id = blueprint_version_traces.blueprint_version_trace_version_id

    -- join on the computed definitions of the back reference fields
    LEFT JOIN (
        SELECT
            blueprint_id,
            version_id,
            jsonb_object_agg(field_id, references_list) AS backref_references_definitions
        FROM (
            SELECT
                blueprint_id,
                version_id,
                field_id,
                jsonb_agg(
                    jsonb_build_object(
                        'backref_reference_bp_id', backref_reference_bp_id,
                        'backref_reference_field_id', backref_reference_field_id
                    )
                ) AS references_list
            FROM (
                SELECT
                    blueprint_versions.blueprint_version_blueprint_id AS blueprint_id,
                    blueprint_versions.blueprint_version_version_id AS version_id,
                    fields_definitions.key AS field_id,
                    backref_reference_entry.value #>> '{blueprintId}' AS backref_reference_bp_id,
                    backref_reference_entry.value #>> '{fieldId}' AS backref_reference_field_id
                FROM
                    blueprint_versions blueprint_versions,
                    jsonb_each(blueprint_versions.blueprint_version_json #> '{fieldDefinitions}') AS fields_definitions,
                    jsonb_array_elements(fields_definitions.value #> '{references}') AS backref_reference_entry
                WHERE
                    fields_definitions.value #>> '{fieldType}' = 'REFERENCE'
                    AND fields_definitions.value #>> '{sourceType}' = 'COMPUTE'
            ) backref_references_flatten
            GROUP BY blueprint_id, version_id, field_id
        ) backref_references_per_field
        GROUP BY blueprint_id, version_id
        OFFSET 0
    ) backref_references_definitions
    ON
        backref_references_definitions.blueprint_id = artifacts.artifact_blueprint_id
        AND backref_references_definitions.version_id = artifacts.artifact_version_id

    -- compute references
    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.reference_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.reference_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.reference_relationship_artifact_id_to
                    ORDER BY rel.reference_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM reference_relationships rel
            WHERE
                rel.reference_relationship_artifact_id_from = artifacts.artifact_id
                AND rel.reference_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.reference_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.reference_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
    ) references_relationships_for_artifact ON TRUE

    -- compute back references
    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                backrefs.field_id,
                backrefs.artifact_id_from_list
            ) AS relationships_object
        FROM (
            SELECT
                field_id,
                COALESCE(
                    jsonb_agg(DISTINCT defs.artifact_id_from),
                    jsonb_build_array()
                ) AS artifact_id_from_list
            FROM (
                SELECT
                    rel.reference_relationship_artifact_id_from AS artifact_id_from,
                    backref_references.field_ids_list AS field_ids_list

                FROM
                    (
                        SELECT
                            jsonb_agg(backref_references_per_field.key) AS field_ids_list,
                            backref_references_flatten #>> '{backref_reference_bp_id}' AS backref_reference_bp_id,
                            backref_references_flatten #>> '{backref_reference_field_id}' AS backref_reference_field_id
                        FROM
                            jsonb_each(backref_references_definitions.backref_references_definitions) AS backref_references_per_field,
                            jsonb_array_elements(backref_references_per_field.value) AS backref_references_flatten
                        GROUP BY
                            backref_references_flatten #>> '{backref_reference_bp_id}',
                            backref_references_flatten #>> '{backref_reference_field_id}'
                    ) backref_references

                    INNER JOIN LATERAL (
                        SELECT rel.reference_relationship_artifact_id_from
                        FROM reference_relationships rel
                        WHERE
                            rel.reference_relationship_artifact_id_to = artifacts.artifact_id
                            AND rel.reference_relationship_json #>> '{relationshipData, fieldId}' = backref_references.backref_reference_field_id
                    ) rel ON TRUE

                    INNER JOIN LATERAL (
                        SELECT 1
                        FROM artifacts artifact_from
                        WHERE
                            artifact_from.artifact_id = rel.reference_relationship_artifact_id_from
                            AND artifact_from.artifact_blueprint_id = backref_references.backref_reference_bp_id
                        LIMIT 1
                    ) artifact_from ON TRUE
            ) defs,
            jsonb_array_elements_text(defs.field_ids_list) AS field_id
            GROUP BY field_id
        ) backrefs
        LIMIT 1
    ) backreferences_relationships_for_artifact ON TRUE

    -- compute the uploaded files relationships
    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.uploaded_file_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.uploaded_file_relationship_uploaded_file_id
                    ORDER BY rel.uploaded_file_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM uploaded_file_relationships rel
            WHERE
                rel.uploaded_file_relationship_artifact_id = artifacts.artifact_id
                AND rel.uploaded_file_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
        LIMIT 1
    ) uploaded_files_relationships_for_artifact ON TRUE

    -- compute the timeseries relationships
    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.time_series_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.time_series_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.time_series_relationship_time_series_id
                    ORDER BY rel.time_series_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM time_series_relationships rel
            WHERE
                rel.time_series_relationship_artifact_id = artifacts.artifact_id
                AND rel.time_series_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.time_series_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.time_series_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
        LIMIT 1
    ) time_series_relationships_for_artifact ON TRUE

    -- compute the attachment references
    LEFT JOIN LATERAL (
        SELECT
            jsonb_agg(
                rel.reference_relationship_artifact_id_to
                ORDER BY rel.reference_relationship_json #> '{relationshipData, index}' ASC
            ) AS relationships_array
        FROM reference_relationships rel
        WHERE
            rel.reference_relationship_artifact_id_from = artifacts.artifact_id
            AND rel.reference_relationship_json #>> '{relationshipData, type}' = 'attachment'
        LIMIT 1
    ) attachments_references_relationships_for_artifact ON TRUE

    -- compute the attachment uploaded files
    LEFT JOIN LATERAL (
        SELECT
            jsonb_agg(
                rel.uploaded_file_relationship_uploaded_file_id
                ORDER BY rel.uploaded_file_relationship_json #> '{relationshipData, index}' ASC
            ) AS relationships_array
        FROM uploaded_file_relationships rel
        WHERE
            rel.uploaded_file_relationship_artifact_id = artifacts.artifact_id
            AND rel.uploaded_file_relationship_json #>> '{relationshipData, type}' = 'attachment'
        LIMIT 1
    ) attachments_uploaded_files_relationships_for_artifact ON TRUE

    -- compute the whole artifact_json_with_relationships
    LEFT JOIN LATERAL (
        SELECT
            -- building the whole artifact JSON including basic fields, references, backreferences, and attachments (references and uploaded files)
            (
                artifacts.artifact_json
                    -------------------------------------------------- build the final fields map in the artifact JSON
                    || jsonb_build_object(
                        'fields',
                        ----------------------------------------------------------- first, need to add the existing fields (internal fields)
                        artifacts.artifact_json #> '{fields}'
                        ---------------------------------------------------- add the references
                        || COALESCE(references_relationships_for_artifact.relationships_object, jsonb_build_object())
                        -------------------------- add the back references
                        || COALESCE(backreferences_relationships_for_artifact.relationships_object, jsonb_build_object())
                        ---------------------------------------------------- add the uploaded files
                        || COALESCE(uploaded_files_relationships_for_artifact.relationships_object, jsonb_build_object())
                        ---------------------------------------------------- add the timeseries
                        || COALESCE(time_series_relationships_for_artifact.relationships_object, jsonb_build_object())
                    )
                    ------------------------------------------- add attachments map in the artifact JSON
                    || jsonb_build_object(
                        'attachments',
                        jsonb_build_object(
                            -------------------------------------- reference attachments
                            'references', COALESCE(attachments_references_relationships_for_artifact.relationships_array, jsonb_build_array()),
                            -------------------------------------- uploaded files attachments
                            'files', COALESCE(attachments_uploaded_files_relationships_for_artifact.relationships_array, jsonb_build_array())
                        )
                    )
            ) AS artifact_json_with_relationships
        LIMIT 1
    ) artifact_json_with_relationships ON TRUE

    LEFT JOIN LATERAL (
        SELECT COALESCE(jsonb_agg(signoffs.signoff_json), jsonb_build_array()) AS signoffs
        FROM signoffs signoffs
        WHERE signoffs.signoff_artifact_id = artifacts.artifact_id
        LIMIT 1
    ) signoffs ON TRUE

    LEFT JOIN LATERAL (
        SELECT user_mappings.user_mapping_user_login
        FROM user_mappings user_mappings
        WHERE user_mappings.user_mapping_artifact_id = artifacts.artifact_id
        LIMIT 1
    ) user_mappings ON TRUE

    LEFT JOIN LATERAL (
        SELECT group_mappings.group_mapping_group_name
        FROM group_mappings group_mappings
        WHERE group_mappings.group_mapping_artifact_id = artifacts.artifact_id
        LIMIT 1
    ) group_mappings ON TRUE

    LEFT JOIN LATERAL (
        SELECT global_api_key_mappings.global_api_key_mapping_api_key_id
        FROM global_api_key_mappings global_api_key_mappings
        WHERE global_api_key_mappings.global_api_key_mapping_artifact_id = artifacts.artifact_id
        LIMIT 1
    ) global_api_key_mappings ON TRUE

WHERE artifact_ids IS NULL OR artifacts.artifact_id = any(artifact_ids);

END;
$func$;

