(function() {
'use strict';


const app = angular.module('dataiku.catalog');


const FACET_FIELDS_DISPLAY_NAMES = Object.freeze({
    'projectName': 'Project',
    'type_raw': 'Type',
    'numColumns': 'Columns',
    'usedIn': 'Used in',
    'user': 'Contributors',
    'storedAs': 'Stored as',
    'projectKey.raw': 'Project',
    'workspaceKey': 'Workspace',
    'tag.raw': 'Tags',
    'catalog.raw': 'DB Catalog',
    'connection.raw': 'Connection',
    'schema.raw': 'DB Schema',
    'objectType': 'Object type',
    'closed': 'Status'
});


app.controller('DiscussionsInboxController', function($controller, $scope, $rootScope, DataikuAPI, TopNav, Notification) {

    function searchWrapper(...args) {
        return DataikuAPI.discussions.inbox.search.apply(this, args)
            .success(function(data) {
                $scope.unreadDiscussionFullIds = data.unreadDiscussionFullIds || [];
            }); // No need for error handling here, done in _CatalogControllerBase
    }
    $controller("_CatalogControllerBase", {$scope: $scope, searchEndpoint: searchWrapper});
    $controller('_InboxCatalogSupportController', {$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "inbox.conversations", "items", null);

    $scope.inboxPage = true;

    const projectNames = {};
    $scope.users = {};
    $scope.tagMaps = {};
    $scope.unreadDiscussionFullIds = [];

    $scope.query.facets.closed = [0]; // By default, only show open discussions

    $scope.unread = function(item) {
        const fullId = item._source.projectKey+'.'+item._source.discussionId;
        return $scope.unreadDiscussionFullIds.includes(fullId);
    };

    DataikuAPI.taggableObjects.listAllTags()
        .success(function(data) {
            $scope.tagMaps = data;
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.security.listUsers()
        .success(function(data) {
            angular.forEach(data, user => $scope.users[user.login] = user.displayName);
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.projects.list()
        .success(function(data) {
            angular.forEach(data, project =>projectNames[project.projectKey] = project.name);
        })
        .error(setErrorInScope.bind($scope));

    const ackListenerDestroyer = Notification.registerEvent('discussion-ack', function(evtType, message) {
        if (message.user != $rootScope.appConfig.login) {
            return; // Only ack current user stuff
        }
        const index = $scope.unreadDiscussionFullIds.indexOf(message.projectKey + '.' + message.discussionId);
        if (index > -1) {
            $scope.unreadDiscussionFullIds.splice(index, 1);
        }
    });
    const replyListenerDestroyer = Notification.registerEvent('discussion-reply', function(evtType, message) {
        if (message.user == $rootScope.appConfig.login) {
            return; // Don't mark current user's messages as unread
        }
        const index = $scope.unreadDiscussionFullIds.indexOf(message.projectKey + '.' + message.discussionId);
        if (index == -1) {
            $scope.unreadDiscussionFullIds.push(message.projectKey + '.' + message.discussionId);
        }
    });

    $scope.$on('$destroy', function() {
        ackListenerDestroyer();
        replyListenerDestroyer();
    });
});


// Required stuff for catalog based UI compatibility
app.controller('_InboxCatalogSupportController', function($scope, $location, StateUtils, $filter, TypeMappingService) {
    $scope.hasNavigator = item => false;

    $scope.isItemSelectable = item => false;

    $scope.getTorFromSource = (src) => ({
        type: src.tor.split('.')[1], // we cannot use src.objectType because it's the IndexableType of the object
        id: src.objectId,
        projectKey: src.projectKey,
        workspaceKey: src.workspaceKey,
    });

    $scope.getLink = function(item) {
        const src = item._source;
        return StateUtils.href.taggableObject($scope.getTorFromSource(src), {discussionId: src.discussionId});
    };

    $scope.goToItem = function(item) {
        $location.path($scope.getLink(item));
    };

    $scope.mapItemToIcon = function(item, size) {
        // those icons are not mapped yet so let's do something dirty for now, otherwise it will have impact on the whole application
        if (item.key === 'chart') {
            // TODO : remove this hack when implementing [sc-243801]
            return 'dku-icon-chart-' + size;
        }
        return TypeMappingService.mapTypeToIcon(item.key, size);
    }

    $scope.sortBy = [
        {
            label: 'Last reply',
            value: item => item._source.lastReplyTime
        },
        {
            label: 'Read',
            value: item => $scope.unread(item)
        }
    ];
    $scope.sortOptions = {
        column: $scope.sortBy[0].value,
        reverse: true
    };

    $scope.formatFacetField = function(field) {
        return FACET_FIELDS_DISPLAY_NAMES[field] || $filter('capitalize')(field);
    };

    $scope.formatFacetValue = function(value, facet) {
        if (facet === 'closed') {
            return value ? 'Resolved discussions' : 'Opened discussions';
        } else if (facet === 'objectType') {
            return $filter('capitalize')(userFriendlyType(value));
        }
        return value;
    };

    function userFriendlyType(indexableType) {
        switch (indexableType) {
            case 'lambda_service':
                return 'API Service';
            case 'gen_ai_model':
                return 'gen AI model';
            default:
                return indexableType.replace(/_/g, " ");
        }
    };

    $scope.formatItemName = function(item, inList) {
        const src = item._source;
        // Comes from _source, encode HTML entities in order to display attributes like <stuff
        const topic = (item.highlight && item.highlight['discussions.topic']) ? item.highlight['discussions.topic'][0] : ($filter('escapeHtml')(((src.discussions && src.discussions.length && src.discussions[0].topic) ? src.discussions[0].topic : "Unnamed discussion")));
        const title = topic + " <small>on " + userFriendlyType(src.objectType) + "</small> " + $filter('escapeHtml')(src.objectName);
        return title;
    };

    $scope.itemCount = function() {
        const hits = $scope.results && $scope.results.hits ? $scope.results.hits.total : 0;
        return '<strong>' + hits + '</strong> discussion' + (hits > 1 ? 's' : '');
    };

    $scope.selectInput = function() {
        $(".catalog-search-input").select();
    };

    $scope.inboxFacetFilter = function(search) {
        return item => item.key !== undefined && (!search || item.key.toLowerCase().includes(search.toLowerCase()));
    };
});

})();
