(function () {
  "use strict";

  angular.module("dataiku.opals").factory("StateObserverService", StateObserverService);

  function isIterable(val) {
    return (
      typeof Symbol !== "undefined" &&
      Symbol &&
      "iterator" in Symbol &&
      val != null &&
      typeof val[Symbol.iterator] === "function"
    );
  }

  /**
   * Ponyfill for AggregateError not available on our test env
   */
  class AggregateError extends Error {
    constructor(errors, message = "") {
      super(message);
      if (!Array.isArray(errors) && !isIterable(errors)) {
        throw new TypeError(`${errors} is not an iterable`);
      }
      this.errors = [...errors];
    }
  }

  /**
   * Ponyfill for Promise.any method not available on our test env
   * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/any
   * @param {*} promises
   * @returns
   */
  function pAny(promises) {
    const PromiseContext = this !== null && this !== void 0 ? this : Promise;
    return PromiseContext.all(
      promises.map((promise) => {
        var _promise$then;
        const wrappedPromise =
          ((_promise$then = promise === null || promise === void 0 ? void 0 : promise.then) !== null &&
          _promise$then !== void 0
            ? _promise$then
            : null) !== null
            ? promise
            : PromiseContext.resolve(promise);
        try {
          return wrappedPromise.then(
            (value) => {
              throw value;
            },
            (error) => error
          );
        } catch (error) {
          return error;
        }
      })
    ).then(
      (reasons) => {
        throw new AggregateError(reasons, "Every promise rejected");
      },
      (value) => value
    );
  }

  function StateObserverService(
    $rootScope,
    $state,
    OpalsService,
    RecipeContextService,
    NotebookContextService,
    WebappContextService,
    ProjectContextService,
    WorkspaceContextService,
    Debounce,
  ) {
    function init() {
      const processState = Debounce().withDelay(500, 500).wrap(function(toState, toParams) {
        const stateInfo = { toState, toParams, context: [toState.name] };
        // we may want to have this iterable of promise run trough a allSettled to check if more than one promise resolved (possible conflict)
        const stateEnrichmentPromises = [
          RecipeContextService.context(stateInfo),
          NotebookContextService.context(stateInfo),
          WebappContextService.context(stateInfo),
          ProjectContextService.context(stateInfo),
          WorkspaceContextService.context(stateInfo),
        ];
        pAny(stateEnrichmentPromises)
          .then((enrichedStateInfo) => {
            OpalsService.sendDssStateChange({
              stateName: enrichedStateInfo.toState.name,
              stateParams: enrichedStateInfo.toParams,
              context: enrichedStateInfo.context,
            });
          })
          .catch(() => {
            OpalsService.sendDssStateChange({
              stateName: stateInfo.toState.name,
              stateParams: stateInfo.toParams,
              context: stateInfo.context,
            });
          });
      });

      function start() {
        // we process the current state
        processState($state.current, $state.params);
        // we observe the route state
        $rootScope.$on("$stateChangeSuccess", function (_, toState, toParams) {
          processState(toState, toParams);
        });
        // we observe right panel state
        $rootScope.$on("opalsCurrentRightPanelTab", function (_, tab) {
          OpalsService.sendDssContextChange(tab);
        });

        $rootScope.$on("opalsCurrentTab", function (_, tab) {
          OpalsService.sendDssContextChange(tab);
        });
      }

      start();
    }

    return { init };
  }
})();
