describe("Collection Filtering service", function(){
	beforeEach(module('dataiku.services'));
	beforeEach(module('dataiku.common.func'));
    beforeEach(module('dataiku.mock'));
    
    function checkFilter(expectedMatching, expectedNotMatching, userFilter, params) {
        if (!Array.isArray(expectedMatching)) expectedMatching = [expectedMatching];
        if (!Array.isArray(expectedNotMatching)) expectedNotMatching = [expectedNotMatching];
        inject(function(CollectionFiltering) {
            expect(CollectionFiltering.filter(
                [...expectedMatching, ...expectedNotMatching],
                userFilter,
                params
            )).toEqual(expectedMatching);
        });
    }

    it("should filter on basic field user object query", function() {
        var item1 = { a:'aa', b:{c:'1'}};
        var item2 = { a:'aa', b:{c:'2'}};
        var others = [
            { a:'bb', b:{c:'1',d:34} },
            { a:'cc', b:{e:90809} }
        ];
        checkFilter(
            [item1, item2], 
            others, 
            { a: 'aa', }, 
            {}
        );
    });

    it("should filter on multiple fields user queries", function() {
        var item1 = { a:'aa', b:{c:'1'}};
        var others = [
            { a:'bb', b:{c:'1',d:34} },
            { a:'cc', b:{e:90809} }
        ];
        checkFilter(
            item1, 
            others, 
            {
                a: 'aa',
                b: { c : '1'}
            }, 
            {}
        );
    });
    
    it("should filter on user not query", function() {
        checkFilter(
            [
                { a:'cc', b:{e:90809}},
                { a:'bb', b:{c:'1',d:34}},
            ],
            [   
                { a:'aa', b:{c:'1'}},
                { a:'bb', b:{c:'aa'}},
            ],
            {
                userQuery:'not:aa',
            }, 
            {
                userQueryTargets:['a','b.c'] // fails without userQueryTargets for some reason
            }
        );
    });
    
    it("should allow userQueryTargets to limit the scope of userQuery", function() {
        checkFilter(
            [
                { a:'cc', b:{e:90809}, c: 'aa'},
                { a:'bb', b:{c:'1',d:34}},
                { a:'bb', b:{c:'aa'}},
            ],
            [   
                { a:'aa', b:{c:'1'}},
            ],
            {
                userQuery:'not:aa',
            }, 
            {
                userQueryTargets:['a']
            }
        );
    });

    it("should fill userQueryResult correctly on basic field user object query (with an array of not-field-specific words)", function() {
        const params = {}

        inject(function(CollectionFiltering) {
            CollectionFiltering.filter(
                [],
                { userQuery: 'tutu name:coucou toto' }, 
                params
            )
            expect(params.userQueryResult).toEqual(['tutu', 'toto'])
        });
    });

    it("should filter on multiple and nested fields", function() {
        checkFilter(
            [{ a:'aa', b: {c:2}}],
            [
                { a:'aa', b:{c:1} },
                { a:'bb', b:{c:1,d:34} },
                { a:'cc', b:{e:90809} },
                { a:'dd', b:{c:1} },
                { b:{c:1,e:'ee'} }
            ],
            {
                userQuery:'a:aa b.c:2'
            }, 
            {}
        );
    });
    
    it("should filter out items matching '-field:' query", function() {
        checkFilter(
            [
                { a:'bb', b:{c:1,d:34} },
                { a:'cc', b:{e:90809} },
                { a:'dd', b:{c:1} },
                { b:{c:1,e:'ee'} }
            ],
            [
                { a:'aa', b:{c:2}}, { a:'aa', b:{c:1} }
            ],
            {
                userQuery:'-a:aa'
            }, 
            {}
        );
    });
    
    it("should not apply userQueryTargets to '-field:' queries", function() {
        checkFilter(
            [
                { a:'dd', b:{c:1}, c: 21},
                { b:{c:1, e:'aa'}, d: 'aa' }
            ],
            [
                { a:'bb', b:{c:1,d:34}, c: 12},
                { a:'aa', b:{c:2}}, 
                { a:'aa', b:{c:1}}
            ],
            {
                userQuery:'-a:aa -c:12' 
            }, 
            {
                userQueryTargets: ["a"]
            }
        );
    });

    it("should filter on excluding tag with nested tag", function() {
        checkFilter(
            { a:'cc', b: { e:90809 }},
            [
                { a:'aa', b: {e: 90809} }, 
                { a:'aa', b:{c:1} },
                { a:'bb', b:{c:1,d:34} },
                { b: { c:1, e:'ee'} }
            ],
            {
                userQuery:'-a:aa b.e:90809'
            }, 
            {}
        );
    });

    it("should filter on tag with regex query", function() {
        checkFilter(
            [
                { a:'this is another very long string', b:{c:2} },
                { a:'this is a very long string', b:{c:1,d:34} },
            ],
            [
                { a: 'this is a long string', b:{c:1} },
                { a: 'v12345', b:{e:90809} },
                { a: 'v1234', b:{c:1} },
                { b: {c: 'this is a very long string', e:'this is a long string'} }
            ],
            {
                userQuery:'a:/this.*very/',
            }, 
            {
                userQueryTargets:['a','b.c']
        	}
        );
    });

    it("should filter on excluding tag with regex query", function() {
        checkFilter(
            [
                { a:'this is a long string', b:{c:1} },
                { a:'v12345', b: {e:90809} },
                { b:{c:1,e:'this is a very long string'} }
            ],
            [
                { a:'this is another very long string', b:{c:2} },
                { a:'this is a very long string', b:{c:1,d:34} },
            ],
            {
                userQuery:'-a:/this.*very/',
            }, 
            {}
        );
    });

    it("should filter on excluding tag with target remapping", function() {
        checkFilter(
            [
                { b:{c:1,e:'this is a long string'} }
            ],
            [
                { a:'this is a long string', b:{c:1} }, // filtered out by bb
                { a:'this is another very long string', b:{e:'long string'} }, // filtered out by -aa
            ],
            {
                userQuery:'-aa:/this.*very/ qq:long',
            },
            {
                userQueryTargets:['a','b.c'],
                propertyRules:{aa:'a', qq:'b.e'}
            }
        );
    });

    it("should filter everything when no regex match", function() {
        checkFilter(
            [],
            [
                { a:'v12345', b:{e:90809} },
                { a:'v1234', b:{c:1} },
                { a:'this is another very long string', b:{c:2} },
                { a:'this is a very long string', b:{c:1,d:34} },
                { a:'this is a long string', b:{c:1} },
                { b: {c:1,e:'this is a long string'} }
            ],
            {
                userQuery:'/NothingTHERETOBESEEN/',
            }, 
            {}
        );
    });

    it("should filter when search for property that object does not have", function() {
        checkFilter(
            [],
            [
                { a:'v12345', b:{e:90809} },
                { a:'this is another very long string', b:{c:2} },
                { b: {c:1,e:'this is a long string'} }
            ],
            {
                userQuery:'nothing:/.*/',
            }, 
            {}
        );
    });
    
    it("should not filter when excluding tag for property that object does not have", function() {
        checkFilter(
            [
                { a:'v12345', b:{e:90809} },
                { a:'this is another very long string', b:{c:2} },
                { b: {c:1,e:'this is a long string'} }
            ], 
            [],
            {
                userQuery:'-nothing:/.*/',
            }, 
            {}
        );
    });
    
    it("should filter depending on propertyRules remapping", function() {
        checkFilter(
            [
                { a:'this is another very long string', b:{c:2} }
            ],
            [
                { a:'this is a long string', b:{c:1} },
                { b:{c:1,e:'this is a long string'} }
            ],
            {
                userQuery:'aa:long -qq:1',
            }, 
            {
                propertyRules:{aa:'a', qq:'b.c'}
            }
        );
    });

    it('should filter on userQuery', () => {
        checkFilter(
            { id: '1', a:'aab11', b: {c:'bbb1'} },
            { id: '1', a:'aab12', b: {c:'bbb2'} },
            { userQuery: 'aab11' },
            {},
        )
    });

    it('should filter on userQuery with deep field', () => {
        checkFilter(
            { id: '1', a:'aab11', b: {c:'bbb1'} },
            { id: '1', a:'aab12', b: {c:'bbb2'} },
            { userQuery: 'bbb1' },
            {},
        )
    });

    it('should filter items that match all words from userQuery across fields', () => {
        checkFilter(
            [{ a:'aaa11', b: {c:'bbb1'} }, { a:'bbb1 aaa11', b: {c:'bbb2'} }],
            [{ a:'aaa11', b: {c:'bbb2'} }, { a:'aaa12', b: {c:'bbb1'} }],
            { userQuery: 'aaa11 bbb1' },
            {},
        )
    });

    it('should filter on explicit field used along userQuery', () => {
        checkFilter(
            { a:'aab11', b: {c:'bbb1'} },
            [{ a:'aab11', b: {c:'bbb2'} }, { a:'aab12', b: {c:'bbb1'} }],
            { a: 'aab11', userQuery: 'bbb1' },
            {},
        )
        checkFilter(
            { id: '1', a:'aab11', b: {c:'bbb1'} },
            [],
            { a: 'aab1', userQuery: 'aab1' },
            {},
        )
    });

    it('should allow userQueryTargets to limit the scope of userQuery', () => {
        checkFilter(
            { a:'aab11', b: {c:'bbb1'} },
            { a:'aab12', b: {c:'bbb2'} },
            { userQuery: 'b1' },
            { userQueryTargets: 'b.c' },
        );
    });

    it('should allow explicit fields to ignore the scope of userQueryTargets', () => {
        checkFilter(
            { a:'aab11', b: {c:'bbb1'} },
            { a:'aab12', b: {c:'bbb2'} },
            { userQuery: 'aab1', 'b.c': 'bbb1' },
            { userQueryTargets: 'a' },
        );
    });

    it('should allow explicit fields in userQuery to ignore the scope of userQueryTargets', () => {
        checkFilter(
            { a:'aab11', b: {c:'bbb1'} },
            { a:'aab12', b: {c:'bbb2'} },
            { userQuery: 'aab1 b.c:bbb1' },
            { userQueryTargets: 'a' },
        );
    });

    it('should match an item if a nested array contains one matching element', () => {
        checkFilter(
            { a:'aab11', b: [{c: 'bbb1'}, {c: 'bbb2'}] },
            [{ a:'aab12', b: [] }, { a:'aab12', b: [{c: 'bbb1'}] }],
            { userQuery: 'bbb2' },
            {},
        );
    });

    it('should match an item if a nested array contains one matching element, with userQueryTargets', () => {
        checkFilter(
            [
                { a:'aab11', b: [{c: 'bbb1'}, {c: 'bbb2'}] },
                { a:'aab12', b: [{c: {d: 'bbb2'}}] } // deeper in b.c
            ],
            [
                { a:'aab12', b: [{d: 'bbb2'}] }, // b.d, not b.c
            ],
            { userQuery: 'bbb2' },
            { userQueryTargets: 'b.c' },
        );
    });
})
