import pmdarima
from statsmodels.tsa.arima.model import ARIMA
from statsmodels.tsa.statespace.sarimax import SARIMAX

from dataiku.base.utils import package_is_at_least

# We only support retrieving coefficients for SARIMAX models, which are used by pmdarima's auto_arima method starting from version 1.5.0
SUPPORTS_MODEL_COEFFICIENTS = package_is_at_least(pmdarima, "1.5.0")


def dku_auto_arima(
    target_values, external_features_values, start_p, max_p, d, max_d, start_q, max_q, start_P, max_P, D, max_D,
    start_Q, max_Q, max_order, stationary, maxiter, m, information_criterion, test, seasonal_test, method, seasonal
):
    if package_is_at_least(pmdarima, "1.8.0"):
        # Starting from version 1.8.0 the exogenous argument is deprecated in favor of X
        # Also, "trace" becomes an int to express a trace level instead of a boolean
        return pmdarima.auto_arima(
            target_values, X=external_features_values, start_p=start_p, max_p=max_p, d=d, max_d=max_d,
            start_q=start_q, max_q=max_q, start_P=start_P, max_P=max_P, D=D, max_D=max_D, start_Q=start_Q, max_Q=max_Q,
            max_order=max_order, stationary=stationary, maxiter=maxiter, m=m, information_criterion=information_criterion, test=test,
            seasonal_test=seasonal_test, method=method, seasonal=seasonal, trace=2)
    else:
        return pmdarima.auto_arima(
            target_values, exogenous=external_features_values, start_p=start_p, max_p=max_p, d=d, max_d=max_d,
            start_q=start_q, max_q=max_q, start_P=start_P, max_P=max_P, D=D, max_D=max_D, start_Q=start_Q, max_Q=max_Q,
            max_order=max_order, stationary=stationary, maxiter=maxiter, m=m, information_criterion=information_criterion, test=test,
            seasonal_test=seasonal_test, method=method, seasonal=seasonal, trace=True)


def instantiate_arima_model(target_values, external_features_values, auto_arima_params):
    if package_is_at_least(pmdarima, "1.5.0"):
        # Starting from version 1.5.0 pmdarima's auto_arima method
        # only uses SARIMAX under the hood
        return SARIMAX(
            endog=target_values,
            exog=external_features_values,
            order=auto_arima_params["order"],
            seasonal_order=auto_arima_params["seasonal_order"] or (0, 0, 0, 0),  # (0, 0, 0, 0) is ARIMA default for non-seasonal
            trend=auto_arima_params["trend"],
        )
    else:
        return ARIMA(
            endog=target_values,
            exog=external_features_values,
            order=auto_arima_params["order"],
            seasonal_order=auto_arima_params["seasonal_order"] or (0, 0, 0, 0),  # (0, 0, 0, 0) is ARIMA default for non-seasonal
            trend=auto_arima_params["trend"],
        )
