"""Dku Pickle Wrapper

This script is supposed to be used to perform all pickling operations in
dip.

Historically this script was part of the doctor module but to allow Python
processes to use this script without importing all the doctor dependencies,
generic functions such as the ones below were moved here.

Since many users have code envs that do not have cloudpickle installed,
the incentive to use this script instead of importing pickle or cloudpickle
directly is to let the user benefit from the power of cloudpickle.dump
while minimizing the chances to break things for users that didn't install
cloudpickle.

It also gives us better control and lets us provide explicit logs in case
of problems regarding pickle/cloudpickle.
"""

import logging
from pickle import Unpickler
import sklearn

from dataiku.base.utils import package_is_at_least

logger = logging.getLogger(__name__)


def dump(pkl_object, pkl_file):
    try:
        import cloudpickle as pickle
        logger.info("Setting cloudpickle as the pickle dumping tool")
    except ImportError:
        logger.warning("Cloudpickle is not installed on this code env. Falling back on pickle for dumping json.")
        try:
            import cPickle as pickle
        except:
            import pickle
    """
    Serializes an object in the given file using pickle or cloudpickle (if available)
    :param pkl_object: object to serialize
    :param pkl_file: pickle file to write
    :param protocol: pickle.dump's "protocol" attribute (cf. official pickle doc)
    """
    pickle.dump(pkl_object, pkl_file)


def load(pkl_file):
    """
    Deserializes an object in the given file using pickle
    :param pkl_file: pickle file to deserialize
    :return deserialized object
    """
    try:
        return ModelUnpickler(pkl_file).load()
    except UnicodeDecodeError:
        raise PickleLoadException(u"Failed to unpickle {}. You might have been trying to load a resource saved in a python 2 code environment with a python 3 one.".format(pkl_file.name))


class PickleLoadException(Exception):
    pass

# Inherit object for python2.7 inheritance compliance
class ModelUnpickler(Unpickler, object):
    """
    Tries to load models.
    # Primarily focuses on compatibility between sklearn pickes (from sklearn 0.20 in sk 0.24 and sk 1.0 formats.)
    It search for classes in the normal way, but when a module is not found it will try from another path.
    It's known to fail on GBT, SVM based and KNN models

    Known useful paths remapping :
    'sklearn.cluster.k_means_' : 'dataiku.doctor.utils.skcompat.UnpickableKMeans' in order to have a hacked __set_state__ during unpickling
    'sklearn.ensemble.forest': 'sklearn.ensemble._forest',
    'sklearn.ensemble.gradient_boosting': 'sklearn.ensemble._gradient_boosting',
    'sklearn.linear_model.base': 'sklearn.linear_model._base',
    'sklearn.linear_model.coordinate_descent': 'sklearn.linear_model._coordinate_descent',
    'sklearn.linear_model.least_angle': 'sklearn.linear_model._least_angle',
    'sklearn.linear_model.logistic': 'sklearn.linear_model._logistic',
    'sklearn.linear_model.ridge': 'sklearn.linear_model._ridge',
    'sklearn.linear_model.sgd_fast': 'sklearn.linear_model._sgd_fast'
    'sklearn.linear_model.stochastic_gradient': 'sklearn.linear_model._stochastic_gradient',
    'sklearn.neighbors.classification': 'sklearn.neighbors._classification',
    'sklearn.neighbors.kd_tree': 'sklearn.neighbors._kd_tree',
    'sklearn.neighbors.regression': 'sklearn.neighbors._regression',
    'sklearn.neural_network.multilayer_perceptron': 'sklearn.neural_network._multilayer_perceptron',
    'sklearn.preprocessing.label': 'sklearn.preprocessing._label',
    'sklearn.svm.classes': 'sklearn.svm._classes',
    'sklearn.tree.tree': 'sklearn.tree._classes',
    'dataiku.doctor.timeseries.models.gluonts': 'dataiku.doctor.timeseries.models.gluonts.mxnet' - This was moved in 13.2.0
    """

    def __init__(self, file, **kwargs):
        super(ModelUnpickler, self).__init__(file, **kwargs)

    def map_module_name(self, module, name):
        if module == 'sklearn.tree.tree' or module == 'sklearn.tree._classes':
            if package_is_at_least(sklearn, "1.4"):
                if name == 'DecisionTreeRegressor':
                    return 'dataiku.doctor.utils.skcompat', 'UnpicklableDecisionTreeRegressor'
                elif name == 'DecisionTreeClassifier':
                    return 'dataiku.doctor.utils.skcompat', 'UnpicklableDecisionTreeClassifier'
                elif name == 'ExtraTreeClassifier':
                    return 'dataiku.doctor.utils.skcompat', 'UnpicklableExtraTreeClassifier'
                elif name == 'ExtraTreeRegressor':
                    return 'dataiku.doctor.utils.skcompat', 'UnpicklableExtraTreeRegressor'
                else:
                    return 'sklearn.tree._classes', name
            else:
                return 'sklearn.tree._classes', name
        elif module == 'sklearn.neighbors.dist_metrics':
            # scikit 0.24 moved dist_metrics
            new_module = 'sklearn.metrics._dist_metrics'
            new_name = name
            # scikit 1.3 renamed _dist_metrics.EuclideanDistance
            if package_is_at_least(sklearn, "1.3") and name == "EuclideanDistance":
                new_name = "EuclideanDistance64"
            return new_module, new_name
        elif module == 'sklearn.tree._tree' and name == "Tree":
            return 'dataiku.doctor.utils.skcompat', 'UnpicklableTree'
        elif (
            module == "sklearn.ensemble.gradient_boosting"  # case scikit-learn < 0.22
            or module == "sklearn.ensemble._gb"  # case scikit-learn >= 0.22
            or module == "sklearn.ensemble._gb_losses" # case scikit-learn < 1.4
        ):
            if name == 'GradientBoostingClassifier':
                return 'dataiku.doctor.utils.skcompat', 'UnpicklableGradientBoostingClassifier'
            elif name == 'GradientBoostingRegressor':
                return 'dataiku.doctor.utils.skcompat', 'UnpicklableGradientBoostingRegressor'
            elif name == 'LogOddsEstimator':
                return 'dataiku.doctor.utils.skcompat', 'LogOddsEstimatorDummyClassifier'
            elif name == 'ScaledLogOddsEstimator':
                return 'dataiku.doctor.utils.skcompat', 'ScaledLogOddsEstimatorDummyClassifier'
            elif name == 'PriorProbabilityEstimator':
                return 'dataiku.doctor.utils.skcompat', 'PriorProbabilityEstimatorDummyClassifier'
            elif name == 'QuantileEstimator':
                return 'dataiku.doctor.utils.skcompat', 'QuantileEstimatorDummyRegressor'
            elif name == 'MeanEstimator':
                return 'dataiku.doctor.utils.skcompat', 'MeanEstimatorDummyRegressor'
            elif name == 'MultinomialDeviance':
                if package_is_at_least(sklearn, "1.4"):
                    return 'dataiku.doctor.utils.skcompat', 'RemovedGradientBoostingLoss'
                else:
                    return 'dataiku.doctor.utils.skcompat', 'Sk020MultinomialDeviance'
            else:
                if package_is_at_least(sklearn, "1.4"):
                    return 'dataiku.doctor.utils.skcompat', 'RemovedGradientBoostingLoss'
                else: # Handles other loss functions like BinomialDeviance
                    return 'sklearn.ensemble._gb_losses', name
        elif module == "sklearn.cluster.k_means_":
            return 'dataiku.doctor.utils.skcompat', 'UnpickableKMeans'
        elif module == "sklearn.neighbors.classification" and name == 'KNeighborsClassifier':
            return 'dataiku.doctor.utils.skcompat', 'UnpickableKNeighborsClassifier'
        elif module == 'sklearn.calibration':
            if name == '_CalibratedClassifier':
                return 'dataiku.doctor.utils.skcompat', 'UnpicklableCalibratedClassifier'
            # out of options, just return None
        elif module == 'sklearn.isotonic' and name == 'IsotonicRegression':
            return 'dataiku.doctor.utils.skcompat', 'UnpicklableIsotonicRegression'
        elif module == 'sklearn.svm.classes':
            if name == 'SVR':
                return 'dataiku.doctor.utils.skcompat', 'UnpicklableSVR'
            if name == 'SVC':
                return 'dataiku.doctor.utils.skcompat', 'UnpicklableSVC'
            # out of options, just return None
        elif module == 'sklearn.linear_model._stochastic_gradient' and name == 'SGDClassifier':
            return 'dataiku.doctor.utils.skcompat', 'UnpicklableSGDClassifier'
        elif module == 'dataiku.doctor.timeseries.models.gluonts.simple_feed_forward' and name == 'DkuSimpleFeedForwardEstimator':
            return 'dataiku.doctor.timeseries.models.gluonts.mxnet.simple_feed_forward', "DkuSimpleFeedForwardEstimator"
        elif module == 'dataiku.doctor.timeseries.models.gluonts.deepar' and name == 'DkuDeepAREstimator':
            return 'dataiku.doctor.timeseries.models.gluonts.mxnet.deepar', "DkuDeepAREstimator"
        elif module == 'dataiku.doctor.timeseries.models.gluonts.mq_cnn' and name == 'DkuMQCNNEstimator':
            return 'dataiku.doctor.timeseries.models.gluonts.mxnet.mq_cnn', "DkuMQCNNEstimator"
        elif module == 'dataiku.doctor.timeseries.models.gluonts.transformer' and name == 'DkuTransformerEstimator':
            return 'dataiku.doctor.timeseries.models.gluonts.mxnet.transformer', "DkuTransformerEstimator"
        elif 'sklearn.' in module:
            last_submodule = module.rindex(".")
            return u"{}_{}".format(module[0:last_submodule + 1], module[last_submodule + 1:]), name
        elif module == 'statsmodels.tsa._stl' and name == 'STL':
            return 'statsmodels.tsa.stl._stl', 'STL' # case statsmodel > 0.14
        elif module == 'keras.src.preprocessing.text':
            return 'keras.src.legacy.preprocessing.text', name
        # no mapping, developers should add a mapping or this pickle will remain un-unpickable
        return None, None

    @staticmethod
    def forced_mapping(module, name):
        forced_customunpicklers = {'sklearn.calibration': ['_CalibratedClassifier'],
                                   'sklearn.isotonic': ['IsotonicRegression'],
                                   'sklearn.ensemble._gb': ['GradientBoostingClassifier', 'GradientBoostingRegressor'],
                                   'sklearn.tree._tree': ['Tree'],
                                   'sklearn.tree._classes': ['DecisionTreeClassifier', 'DecisionTreeRegressor', 'ExtraTreeClassifier', 'ExtraTreeRegressor'],
                                   'sklearn.linear_model._stochastic_gradient': ['SGDClassifier']}
        return name in forced_customunpicklers.get(module, [])

    def find_class(self, module, name):
        if self.forced_mapping(module, name):
            mapped_module, mapped_name = self.map_module_name(module, name)
            return super(ModelUnpickler, self).find_class(mapped_module, mapped_name)
        else:
            try:
                return super(ModelUnpickler, self).find_class(module, name)
            except ModuleNotFoundError:
                mapped_module, mapped_name = self.map_module_name(module, name)
                if not mapped_module or not mapped_name:
                    logger.warning(u"Couldn't unpickle {}.{} and there is no remapping to try".format(module, name))
                    raise
                logger.warning(u"Couldn't unpickle {}.{}, retrying with {}.{}".format(module, name, mapped_module, mapped_name))
                return super(ModelUnpickler, self).find_class(mapped_module, mapped_name)

    def load(self):
        return super(ModelUnpickler, self).load()
