(function(){
'use strict';

var app = angular.module('dataiku.ml.report', []);

app.constant("AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS", [
{
    name: 'sagemaker',
    fullName: 'Amazon SageMaker',
    shortName: 'SageMaker',
    savedModelType: 'PROXY_MODEL',
    connectionType: 'SageMaker',
    canAuthenticateFromEnvironment: true,
    icon: 'dku-icon-model-sagemaker',
    inputFormats: [
        {
            name: 'INPUT_SAGEMAKER_CSV',
            displayName: 'SageMaker - CSV'
        },
        {
            name: 'INPUT_SAGEMAKER_JSON',
            displayName: 'SageMaker - JSON'
        },
        {
            name: 'INPUT_SAGEMAKER_JSON_EXTENDED',
            displayName: 'SageMaker - JSON (extended)'
        },
        {
            name: 'INPUT_SAGEMAKER_JSONLINES',
            displayName: 'SageMaker - JSONLINES'
        },
        {
            name: 'INPUT_DEPLOY_ANYWHERE_ROW_ORIENTED_JSON',
            displayName: 'Deploy Anywhere - Row oriented JSON'
        },
    ],
    outputFormats: [
        {
            name: 'OUTPUT_SAGEMAKER_CSV',
            displayName: 'SageMaker - CSV'
        },
        {
            name: 'OUTPUT_SAGEMAKER_ARRAY_AS_STRING',
            displayName: 'SageMaker - Array as string'
        },
        {
            name: 'OUTPUT_SAGEMAKER_JSON',
            displayName: 'SageMaker - JSON'
        },
        {
            name: 'OUTPUT_SAGEMAKER_JSONLINES',
            displayName: 'SageMaker - JSONLINES'
        },
        {
            name: 'OUTPUT_DEPLOY_ANYWHERE_JSON',
            displayName: 'Deploy Anywhere - JSON'
        },
    ]
},
{
    name: 'azure-ml',
    fullName: 'Azure Machine Learning',
    shortName: 'Azure ML',
    savedModelType: 'PROXY_MODEL',
    connectionType: 'AzureML',
    canAuthenticateFromEnvironment: true,
    icon: 'dku-icon-model-azureml',
    inputFormats: [
        {
            name: 'INPUT_AZUREML_JSON_INPUTDATA',
            displayName: 'Azure ML - JSON (input_data)'
        },
        {
            name: 'INPUT_AZUREML_JSON_INPUTDATA_DATA',
            displayName: 'Azure ML - JSON (input_data with data and columns)'
        },
        {
            name: 'INPUT_AZUREML_JSON_WRITER',
            displayName: 'Azure ML - JSON (Inputs/data)'
        },
        {
            name: 'INPUT_DEPLOY_ANYWHERE_ROW_ORIENTED_JSON',
            displayName: 'Deploy Anywhere - Row oriented JSON'
        },
    ],
    outputFormats: [
        {
            name: 'OUTPUT_AZUREML_JSON_OBJECT',
            displayName: 'Azure ML - JSON (Object)'
        },
        {
            name: 'OUTPUT_AZUREML_JSON_ARRAY',
            displayName: 'Azure ML - JSON (Array)'
        },
        {
            name: 'OUTPUT_DEPLOY_ANYWHERE_JSON',
            displayName: 'Deploy Anywhere - JSON'
        },
    ],
},
{
    name: 'vertex-ai',
    fullName: 'Google Vertex AI',
    shortName: 'Vertex AI',
    savedModelType: 'PROXY_MODEL',
    connectionType: 'VertexAIModelDeployment',
    canAuthenticateFromEnvironment: true,
    icon: 'dku-icon-model-google-vertex',
    inputFormats: [
        {
            name: 'INPUT_VERTEX_DEFAULT',
            displayName: 'Vertex - default'
        }
    ],
    outputFormats: [
        {
            name: 'OUTPUT_VERTEX_DEFAULT',
            displayName: 'Vertex - default'
        }
    ],
},
{
    name: 'databricks',
    fullName: 'Databricks',
    shortName: 'Databricks',
    savedModelType: 'PROXY_MODEL',
    connectionType: 'DatabricksModelDeployment',
    canAuthenticateFromEnvironment: false,
    icon: 'dku-icon-model-databricks',
    inputFormats: [
        {
            name: 'INPUT_RECORD_ORIENTED_JSON',
            displayName: 'Databricks - Record oriented JSON'
        },
        {
            name: 'INPUT_SPLIT_ORIENTED_JSON',
            displayName: 'Databricks - Split oriented JSON'
        },
        {
            name: 'INPUT_TF_INPUTS_JSON',
            displayName: 'Databricks - TS Inputs JSON'
        },
        {
            name: 'INPUT_TF_INSTANCES_JSON',
            displayName: 'Databricks - TF Instances JSON'
        },
        {
            name: 'INPUT_DATABRICKS_CSV',
            displayName: 'Databricks - CSV'
        }
    ],
    outputFormats: [
        {
            name: 'OUTPUT_DATABRICKS_JSON',
            displayName: 'Databricks - JSON'
        },
    ]
},
{
    name: 'mlflow',
    fullName: 'MLflow',
    shortName: 'MLflow',
    savedModelType: 'MLFLOW_PYFUNC',
    icon: 'dku-icon-model-mlflow'
},
{
    name: 'finetuned',
    fullName: 'Fine-tuned LLM' ,
    shortName: 'Fine-tuned LLM',
    savedModelType: 'LLM_GENERIC',
    icon: 'dku-icon-saved-model-fine-tuning'
}]);

app.controller("_GenAiPredictionModelReportController", function($scope, $controller, isLLMEvaluation, isAgentEvaluation) {

    $controller("_ModelReportControllerBase", {$scope:$scope});

    $scope.uiState = $scope.uiState || {};

    $scope.isLLMEvaluation = function() {
        return isLLMEvaluation;
    }

    $scope.isAgentEvaluation = function() {
        return isAgentEvaluation;
    }
});

/**
 * Controller for displaying results screen of a prediction model,
 * either in a PMLTask or a PredictionSavedModel
 *
 * Requires: $stateParams.fullModelId or $scope.fullModelId
 *
 * Must be inserted in another controller.
 */
app.controller("_PredictionModelReportController", function($scope, $controller, Assert, DataikuAPI,
    Debounce, $stateParams, ActivityIndicator, Fn, TopNav, BinaryClassificationModelsService,
    PartitionedModelsService, ActiveProjectKey, FullModelLikeIdUtils, SavedModelsService, MLDiagnosticsService,
    ModelEvaluationUtils, ModelDataUtils, AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS, CachedAPICalls) {

    $controller("_ModelReportControllerBase", {$scope:$scope});

    $scope.uiState = $scope.uiState || {};

    $scope.ModelDataUtils = ModelDataUtils;

    $scope.areMetricsWeighted = function() {
        return $scope.modelData && $scope.modelData.coreParams && !!$scope.modelData.coreParams.weight && ($scope.modelData.coreParams.weight.weightMethod == "SAMPLE_WEIGHT" || $scope.modelData.coreParams.weight.weightMethod == "CLASS_AND_SAMPLE_WEIGHT");
    };

    $scope.areMetricsInversePropensityWeighted = function() {
        return $scope.modelData && $scope.modelData.modeling.metrics.causalWeighting == "INVERSE_PROPENSITY";
    };

    $scope.getSampleWeightVariable = function() {
        return $scope.areMetricsWeighted() ? $scope.modelData.coreParams.weight.sampleWeightVariable : undefined;
    };

    $scope.printWeightedIfWeighted = function() {
        return $scope.areMetricsWeighted() ? "Weighted" : "";
    };

    $scope.isTimeOrderingEnabled = function() {
        return $scope.modelData && $scope.modelData.coreParams && !!$scope.modelData.coreParams.time && $scope.modelData.coreParams.time.enabled;
    };

    $scope.getAggregationExplanation = function(metricName, displayName) {
        return PartitionedModelsService.getAggregationExplanation(metricName, displayName || metricName, $scope.uiState.currentMetricIsCustom);
    };

    $scope.trainedOnAllData = function() {
        return $scope.modelData.splitDesc && $scope.modelData.splitDesc.fullRows > 0 || $scope.modelData.trainInfo.fullRows > 0;
    };

    // Cost Matrix Gain
    $scope.updateCMG = function() {
        Assert.inScope($scope, 'modelData');
        if (!$scope.isBinaryClassification()) {
            return;
        }
        _updateCmgForGivenPerf($scope.modelData.perf);
        if ($scope.modelData.perfWithoutOverrides) {
            _updateCmgForGivenPerf($scope.modelData.perfWithoutOverrides);
        }
    }

    function _updateCmgForGivenPerf(perf) {
        if (!perf || !perf.perCutData) {
            return;
        }

        // Compute and set CMG
        perf.perCutData.cmg = ModelDataUtils.computeCmgForGivenPerf($scope.modelData, perf);

        // Update currentCutData if applicable
        if ($scope.currentCutData) {
            $scope.currentCutData.cmg = perf.perCutData.cmg[$scope.currentCutData.index];
        }

        // Update format if applicable
        if (perf.perCutData.format) {
            const e = d3.extent(perf.perCutData.cmg);
            perf.perCutData.format[2] = e[1] - e[0] > 10 ? '1g' : '.02f';
        }
    }

    function prepareFormat(modelData) {
        try {

            const pcd = modelData.perf ? modelData.perf.perCutData : null;
            const pdd = modelData.perf ? modelData.perf.densityData : null;
            const predInfo = modelData.predictionInfo ? modelData.predictionInfo : null;
            const tr = modelData.preprocessing ? modelData.preprocessing.target_remapping : null;
            if (pcd) { // set default format for x, both y axes, and CMG in tooltip
                pcd.format = ['.02f', '.02f', '.02f', '.02f'];
            }
            modelData.classes = tr && tr.length ? tr.map(
                (c) => tr[c.mappedValue].sourceValue
            ) : (pdd ? Object.keys(pdd) : null);
            if (pdd) { // Probability densities: make X coordinates
                pdd.x = pdd[modelData.classes[0]].actualIsNotThisClass.map(function(_, i, a) { return i / a.length; });
            }
            if (predInfo && predInfo.probabilityDensities) {
                predInfo.x = predInfo.probabilityDensities[modelData.classes[0]].density.map(function(_, i, a) { return i / a.length; });
            }
        } catch (ignored) { /* Nothing for now */ }
    }

    function prepareProxyModelSummary(modelData) {
        if (!SavedModelsService.isProxyModel(modelData)) {
            return;
        }
        const protocol = modelData.proxyModelConfiguration.protocol;
        if (!protocol) {
            return;
        }
        const details = AVAILABLE_EXTERNAL_MODEL_TYPES_DETAILS.find(t => t.name === protocol);
        if (modelData.inputFormat) {
            if (details.inputFormats) {
                const inputFormatObj = details.inputFormats.find(i => i.name === modelData.inputFormat);
                if (inputFormatObj) {
                    modelData.inputFormatDisplayName = inputFormatObj.displayName
                }
            }
        }
        if (modelData.outputFormat) {
            if (details.outputFormats) {
                const outputFormatObj = details.outputFormats.find(i => i.name === modelData.outputFormat);
                if (outputFormatObj) {
                    modelData.outputFormatDisplayName = outputFormatObj.displayName
                }
            }
        }
    }

    $scope.currentGraphData = {};

    function isPrediction() {
        return ($scope.modelData  && $scope.modelData.coreParams && this.indexOf($scope.modelData.coreParams.prediction_type) !== -1)
            || ($scope.evaluationDetails && $scope.evaluationDetails.evaluation && this.indexOf($scope.evaluationDetails.evaluation.predictionType) !== -1);
    }

    $scope.isBinaryClassification = isPrediction.bind(['BINARY_CLASSIFICATION']);
    $scope.isMulticlass           = isPrediction.bind(['MULTICLASS', 'DEEP_HUB_IMAGE_CLASSIFICATION']);
    $scope.isForecast             = isPrediction.bind(['TIMESERIES_FORECAST']);
    $scope.isClassification  = () => $scope.isMulticlass() || $scope.isBinaryClassification();
    $scope.isRegression           = isPrediction.bind(['REGRESSION']);
    $scope.isPrediction           = Fn.cst(true);
    $scope.isKerasDl =         () => $scope.modelData && $scope.modelData.coreParams && $scope.modelData.coreParams.backendType === "KERAS";
    $scope.isCausalPrediction     = isPrediction.bind(['CAUSAL_BINARY_CLASSIFICATION', 'CAUSAL_REGRESSION']);
    $scope.isCausalClassification = isPrediction.bind(['CAUSAL_BINARY_CLASSIFICATION']);

    // Only relevant for causal predictions
    $scope.isMultiValueTreatment = function() {
        return ($scope.modelData  && $scope.modelData.coreParams && $scope.modelData.coreParams.enable_multi_treatment && $scope.modelData.coreParams.treatment_values.length > 2);
    };

    $scope.hasMultipleFoldsOrIdentifiers = function () {
        return $scope.hasMultipleFolds() || $scope.hasMultipleIdentifiers();
    };

    $scope.hasMultipleIdentifiers = function () {
        return $scope.isForecast() && $scope.modelData.coreParams.timeseriesIdentifiers.length;
    };

    $scope.hasMultipleFolds = function () {
        return $scope.isForecast() && ((!$scope.modelData.coreParams.customTrainTestSplit && $scope.modelData.splitDesc.params.kfold) ||
            ($scope.modelData.coreParams.customTrainTestSplit && $scope.modelData.coreParams.customTrainTestIntervals.length > 1));
    }

    $scope.isMLBackendType = function(mlBackendType) {
        return $scope.modelData.coreParams.backendType === mlBackendType;
    };

    $scope.supportsOverrides = function() {
        return !$scope.isExternalMLflowModel() &&
            isPrediction.bind(['REGRESSION', 'BINARY_CLASSIFICATION', 'MULTICLASS'])() &&
            $scope.isMLBackendType("PY_MEMORY") &&
            !PartitionedModelsService.isPartitionedModel($scope.modelData);
    }

    $scope.supportsModelView = function(skinId) {
        const ensembleUnsupportedIds = ['error-analysis'];
        return !$scope.isPartitionedModel() && !(ensembleUnsupportedIds.includes(skinId) && ModelDataUtils.isEnsemble($scope.modelData));
    };

    $scope.staticModelSkins = [
        {
            id: 'stress-test-center-view',
            ownerPluginId: 'model-stress-test',
            desc: {
                roles: getStaticModelSkinRoles(),
                meta: {
                    label: 'Stress test center',
                    emptyState: {
                        animationPath: '/static/dataiku/images/analysis/stress-test-center-empty-state.json',
                        text: 'This plugin lets you randomly apply distribution shifts and feature corruptions, and check their impact on model performance.'
                    },
                    reportSection: 'PERFORMANCE'
                }
            },
        },
        {
            id: 'error-analysis',
            ownerPluginId: 'model-error-analysis',
            desc: {
                roles: getStaticModelSkinRoles(),
                meta: {
                    label: 'Model error analysis',
                    emptyState: {
                        animationPath: '/static/dataiku/images/analysis/model-error-analysis-empty-state.json',
                        text: 'This plugin helps debug model performance by identifying subpopulations on which it underperforms.'
                    },
                    reportSection: 'PERFORMANCE'
                }
            },
        },
        {
            id: 'model-fairness-view',
            ownerPluginId: 'model-fairness-report',
            desc: {
                roles: getStaticModelSkinRoles(['BINARY_CLASSIFICATION']),
                meta: {
                    label: 'Model fairness report',
                    emptyState: {
                        animationPath: '/static/dataiku/images/analysis/model-fairness-report-empty-state.json',
                        text: 'This plugin helps you measure your model fairness with regards to a specific group and a given outcome.'
                    },
                    reportSection: 'EXPLAINIBILITY'
                }
            }
        }
    ];

    function getStaticModelSkinRoles(predictionTypes = ['BINARY_CLASSIFICATION', 'MULTICLASS',  'REGRESSION']) {
        const backendTypes = ['PY_MEMORY'];
        const contentType = 'prediction';

        return [{
            backendTypes,
            contentType,
            predictionTypes,
            pathParamsKey: 'versionId',
            type: 'SAVED_MODEL',
        }, {
            backendTypes,
            contentType,
            predictionTypes,
            type: 'ANALYSIS'
        }];
    }

    // Ensure that this controller is correctly initialized for other sub-controllers
    $scope.predictionModelReportControllerInitDone = false;
    if ($scope.modelData) {
        $scope.fullModelId = $scope.fullModelId || $scope.modelData.fullModelId;
        prepareFormat($scope.modelData);
        prepareProxyModelSummary($scope.modelData);
        $scope.headTaskCMW = $scope.modelData.headTaskCMW;
        $scope.updateCMG();

        if ($scope.modelData.userMeta.activeClassifierThreshold) {
            updateGraphData($scope.modelData.userMeta.activeClassifierThreshold);
        }

        if ($scope.mlTasksContext) {
            $scope.mlTasksContext.model = $scope.modelData;
        }
        if ($scope.smContext) {
            $scope.smContext.model = $scope.modelData;
        }

        $scope.predictionModelReportControllerInitDone = true;
    } else {

        const fullModelId = $stateParams.fullModelId || $scope.fullModelId;
        const treatment = $stateParams.treatment || null;
        const p = DataikuAPI.ml.prediction.getModelDetails(fullModelId, treatment).success(function(modelData) {
            prepareFormat(modelData);
            prepareProxyModelSummary(modelData);
            $scope.modelData = modelData;
            $scope.headTaskCMW = modelData.headTaskCMW;
            $scope.updateCMG();
            CachedAPICalls.pmlDiagnosticsDefinition.then(pmlDiagnosticsDefinition => {
                $scope.diagnosticsDefinition = pmlDiagnosticsDefinition($scope.modelData.coreParams.backendType, $scope.modelData.coreParams.prediction_type);
            });
            // Normally, modelData.modelEvaluation is always empty here, but it seems reasonable to check anyway
            // to make sure that it is not possible to display model evaluation diagnostics instead of the MES-evaluation diagnostics.
            if ($scope.isExternalMLflowModel() && !modelData.modelEvaluation) {
                $scope.evaluationDiagnostics = MLDiagnosticsService.groupByType(modelData.mlDiagnostics);
                $scope.evaluationDiagnosticsCount = MLDiagnosticsService.countDiagnostics(modelData);
            }

            if (modelData.userMeta) {
                TopNav.setPageTitle(modelData.userMeta.name + " - Analysis");
                updateGraphData($scope.modelData.userMeta.activeClassifierThreshold);
            }

            if ($scope.mlTasksContext) {
                $scope.mlTasksContext.model = modelData;
            }
            if ($scope.smContext) {
                $scope.smContext.model = modelData;
            }

            if ($scope.onLoadSuccess) $scope.onLoadSuccess(); // used by saved-model-report-insight-tile

            $scope.predictionModelReportControllerInitDone = true;
        }).error(setErrorInScope.bind($scope))
          .error(function(data, status, headers, config, statusText) {
            $scope.predictionModelReportControllerInitDone = true;
            if ($scope.onLoadError) $scope.onLoadError(data, status, headers, config, statusText);
        });

        if ($scope.noSpinner) p.noSpinner(); // used by saved-model-report-insight-tile
    }

    $scope.colors  = window.dkuColorPalettes.discrete[0].colors // adjacent colors are too similar
        .filter(function(c, i) { return i % 2 === 0; });        // take only even-ranked ones

    $scope.hasVariableImportance = function() {
        if (!$scope.modelData) return false;
        const iperf = $scope.modelData.iperf;
        return !!(iperf && iperf.rawImportance && iperf.rawImportance.variables && iperf.rawImportance.variables.length);
    };

    $scope.hasGlobalExplanations = function() {
        if ($scope.isForecast() || $scope.isKerasDl() || $scope.isCausalPrediction()) return false;
        return !!($scope.modelData && $scope.modelData.globalExplanationsAbsoluteImportance);
    };

    $scope.supportsShapleyFeatureImportance = isPrediction.bind(['BINARY_CLASSIFICATION', 'REGRESSION', 'MULTICLASS']);

    $scope.hasRawCoefficients = function() {
        if (!$scope.modelData) return false;
        const iperf = $scope.modelData.iperf;
        return !!(iperf && iperf.lmCoefficients);
    };

    $scope.hasProbabilityDensities = function() {
        return $scope.isClassification() &&
            $scope.modelData &&
            $scope.modelData.perf &&
            $scope.modelData.perf.densityData &&
            $scope.modelData.perf.densityData.x
    };

    $scope.hasNoAssociatedModel = function() {
        return !!$scope.evaluationDetails
            && ($scope.evaluationDetails.evaluation.modelType=='EXTERNAL'
                || $scope.evaluationDetails.backingModelVersionDeleted
                || $scope.fullModelId);
    }

    const ufiMinBackGroundSize = 25;

    $scope.canComputeUfi = function(){
        try {
            if ($scope.trainedOnAllData()) {
                return ($scope.modelData.splitDesc.fullRows || $scope.modelData.trainInfo.fullRows) >= ufiMinBackGroundSize;
            } else {
                return ($scope.modelData.splitDesc.trainRows || $scope.modelData.trainInfo.trainRows) >= ufiMinBackGroundSize;
            }
        } catch (e) {
            // If we don't know the number of rows in the training set, tentatively say UFI is computable
            return true;
        }
    }

    $scope.ufiNotAvailableMessage = function (method) {
        if ("ICE" !== method ) {
            method = "Shapley"
        }
        if ($scope.isProxyModel()) {
            return method + " feature importance was skipped when creating the external model version.";
        } else if ($scope.isMLflowModel()) {
            return method + " feature importance was skipped when importing the MLflow model.";
        } else if ($scope.canComputeUfi()) {
            return method + " feature importance was skipped during training.";
        } else {
            return method + " feature importance isn't available because your model was trained on less than " + ufiMinBackGroundSize + " rows."
        }
    }

    $scope.hasNoAssociatedModelText = function() {
        return ModelEvaluationUtils.hasNoAssociatedModelText($scope.evaluationDetails, $scope.fullModelId);
    }

    // Causal Prediction
    $scope.hasPropensityModel = function() {
        return $scope.isCausalPrediction() && $scope.modelData && $scope.modelData.modeling &&
               $scope.modelData.modeling.propensityModeling && $scope.modelData.modeling.propensityModeling.enabled;
    };

    // Causal Prediction with propensity model
    $scope.isNonRandomizationDetected = function() {
        const test = $scope.modelData.perf.propensityPerf.binomialTreatmentTest;
        return test.pValue <= (1 - test.confidenceLevel);
    };

    $scope.hasFeatures = function() {
        const features = ($scope.modelData && $scope.modelData.preprocessing && $scope.modelData.preprocessing.per_feature) || {};
        return !(Object.keys(features).length === 0);
    }

    $scope.tabNotAvailableText = function(tabName) {
        const tabsMLflowExcluded = ["train", "coefficients", "grid_search", "variables", "algorithm", "overrides_metrics"];
        const tabsMLflowPerfUnneeded = ["features"];
        const tabsModelNeeded = ["train", "pdp_plot", "individual_explanations", "interactive_scoring", "coefficients", "grid_search", "variables", "features", "algorithm", "autoarima_orders", "overrides_metrics", "timeseries_model_coefficients", "timeseries_information_criteria", "timeseries_interactive_scoring"];
        const tabsProbaNeeded = ["pdp_plot", "individual_explanations", "feature_importance", "interactive_scoring", "bc_decision_chart", "bc_lift", "c_calibration"];
        const tabsPerfUnneeded = ["input_data_drift", "prediction_drift", "ensemble_summary", "tree_summary", "variables", "coefficients", "pdp_plot", "interactive_scoring", "features", "algorithm", "autoarima_orders", "timeseries_model_coefficients", "timeseries_information_criteria", "timeseries_interactive_scoring", "train", "grid_search", "individual_explanations"];
        const tabsPredictionsNeeded = ["individual_explanations", "feature_importance"];
        const tabsProbaDensityNeeded = ["individual_explanations", "feature_importance"];
        const tabsPredictionDistributionNeeded = ["prediction_drift"];
        const tabsPredictionPDFNeeded = ["prediction_drift"];
        const tabsCompatibleReferenceNeeded = ["prediction_drift", "performance_drift"]
        const tabsKerasExcluded = ["individual_explanations", "feature_importance", "overrides_metrics"];
        const tabsModelNeededWithCustomPreprocessing = ["subpopulation"];
        const tabNoEnsemble = ["overrides_metrics", "error-analysis"];
        const tabsNoModelSer = ["summary", "input_data_drift"];
        if ($scope.evaluationDetails && !$scope.evaluationDetails.evaluation.predictionType) {
            return "Not available for an evaluation of a non classification or regression model";
        }
        if (tabsKerasExcluded.includes(tabName) && $scope.modelData && $scope.modelData.backendType === "KERAS") {
            return "Not available for Deep learning models";
        }
        if ((tabsModelNeeded.includes(tabName) || (hasCustomPreprocessing($scope.modelData) && tabsModelNeededWithCustomPreprocessing.includes(tabName))) && $scope.hasNoAssociatedModel()) {
            return $scope.hasNoAssociatedModelText();
        }

        if ($scope.isExternalMLflowModel()) {
            if (tabsMLflowExcluded.includes(tabName)) {
                return "Not available for MLflow models";
            }
            if (!tabsMLflowPerfUnneeded.includes(tabName) && $scope.modelData && !$scope.modelData.modelEvaluation && !$scope.modelData.perf) {
                return "Model was not evaluated.";
            }
            if (tabName == "features" && !$scope.hasFeatures()) {
                return "No features in this model's metadata";
            }
        }
        if (tabNoEnsemble.includes(tabName) && $scope.getAlgorithm() && $scope.getAlgorithm().endsWith('_ENSEMBLE'))
        {
            return "Not available for Ensemble models";
        }
        if ($scope.modelData && $scope.modelData.modelEvaluation && $scope.modelData.modelEvaluation && !$scope.modelData.modelEvaluation.hasModel && !tabsNoModelSer.includes(tabName)){
            return "Not available for evaluations without model."
        }
        // /!\ if no perf, tab is deactivated except for perfLessTabs
        if ($scope.modelData &&
                // Careful: for a model evaluation, perf can have some things in it even if dontComputePerformance is true: custom metrics!
                // Custom evaluation recipe metrics are always stored in perf but can actually be drift metrics.
                (!$scope.modelData.perf || $scope.evaluationDetails?.evaluation?.evaluateRecipeParams?.dontComputePerformance) &&
                !tabsPerfUnneeded.includes(tabName)) {
            return "Model performance was not computed (disabled or ground truth was missing)";
        }

        if ($scope.uiState.driftState && $scope.uiState.driftState.selectedReference && !$scope.uiState.driftState.selectedReference.isCompatibleReference && tabsCompatibleReferenceNeeded.includes(tabName)) {
            return "The selected drift reference is incompatible with prediction and performance drift";
        }

        const modelDataContext = ModelDataUtils.isFromModelEvaluation($scope.modelData) ? "Model evaluation" : "Model";
        // /!\ if there are no prediction statistics for regression with predictions and the tab requires it, it is deactivated
        if ($scope.isRegression() && $scope.modelData && $scope.modelData.predictionInfo && !$scope.modelData.predictionInfo.predictions && tabsPredictionsNeeded.includes(tabName)) {
            return modelDataContext + " does not have prediction statistics with regression predictions";
        }
        // /!\ if there are no prediction statistics for regression with predictions and the tab requires it, it is deactivated
        if ($scope.isRegression() && $scope.modelData && $scope.modelData.predictionInfo && !$scope.modelData.predictionInfo.pdf && tabsPredictionPDFNeeded.includes(tabName)) {
            return modelDataContext + " does not have prediction statistics with probability density function";
        }
        // /!\ if there are no probability densities and the tab requires it, it is deactivated
        if ($scope.isClassification() && $scope.modelData &&
         $scope.modelData.predictionInfo &&
          (!$scope.modelData.predictionInfo.probabilityDensities || Object.keys($scope.modelData.predictionInfo.probabilityDensities).length == 0) &&
           tabsProbaDensityNeeded.includes(tabName)) {
            return modelDataContext + " does not have prediction statistics with probability densities";
        }
        // /!\ if there are no prediction distribution and the tab requires it, it is deactivated
        if ($scope.isClassification() && $scope.modelData &&
        $scope.modelData.predictionInfo &&
         (!$scope.modelData.predictionInfo.predictedClassCount || Object.keys($scope.modelData.predictionInfo.predictedClassCount).length == 0) &&
         tabsPredictionDistributionNeeded.includes(tabName)) {
           return modelDataContext + " does not have prediction statistics with prediction distribution";
       }
        if ($scope.isClassification() && tabsProbaNeeded.includes(tabName)) {
            if (!ModelDataUtils.hasProbas($scope.modelData)) {
                return "Not available for non-probabilistic models";
            }
            if($scope.modelEvaluation && !$scope.modelEvaluation.outputProbabilities) {
                return "Not available for non-probabilistic evaluations";
            }
        }
        if (tabName == "subpopulation" && $scope.isMulticlass()) {
            return "Not available for multi-class classification";
        }
        return null;
    }

    function hasCustomPreprocessing(modelData){
        if (!modelData || !modelData.preprocessing || !modelData.preprocessing.per_feature){
            return;
        }
        return Object.values(modelData.preprocessing.per_feature).some(column =>
            (column.numerical_handling && column.numerical_handling === "CUSTOM") ||
            (column.category_handling && column.category_handling === "CUSTOM") ||
            (column.text_handling && column.text_handling === "CUSTOM")
        );
    }

    $scope.hasROCCurve = function() {
        return ($scope.isBinaryClassification() && $scope.modelData && $scope.modelData.perf && $scope.modelData.perf.rocVizData) ||
            ($scope.isMulticlass() && Object.keys($scope.modelData && $scope.modelData.perf && $scope.modelData.perf.oneVsAllRocCurves || {}).length);
    };

    $scope.hasPRCurve = function() {
        return ($scope.isBinaryClassification() && $scope.modelData && $scope.modelData.perf && $scope.modelData.perf.prVizData) ||
            ($scope.isMulticlass() && Object.keys($scope.modelData && $scope.modelData.perf && $scope.modelData.perf.oneVsAllPrCurves || {}).length);
    };

    $scope.isExternalMLflowModel = function() {
        return SavedModelsService.isExternalMLflowModel($scope.modelData);
    }

    $scope.isLLMEvaluation = function() {
        return false;
    }

    $scope.isAgentEvaluation = function() {
        return false;
    }

    $scope.isMLflowModel = function() {
        return SavedModelsService.isMLflowModel($scope.modelData);
    }

    $scope.isProxyModel = function() {
        return SavedModelsService.isProxyModel($scope.modelData);
    }

    $scope.isModelWithSingleTree = function() {
        return ['MLLIB_DECISION_TREE', 'DECISION_TREE_CLASSIFICATION', 'DECISION_TREE_REGRESSION'].indexOf($scope.getAlgorithm()) >=0;
    };

    $scope.isModelWithMultipleTrees = function(){
         var algo = $scope.getAlgorithm();
         return ['GBT_REGRESSION', 'GBT_CLASSIFICATION', 'RANDOM_FOREST_CLASSIFICATION', 'RANDOM_FOREST_REGRESSION',
            'MLLIB_GBT', 'MLLIB_RANDOM_FOREST'].indexOf(algo) >=0 && ! (
                algo == 'MLLIB_RANDOM_FOREST' && $scope.isMulticlass() //because we can't create summary correctly for this case
            );
    };

    $scope.isAutoArimaModel = function() {
        return $scope.getAlgorithm() === 'AUTO_ARIMA';
    };

    $scope.isTimeseriesModelWithInformationCriteria = function () {
        return ['AUTO_ARIMA', 'ETS', 'SEASONAL_LOESS'].includes($scope.getAlgorithm());
    }

    $scope.isTimeseriesModelWithCoefficients = function() {
        return $scope.modelData?.iperf?.modelCoefficients;
    };

    $scope.optimalThresholdDefined = function() {
        return ($scope.isExternalMLflowModel() && typeof $scope.modelData.perf.optimalThreshold !== "undefined")
            || (!$scope.evaluationDetails && !$scope.isExternalMLflowModel())
            || ($scope.evaluationDetails && $scope.evaluationDetails.evaluation.thresholdAutoOptimized);
    }

    $scope.getDefaultThreshold = function() {
        // model evaluation when getting the data we assign:
        // $scope.modelData = data.evaluationDetails.details;
        // so we have the perf always available like this
        return $scope.optimalThresholdDefined() ?
            $scope.modelData.perf.optimalThreshold:
                ($scope.isExternalMLflowModel()?
                    $scope.modelData.modeling.forcedClassifierThreshold:
                    $scope.evaluationDetails.evaluation.activeClassifierThreshold);
    }

    $scope.getThresholdBackToWhat = function() {
        return $scope.optimalThresholdDefined() ?'optimal':'saved value';
    }

    $scope.hasThresholdSlider = function() {
        return $scope.isBinaryClassification()
            && $scope.modelData.backendType!=='VERTICA'
            && ModelDataUtils.hasProbas($scope.modelData)
            && !(
                $scope.isPartitionedModel()  &&
                // This line checks if we are on a partition tab or on the base one without depending on the scope hell
                !FullModelLikeIdUtils.isPartition(FullModelLikeIdUtils.getFmi($scope))
            );
    }

    $scope.uiState.displayMode = "records";

    $scope.getMaybeWeighted = function(x) {
        if (typeof x !== 'number') {
            return x; // for when it's percentage
        }
        return $scope.areMetricsWeighted() ? x.toFixed(2) : x.toFixed(0);
    };

    $scope.currentGraphData = {};
    function updateGraphData(nv) {
        if (nv === undefined) return;
        $scope.currentCutData = BinaryClassificationModelsService.findCutData($scope.modelData.perf, nv);
        angular.forEach($scope.currentCutData, // capitalized => graphed
            function(v, k) { if (k.charCodeAt(0) <= 90) {
              if ($scope.areMetricsWeighted()) this["Weighted " + k] = v;
              else this[k] = v;
          }},
          $scope.currentGraphData);
    }
    $scope.$watch("modelData.userMeta.activeClassifierThreshold", updateGraphData);

    $scope.isLearningCurveSupported = function() {
        const fmi = FullModelLikeIdUtils.getFmi($scope);
        const evaluationId = $stateParams.evaluationId;
        return isPrediction.bind(['BINARY_CLASSIFICATION', 'MULTICLASS', 'REGRESSION'])()
            && $scope.isMLBackendType("PY_MEMORY")
            && !evaluationId
            && (
                FullModelLikeIdUtils.isAnalysis(fmi)
                || FullModelLikeIdUtils.isSavedModel(fmi)
            )
            && (
                !PartitionedModelsService.isPartitionedModel($scope.modelData)
                || (PartitionedModelsService.isPartitionedModel($scope.modelData) && FullModelLikeIdUtils.isPartition(fmi))
            )
            && SavedModelsService.isVisualMLModel($scope.modelData) // otherwise we could have mlflow models
            && !ModelDataUtils.isEnsemble($scope.modelData);
    }

    // Handling of save
    if (!$scope.readOnly) {
        const simpleApiResult = function(msg, r) {
                r.success(function(){ ActivityIndicator.success(msg) })
                 .error(setErrorInScope.bind($scope));
        };
        const debouncedUpdateCMW = Debounce().withDelay(400, 1000).wrap(function() {
            if (!$scope.savedModel && !$scope.evaluationDetails) {
                const fmi = FullModelLikeIdUtils.getFmi($scope);
                simpleApiResult("Weights saved", DataikuAPI.analysis.pml.saveCostMatrixWeights(fmi, $scope.headTaskCMW));
            }
            $scope.updateCMG();
        });
        const saveMeta = function() {
            if ($scope.readOnly) return;
            if ($scope.evaluationDetails) {
                const mesId = $stateParams.mesId || $scope.mesId;
                const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
                let fme = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);
                simpleApiResult("Saved", DataikuAPI.modelevaluations.saveEvaluationModelUserMeta(
                        fme, $scope.modelData.userMeta));
            } else {
                simpleApiResult("Saved", DataikuAPI.ml.saveModelUserMeta(
                        $stateParams.fullModelId || $scope.fullModelId, $scope.modelData.userMeta));
            }
        };
        const debouncedSaveMeta = Debounce().withDelay(400, 1000).wrap(saveMeta);
        const saveIfDataChanged = function(nv, ov) { if (nv && ov && !$scope.evaluationDetails && !_.isEqual(nv,ov)) { this.call();} }; // only applies to saved models
        $scope.$watch("headTaskCMW", saveIfDataChanged.bind(debouncedUpdateCMW), true);
        $scope.$watch("modelData.userMeta.activeClassifierThreshold", saveIfDataChanged.bind(debouncedSaveMeta), true);
    
        $scope.$watch("modelData.userMeta", function(nv, ov){
            if (nv && ov && (nv.name != ov.name || nv.description != ov.description || !_.isEqual(nv.labels, ov.labels))) {
                debouncedSaveMeta();
            }
        }, true)

        const saveEvaluationUserMeta = function() {
            const mesId = $stateParams.mesId || $scope.mesId;
            const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
            let fme = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId);
            simpleApiResult("Saved", DataikuAPI.modelevaluations.saveEvaluationUserMeta(
                fme, $scope.evaluationDetails.evaluation.userMeta));
        }
        const debouncedSaveEvaluationUserMeta = Debounce().withDelay(400, 1000).wrap(saveEvaluationUserMeta);
        $scope.$watch("evaluationDetails.evaluation.userMeta", function(nv, ov){
            if (nv && ov && nv != ov) {
                debouncedSaveEvaluationUserMeta();
            }
        }, true)
    }
});

app.controller("UpliftCurvesController", function($scope, $stateParams, $state) {
    const isDashboardTile = !!$stateParams.dashboardId;

    $scope.perf = $scope.modelData.perf.causalPerf;

    function getCurveDisplayMode() {
        const advancedOptions = $scope.tile && $scope.tile.tileParams && $scope.tile.tileParams.advancedOptions || {};
        return advancedOptions.upliftCurve && advancedOptions.upliftCurve.displayMode;
    }

    if (isDashboardTile) {
        $scope.uiState.displayedUpliftCurve = getCurveDisplayMode();
    } else {
        $scope.uiState.displayedUpliftCurve = $stateParams.curveId || 'cumulativeUplift';
    }

    $scope.getUpliftCurveDisplayMode = function() {
        // If in dashboard tile
        if (isDashboardTile) return getCurveDisplayMode();

        // If in model report: Lab, saved model, insight
        return $scope.uiState.displayedUpliftCurve;
    };

    $scope.selectUpliftCurve = function(curveId) {
        $scope.uiState.displayedUpliftCurve = curveId;
        $state.go($state.current, { curveId });
    };
});

app.component("computedOnRows", {
    bindings: {
        trainInfo: '<',
        evaluation: '<',
        isOnPartitionedBaseModel: '<'
    },
    templateUrl: "/templates/ml/prediction-model/computed-on-rows.html"
});

app.component("thresholdSlider", {
    bindings: {
        originalThreshold: '<',
        backToWhat: '<',
        activeClassifierThreshold: '='
    },
    templateUrl: '/templates/ml/prediction-model/threshold-slider.html',
    controller: function thresholdSliderController () {
        const $ctrl = this;

        $ctrl.resetToThresholdOriginal = function () {
            $ctrl.activeClassifierThreshold = $ctrl.originalThreshold;
        }
    }
});

app.service('PMLCVParamsUtils', function(PMLSettings, SamplingData) {
    const samplingMethods = arr2obj(SamplingData.streamSamplingMethods);
    const partitionSelectionMethods = arr2obj(SamplingData.partitionSelectionMethods);

    // Make a nice array for train & test policy

    function selection(prefix, ds, sel) {
        if (ds) {
            this.push([prefix + 'dataset',  ds]);
        }
        this.push([prefix + 'sampling method', samplingMethods[sel.samplingMethod]]);
        this.push([prefix + 'partitions', partitionSelectionMethods[sel.partitionSelectionMethod]]);
        if (sel.partitionSelectionMethod === 'SELECTED') {
            this[this.length-1].push(': ', sel.selectedPartitions.join(', '));
        }
        if (['HEAD_SEQUENTIAL', 'RANDOM_FIXED_NB', 'COLUMN_BASED', 'COLUMN_ORDERED'].indexOf(sel.samplingMethod) >= 0) {
            this.push([prefix + 'record limit', sel.maxRecords]);
            if (sel.samplingMethod === 'COLUMN_BASED') {
                this.push([prefix + 'column', sel.column]);
            } else if (sel.samplingMethod === 'COLUMN_ORDERED') {
                this.push([prefix + 'sorted by', sel.column]);
            }
        } else if (sel.samplingMethod === 'RANDOM_FIXED_RATIO') {
            this.push([prefix + 'sampling ratio', sel.targetRatio]);
        }
    }

    function makeCVParams(split, coreParams, gridSearchParams) {
        const predictionType = coreParams.prediction_type
        let ttPolicy = split.ttPolicy;
        const cvParams = [];
        if (ttPolicy === 'SPLIT_SINGLE_DATASET') {
            ttPolicy = split.ssdDatasetSmartName ? 'SPLIT_OTHER_DATASET' : 'SPLIT_MAIN_DATASET';
        } else if (ttPolicy === 'EXPLICIT_FILTERING_SINGLE_DATASET') {
            ttPolicy = split.efsdDatasetSmartName ? 'EXPLICIT_FILTERING_SINGLE_DATASET_OTHER' : 'EXPLICIT_FILTERING_SINGLE_DATASET_MAIN';
        }

        if (predictionType !== 'TIMESERIES_FORECAST') {
            cvParams.push(["policy", arr2obj(PMLSettings.task.trainTestPolicies)[ttPolicy]]);
        }

        switch (ttPolicy) {
            case 'SPLIT_MAIN_DATASET':
            case 'SPLIT_OTHER_DATASET':
                selection.call(cvParams, '', split.datasetSmartName, split.ssdSelection);
                if (predictionType !== 'TIMESERIES_FORECAST') {
                    cvParams.push(['split mode', arr2obj(PMLSettings.task.splitModes)[split.ssdSplitMode]]);
                }

                if (coreParams?.customTrainTestSplit) {
                    cvParams.push(['number of custom folds', coreParams.customTrainTestIntervals.length]);
                } else if (split.kfold) {
                    cvParams.push(['number of folds', split.nFolds]);
                    cvParams.push(['stratified', split.ssdStratified ? 'Yes' : 'No']);
                    cvParams.push(['grouped', split.ssdGrouped ? 'Yes' : 'No']);
                    if (split.ssdGrouped) {
                        cvParams.push(['group column', split.ssdGroupColumnName]);
                    }
                    if (predictionType === 'TIMESERIES_FORECAST' && gridSearchParams) {
                        cvParams.push(['fold offset', gridSearchParams.foldOffset ? 'Yes' : 'No']);
                        cvParams.push(['equal duration train set folds', gridSearchParams.equalDurationFolds ? 'Yes' : 'No']);
                    }
                } else if (predictionType !== 'TIMESERIES_FORECAST') {
                    cvParams.push(['train ratio', split.ssdTrainingRatio]);
                }

                if (predictionType !== 'TIMESERIES_FORECAST') {
                    cvParams.push(['random seed', split.ssdSeed]);
                }
                break;
            case 'EXPLICIT_FILTERING_SINGLE_DATASET':
                selection.call(cvParams, 'Train ', split.eftdTrain.datasetSmartName, split.eftdTrain.selection);
                selection.call(cvParams, 'Test  ', split.eftdTest .datasetSmartName, split.eftdTest .selection);
                break;
            case 'EXPLICIT_FILTERING_TWO_DATASETS':
                selection.call(cvParams, 'Train ', split.eftdTrain.datasetSmartName, split.eftdTrain.selection);
                selection.call(cvParams, 'Test  ', split.eftdTest .datasetSmartName, split.eftdTest .selection);
                break;
            case 'FIXED_ID_BASED': break;
        }
        return cvParams;
    }
    return { makeCVParams };
});

app.filter('getETSShortName', function() {
    /**
     * Function to turn the final parameters of an ETS model into the standard representation of ETS models:
     * @returns {string}
     */
    return function(ets_params) {
        let res = "";
        res += getETSLetter(ets_params.error);
        res += getETSLetter(ets_params.trend);
        res += getETSLetter(ets_params.damped_trend);
        res += getETSLetter(ets_params.seasonal);
        return res;
    }

    function getETSLetter(string) {
        // Convert parameters to the ETS model representation standard
        // To be kept in sync with ETSHyperparametersSpace's param_to_str method
        switch (string) {
            case "add":
                return "A";
            case "mul":
                return "M";
            case "none":
                return "N";
            case "true":
                return "d"; // damped_trend is enabled
            case "false":
                return ""; // damped_trend is disabled
        }
    }
})

app.controller('PMLReportTrainController', function($scope, MLDiagnosticsService, FullModelLikeIdUtils, PMLCVParamsUtils, ExportModelDatasetService,
                                                    GpuUsageService) {
    const split = $scope.modelData.splitDesc.params;
    $scope.mti = $scope.modelData.trainInfo;
    $scope.diagnostics = MLDiagnosticsService.groupByType($scope.modelData.mlDiagnostics);
    $scope.didUseGpu = false;

    if ($scope.modelData.coreParams.executionParams &&
        $scope.modelData.coreParams.executionParams.gpuConfig &&
        $scope.modelData.coreParams.executionParams.gpuConfig.params) {
        $scope.didUseGpu = $scope.modelData.coreParams.executionParams.gpuConfig.params.useGpu;
    }

    $scope.isMLBackendType = function(mlBackendType) {
        return $scope.modelData.coreParams.backendType === mlBackendType;
    };

    $scope.getFormattedDisabledCapabilities = function() {
        return $scope.modelData.coreParams.executionParams.gpuConfig.disabledCapabilities.map(item => GpuUsageService.CAPABILITIES[item].name);
    };

    $scope.canDisplayDiagnostics = function() {
        const modelData = $scope.modelData;
        if (FullModelLikeIdUtils.isPartition(modelData.fullModelId)) {
            return true;  // Always display diagnostics on a partition
        }

        // We cannot know from a Saved Model if we are on a partition base or not, so use smOrigin to check that
        if (!angular.isUndefined(modelData.smOrigin) && !angular.isUndefined(modelData.smOrigin.fullModelId)) {
            return !FullModelLikeIdUtils.isAnalysisPartitionBaseModel(modelData.smOrigin.fullModelId);  // Do not display Diagnostics on partition base
        }

        return !FullModelLikeIdUtils.isAnalysisPartitionBaseModel(modelData.fullModelId);
    };

    $scope.cvParams = PMLCVParamsUtils.makeCVParams(split, $scope.modelData.coreParams, $scope.modelData.modeling.grid_search_params);

    $scope.exportTrainTestSets = function() {
        ExportModelDatasetService.exportTrainTestSets($scope, $scope.modelData.fullModelId);
    };

    $scope.exportTrainTestSetsForbiddenReason = function() {
        return ExportModelDatasetService.exportTrainTestSetsForbiddenReason($scope.modelData);
    };

    $scope.isTimeseriesForecastWithBothKfold = function () {
        if (!$scope.isForecast()) return false;
        return split.kfold && $scope.modelData.modeling.grid_search_params.mode == 'TIME_SERIES_KFOLD';
    };
});

// For elements that are instantly loaded (raw text for example) we signal right away to Puppeteer that they are available for content extraction
app.directive('puppeteerHookElementContentLoaded', function() {
    return {
        scope: false,
        restrict: 'A',
        link: function($scope) {
            // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
            $scope.puppeteerHook_elementContentLoaded = true;
        }
    };
});

app.controller('DecisionChartController', function($scope, $stateParams, MLChartsCommon) {
    $scope.pcd = $scope.modelData.perf.perCutData;
    if (!$scope.modelData.trainInfo.kfold) {
        return;
    }

    // wrap lines with confidence area
    $scope.svgCallback = MLChartsCommon.makeSvgAreaCallback(scope =>
        scope.theData
            .map(series => Object.assign({}, series, { std: $scope.pcd[series.key.toLowerCase() + 'std'] }))
            .filter(_ => _.std) // only series with a stddev series
            .map(series => ({
                color: series.color,
                values: series.values
                    // filter NaN data to prevent erroneous path
                    .filter(d => typeof d.y === 'number' && !isNaN(d.y))
                    .map((d, i) => ({
                        x: d.x,
                        y0: Math.max(0, d.y - 2 * series.std[i]),
                        y1: Math.min(1, d.y + 2 * series.std[i])
                    }))
            }))
    );
});


app.controller('ClassificationDensityController', function($scope) {
    var pdd = $scope.modelData.perf.densityData;
    $scope.colorsRep = $scope.colors.slice(0, 2).concat($scope.colors.slice(0, 2))
    $scope.setDensityClass = function(nc) {
        var dd = pdd[nc];
        $scope.densityClass = nc;
        const y1 = dd.actualIsNotThisClass.length > 0 ? dd.actualIsNotThisClass : dd.actualIsThisClass.map(i => 0)
        const y2 = dd.actualIsThisClass.length > 0 ? dd.actualIsThisClass : dd.actualIsNotThisClass.map(i => 0)
        $scope.ys = [y1, y2];
        $scope.xm = [dd.actualIsNotThisClassMedian, dd.actualIsThisClassMedian];
        $scope.labels = $scope.isMulticlass() ? ['For all classes but ' + nc, 'For class ' + nc]
                : ['For class ' + $scope.modelData.classes[0], 'For class ' + $scope.modelData.classes[1]];
    }
    $scope.setDensityClass($scope.modelData.classes[$scope.isMulticlass() ? 0 : 1]);
});


app.factory("FeatureImportanceService", function($filter){
    return {
        build: function(rawImportance, colors) {
            let imp = rawImportance;
            let rgb = colors[0];
            const name = $filter('mlFeature');

            rgb = rgb.replace('#', '').replace(/^([0-9a-f])([0-9a-f])([0-9a-f])$/, '$1$1'); // 6-digit hex
            rgb = parseInt(rgb, 16);
            rgb = ['rgba(', rgb >> 16, ',', rgb >> 8 & 255, ', ', rgb & 255, ', '].join('');

            if ('variables' in imp) {
                imp = imp.variables.map(function(v, i) { return { rawVarName: v, varName: name(v), impVal: this[i] }; }, imp.importances);
            } else {
                imp = Object.keys(imp).map(function(v) { return { varName: name(v), impVal: imp[v] }; });
            }
            imp = $filter('orderBy')(imp, '-impVal');

            const filtered_imp = imp.slice(0, 20); // If not filtered already, filter to top 20 features
            const importances = filtered_imp.map(function(o) { return [o.varName, o.impVal]; });
            const fades       = filtered_imp.map(function(o) { return rgb + Math.min(o.impVal / .3 + .4, 1) + ')'; });
            return [importances, fades, imp];
        }
    }
})

app.controller('FeatureImportanceController', function($scope, DataikuAPI, $stateParams, $state, ExportUtils, FeatureImportanceService,
    FullModelLikeIdUtils, ActiveProjectKey, FutureProgressModal, WT1, $q, StateUtils, ColumnGeneratorService, LocalStorage) {

    $scope.isKFolding = $scope.modelData.trainInfo.kfold;
    $scope.variableImportanceOnlyForTreeBasedModels = !$scope.isCausalPrediction();

    if ($scope.supportsShapleyFeatureImportance() && !$scope.isDashboardTile) {
        $scope.uiState.importanceDisplayMode = $stateParams.displayMode ||  'globalExplanations';
    }

    getGlobalExplanations()
        .catch(setErrorInScope.bind($scope));

    getGlobalExplanationsFacts()
        .catch(setErrorInScope.bind($scope))

    $scope.isDashboardTile = !!$stateParams.dashboardId;
    const objectId = $scope.isDashboardTile ? `insightId.${$scope.insight.id}` : `fullModelId.${$scope.modelData.fullModelId}`;

    if ($scope.isMulticlass()) {
        // The classes that do not appear in test set should not be selectable to compute explanations
        $scope.selectableClasses = Object.keys($scope.modelData.predictionInfo.predictedClassCount);
        $scope.selectedClass = LocalStorage.get(`dss.ml.featureImportance.${objectId}.selectedClass`);
        if (!$scope.selectedClass) {
            $scope.selectedClass = $scope.selectableClasses[0];
        }
    } else {
        $scope.selectedClass = 'unique';
    }

    $scope.goToEvaluatedFeatureImportance = function() {
        if ($scope.insight) { // Dashboard tile
            StateUtils.go.savedModelVersion(
                'PREDICTION', $scope.insight.params.savedModelSmartId, $scope.insight.$fullModelId , $scope.insight.$savedModel.projectKey, null, { tab: '.tabular-feature_importance'}
            );
            return;
        }
        if (!$scope.mesContext.evaluationFullInfo.evaluation) return;
        const modelRef = $scope.mesContext.evaluationFullInfo.evaluation.modelRef;
        StateUtils.go.savedModelVersion('PREDICTION', modelRef.smId, modelRef.fullId, modelRef.projectKey, null, {tab: '.tabular-feature_importance'});
    }

    $scope.scrollToGraph = (selectedGraph) => {
        if (!selectedGraph) { return }
        const selectedGraphDOM = document.getElementById(selectedGraph);
        if (selectedGraphDOM) {
            selectedGraphDOM.scrollIntoView({ behavior: 'smooth' });
        }
    };

    $scope.selectDisplayMode = (displayMode) => {
        $scope.uiState.importanceDisplayMode = displayMode;
        $state.go($state.current, { displayMode });
    }

    function formatGlobalExplanations(explanationResults) {
        if (explanationResults && Object.keys(explanationResults).length) {
            $scope.explanationsAll = explanationResults.explanations;
            $scope.explanations = $scope.explanationsAll[$scope.isMulticlass() ? $scope.selectedClass : "unique"];
            $scope.observations = explanationResults.observations;
            const absoluteImportance = explanationResults.absoluteImportance;

            return DataikuAPI.ml.prediction.getColumnImportance($stateParams.fullModelId || $scope.fullModelId).then(function (columnImportanceResp) {

                /* We're going to compute the ratio of displayed column importance VS total importance.
                If the surrogate importance is available (= columnImportanceResp.data is defined) , it means we only have 20 columns worth of shap values, so we display the ratio of top 20 surrogate importance Vs total of surrogate importance.
                Otherwise, if surrogate importance isn't available, it means we have all the shap values and can compute the ratio based on that.
                 */
                let knownImportantColumns;
                if(columnImportanceResp.data ){
                    knownImportantColumns =columnImportanceResp.data.columns.map((column, index) => [column, columnImportanceResp.data.importances[index]]);
                } else {
                    knownImportantColumns = Object.entries(absoluteImportance);
                }
                $scope.roundedTo20Features = knownImportantColumns.length > 20;

                knownImportantColumns.sort((a, b) => b[1] - a[1]);
                //At this point, knownImportantColumns is always an array of arrays looking like [["columnA",0.4],["columnB":0.3],...] sorted by descending importance
                const knownColumnImportances = knownImportantColumns.map(arr => arr[1]);
                const top20ColumnImportances = knownColumnImportances.slice(0, 20).reduce((partialSum, a) => partialSum + a, 1e-20);
                const totalColumnImportance = knownColumnImportances.reduce((partialSum, a) => partialSum + a, 1e-20);
                const top20RatioOfImportance = top20ColumnImportances / totalColumnImportance;
                $scope.roundedPercentageOfTotalmportance = Math.round(top20RatioOfImportance * 1000) / 10;

                //Get the %age (among the top 20) of importance for the top 20 shap important columns
                const top20ShapImportantColumns = Object.entries(absoluteImportance).sort((a, b) => b[1] - a[1]).slice(0,20)
                const top20TotalShapImportance = top20ShapImportantColumns.reduce((partialSum, a) => partialSum + a[1], 0);
                const top20DisplayedImportances = Object.fromEntries(top20ShapImportantColumns.map(([col, val]) => [col, val / top20TotalShapImportance * top20RatioOfImportance]));
                // Get the top column names for the beeswarm plot
                $scope.topColumns = Object.entries(top20DisplayedImportances).map(v => v[0]);
                const arr = FeatureImportanceService.build(top20DisplayedImportances, $scope.colors);
                $scope.globalExplanationsImportance = arr[0];
                $scope.globalExplanationsFades = arr[1];
                $scope.globalExplanationsComputed = true;
                $scope.scatterColumns = ColumnGeneratorService.getPossibleColumns(
                    $scope.topColumns,
                    ['NUMERIC', 'CATEGORY', 'TEXT', 'VECTOR'],
                    ['INPUT'],
                    $scope.modelData.preprocessing.per_feature
                );
                $scope.scatterColumnNames = $scope.scatterColumns.map(v => v.name);
                // Initialize to the most important column that can be selected
                $scope.selectedColumn = LocalStorage.get(`dss.ml.featureImportance.${objectId}.selectedColumn`);
                if (!$scope.selectedColumn) {
                    $scope.selectedColumn = $scope.topColumns.find(x => $scope.scatterColumnNames.includes(x));
                }
                // Reverse order of topColumns for beeswarm plot
                $scope.topColumns.reverse();
            });
        }
        return $q.resolve();
    }

    $scope.compute = function() {
        DataikuAPI.ml.prediction.globalExplanationsComputationStart($stateParams.fullModelId || $scope.fullModelId).success((result) => {
            FutureProgressModal.show($scope, result, "Computing Shapley feature importance").then((explanationResults) => {
                getGlobalExplanationsFacts();
                formatGlobalExplanations(explanationResults);
            });
        }).error(setErrorInScope.bind($scope));
        WT1.event("doctor-compute-shapley-feature-importance", {
            algorithm: $scope.modelData.modeling.algorithm,
        });
    };

    function getGlobalExplanations() {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        if (mesId) {
            return DataikuAPI.modelevaluations.getGlobalExplanations(FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId))
                .then(({ data }) => {
                    return formatGlobalExplanations(data);
                });
        } else {
            return DataikuAPI.ml.prediction.getGlobalExplanations($stateParams.fullModelId || $scope.fullModelId).then(({ data }) => {
                return formatGlobalExplanations(data);
            });
        }
    }

    function getGlobalExplanationsFacts() {
        const mesId = $stateParams.mesId || $scope.mesId;
        const evaluationId = $stateParams.evaluationId || $scope.evaluationId;
        if (mesId) {
            return DataikuAPI.modelevaluations.getGlobalExplanationsFacts(FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), mesId, evaluationId))
                .then(({ data }) => {
                    $scope.perClassFacts = data.perClassFacts;
                });
        } else {
            return DataikuAPI.ml.prediction.getGlobalExplanationsFacts($stateParams.fullModelId || $scope.fullModelId).then(({ data }) => {
                $scope.perClassFacts = data.perClassFacts;
            });
        }
    }

    $scope.getVariableImportance = function(){
        const arr = FeatureImportanceService.build($scope.modelData.iperf.rawImportance, $scope.colors);
        $scope.variableImportance = arr[0];
        $scope.variableImportanceFades = arr[1];
        $scope.variableUnfilteredImportance = arr[2];
    }

    $scope.exportVariableImportance = function(){
        const data = $scope.variableUnfilteredImportance.map(function(x){
            return [x.rawVarName, x.varName, x.impVal];
        });
        ExportUtils.exportUIData($scope, {
            name : "Variable importance for model: " + $scope.modelData.userMeta.name,
            columns : [
                { name : "feature_name", type : "string"},
                { name : "feature_description", type : "string"},
                { name : "importance", type : "double"}
            ],
            data : data
        }, "Export variable importances");
    }

    $scope.exportGlobalExplanations = function(){
        ExportUtils.exportUIData($scope, {
            name : "Shapley feature importance for model: " + $scope.modelData.userMeta.name,
            columns : [
                { name : "feature_name", type : "string"},
                { name : "importance", type : "double"}
            ],
            data : $scope.globalExplanationsImportance,
        }, "Export Shapley feature importance");
    }

    function getRawExplanationsColumnsAndData(observations, explanationsAll) {
        const targetClasses = Object.keys(explanationsAll);
        const explanationFeatures = Object.keys(explanationsAll[targetClasses[0]]);
        const obsFeatures = Object.keys(observations);
        const nbObservations = observations[obsFeatures[0]].length;
        const observationsWithExplanations = [];
        for (let observationIdx = 0; observationIdx < nbObservations; observationIdx++) {
            const featureValues = obsFeatures.map(feature => observations[feature][observationIdx]);
            targetClasses.forEach(targetClass=> {
                const explanationValues = explanationFeatures.map(feature => explanationsAll[targetClass][feature][observationIdx]);
                const observationWithExplanations = featureValues.concat(explanationValues);
                if ($scope.isMulticlass()) {
                    observationWithExplanations.push(targetClass);
                }
                observationsWithExplanations.push(observationWithExplanations);
            });
        }
        const allColumns = obsFeatures.concat(explanationFeatures.map(feature => "shapley_" + feature));
        if ($scope.isMulticlass()) {
            allColumns.push("shapley__class");
        }
        return [
            allColumns.map(column => { return { name: column, type: "string" }}),
            observationsWithExplanations
        ];
    }
    $scope.exportRawExplanations = function() {
        const [columns, data] = getRawExplanationsColumnsAndData($scope.observations, $scope.explanationsAll);
        ExportUtils.exportUIData($scope, {
            name: "Raw shapley feature importance for model: " + $scope.modelData.userMeta.name,
            columns: columns,
            data: data,
        }, "Export raw explanations");
    }
    $scope.onChangeSelectedClass = function(selectedClass) {
        $scope.explanations = $scope.explanationsAll[$scope.isMulticlass() ? selectedClass : 'unique'];
        LocalStorage.set(`dss.ml.featureImportance.${objectId}.selectedClass`, selectedClass);
    }

    $scope.onChangeSelectedColumn = function(selectedColumn) {
        LocalStorage.set(`dss.ml.featureImportance.${objectId}.selectedColumn`, selectedColumn);
    }
});

app.component("globalExplanationsScatterPlot", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/scatter_plot.html',
    bindings: {
        explanations: '<',
        observations: '<',
        columns: '<',
        perFeature: '<',
        loadedStateField: '=?',
    },
    controller: function($scope, Debounce, $element, PuppeteerLoadedService) {
        const $ctrl = this;

        $ctrl.onChartInit = function($event) {
            // this allows us to query the chart instance later to get its options
            // https://stackoverflow.com/a/68715288
            $ctrl.echartsInstance = $event;

            const debouncedSetPuppeteerField = PuppeteerLoadedService.getDebouncedSetField($scope, $element, $ctrl.loadedStateField);

            // Signal to Puppeteer that the chart has been rendered and is thus available for content extraction
            $ctrl.echartsInstance.on('rendered', () => {
                debouncedSetPuppeteerField();
            });
        };

        $ctrl.show = function($event, show) {
            const chartOptions = $ctrl.echartsInstance.getOption();
            const seriesIndex = $event.seriesIndex;

            if (chartOptions.visualMap[seriesIndex].type === 'piecewise' && chartOptions.visualMap[seriesIndex].showLegendOnHover) {
                chartOptions.visualMap[seriesIndex].show = show;
                $ctrl.echartsInstance.setOption(chartOptions);
            }

            if (chartOptions.visualMap[seriesIndex].type === 'piecewise' && chartOptions.visualMap[seriesIndex].showGroupOnHover) {
                const dataIndexes = [];
                $ctrl.beeswarmDataCategorical[$event.data[2]].values.forEach((a,index) => {
                    if(a[3] === $event.data[3]) {
                        dataIndexes.push(index);
                    }
                })
                $ctrl.echartsInstance.dispatchAction({
                    type: show ? 'highlight' : 'downplay',
                    seriesIndex: seriesIndex,
                    dataIndex : dataIndexes
                });
            }
        }

        $ctrl.$onInit = function () {
            $ctrl.chartOptions = buildChartOptions();
        }

        $ctrl.$onChanges = function () {
            $ctrl.chartOptions = buildChartOptions();
        }
        let colMax;
        let colMin;

        function getCategoricalUniqueValues(colObservations) {
            const colObservationsSet = new Set(colObservations);
            const colObservationsLower = new Set([...colObservationsSet].map(x => x.toLowerCase()));
            const booleans = colObservationsLower === new Set(["true", "false"])
                || colObservationsLower === new Set(["true", "false", ""]);

            let sortedValueCounts;
            if (booleans) {
                // ensure colouring for True/false always consistent across features,
                sortedValueCounts = [...colObservationsSet].sort().reverse();
            } else {
                const valueCounts = {}
                colObservations.forEach((val) => {
                    if (val in valueCounts) {
                        valueCounts[val] += 1;
                    } else {
                        valueCounts[val] = 1;
                    }
                });

                sortedValueCounts = Object.entries(valueCounts)
                    .sort(function(a, b) {
                        if (a[0] === "") return 1
                        else if (b[0] === "") return -1
                        else return b[1] - a[1]
                    } )
                    .map((item) => item[0]);
            }
            return sortedValueCounts;
        }

        function buildChartOptions() {
            // Get uprounded highest absolute value for x-axis limits (If the max is 0.05, display till 0.1)
            const max = Math.ceil(Math.max(...(Object.values($ctrl.explanations).flat().map(Math.abs))) * 10) / 10;
            const beeswarmData = [];
            $ctrl.beeswarmDataCategorical = {};
            // Use "random" numbers between 0.2 and -0.2 to jitter the data points
            const jitterRandomNums = [0.1871, 0.0472, 0.0488, 0.1945, -0.141, 0.0865, 0.1165, -0.0983, -0.1128, 0.0171, 0.0554, -0.0276, 0.0561, -0.0573, -0.1523, -0.1426, 0.0672, -0.0154, 0.0789, 0.177, -0.0215, 0.0533, 0.1659, 0.181, -0.0754, 0.1079, 0.0295, -0.1167, -0.1225, -0.1451, 0.0422, -0.1081, 0.1312, 0.0852, -0.1916, 0.1141, -0.1029, -0.1247, -0.1448, 0.0517, -0.0597, -0.1888, -0.1868, 0.088, -0.1844, 0.0529, -0.1984, -0.087, -0.1167, -0.1905];

            for (let i = 0; i < $ctrl.columns.length; i++) {
                const col = $ctrl.columns[i];
                const colExplanations = $ctrl.explanations[col];
                const colObservations = $ctrl.observations[col];
                const colFeatType = $ctrl.perFeature[col].type
                // Check if all colObservations can be turned into numbers or are empty strings
                // For example ['', '2.0', '71.0', '30.0', '29.0', '42.0', '30.0', '', '',] will be True
                const treatAsNumeric = (colFeatType === 'NUMERIC') && colObservations.every((x) => {
                    return x === "" || !isNaN(x);
                });
                // Get max & min values for this column
                if (treatAsNumeric) {
                    // To be insensitive to outliers, use the 95th/5th percentile instead of the max/min
                    // Sort without mutating the original array
                    const colObservationsSorted = [...colObservations].sort((a, b) => a - b);
                    colMax = d3.quantile(colObservationsSorted, 0.95);
                    colMin = d3.quantile(colObservationsSorted, 0.05);
                } else {
                    $ctrl.beeswarmDataCategorical[col] = {
                        values: [],
                        uniqueValues: getCategoricalUniqueValues(colObservations)
                    }
                }
                for (let j = 0; j < colExplanations.length; j++) {
                    const randomNum = jitterRandomNums[j % jitterRandomNums.length];

                    // Initialize to 0.5 in case it is not a number or max = min
                    let featValue = 0.5;
                    if (treatAsNumeric && colObservations[j] != "" && colMax != colMin) {
                        // Cap value at the percentiles
                        featValue = Math.min(Math.max(colObservations[j], colMin), colMax);
                        // Force featValue into range from 0 - 1
                        featValue = (featValue - colMin) / (colMax - colMin);
                    }
                    if (!treatAsNumeric) {
                        // These values will be colored grey regardless of feature value
                        $ctrl.beeswarmDataCategorical[col].values.push([colExplanations[j], i + 1 + randomNum, col, colObservations[j]]);
                    } else {
                        beeswarmData.push([colExplanations[j], i + 1 + randomNum, featValue, colObservations[j]]);
                    }
                }
            }
            // Add an empty string at the beginning and end to columns for space around the axes
            const columnsAugmented = ["", ...$ctrl.columns, ""];

            const series = [{
                type: 'scatter',
                data: beeswarmData,
                symbolSize: 5,
                name: 'numerical'
            }];

            const visualMap = [
                // visualMap for numeric features in red & blue
                {
                    dimension: '2',
                    hoverLink: false, // Do not show the dot on the visualMap when hovering
                    inRange: {
                        color: ['#3A69DA', '#C6302A']
                    },
                    left: '0%',
                    max: 1,
                    min: 0,
                    orient: 'horizontal',
                    seriesIndex: 0,
                    text: ['High numerical value', 'Low numerical value'],
                    textStyle: {
                        color: '#000',
                        fontFamily: 'SourceSansPro'
                    },
                    top: '0%',
                    type: 'continuous'
                }
            ];

            Object.entries($ctrl.beeswarmDataCategorical).forEach(([_, value], index) => {
                const seriesIndex = index + 1;
                series.push({
                    type: 'scatter',
                    data: value.values,
                    symbolSize: 5,
                    dimensions: [
                        null,
                        null,
                        null,
                        { type: 'ordinal' }
                      ]
                  
                })

                if (value.uniqueValues.length > 6) {
                    const categories =  value.uniqueValues;

                    visualMap.push({
                        bottom: 10,
                        categories: categories,
                        dimension: '3',
                        hoverLink: false,
                        inRange: {
                            color: '#808080'
                        },
                        itemHeight: 20,
                        left: '82%',
                        seriesIndex: seriesIndex,
                        show: false,
                        showGroupOnHover: true,
                        showLabel: false,
                        showLegendOnHover: false,
                        top: '2%',
                        type: 'piecewise'
                    });
                } else {
                    const colors = ['#F8A217',
                        '#23A373',
                        '#82BCE6',
                        '#FBCD22',
                        '#9061B8',
                        '#8ABB4C',
                    ]
                    let inUseColours;

                    if (value.uniqueValues.includes("")) {
                        inUseColours = colors.slice(0, value.uniqueValues.length - 1);
                        inUseColours.push("#999999");
                    } else{
                        inUseColours = colors.slice(0, value.uniqueValues.length);
                    }
                    visualMap.push(
                        {
                            bottom: 10,
                            categories: value.uniqueValues,
                            dimension: '3',
                            formatter: function(value) {
                                const sanitized = sanitize(value);
                                if (sanitized === "") {
                                    return "Missing Values";
                                }
                                if (sanitized.length > 10) {
                                    return sanitized.substring(0, 7) + '...';
                                } else {
                                    return sanitized;
                                }
                            },
                            hoverLink: false,
                            inRange: {
                                color: inUseColours,
                            },
                            itemHeight: 20,
                            left: '0%',
                            orient: 'horizontal',
                            padding: 4,
                            selectedMode: false,
                            seriesIndex: seriesIndex,
                            show: false,
                            showGroupOnHover: true,
                            showLegendOnHover: true,
                            textGap: 4,
                            top: '6%',
                            type: 'piecewise'
                        }
                    );
                }
            });

            return {
                // Make dot appear with an electric blue shining border to make it obvious when hovering over it
                emphasis: {
                    itemStyle: {
                        borderColor: '#3B99FC',
                        borderWidth: 1
                    }
                },
                // Adopts a similar style as ml-chart-container--small
                grid: {
                    left: '150px',
                    right: 20,
                    top: 70
                },
                series: series,
                textStyle: {
                    color: '#000000',
                    fontFamily: 'SourceSansPro'
                },
                tooltip: {
                    borderColor: "#F2F2F2",
                    borderRadius: 0,
                    extraCssText: 'box-shadow: 0px 5px 6px rgba(0, 0, 0, 0.1);',
                    formatter: (param) => {
                        const shapleyValue = param.value[0].toFixed(3);
                        const columnIdx = param.value[1];
                        // Round columnIdx to nearest Int to remove jitter
                        const column = $ctrl.columns[Math.round(columnIdx - 1)];
                        const observation = param.value[3];
                        const observationString = observation.toString().length > 50 ? observation.toString().substring(0, 50) + "..." : observation.toString();
                        const dotColor = param.color;

                        let html = '<table class="global-explanations-scatter__tooltip-table">';
                        html += '<tr><td>' + sanitize(column) + ' <i class="icon-circle" style="color: ' + dotColor + '"></td><td><strong>' + sanitize(observationString) + '</strong></td></tr>'
                        html += '<tr><td>Shapley value</td><td><strong>' + shapleyValue + '</strong></td></tr>'
                        html += "</table>";

                        return html;
                    },
                    trigger: 'item'
                },
                visualMap: visualMap,
                xAxis: {
                    // Add space at bottom & top
                    axisLabel: {
                        margin: 30
                    },
                    axisLine: {
                        show: false
                    },
                    axisTick: {
                        show: true
                    },
                    max: max,
                    min: -max,
                    nameGap: 10,
                    nameLocation: 'middle',
                    splitLine: {
                        show: false
                    },
                    type: 'value'
                },
                yAxis: {
                    // If the tick name is too long shorten it with ...
                    axisLabel: {
                        formatter: function(value, index) {
                            const colName = columnsAugmented[value];
                            return colName.length > 20 ? colName.substring(0, 20) + "..." : colName;
                        }
                    },
                    axisLine: {
                        show: true
                    },
                    axisTick: {
                        alignWithLabel: true,
                        show: false
                    },
                    data: columnsAugmented,
                    max: columnsAugmented.length - 1,
                    min: 0,
                    splitLine: {
                        show: true
                    },
                    // Show all ticks
                    splitNumber: columnsAugmented.length - 1,
                    type: 'value'
                }
            };
        }
    }
});

app.component("globalExplanationsDependencePlot", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/scatter_plot.html',
    bindings: {
        explanations: '<',
        observations: '<',
        column: '<',
        type: '<',
        loadedStateField: '=?',
    },
    // Plots the feature values against the shapley explanations for a single feature
    controller: function ($scope, Debounce, $element, PuppeteerLoadedService) {
        const $ctrl = this;
        $ctrl.$onInit = function () {
            $ctrl.chartOptions = buildChartOptions();
        }
        $ctrl.$onChanges = function () {
            $ctrl.chartOptions = buildChartOptions();
        }

        $ctrl.onChartInit = function($event) {
            // this allows us to query the chart instance later to get its options
            // https://stackoverflow.com/a/68715288
            $ctrl.echartsInstance = $event;

            const debouncedSetPuppeteerField = PuppeteerLoadedService.getDebouncedSetField($scope, $element, $ctrl.loadedStateField);

            // Signal to Puppeteer that the chart has been rendered and is thus available for content extraction
            $ctrl.echartsInstance.on('rendered', () => {
                debouncedSetPuppeteerField();
            });
        };


        function buildChartOptions() {
            const colExplanations = $ctrl.explanations[$ctrl.column]; // y-axis
            const colObservations = $ctrl.observations[$ctrl.column]; // x-axis

            const xAxis = {
                type: 'value',
                nameLocation: 'middle',
                nameGap: 30,
                splitLine: {
                    show: true
                },
                axisTick: {
                    alignWithLabel: true,
                    show: true
                },
                name: $ctrl.column
            }

            const data = [];
            if ($ctrl.type === "TEXT" || $ctrl.type === "CATEGORY" || $ctrl.type === "VECTOR") {
                const counts = {};
                for (let i = 0; i < colObservations.length; i++) {
                    counts[colObservations[i]] = 1 + (counts[colObservations[i]] || 0);
                }
                const sortedCounts = Object.entries(counts).sort((a, b) => b[1] - a[1]);
                // If there are more than 5 categories, plot the 4 most common + Other else the top 5
                const topCategories = sortedCounts.length > 5 ? sortedCounts.slice(0, 4).map(x => x[0]) : sortedCounts.map(x => x[0]);
                const axisLabels = ["", ...sortedCounts.length > 5 ? topCategories.concat(["Other"]) : topCategories, ""];
                for (let i = 0; i < colObservations.length; i++) {
                    if (topCategories.includes(colObservations[i])) {
                        data.push([topCategories.indexOf(colObservations[i]) + 1, colExplanations[i], colObservations[i]]);
                    } else {
                        data.push([topCategories.length + 1, colExplanations[i], colObservations[i]]);
                    }
                }
                // Modify the xAxis to show categories
                xAxis.data = axisLabels;
                xAxis.min = 0;
                xAxis.max = axisLabels.length - 1;
                xAxis.splitNumber = axisLabels.length - 1;
                xAxis.axisLabel = {
                    formatter: function (value, index) {
                        const colName = axisLabels[value];
                        return colName.length > 20 ? colName.substring(0, 20) + "..." : colName;
                    }
                }
            } else {
                for (let i = 0; i < colExplanations.length; i++) {
                    data.push([colObservations[i], colExplanations[i], colObservations[i]]);
                }
            }

            return {
                tooltip: {
                    trigger: 'item',
                    formatter: (param) => {
                        const shapleyValue = param.value[1];
                        const observation = param.value[2];
                        // Shorten observation if it is a long string, e.g. a text column
                        const observationString = observation.toString().length > 50 ? observation.toString().substring(0, 50) + "..." : observation.toString();
                        return  `${sanitize($ctrl.column)}: ${sanitize(observationString)}<br>`
                            + `Shapley value: ${shapleyValue}<br>`
                    }
                },
                xAxis: xAxis,
                yAxis: {
                    type: 'value',
                    nameLocation: 'middle',
                    nameGap: 75,
                    boundaryGap: ['10%', '10%'],
                    splitLine: {
                        show: true
                    },
                    name: 'Shapley value'
                },
                series: [{
                    type: 'scatter',
                    data: data,
                    symbolSize: 5
                }],
                textStyle: {
                    color: '#000000',
                    fontFamily: 'SourceSansPro'
                },
                grid: {
                    left: '150px',
                    right: 20
                }
            };
        }
    }
});

app.directive("cateHistogram", function($filter) {
    return {
        scope: {
            histogramData: '<',
        },
        template: `<div class="h100">
            <div block-api-error />
            <ng2-lazy-echart [options]="chartOptions" ng-if="chartOptions"></ng2-lazy-echart>
        </div>`,
        link: function(scope) {
            const bins = scope.histogramData.map(function(bin, idx) {
                const mean = (bin.bin_min + bin.bin_max) / 2;
                return [ mean, bin.count, mean, bin.bin_min, bin.bin_max ];
            });

            scope.chartOptions = {
                animation: false,
                textStyle: { fontFamily: 'SourceSansPro' },
                tooltip: {
                    trigger: 'axis',
                    axisPointer: {
                        type: 'shadow',
                    },
                    formatter: function(data) {
                        const series = data[0];
                        const lowerBound = $filter("nicePrecision")(series.data[3], 3);
                        const upperBound = $filter("nicePrecision")(series.data[4], 3);
                        // np.histogram forms half-open bins, apart from the last one
                        const firstTooltipLine = series.dataIndex !== (bins.length - 1) ? `[${lowerBound}; ${upperBound})` : `[${lowerBound}; ${upperBound}]`;
                        return `<span class="font-weight-bold">${firstTooltipLine}</span><br/>
                            <i class='icon-circle mright8' style='color: ${series.color}'></i>Count: ${series.data[1]}<br/>`;
                    }
                },
                legend: { show: false },
                grid: { bottom: 88, left: 56, right: 32 },
                xAxis: {
                    name: "Predicted individual treatment effect",
                    nameGap: 40,
                    nameLocation: "middle",
                    type: 'category',
                    axisLabel: {
                        rotate: 45,
                        formatter: value => $filter("nicePrecision")(parseFloat(value), 3),
                    },
                },
                yAxis: {
                    name: "Number of occurrences",
                    nameGap: 40,
                    nameLocation: "middle",
                    type: "value",
                },
                series: [
                    {
                        data: bins, type: 'bar',
                    }
                ],
            };

            const COLORS = {
                green:'#81C141',   // @digital-green-base
                red: '#CE1228',    // @error-red-base
            };
            scope.chartOptions.visualMap = [{
                type: 'piecewise',
                orient: "horizontal",
                pieces: [
                    { max: 0, color: COLORS.red, label: 'Negative effect' },
                    { min: 0, color: COLORS.green, label: 'Positive effect' },
                ],
                dimension: 2,
            }];
        }
    };
});

app.directive("propensityPositivityChart", function() {
    return {
        scope: {
            data: '<',
        },
        template: `<div class="h100">
            <div block-api-error />
            <ng2-lazy-echart [options]="chartOptions" ng-if="chartOptions"></ng2-lazy-echart>
        </div>`,
        link: function(scope) {
            const probaDistribsTreated = scope.data.probaDistribs[1];
            const probaDistribsControl = scope.data.probaDistribs[0];
            const bins = scope.data.bins;
            const nBins = scope.data.bins.length;
            const COLORS = {
                blue: "#1F77B4",   // @custom-ml-chart-blue
                orange: "#FF7F0E", // @custom-ml-chart-orange
            };

            scope.chartOptions = {
                textStyle: { fontFamily: 'SourceSansPro' },
                tooltip: {
                    trigger: 'axis',
                    axisPointer: {
                        type: 'shadow',
                    },
                    formatter: function(params) {
                        const [firstSeries, secondSeries] = params;
                        const idx = firstSeries.dataIndex;
                        let tooltipContent = `<span class='text-debug'>[${idx / (nBins-1)}, ${(idx+1) / (nBins-1)}]</span><br/>`;
                        tooltipContent += `${firstSeries.seriesName}: <span class='font-weight-bold'>${firstSeries.data}</span>`;
                        if (secondSeries) {
                            tooltipContent += `<br/>${secondSeries.seriesName}: <span class='font-weight-bold'>${secondSeries.data}</span>`;
                        }
                        return tooltipContent;
                    }
                },
                legend: {},
                grid: { bottom: 88, left: 56, right: 32 },
                xAxis: {
                    name: "Propensity",
                    nameGap: 24,
                    nameLocation: "middle",
                    type: 'category',
                    data: bins
                },
                yAxis: {
                    scale: true,
                    name: "Number of occurrences",
                    nameGap: 40,
                    nameLocation: "middle",
                    type: "value",
                },
                series: [
                    {
                        name: 'Treated',
                        type: 'bar',
                        data: probaDistribsTreated,
                        color: COLORS.blue,
                    },
                    {
                        name: 'Control',
                        type: 'bar',
                        data: probaDistribsControl,
                        color: COLORS.orange,
                    }
                ]
                };
        }
    };
});


app.directive("upliftMetricChart", function() {
    return {
        scope: {
            data: '<',
            metric: '@',
            netUplift: '<?',
            netUpliftPoint: '<?',
            testAte: '<',
            normalized: '<',
        },
        template: `<div class="h100">
            <div block-api-error />
            <ng2-lazy-echart [options]="chartOptions" ng-if="chartOptions"></ng2-lazy-echart>
        </div>`,
        link: function(scope) {
            // Compute the non normalized version of the curves + net uplift point
            // These values can be later be normalized in updateGraph if scope.normalized is true
            const curveData = scope.data.map(dataPoint => [ dataPoint.x, dataPoint.y ]);
            const lastDataPointIdx = scope.data.length - 1;
            const randomAssignmentData = scope.data.map((dataPoint, idx) => [ dataPoint.x, scope.testAte * idx / lastDataPointIdx ]);
            const netUpliftAtSpecifiedLevel = (scope.netUplift != undefined && scope.netUpliftPoint != undefined) ? scope.netUplift + scope.testAte * scope.netUpliftPoint : null;

            const COLORS = {
                blue: "#1F77B4",   // @custom-ml-chart-blue
                orange: "#FF7F0E", // @custom-ml-chart-orange
                green: "#4caf50",  // @success-green-base
            };

            const toFixed = (number, precision) => parseFloat(number.toFixed(precision));
            function tooltipContentForSeries(series) {
                return `<i class='icon-circle mright8' style='color: ${series.color}'></i>
                    ${series.seriesName}: ${toFixed(series.data[1], 3)}<br/>`;
            };

            function updateGraph(curveData, randomAssignmentData, netUpliftAtSpecifiedLevel) {
                if (scope.normalized) {
                    const normalizationScale = Math.abs(scope.testAte);
                    if (normalizationScale > 0) { // extra carefulness, scale should always be != 0 as otherwise normalized cannot be set to true (switch disabled)
                        curveData = curveData.map(dataPoint => [ dataPoint[0], dataPoint[1] / normalizationScale]);
                        randomAssignmentData = randomAssignmentData.map(dataPoint => [ dataPoint[0], dataPoint[1] / normalizationScale]);
                        if (netUpliftAtSpecifiedLevel != null) {
                            netUpliftAtSpecifiedLevel /= normalizationScale;
                        }
                    }
                }

                const currentModelSeries = {
                    type: 'line',
                    symbol: 'none',
                    data: curveData,
                    name: scope.metric,
                    color: COLORS.blue,
                };

                // On cumulative gain chart only
                if (netUpliftAtSpecifiedLevel != null) {
                    currentModelSeries.markLine = {
                        data: [
                            {
                                xAxis: scope.netUpliftPoint * 100,
                                label: { color: COLORS.green, padding: -4, fontSize: 10 }
                            },
                            {
                                yAxis: netUpliftAtSpecifiedLevel,
                                label: {
                                    align: "center",
                                    color: COLORS.green,
                                    fontSize: 10,
                                    formatter: `Uplift at ${scope.netUpliftPoint * 100}%: ${toFixed(netUpliftAtSpecifiedLevel, 3)}\n`,
                                    lineHeight: 14,
                                    padding: [0, 0, 0, 112],
                                    position: "start",
                                }
                            },
                        ],
                        lineStyle: { type: 'dashed', width: 1, color: COLORS.green },
                        precision: 10,
                        silent: true,
                        symbol: 'none',
                    };
                }

                const randomModelSeries = {
                    type: 'line',
                    symbol: 'none',
                    data: randomAssignmentData,
                    name: "Random",
                    color: COLORS.orange,
                    lineStyle: { type: 'dashed' },
                };

                scope.chartOptions = {
                    animation: false,
                    legend: {
                        type: "scroll",
                        y: 16,
                        x: "center",
                        itemStyle: { opacity: 0 }
                    },
                    tooltip: {
                        trigger: 'axis',
                        textStyle: { fontSize: 13 },
                        formatter: function(params) {
                            const [ metricSeries, randomSeries ] = params;

                            let tooltipContent = `<b>${toFixed(metricSeries.data[0], 4)}% of the population</b><br/>`;
                            tooltipContent += tooltipContentForSeries(metricSeries);
                            tooltipContent += tooltipContentForSeries(randomSeries);
                            return tooltipContent;
                        }
                    },
                    textStyle: { fontFamily: 'SourceSansPro' },
                    grid: { right: 4 },
                    xAxis: {
                        axisLine: { onZero: false },
                        axisLabel: {
                            rotate: 45,
                            formatter: val => toFixed(val, 3)
                        },
                        name: "Fraction of the test observations, sorted by decreasing predicted individual effect\n(% of total test observations)",
                        nameGap: 24,
                        nameLocation: "middle",
                    },
                    yAxis: {
                        axisLabel: { formatter: val => toFixed(val, 3) },
                        scale: true,
                        name: "Cumulative effect" + (scope.normalized ? " (ratio of the ATE on the test set)" : ""),
                        nameGap: 36,
                        nameLocation: "middle",
                        type: "value",
                    },
                    series: [ currentModelSeries, randomModelSeries ]
                };
            }

            scope.$watch("normalized", function() {
                updateGraph(curveData, randomAssignmentData, netUpliftAtSpecifiedLevel);
            });
        }
    };
});

app.controller('_snippetMetricsCommon', function($scope, PartitionedModelsService, ModelDataUtils) {
    /* Compute various stats with precise number of decimals for display in table */

    function getMetric(metricFieldName) {
        return $scope.display.metrics.find(m => metricFieldName === m.fieldName);
    }

    $scope.getAggregationExplanation = function(metric) {
        const metricName = metric.metricName || metric.fieldName.toUpperCase();
        const displayName = metric.shortName || metric.name || metric.fieldName;
        return PartitionedModelsService.getAggregationExplanation(metricName, displayName, false, $scope.display.predictionType === 'TIMESERIES_FORECAST');
    }

    $scope.getCustomMetricAggregationExplanation = function(metric) {
        return PartitionedModelsService.getAggregationExplanation(metric.name, metric.name, true, $scope.display.predictionType === 'TIMESERIES_FORECAST');
    }

    function getNumDecimalsFromMetric(numDecimalsOrMetric) {
        if (!numDecimalsOrMetric) { return 0; }

        if (typeof numDecimalsOrMetric === "number") {
            return numDecimalsOrMetric;
        }

        let metric = getMetric(numDecimalsOrMetric);
        let numDecimals = $scope.display.numDecimalsMetrics[metric.fieldName];

        if (metric.minDecimals) {
            numDecimals = Math.max(metric.minDecimals, numDecimals);
        }
        if (metric.maxDecimals) {
            numDecimals = Math.min(metric.maxDecimals, numDecimals);
        }

        return numDecimals;
    }

    $scope.formatMetric = function(value, metricFieldName, forcedNumDecimals) {
        let metric = getMetric(metricFieldName);

        let numDecimals;
        if (forcedNumDecimals) {
            numDecimals = forcedNumDecimals;
        } else {
            numDecimals = getNumDecimalsFromMetric(metricFieldName);
        }

        const modelHasProbas = ModelDataUtils.hasProbas($scope.modelData);
        if (angular.isUndefined(value)
            || (value === 0 && metric.ignoreZero)
            || (!modelHasProbas && metric.needsProbability)
        ) {
            return '-';
        }

        let exp = value > 10000;
        if (exp) {
            numDecimals = 4;
        }

        if (metric.percentage) {
            return getDetailedPercent(value, numDecimals).toFixed(numDecimals) + " %";
        } else {
            return getDetailedValue(value, numDecimals)[exp ? 'toPrecision': 'toFixed'](numDecimals);
        }
    };

    function getDetailedValue(value, numDecimals) {
        return (Math.round(value * Math.pow(10, numDecimals)) / Math.pow(10, numDecimals));
    }

    function getDetailedPercent (p, numDecimals) {
        return getDetailedValue(p * 100, numDecimals);
    }

    function getDiffWithAllDataset(metric, numDecimals) {
        switch (metric) {
            case 'actual':
                return getDetailedPercent($scope.data.perf.singleMetrics.actPos["ratio"], numDecimals)
                    - getDetailedPercent($scope.allDatasetPerf.singleMetrics.actPos["ratio"], numDecimals);
            case 'predicted':
                return getDetailedPercent($scope.data.perf.singleMetrics.predPos["ratio"], numDecimals)
                    - getDetailedPercent($scope.allDatasetPerf.singleMetrics.predPos["ratio"], numDecimals);
            default:
                return 0;
        }
    }

    $scope.isAboveAllDataset = function(metric) {
        return getDiffWithAllDataset(metric, getNumDecimalsFromMetric(metric)) > 0;
    };

    $scope.isBelowAllDataset = function(metric) {
        return getDiffWithAllDataset(metric, getNumDecimalsFromMetric(metric)) < 0;
    };

    $scope.getAbsoluteDiffWithAllDataset = function(metric, numDecimalsOrMetric) {
        const numDecimals = getNumDecimalsFromMetric(numDecimalsOrMetric);
        return Math.abs(getDiffWithAllDataset(metric, numDecimals).toFixed(numDecimals)) + "\u00A0%";
    };

});

app.directive("partitionSummaryValue", function() {
    return {
        restrict: 'E',
        templateUrl: "/templates/ml/prediction-model/partition_summary-value.html",
        scope: {
            allDatasetPerf: "=",
            data : "=",
            threshold: "=",
            colors: "=",
            display: "=",
            modelData: "=",
            partitionStates: '='
        },
        controller: function($scope, $controller) {
            $controller("_snippetMetricsCommon", {$scope: $scope});

            // VISUAL HELPERS

            $scope.getLinearGradient = function(ratio) {
                return 'linear-gradient(to right, #c6e8d3 0%, #c6e8d3 '+ (ratio * 100) +'%,rgba(0, 0, 0, 0) '+ (ratio * 100) +'%, rgba(0, 0, 0, 0) 100%)';
            };

            // INIT

            $scope.uiState = {
                isExpanded: false
            }

            $scope.getCustomMetricResult = function(customMetricResults, customMetricName) {
                return customMetricResults.find(metricResult => metricResult.metric.name === customMetricName);
            }

            $scope.onClick = function(event) {
                if ($scope.data.excluded) return;
                if ($scope.display.predictionType === 'TIMESERIES_FORECAST') return;

                // If click event originates from info icon, we do not want to expand/hide subpop table row content,  as
                // a popover is being shown/hidden.
                if (event.originalEvent.composedPath().some(_ => _.className === "icon-info-sign")) return;
                $scope.uiState.isExpanded = !$scope.uiState.isExpanded;
            }
        }
    }
});


app.controller('VariableCoefficientController', function($scope, $filter, ListFilter, Debounce, ExportUtils, getNameValueFromMLFeatureFilter) {
    $scope.uiState = {
        advanced: false
    }
    var coefs = $scope.modelData.iperf.lmCoefficients.variables.map(function(v, i) {
            var splitedFeature = getNameValueFromMLFeatureFilter(v),
                o = {
                        full: v,
                        name: splitedFeature.name,
                        value: splitedFeature.value,
                        coef: this.coefs[i],
                        coefRescaled: this.rescaledCoefs ? this.rescaledCoefs[i] : undefined,
                        abs: Math.abs(this.coefs[i]),
                        rescaledAbs:  this.rescaledCoefs ? Math.abs(this.rescaledCoefs[i]) : undefined
                   };
            if (this.tstat)  { o.tstat = this.tstat[i]; }
            if (this.pvalue) { o.pvalue = this.pvalue[i]; }
            if (this.stderr) { o.stderr = this.stderr[i]; }
            if (this.rescaledStderr) { o.rescaledStderr = this.rescaledStderr[i]; }
            return o;
        }, $scope.modelData.iperf.lmCoefficients),
        maxCoef = Math.max.apply(Math, $scope.modelData.iperf.lmCoefficients.coefs.map(Math.abs)),
        maxRescaledCoef = $scope.modelData.iperf.lmCoefficients.rescaledCoefs ?
            Math.max.apply(Math, $scope.modelData.iperf.lmCoefficients.rescaledCoefs.map(Math.abs)) : undefined,
        filteredCoefs = coefs;

    function getVars() {
        filteredCoefs = !$scope.coefFilter ? coefs : ListFilter.filter(coefs, $scope.coefFilter);
        sortVars();
    }
    function sortVars() {
        var sort = ['+name', '+value'];
        var by = $scope.sort.by;
        if(!$scope.displayOptions.showRawCoefs){
            if(by == "abs"){
                by = "rescaledAbs";
            } else if(by == "coef") {
                by = "coefRescaled";
            }
        }

        if ($scope.sort.by !== 'name') { sort.unshift('+' + by); }
        sort[0] = ($scope.sort.reverse ? '-' : '+') + sort[0].substring(1);
        $scope.pagination.list = $filter('orderBy')(filteredCoefs, sort);
        $scope.pagination.page = 1;
        getCoeffs();
    }
    function getCoeffs() {
        $scope.pagination.update();
        $scope.coefs = $scope.pagination.slice;
    }

    $scope.exportCoefficients = function(){
        let lmc = $scope.modelData.iperf.lmCoefficients;
        let f = $filter("mlFeature");
        let data;
        if ($scope.uiState.advanced) {
            data = $filter("orderBy")(coefs, "-abs").map(function (x) {
                return [x.full, f(x.full), $scope.displayOptions.showRawCoefs ? x.coef : x.coefRescaled,
                    x.stderr === 0 ? "" : x.stderr,
                    x.tstat === 0 ? "" : x.tstat,
                    x.pvalue === 0 ? "" : x.pvalue]
            });
            data.push(["Intercept", null,
                    $scope.displayOptions.showRawCoefs  ? lmc.interceptCoef : lmc.rescaledInterceptCoef,
                    lmc.interceptStderr === 0 ? "" : lmc.interceptStderr,
                    lmc.interceptTstat === 0 ? "" : lmc.interceptTstat,
                    lmc.interceptPvalue === 0 ? "" : lmc.interceptPvalue
                    ]);

        } else {
            data = $filter("orderBy")(coefs, "-abs").map(function (x) {
                return [x.full, f(x.full), $scope.displayOptions.showRawCoefs ? x.coef : x.coefRescaled]
            });
            data.push(["Intercept", null,
                    $scope.displayOptions.showRawCoefs  ? lmc.interceptCoef : lmc.rescaledInterceptCoef]
                    );
        }

        let columns = [
                { name : "feature_name", type : "string"},
                { name : "feature_description", type : "string"},
                { name : "coefficient", type : "double"}
        ];
        if ($scope.uiState.advanced) {
            columns.push({ name : "stderr", type : "double"});
            columns.push({ name : "tstat", type : "double"});
            columns.push({ name : "pvalue", type : "double"});
        }

        ExportUtils.exportUIData($scope, {
            columns : columns,
            data : data
        }, "Export coefficients");
    };

    $scope.sorts = { name: 'Name', coef: 'Coefficient', abs: '| Coefficient |' };
    if ($scope.modelData.iperf.lmCoefficients.pvalue) {
        $scope.sorts['pvalue'] = 'Trust';
    }
    $scope.sort = { by: 'abs', reverse: true };
    $scope.baseWidth = function(){
        return 50 / ($scope.displayOptions.showRawCoefs ? maxCoef : maxRescaledCoef);
    };
    $scope.displayPossibleSmall = function(value){
        if(value < 1e-4) {
            return "< 1e-4";
        } else {
            return value.toFixed(4);
        }
    };
    $scope.getCoef = function(c){
         return $scope.displayOptions.showRawCoefs ? c.coef : c.coefRescaled;
    };
    var lmc = $scope.modelData.iperf.lmCoefficients;
    $scope.getIntercept = function(){
        return $scope.displayOptions.showRawCoefs ? lmc.interceptCoef : lmc.rescaledInterceptCoef;
    };
    $scope.getInterceptStderr = function(){
        return $scope.displayOptions.showRawCoefs ? lmc.interceptStderr : lmc.rescaledInterceptStderr;
    };
    $scope.getInterceptTstat = function(){
        return $scope.displayOptions.showRawCoefs ? lmc.interceptTstat : lmc.rescaledInterceptTstat;
    };
    $scope.getInterceptPvalue = function(){
        return $scope.displayOptions.showRawCoefs ? lmc.interceptPvalue : lmc.rescaledInterceptPvalue;
    };
    $scope.getStderr = function(c){
        return $scope.displayOptions.showRawCoefs ? c.stderr : c.rescaledStderr;
    };
    $scope.getAbs = function(c){
        return $scope.displayOptions.showRawCoefs ? c.abs : c.rescaledAbs;
    };
    $scope.coefFilter = '';
    $scope.displayOptions = {
        showRawCoefs: !$scope.modelData.iperf.lmCoefficients.rescaledCoefs
    }

    $scope.pagination = new ListFilter.Pagination([], 50);
    $scope.$watch('coefFilter', Debounce().withScope($scope).withDelay(75,150).wrap(getVars), true);
    $scope.$watch('sort', sortVars, true);
    $scope.$watch('displayOptions', sortVars, true);
    $scope.$watch('pagination.page', getCoeffs);
    getVars();
    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
    $scope.puppeteerHook_elementContentLoaded = true;
});

app.controller('MultiClassConfusionMatrixController', function($scope, Fn, $filter, Assert) {
    Assert.trueish($scope.modelData.perf, 'no modelData.perf');
    Assert.trueish($scope.modelData.perf.confusion, 'no confusion matrix data');
    var perActual = $scope.modelData.perf.confusion.perActual;
    $scope.total = $scope.modelData.perf.confusion.totalRows;
    if ($scope.modelData.classes && $scope.modelData.classes.length) {
        $scope.cs = $scope.modelData.classes;
    } else {
        $scope.cs = $scope.modelData.perf.classes;
    }
    $scope.n = $scope.cs.length;
    $scope.displayMode = 'actual';

    var predictedClassCount = $scope.cs.map(Fn.cst(0)),
        all100 = $scope.cs.map(Fn.cst('100\u00a0%')),
        smartPC = $filter('smartPercentage'),
        data = {};
    data.records = $scope.cs.map(function(ca, i) { return $scope.cs.map(function(cp, j) {
            if (!perActual[ca] || !perActual[ca].perPredicted[cp]) return 0;
            predictedClassCount[j] += perActual[ca].perPredicted[cp];
            return perActual[ca].perPredicted[cp];
        }); });
    data.actual = data.records.map(function(cps, i) {
        if (!perActual[$scope.cs[i]]) return cps.map(_ => '-');
        return cps.map(function(cp) {
            return this > 0 ? smartPC(cp / this, 0, true) : '-';
        }, perActual[$scope.cs[i]].actualClassCount);
    });
    data.predicted = data.records.map(function(cps) { return cps.map(function(cp, j) {
            return predictedClassCount[j] > 0 ? smartPC(cp / predictedClassCount[j], 0, true) : '-';
        });
    });

    //if the  number of classes is large, don't display the table to avoid crashing the browser
    $scope.tableHidden = data.records.length > 50;
    $scope.showTable = function(){
        $scope.tableHidden = false;
    };
    $scope.data = data;
    $scope.sumActual = {
        records: $scope.cs.map(Fn.dict(perActual,{})).map(Fn.prop('actualClassCount')).map(x=> isNaN(x) ? 0 : x),
        actual: all100
    };
    $scope.sumPredicted = {
        records: predictedClassCount,
        predicted: all100
    };
    $scope.total = $scope.sumActual.records.reduce(function(x, y) {return Number(x)+Number(y);}, 0)

    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
    $scope.puppeteerHook_elementContentLoaded = true;
});


app.controller('ROCCurveController', function($scope, ExportUtils) {
    $scope.setLabels = function () {
        if ($scope.areMetricsWeighted()) {
            $scope.data.xlabel = 'Weighted False Positive Rate';
            $scope.data.ylabel = 'Weighted True Positive Rate';
        } else {
            $scope.data.xlabel = 'False Positive Rate';
            $scope.data.ylabel = 'True Positive Rate';
        }

    }

    const perf = $scope.modelData.perf;

    if ($scope.isMulticlass()) {
        $scope.setRocClass = function(nv) {
            $scope.data = [perf.oneVsAllRocCurves[nv]];
            $scope.rocAuc = perf.oneVsAllRocAUC[nv];
            $scope.setLabels()
        };
        $scope.rocClass = $scope.modelData.classes[0];
        $scope.setRocClass($scope.rocClass);
        $scope.auc = perf.metrics.mrocAUC;
        $scope.aucstd = perf.metrics.mrocAUCstd;
    } else {
        $scope.data = perf.rocVizData;
        $scope.auc = perf.tiMetrics.auc;
        $scope.aucstd = perf.tiMetrics.aucstd;
        $scope.setLabels();
    }

    $scope.exportROCData = function(){
        var data = $scope.data[0].map(function(x){
            return [ x.x, x.y, x.p > 1 ? 1 : x.p ]
        })
        ExportUtils.exportUIData($scope, {
            name : "ROC data for model: " + $scope.modelData.userMeta.name,
            columns : [
                { name : "False positive rate", type : "double"},
                { name : "True positive rate", type : "double"},
                { name : "Proba threshold", type : "double"}
            ],
            data : data
        }, "Export ROC curve");
    }
    // Registration for RocAndPrCurvesController embedding support
    $scope.$on('exportData', function () {
        $scope.exportROCData();
    });
});

app.controller('PrecisionRecallCurveController', function($scope, ExportUtils) {
    $scope.setLabels = function () {
        $scope.data.xlabel = 'Recall';
        $scope.data.ylabel = 'Precision';
        if ($scope.areMetricsWeighted()) {
            $scope.data.xlabel = 'Weighted ' + $scope.data.xlabel;
            $scope.data.ylabel = 'Weighted ' + $scope.data.ylabel;
        }
    }

    const perf = $scope.modelData.perf;
    if ($scope.isMulticlass()) {
        $scope.setPrClass = function(nv) {
            $scope.data = [perf.oneVsAllPrCurves[nv]];
            $scope.classAveragePrecision = perf.oneVsAllAveragePrecision[nv];
            $scope.setLabels();
        };
        $scope.prClass = $scope.modelData.classes[0];
        $scope.setPrClass($scope.prClass);
        $scope.averagePrecision = perf.metrics.averagePrecision;
        $scope.averagePrecisionstd = perf.metrics.averagePrecisionstd;
    } else {
        $scope.data = perf.prVizData;
        $scope.averagePrecision = perf.tiMetrics.averagePrecision;
        $scope.averagePrecisionstd = perf.tiMetrics.averagePrecisionstd;
        $scope.setLabels();
    }

    $scope.hasFolds = $scope.modelData.trainInfo.kfold;

    $scope.exportPRData = function(){
        var data = $scope.data[0].bins.map(function(x){
            return [ x.x, x.y, x.p > 1 ? 1 : x.p ]
        })
        ExportUtils.exportUIData($scope, {
            name : "Precision-Recall data for model: " + $scope.modelData.userMeta.name,
            columns : [
                { name : "Recall", type : "double"},
                { name : "Precision", type : "double"},
                { name : "Proba threshold", type : "double"}
            ],
            data : data
        }, "Export Precision-Recall curve");
    }
    // Registration for RocAndPrCurvesController embedding support
    $scope.$on('exportData', function () {
        $scope.exportPRData();
    });
});

app.controller('RocAndPrCurvesController', function($scope, $state, $stateParams) {
    $scope.curves = [
        {
            curveId: 'roc',
            fullName: 'ROC curve',
            buttonName: 'ROC curve',
            template: '/templates/ml/prediction-model/c_roc.html',
            available: $scope.hasROCCurve()
        },
        {
            curveId: 'pr',
            fullName: 'Precision-Recall curve',
            buttonName: 'PR curve',
            template: '/templates/ml/prediction-model/c_precision_recall.html',
            available: $scope.hasPRCurve()
        }
    ]
    if ($stateParams.curveId) {  // retrieve displayed curve from url option
        $scope.displayedCurve = $scope.curves.find(c => c.curveId === $stateParams.curveId);
    }
    $scope.displayedCurve = $scope.displayedCurve || $scope.curves.find(c => c.available) || $scope.curves[0];
    $scope.embeddedMode = true;

    $scope.selectCurve = function(curve) {
        $scope.displayedCurve = curve;
        $state.go($state.current, {curveId: curve.curveId}, {reload: false});
    }

    $scope.exportData = function() {
        $scope.$broadcast('exportData');
    }
});

app.controller('CalibrationCurveController', function($scope, ExportUtils) {
    const perf = $scope.perf;
    $scope.setLabels = function() {
        if ($scope.areMetricsWeighted()) {
            $scope.data.xlabel = 'Weighted Average of Predicted Probability for Positive Class';
            $scope.data.ylabel = 'Weighted Frequency of Positive Class';
        } else {
            $scope.data.xlabel = 'Average of Predicted Probability for Positive Class';
            $scope.data.ylabel = 'Frequency of Positive Class';
        }
    }

    if ($scope.isMulticlass()) {
        $scope.setCalibrationClass = function(nv) {
            $scope.data = [perf.oneVsAllCalibrationCurves[nv]];
            $scope.calibrationLoss = perf.oneVsAllCalibrationLoss[nv];
            $scope.setLabels();
        };
        $scope.calibrationClass = $scope.modelData.classes[0];
        $scope.setCalibrationClass($scope.calibrationClass);
        $scope.mCalibrationLoss = perf.metrics.mCalibrationLoss;
        $scope.mCalibrationLossStd = perf.metrics.mCalibrationLossstd;
    } else {
        $scope.data = [perf.calibrationData];
        $scope.calibrationLoss = perf.tiMetrics.calibrationLoss;
        $scope.setLabels();
    }
    var calibrationMethod = $scope.modelData.coreParams && $scope.modelData.coreParams.calibration ? $scope.modelData.coreParams.calibration.calibrationMethod : undefined;
    if (calibrationMethod==="ISOTONIC") {
        $scope.uiState.calibrationMethod = "Isotonic Regression";
    } else if (calibrationMethod==="SIGMOID") {
        $scope.uiState.calibrationMethod = "Sigmoid (Platt scaling)";
    } else {
      $scope.uiState.calibrationMethod = "No calibration";
    }

    $scope.exportCalibrationData = function(){
        const data = $scope.data[0].filter(_ => _.n > 0).map(_ => [_.x, _.y, _.n]);
        ExportUtils.exportUIData($scope, {
            name : "Calibration data for model: " + $scope.modelData.userMeta.name,
            columns : [
                { name : "Average of Predicted Probability for Positive Class", type : "double"},
                { name : "Frequency of Positive Class", type : "double"},
                { name : "Count of Records", type : "double"}
            ],
            data : data
        }, "Export Calibration Curve");
    }

    $scope.hasCalibrationData = function(){
        return (typeof(perf.calibrationData) !== "undefined") ||
               (typeof(perf.oneVsAllCalibrationCurves) !== "undefined");
    }
});

app.service("ResidualsPlottingService", function() {

    const BLUE_COLOR = "#1F77B4";
    const ORANGE_COLOR = "#FF7F0E";

    return {
        createResidualsBarChartConfig, getQQPlotConfig
    }

    function createResidualsBarChartConfig(residuals, isStandardized=true, residualsMean=null, residualsStd=null) {
        const zeroResiduals = residuals.every(value => value === 0.0);
        // Don't use Math.min/max(...residuals) to avoid "RangeError: Maximum call stack size exceeded" when residuals are a big array.
        let minResidual, maxResidual;
        if (residuals && residuals.length > 0) {
            minResidual = maxResidual = residuals[0];
            for (let i = 1; i < residuals.length; i++) {
                if (residuals[i] < minResidual) minResidual = residuals[i];
                if (residuals[i] > maxResidual) maxResidual = residuals[i];
            }
        } else {
            minResidual = maxResidual = 0.0;
        }
        const x = d3.scale.linear().domain([minResidual, maxResidual]).nice();
        const data = d3.layout.histogram()
            .frequency(0)
            .bins(x.ticks(20))
            (residuals);
        const bars = data.filter(d => d.length > 0).map(d => {
            return { min: d.x, max: d.x + d.dx, count: d.length };
        });

        const scaleFactor = residuals.length * ((Math.abs(maxResidual - minResidual)) / data.filter(d => d.length > 0).length);

        function generateNormalDistribution(mean, stdDev, xs) {
            let data = [];
            xs.forEach(x => {
                let fx = (1 / Math.sqrt(2 * Math.PI * Math.pow(stdDev, 2))) * Math.exp(- Math.pow(x - mean, 2) / (2 * Math.pow(stdDev, 2)))
                data.push([x, fx * scaleFactor]);
            })
            return data;
        }

        const bins = bars.map(b => [(b.max + b.min) / 2, b.count]);

        const title = zeroResiduals ?
                {
                    text: "All residuals are 0",
                    left: "center",
                    top: "center",
                    textStyle: { fontWeight: 'bold', fontSize: 12 }
                } :
                {
                    text: isStandardized ? "Standardized residuals distribution" : "Residuals distribution",
                    textAlign: "left",
                    textStyle: { fontWeight: 'bold', fontSize: 12 }
                };

        const graphOptions = {
            grid: { top: 32, bottom: 40 , right: 8, left: 48 },
            title,
            xAxis: [
                {
                    name: isStandardized ? "Standardized residual value" : "Residual value",
                    type: 'value',
                    nameLocation: "middle",
                    nameGap: 24
                }
            ],
            yAxis: [
                {
                    name: "Count",
                    nameRotate: 90,
                    nameGap: 36,
                    type: 'value',
                    nameLocation: "center",
                    axisLabel: {
                        formatter: function (value) {
                            if(residuals && residuals.length > 0) {
                                return `${(value * 100 / residuals.length).toFixed(0)}%`
                            } else {
                                return `${value}`
                            }
                        }
                    }
                }
            ],
            tooltip: {
                trigger: 'axis',
                confine: true,
                formatter: params => {
                    return params.map(p => {
                        if(residuals && residuals.length > 0) {
                            const percent = (p.value[1] * 100 / residuals.length).toFixed(2) + "%"
                            return `${p.marker} ${p.seriesName}<span style="float: right; margin-left: 20px"><b> ${percent}</b></span>`;
                        } else {
                            return `${p.marker} ${p.seriesName}<span style="float: right; margin-left: 20px"><b> ${p.value[1]}</b></span>`;
                        }
                    }).join('<br/>');
                }
            },
            series: [
                {
                    name: 'Count',
                    type: 'bar',
                    barWidth: '100%',
                    data: bins,
                    color: BLUE_COLOR
                },
                {
                    name: 'Normal distribution',
                    type: 'line',
                    showSymbol: true,
                    clip: true,
                    data: generateNormalDistribution(isStandardized ? 0 : residualsMean, isStandardized ? 1 : residualsStd, bars.map(b => (b.max + b.min) / 2)),
                    color: ORANGE_COLOR
                }
            ]
        };
        if (!isStandardized) {
            // Add vertical yellow line to display the "Average clipped" (residualsMean)
            graphOptions.series.push(
                {
                    name: 'Mean',
                    type: 'line',
                    markLine: {
                        silent: true,
                        symbol: 'none',
                        label: {
                            show: true,
                            formatter: 'Average (clipped): {c}'
                        },
                        data: [
                            {
                                xAxis: residualsMean,
                                lineStyle: {
                                    color: ORANGE_COLOR,
                                    width: 2,
                                    type: 'dashed'
                                }
                            }
                        ]
                    }
                })
        }
        return graphOptions
    }

    function getQQPlotConfig(stdResiduals, theoreticalQuantiles) {
        return  {
            grid: { top: 32, bottom: 40 , right: 8, left: 48 },
            title: {
                text: "Normal Q-Q Plot",
                textAlign: "left",
                textStyle: { fontWeight: 'bold', fontSize: 12 }
            },
            xAxis: {
                name: "Standardized quantiles",
                nameLocation: "start",
                nameGap: 32,
                nameRotate: 90
            },
            yAxis: {
                name: "Theoretical quantiles",
                nameGap: 24,
                nameLocation: "start"
            },
            tooltip: {
                trigger: 'axis',
                confine: true
            },
            series: [
                {
                    data: stdResiduals.sort(function(a,b) { return a - b;}).map((r, idx) => [theoreticalQuantiles[idx], r]),
                    type: 'scatter',
                    symbolSize: 4,
                    color: BLUE_COLOR,
                    name: 'Standardized quantiles',
                    sampling: 'lttb',
                },
                {
                    data: [
                        [theoreticalQuantiles[0], theoreticalQuantiles[0]],
                        [theoreticalQuantiles[theoreticalQuantiles.length -1], theoreticalQuantiles[theoreticalQuantiles.length -1]]
                    ],
                    type: 'line',
                    showSymbol: false,
                    color: ORANGE_COLOR,
                    name: 'Theoretical quantiles'
                },
            ]
        }
    }

})

app.controller('ErrorDistributionController', function($scope, ResidualsPlottingService, $filter) {

    if (getCookie('dku_graphics_export') === 'true') {
        $scope.isInExport = true
    }

    const rp = $scope.modelData.perf.regression_performance;
    angular.forEach(rp, function (v, k) {    // *_error metrics into scope
        if (k.substr(-6) === '_error' && typeof v === 'number') {
            $scope[k.substr(0, k.length - 6)] = v.toPrecision(5);
        }
    });
    $scope.hasRawMinMax = function () {
        return $scope.modelData.perf.regression_performance.hasOwnProperty("raw_min_error") &&
            $scope.modelData.perf.regression_performance.hasOwnProperty("raw_max_error");
    };
    $scope.hasRawMeanStd = function () {
        return $scope.modelData.perf.regression_performance.hasOwnProperty("raw_average_error") &&
            $scope.modelData.perf.regression_performance.hasOwnProperty("raw_std_error");
    };
    $scope.nicePrecision = n => $filter('nicePrecision')(n, 4);

    const residuals = $scope.modelData.perf?.residuals
    if (residuals) {
        $scope.residualsHistograms = ResidualsPlottingService.createResidualsBarChartConfig(residuals.residuals, false, residuals.residualsMean, residuals.residualsStd)
        $scope.stats = residuals.stats
        $scope.qqplots = ResidualsPlottingService.getQQPlotConfig(residuals.stdResiduals, residuals.theoreticalQuantiles)
    }

    // Old graph for legacy models trained before 14.0
    if (rp.error_distribution) {
        $scope.bars = rp.error_distribution.map(function (p) {
            return {min: p.bin_min, max: p.bin_max, count: p.count};
        });
    }

    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
    setTimeout(function () {
        $scope.puppeteerHook_elementContentLoaded = true;
    }, 2000); // Need time to display animation
});


app.controller("StdModelReportFeaturesHandlingController", function($scope, ListFilter, TimeseriesForecastingUtils, DataikuAPI, EmbeddingService) {
    $scope.filter = {
        query : "",
        pagination : new ListFilter.Pagination([], 40)
    };

    $scope.uiState = {
        showAllPreprocessedFeatures:false
    };

    $scope.getCategoricalEncodingHeader = function(feature) {
        return ["category"].concat($scope.modelData.preprocessingReport.categoricalEncodings[feature]['targetValues']);
    };

    $scope.getLimitedCategoricalEncodingRows = function(feature) {
        const featureEncodings = $scope.modelData.preprocessingReport.categoricalEncodings[feature];
        // Sort by (descending) category counts
        const sortByCountsIndices = featureEncodings.counts.map((count, idx) => {return {idx, count}})
                                                           .sort((a, b) => {return a.count > b.count ? -1 : a.count == b.count ? 0 : 1 })
                                                           .map((obj) => obj.idx)
        return sortByCountsIndices.map((idx) => {
            return [featureEncodings.values[idx]].concat(featureEncodings.encodings[idx]);
        }).slice(0, 20);
    };

    $scope.updateList = function(){
        $scope.filteredList = ListFilter.filter($scope.features, $scope.filter.query);
        $scope.currentPageItems = $scope.filter.pagination.updateAndGetSlice($scope.filteredList);
    };

    $scope.$watch("filter", $scope.updateList, true);
    $scope.embeddingMetadataFetcher = EmbeddingService.metadataFetcher(setErrorInScope.bind($scope));

    $scope.$watch("modelData", function(nv, ov) {
        if (nv) {
            $scope.features = [];
            const algoDoesNotSupportExternalFeatures = TimeseriesForecastingUtils.ALGOS_WITHOUT_EXTERNAL_FEATURES.names.includes($scope.modelData.modeling.algorithm);
            $.each($scope.modelData.preprocessing.per_feature, function(k, v){
                const featurePreprocessing = Object.assign({}, v);
                featurePreprocessing.name = k;

                const hasCategoricalEncodings = nv.preprocessingReport &&
                    "categoricalEncodings" in nv.preprocessingReport &&
                    k in nv.preprocessingReport.categoricalEncodings &&
                    nv.preprocessingReport.categoricalEncodings[k].values.length > 0;

                featurePreprocessing.hasReport =
                    (featurePreprocessing.type == "CATEGORY" && featurePreprocessing.role != "REJECT" && featurePreprocessing.category_handling == "IMPACT" && hasCategoricalEncodings) ||
                    (featurePreprocessing.type == "CATEGORY" && featurePreprocessing.role != "REJECT" && featurePreprocessing.category_handling == "ORDINAL" && hasCategoricalEncodings) ||
                    (featurePreprocessing.type == "CATEGORY" && featurePreprocessing.role != "REJECT" && featurePreprocessing.category_handling == "FREQUENCY" && hasCategoricalEncodings) ||
                    (featurePreprocessing.type == "TEXT" && featurePreprocessing.role != "REJECT" && featurePreprocessing.text_handling == "TOKENIZE_COUNTS")||
                    (featurePreprocessing.type == "TEXT" && featurePreprocessing.role != "REJECT" && featurePreprocessing.text_handling == "SENTENCE_EMBEDDING")||
                    (featurePreprocessing.type == "IMAGE" && featurePreprocessing.role != "REJECT" && featurePreprocessing.image_handling == "EMBEDDING_EXTRACTION")||
                    (featurePreprocessing.type == "TEXT" && featurePreprocessing.role != "REJECT" && featurePreprocessing.text_handling == "TOKENIZE_TFIDF");

                const isUnsupportedExternalFeature = (featurePreprocessing.role === "INPUT" || featurePreprocessing.role === "INPUT_PAST_ONLY") && algoDoesNotSupportExternalFeatures;
                const isUnsupportedPastOnlyFeature = featurePreprocessing.role === "INPUT_PAST_ONLY" && nv?.modeling?.isShiftWindowsCompatible !== true;
                if (isUnsupportedExternalFeature || isUnsupportedPastOnlyFeature) {
                    featurePreprocessing.role = "REJECT";
                }

                if (featurePreprocessing.role === "TIMESERIES_IDENTIFIER") {
                    const resolvedParams = $scope.modelData.actualParams.resolved;
                    const resolvedAlgoParamsWithIdsAsFeatCapability = resolvedParams.gluonts_deepar_timeseries_params ||
                        resolvedParams.gluonts_transformer_timeseries_params ||
                        resolvedParams.gluonts_mqcnn_timeseries_params;
                    featurePreprocessing.alsoHasInputRole = resolvedAlgoParamsWithIdsAsFeatCapability &&
                        resolvedAlgoParamsWithIdsAsFeatCapability.use_timeseries_identifiers_as_features;
                }
                $scope.features.push(featurePreprocessing);
            });
            $scope.updateList();
            $scope.someMonotonicityConstraints = Object.values($scope.modelData.preprocessing.per_feature).some(x => ((x.role == 'INPUT') && (x.type == 'NUMERIC') && (x.numerical_handling == 'REGULAR') && (x.monotonic!='NONE')));
        }
    });
});

app.component("featureGenerationReport", {
    bindings: {
        modelData: "<"
    },
    templateUrl: "/templates/ml/prediction-model/timeseries/feature-generation.html",
    controller: function(ListFilter, TimeseriesForecastingUtils, $scope) {
        var ctrl = this;
        ctrl.prettyTimeUnit = TimeseriesForecastingUtils.prettyTimeUnit;
        ctrl.$onInit = function() {
            ctrl.tables = {
                shiftPagination : new ListFilter.Pagination([], 40),
                windowsPaginations: []
            };
            ctrl.hasAutoShifts = Object.entries(ctrl.modelData.preprocessing.feature_generation.shifts).some(shiftAndFeatureName => shiftAndFeatureName[1].from_horizon_mode == 'AUTO')
            ctrl.autoShiftsParams = ctrl.modelData.preprocessing.feature_generation.auto_shifts_params;
            ctrl.shifts = Object.entries(ctrl.modelData.preprocessing.feature_generation.shifts)
                .map(shiftAndFeatureName => {
                    let copy = angular.copy(shiftAndFeatureName);
                    let feature_name = copy[0];
                    let shift = copy[1];
                    shift.feature_name = feature_name;
                    shift.feature_type = ctrl.modelData.preprocessing.per_feature[feature_name].type;
                    shift.feature_role = ctrl.modelData.preprocessing.per_feature[feature_name].role;
                    return shift;
                })
                .filter(x => x.feature_role !== "REJECT")
                .filter(x => x.from_forecast.length > 0 || (x.from_horizon_mode == 'FIXED' && x.from_horizon.length > 0) || x.from_horizon_mode == 'AUTO');

            ctrl.windows = ctrl.modelData.preprocessing.feature_generation.windows.map(((rawWindow, index) => {
                let window = angular.copy(rawWindow);
                window.operations_list = Object.entries(window.operations_map)
                    .filter(featureNameAndWindowOperations => {
                        let featureName = featureNameAndWindowOperations[0];
                        return TimeseriesForecastingUtils.isWindowCompatible(ctrl.modelData.preprocessing.per_feature[featureName]) &&
                               featureNameAndWindowOperations[1].some(operation => operation.enabled)
                    })
                    .map(featureNameAndWindowOperations => {
                        let res = {};
                        res.feature_name = featureNameAndWindowOperations[0];
                        res.feature_type = ctrl.modelData.preprocessing.per_feature[res.feature_name].type;
                        res.feature_role = ctrl.modelData.preprocessing.per_feature[res.feature_name].role;
                        res.window = featureNameAndWindowOperations[1];
                        return res;
                    });
                ctrl.tables.windowsPaginations.push(new ListFilter.Pagination([], 40));
                window.index = index;
                return window;
            }));

            ctrl.windows = ctrl.windows.filter(x => x.operations_list.length > 0);
            ctrl.updateList = function() {
                ctrl.currentPageShifts = ctrl.tables.shiftPagination.updateAndGetSlice(ctrl.shifts);
                ctrl.windowPages = ctrl.tables.windowsPaginations.map(((windowPagination, index) =>
                    windowPagination.updateAndGetSlice(ctrl.windows[index].operations_list)));
            };
            ctrl.updateList();
            ctrl.getCompatibleOperations = function(feature) {
                return feature.window.filter(operation => (feature.feature_type === "CATEGORY" && operation.operation === "FREQUENCY") ||
                                                           feature.feature_type === "NUMERIC" && operation.operation !== "FREQUENCY");
            }
        };
        $scope.$watch(
            () => ctrl.tables,
            function(newVal, oldVal) {
                ctrl.updateList();
            },
            true
        );
    }
});

app.controller("GridSearchReportController", function($scope, $filter, PMLSettings, ExportUtils, MLChartsCommon, CustomMetricIDService, NumberFormatter, Fn){
    $scope.fieldList = [
        {field:'score', name:'Score', metric: true},
        {field:'scoreStd', name:'Score StdDev', metric: true},
        {field:'fitTime', name:'Fit Time', metric: true},
        {field:'fitTimeStd', name:'Fit Time StdDev', metric: true},
        {field:'scoreTime', name:'Score Time', metric: true},
        {field:'scoreTimeStd', name:'Score Time StdDev', metric:true}
    ];

    $scope.uiState = {
        showConstantColumns: false,
        chartColumn: null,
        chartLogScale: null, // null = N/A (non-numeric dimension)
        showFitTime: false,
        view: '1D' // Initialize the tab view to 1D plots
    };

    $scope.selectView = function(view) {
        $scope.uiState.view = view;
    }

    $scope.$watch('modelData.iperf.gridCells', gridCells => {
        if (!gridCells) return;
        $scope.gridCells = gridCells;
        if (['SVC_CLASSIFICATION', 'SVM_REGRESSION'].includes($scope.modelData.modeling.algorithm)) {
            $scope.gridCells.forEach(function(cell){
                let gamma_value = cell.params.gamma;
                if (typeof gamma_value !== 'undefined' && !['auto', 'scale'].includes(gamma_value)){
                    cell.params['custom_gamma'] = gamma_value;
                    cell.params['gamma'] = 'custom';
                }
            })
        }
        $scope.paramColumns = gridCells.map(_ => Object.keys(_.params))   // [['a', 'b'], ['a', 'c']]
            .reduce((a, b) => a.concat(b.filter(_ => a.indexOf(_) < 0)), []);   // ['a', 'b', 'c']
        $scope.gridData = gridCells.map(cell =>
            $scope.paramColumns.map(_ => cell.params[_]).concat($scope.fieldList.map(_ => cell[_.field]))
                               .map(_ => (_ === null || _ === undefined) ? "" : _)
        );
        $scope.pairwiseDependencies = computeHyperparametersPairwiseDependencies();

        // compute data when hiding columns that don't change
        const columnChanges = $scope.paramColumns.map( (col, i) =>
            $scope.gridData.map(_ => _[i])  // values of that column
                .some((cell, j, values) => j > 0 && cell != values[j-1]) // at least 1 differs from the previous
        );
        $scope.changingParamColumns = $scope.paramColumns.filter((col, i) => columnChanges[i]);
        $scope.changingGridData = $scope.gridData.map((row, j) =>
            row.filter((col, i) => i >= $scope.paramColumns.length || columnChanges[i]));
        if ($scope.changingParamColumns.indexOf($scope.uiState.chartColumn) < 0) {
            $scope.uiState.chartColumn = $scope.changingParamColumns[0];
        }

        const metric = $scope.modelData.modeling.metrics.evaluationMetric;
        if (metric) {
            if (metric === "CUSTOM") {
                const customEvaluationMetricName = $scope.modelData.modeling.metrics.customEvaluationMetricName;
                const customMetric = $scope.modelData.modeling.metrics.customMetrics.find(cM => cM.name === customEvaluationMetricName);
                $scope.uiState.currentMetric = CustomMetricIDService.getCustomMetricId(customMetric.name);
                $scope.uiState.scoreMetric = ' (' + customEvaluationMetricName + ')';
            } else {
                $scope.uiState.scoreMetric = ' (' + (PMLSettings.names.evaluationMetrics[metric] || metric) + ')';
                $scope.uiState.currentMetric = metric;
            }
        } else {
            $scope.uiState.scoreMetric = '';
            $scope.uiState.currentMetric = metric;
        }
    });

    $scope.$watch('uiState.showConstantColumns', showConstantColumns => {
        let tableData;
        let tableColumns;
        if (showConstantColumns) {
            tableColumns = $scope.paramColumns;
            tableData = $scope.gridData;
        } else {
            tableColumns = $scope.changingParamColumns;
            tableData = $scope.changingGridData;
        }

        // Generate the grid data as a list of objects (field name, value) so that it can be interactively ordered by users
        $scope.displayColumns = tableColumns.map(col => ({field: col, name: col, metric:false})).concat($scope.fieldList);
        $scope.displayData = tableData.map(row => {
            return $scope.displayColumns.reduce((result, item, index) => {
                result[item.field] = row[index];
                return result;
            }, {});
        });
    });

    // Formatter for search table values.
    // - displayIntegersAsFloat = true :  3 => 3.000000
    // - displayIntegersAsFloat = false :  3 => 3
    $scope.formatSearchTableValue = function(value, precision, displayIntegersAsFloat) {
        if (angular.isString(value)) {
            return value !== '' ? value : '-';
        }
        var isInteger = (value % 1 == 0);
        return (displayIntegersAsFloat || !isInteger) ? value.toFixed(precision) : value;
    }

    // Simple formatter for train time on tick grid
    function formatTimeShort(maxValue, seconds) {
        let components = [];
        if (maxValue >= 86400) {
            components = [
                {v: Math.floor(seconds / 86400), t: 'd'}, //
                {v: Math.round((seconds % 86400) / 3600), t: 'h' }];
        } else if (maxValue >= 3600) {
            components = [
                {v: Math.floor(seconds / 3600), t: 'h'},
                {v: Math.round((seconds % 3600) / 60), t: 'm'}];
        } else if (maxValue >= 60) {
            components = [
                {v: Math.floor(seconds / 60), t: 'm'},
                {v: Math.round(seconds % 60), t: 's'}];
        } else if (maxValue >= 10) {
            components = [{v: Math.round(seconds), t: 's'}];
        } else if (maxValue >= 1) {
            components = [{v: seconds.toFixed(1), t: 's'}];
        } else if (maxValue >= 0.1) {
            components = [{v: seconds.toFixed(2), t: 's'}];
        } else {
            components = [{v: seconds.toFixed(3), t: 's'}];
        }
        if (seconds === 0) {
            // Special case for 0: print it using the smallest component type.
            return "0" + components[components.length - 1].t;
        } else {
            let result = "";
            components.forEach(function (item) {
                if (item.v > 0) {
                    if (result.length > 0) {
                        result += " ";
                    }
                    result += item.v + item.t;
                }
            });
            return result;
        }
    }
    // Simple formatter for train time in tooltip
    function formatTimeFull(seconds) {
        let str = '';
        if (seconds >= 86400) {
            str += Math.floor(seconds / 86400) + "d ";
            seconds = seconds % 86400;
        }
        if (seconds >= 3600) {
            str += Math.floor(seconds / 3600) + "h ";
            seconds = seconds % 3600;
        }
        if (seconds >= 60) {
            str += Math.floor(seconds / 60) + "m ";
            seconds = seconds % 60;
        }
        return str + seconds.toFixed(3) + 's';
    }

    $scope.$watch('uiState.chartColumn', column => {
        const colIdx = $scope.paramColumns.indexOf(column);
        if (colIdx < 0) return;

        const scoreIdx = $scope.paramColumns.length;
        const fitTimeIdx = scoreIdx + 2;
        const colors = $scope.colors.slice(0, 3); // We need 3 colors: Score, Fit time & Best score marker
        const naturalSort = (a, b) => (a == b ? 0 : (a < b ? -1 : 1));

        // skip null scores
        const validGridData = filterInvalidScoresFromGridData();

        let x       = validGridData.map(_ => _[colIdx]);      // [1,  2,  1,  2]
        let score   = validGridData.map(_ => _[scoreIdx]);    // [.7, .6, .9, .8]
        let fitTime = validGridData.map(_ => _[fitTimeIdx]);  // [10, 15, 20, 25]

        // Group by X (avg, min, max)
        function groupByX(series, x) {
            const values = {};
            const min = {};
            const max = {};
            for (let i in x) {
                const curX = x[i];
                if (! (curX in values)) {
                    values[curX] = [];
                }
                values[curX].push(series[i]);
            }
            for (let curX in values) {
                min[curX] = Math.min(...values[curX]);
                max[curX] = Math.max(...values[curX]);
                values[curX] = values[curX].reduce((a, b) => a + b, 0) / values[curX].length;
            }
            return { avg: values, min, max };
        }
        score   = groupByX(score, x);   // score.avg:   {1: .8, 2: .7}
        fitTime = groupByX(fitTime, x); // fitTime.avg: {1: 15, 2: 20}

        // Build series for plotting
        // Ignore values with empty (curX === "") X value (the parameter is not used/defined at point)
        x = x.sort(naturalSort).filter((curX, i, x) => curX !== "" && (i == 0 || curX != x[i-1])); // [1,  2]
        for (let k of ['avg', 'min', 'max']) {
            score[k]   = x.map(curX => score[k][curX]);    // [.8, .7]
            fitTime[k] = x.map(curX => fitTime[k][curX]);  // [15, 20]
        }
        const minScore = Math.min(...score.min);
        const maxScore = Math.max(...score.max);
        const minAvgScore = Math.min(...score.avg);
        const maxAvgScore = Math.max(...score.avg);
        const maxFitTime = Math.max(...fitTime.max);
        const lib = PMLSettings.sort.lowerIsBetter($scope.uiState.currentMetric, $scope.modelData.modeling.metrics.customMetrics) ? -1 : 1;
        const indexOfBestScore = lib > 0 ? score.avg.indexOf(maxAvgScore) : score.avg.indexOf(minAvgScore);

        /* If the scores are between 0 and 1 and the difference is big, force the scale to [0, 1].
         * Else, use a scale that goes just a bit beyond the boundaries so that the data does not
         * align to the angles of the chart (because it looks a bit ugly)
         */
        const scale = (function() {
            if (minScore >= 0 && maxScore <= 1) {
                if (maxAvgScore - minAvgScore >= 0.2) {
                    return [0, 1];
                } else {
                    const factor = Math.abs(maxScore - minScore) * 0.1;
                    const furtherMax = maxScore > 0 ? (maxScore + factor) : (maxScore - factor);
                    const furtherMin = minScore > 0 ? (minScore - factor) : (minScore + factor);
                    return [Math.max(0, furtherMin), Math.min(1, furtherMax)]
                }
            } else {
                const furtherMax = maxScore > 0 ? maxScore * 1.05 : maxScore * 0.95;
                const furtherMin = minScore > 0 ? minScore * 0.95 : minScore * 1.05;
                return [furtherMin, furtherMax];
            }
        })();
        const scale2 = [0, maxFitTime];

        // Prepare the formats
        const xAreNumbers = x.every(_ => typeof _ === "number");
        const format = [
            "",                                               // x (overwritten in updateChartXScale function)
            minScore > -10 && maxScore < 10 ? '.4g' : '.4s',  // y1
            formatTimeShort.bind(null, maxFitTime),           // y2 (axis)
            formatTimeFull                                    // y2 (tooltip)
        ];

        // svg callbacks: 1. draw area for min/max
        const svgAreaCallback = MLChartsCommon.makeSvgAreaCallback(scope =>
            [{
                color: colors[0],
                values: $scope.chart.x.map((curX, i) => ({x: curX, y0: score.min[i], y1: score.max[i]}))
            }, {
                color: colors[1],
                yScale: scope.ys2.scale.copy().range(scope.chart.yScale().range()),
                values: $scope.chart.x.map((curX, i) => ({x: curX, y0: fitTime.min[i], y1: fitTime.max[i]}))
            }]
        );

        // and 2. color left axis with the color for score and overload the click on legend
        const svgCallback = (svg, scope) => {
            svg.selectAll(".nv-y.nv-axis text").style('fill', colors[0]);

            // Declare this method in $scope to get access to 'svg'.
            $scope.showHideFitTime = function () {
                // Show/hide the curve
                let displayValue = $scope.uiState.showFitTime ? "": "none";
                svg.selectAll("g.nv-lineChart g.nv-line g.nv-series-1").style('display', displayValue);

                // Show/hide the area (if present)
                let area = svg.select('.tubes').selectAll('path');
                if (area && area.length > 0) {
                    area[0][1].style.opacity = $scope.uiState.showFitTime ? '.3' : '0';
                }

                // Show/hide the secondary axis
                svg.select("g.nv-lineChart g.secondary-axis").style('display', displayValue);

                // Show/hide the legend
                svg.selectAll("g.nv-legendWrap g.nv-series:nth-child(2) .nv-legend-symbol").style('fill-opacity', $scope.uiState.showFitTime ? "1" : "0");
            };
            $scope.showHideFitTime(); // Call this method once to initially hide the "fit time" series.

            // Manually handle the click on legend
            scope.chart.legend.updateState(false);
            scope.chart.legend.dispatch.on('legendClick.overload', function (d, index) {
                if (index === 1) { // We only care about showing/hiding the "fit time" series (located at index 1).
                    $scope.$apply(function() {
                        $scope.uiState.showFitTime = !$scope.uiState.showFitTime;
                    });
                }
            });

            // Add tick vertical line to highlight the best score
            if (indexOfBestScore >= 0) {
                const xMarkStrokeWidth = 2;
                let xBestScore = x[indexOfBestScore];
                if (!xAreNumbers) {
                    // For ordinal values, use corresponding index
                    xBestScore = x.indexOf(xBestScore);
                }
                let xTranslate = scope.chart.xScale()(xBestScore);
                // In case the best score is the the first one on the X axis, slightly shift the line so that it
                // does not overlap with the left y-axis.
                if (indexOfBestScore === 0) {
                    xTranslate += xMarkStrokeWidth;
                }
                let xMarkG = svg.select(".nv-lineChart").append("g").attr("class", "x mark")
                    .attr("transform", "translate(" + xTranslate + ", 0)");
                xMarkG.append("path").attr('d', "M0,0 V" + scope.axisHeight)
                    .attr('stroke-width', xMarkStrokeWidth).attr('stroke', colors[2]).attr('stroke-dasharray', '5,3');
            }

            // Format tooltip x values
            if ($scope.uiState.chartLogScale === null) {
                // Use same formatter as x axis for categories
                scope.chart.interactiveLayer.tooltip.headerFormatter($scope.chart.format[0]);
            } else {
                // Use simple formatter for numbers
                scope.chart.interactiveLayer.tooltip.headerFormatter($scope.chart.xNumericFormat);
            }

            svgAreaCallback(svg, scope);

            // Fix tooltips persisting when changing column (https://github.com/krispo/angular-nvd3/issues/530#issuecomment-246745836)
            d3.selectAll('.nvtooltip').remove();
        };
        $scope.chart = {
            format, x, colors, scale, scale2, svgCallback,
            xLabels: x, score: score.avg, fitTime: fitTime.avg
        };

        // ~Smart default for log scale
        if (!xAreNumbers) {
            $scope.uiState.chartLogScale = null;
        } else {
            const min = Math.min(...x);
            const max = Math.max(...x);
            if (min == 0) {
                $scope.uiState.chartLogScale = max > 10;
            } else if (max == 0) {
                $scope.uiState.chartLogScale = min < -10;
            } else {
                $scope.uiState.chartLogScale = Math.abs(max / min) > 10;
            }

            // Determine numeric formatter as a function of min and max values of x
            if ( min <= 0.01 || max >= 100 ) {
                $scope.chart.xNumericFormat = _ => MLChartsCommon.trimTrailingZeros(d3.format(".3e")(_));
            } else {
                $scope.chart.xNumericFormat = _ => MLChartsCommon.trimTrailingZeros(d3.format(".4g")(_));
            }
        }
        updateChartXScale($scope.uiState.chartLogScale);
    });

    $scope.$watch('uiState.showFitTime', function () {
        if ($scope.showHideFitTime) {
            $scope.showHideFitTime();
        }
    });

    function updateChartXScale(chartLogScale) {
        let x = $scope.chart.x;
        if (chartLogScale === null) {
            // cheat for ordinal values using a linear scale & a custom formatter
            $scope.chart.xScale = d3.scale.linear().domain([-0.1, x.length - 0.9]);
            $scope.chart.x = x.map((_, i) => i);
            $scope.chart.xTicks = $scope.chart.x;  // fixed ticks for ordinals
            $scope.chart.format[0] = function(xValue) {  // check multiLineChart directive for format array meaning
                // format x axis values: empty strings for values that are not in categories, label string otherwise
                if (xValue < 0 || xValue > x.length - 1) {
                    return "";
                } else {
                    return $scope.chart.xLabels[xValue]
                }
            };
        } else {
            if (chartLogScale) {
                $scope.chart.xScale = d3.scale.log();
                if (x[0] <= 0) {
                    // protect against <= 0
                    x = x.slice();
                    x[0] = 1e-23;
                    $scope.chart.xScale.clamp(true);
                }
                $scope.chart.xTicks = false; // default ticks for log scale
                // Format x values for log scale
                $scope.chart.format[0] = function(xValue) {
                    if (xValue === x[0] || xValue === x[x.length - 1]){
                        // print exact value for x axis limits
                        return $scope.chart.xNumericFormat(xValue);
                    } else {
                        // print the ticks
                        if (!["1", "2", "4"].includes(d3.format(".0e")(xValue)[0])){
                            // only print ticks 1, 2, and 4
                            return "";
                        }
                        return $scope.chart.xNumericFormat(xValue);
                    }
                }
            } else {
                $scope.chart.xTicks = false; // default ticks for linear scale
                $scope.chart.xScale = d3.scale.linear();
                $scope.chart.format[0] = $scope.chart.xNumericFormat; // simple formatting for linear scale
            }
            $scope.chart.xScale.domain([x[0], x[x.length - 1]]);
        }
    }
    $scope.$watch('uiState.chartLogScale', updateChartXScale);

    $scope.exportGridSearchData = function(){
        if (!$scope.gridCells) return;
        ExportUtils.exportUIData($scope, {
            name : "Hyperparameter search data for model: " + $scope.modelData.userMeta.name,
            columns : $scope.paramColumns.map(_ => ({name: _, type: 'string'}))
                    .concat($scope.fieldList.map(_ => ({name: _.name, type: 'double'}))),
            data : $scope.gridData
        }, "Export hyperparameter search data");
    };

    $scope.hasOptimizedNEstimators = () => {
        const modelingParameters = $scope.mlTasksContext.model.modeling;
        const { algorithm } = modelingParameters;

        if (algorithm.startsWith("XGBOOST")) {
            return modelingParameters.xgboost_grid.enable_early_stopping;
        }

        if (algorithm === "LIGHTGBM_CLASSIFICATION") {
            return modelingParameters.lightgbm_classification_grid.early_stopping;
        }

        if (algorithm === "LIGHTGBM_REGRESSION") {
            return modelingParameters.lightgbm_regression_grid.early_stopping;
        }

        return false;
    };

    function computeHyperparametersPairwiseDependencies() {
        const scoreIdx = $scope.paramColumns.length;

        // Compute indices of all combinations of hyperparams
        const pairsIndices = [].concat(
            ...$scope.paramColumns.map((_, idx) =>
                $scope.paramColumns.slice(idx + 1).map((_, idx2) => [idx, idx + 1 + idx2])
            )
        );

        // Fill pairwise dependencies data with gridData
        let dependencies = Array.from(pairsIndices, function (pairIndices) {
            const [param1Idx, param2Idx] = pairIndices;

            // skip null scores
            const validGridData = filterInvalidScoresFromGridData();

            // Accumulate scores by unique (x, y) values
            const scoresByXY = validGridData.reduce(function (acc, data) {

                const [x, y, score] = [data[param1Idx], data[param2Idx], data[scoreIdx]];
                // Ignore undefined values
                if (x !== '' && y !== '') {
                    // Convert (x, y) pair to string for indexing
                    let xyKey = JSON.stringify([x, y]);
                    if (!(xyKey in acc)) {
                        acc[xyKey] = [];
                    }
                    acc[xyKey].push(score);
                }
                return acc;
            }, {});

            // Average scores by unique (x, y) value
            for (let xyPair in scoresByXY) {
                scoresByXY[xyPair] = scoresByXY[xyPair].reduce((a, b) => a + b, 0) / scoresByXY[xyPair].length;
            }

            // Create separate x, y, score arrays
            const x = Object.keys(scoresByXY).map((_) => JSON.parse(_)[0]);
            const y = Object.keys(scoresByXY).map((_) => JSON.parse(_)[1]);
            const score = Object.values(scoresByXY);

            // Ignore dependencies where one of the parameters has only one value
            // (equivalent to 1D plot)
            const xUnique = new Set(x);
            const yUnique = new Set(y);
            if (xUnique.size === 1 || yUnique.size === 1) {
                return {};
            }

            return {
                xLabel: $scope.paramColumns[param1Idx],
                yLabel: $scope.paramColumns[param2Idx],
                x: x,
                xCategorical: !x.every((_) => typeof _ === 'number'),
                y: y,
                yCategorical: !y.every((_) => typeof _ === 'number'),
                score: score,
            };
        });

        // Clean empty dependencies
        dependencies = dependencies.filter((dependency) => !angular.equals(dependency, {}));

        return dependencies;
    }

    function filterInvalidScoresFromGridData() {
        const scoreIdx = $scope.paramColumns.length;
        return $scope.gridData.filter(x => x[scoreIdx] !== '');
    }
});

app.filter("modelImportantParamName", function(){
    const dict = {
        depth: "Depth",
        min_samples: "Min samples",
        trees: "Trees",
        penalty: "Penalty",
        max_depth: "Max depth",
        criterion: "Split criterion",
        alpha: "Alpha",
        lambda: "Lambda",
        epsilon: "Epsilon",
        gamma: "Gamma",
        C: "C",
        kernel: "Kernel",
        loss: "Loss",
        k: "K",
        distance_weighting: "Distance weighting",
        layer_sizes: "Layer size",
        max_iters: "Max iterations",
        hidden_layers: "Hidden layers",
        activation: "Activation",
        dropout:  "Dropout",
        l1: "L1",
        l2: "L2",
        strategy: "Strategy",
        smoothing: "Smoothing",
        learning_rate: "Learning rate",
        features: "Features",
        solver:  "Solver",
        epochs: "Epochs",
        boosting_type: "Boosting type",
        n_estimators: "Nb. estimators",
        num_leaves: "Nb. leaves",
        units: "Units",
    };

    return function(input) {
        if (input && input in dict) return dict[input];
        return input;
    }
})


app.controller('HyperparametersPairwiseDependenciesController', ['$scope', 'MLChartsCommon', function ($scope, MLChartsCommon) {
    function groupByCategoryReducer(categories) {
        let groupByCategory = function (acc, value, index) {
            let category = categories[index];
            acc[category] = acc[category] || [];
            acc[category].push(value);
            return acc;
        };
        return groupByCategory;
    }

    function groupDependenciesByY(acc, dependency) {
        if (
            (acc.has(dependency.xLabel) && !(!dependency.xCategorical && dependency.yCategorical)) ||
            (dependency.xCategorical && !dependency.yCategorical)
        ) {
            // xLabel already present in acc and not only y is categorical,
            // or only x is categorical => We use y as x axis
            acc.set(dependency.xLabel, acc.get(dependency.xLabel) || []);
            acc.get(dependency.xLabel).push({
                x: dependency.y,
                xLabel: dependency.yLabel,
                xCategorical: dependency.yCategorical,
                y: dependency.x,
                yLabel: dependency.xLabel,
                yCategorical: dependency.xCategorical,
                score: dependency.score,
            });
        } else {
            // Otherwise we use x as x axis
            acc.set(dependency.yLabel, acc.get(dependency.yLabel) || []);
            acc.get(dependency.yLabel).push(dependency);
        }
        return acc;
    }

    function setXScaleMultiline(x) {
        let xScale, xNumericFormat, xScaleFormat, xTicks;

        const min = Math.min(...x);
        const max = Math.max(...x);

        xNumericFormat = MLChartsCommon.makeAxisNumericFormatter(min, max, 3, 1);

        const logScale = min > 0 && max / min > 10;
        if (logScale) {
            xScale = d3.scale.log().domain([min, max]);
            xTicks = [];
            if (max / min >= 1e5) {
                // If ratio is too high, only print major vertical lines
                for (let tickValue of xScale.ticks()) {
                    if (tickValue / 10 ** Math.floor(Math.log10(tickValue)) === 1) {
                        xTicks.push(tickValue);
                    }
                }
                // Show maximum ~4 major ticks
                xTicks = xTicks.filter((_, idx) => {
                    if (idx % Math.floor(xTicks.length / 4) === 0) return true;
                });
            } else {
                xTicks = false; // Default log ticks
            }
            xScaleFormat = function (xValue) {
                if (xValue === x[0] || xValue === x[x.length - 1]) {
                    // print exact value for x axis limits
                    return xNumericFormat(xValue);
                } else {
                    // print the ticks
                    if (d3.format('.0e')(xValue)[0] !== '1') {
                        // only print major ticks (does not hide the vertical line for minor ticks)
                        return '';
                    }
                    return xNumericFormat(xValue);
                }
            };
        } else {
            xTicks = false; // Default linear ticks
            xScale = d3.scale.linear().domain([min, max]);
            xScaleFormat = xNumericFormat;
        }

        return { xScale, xScaleFormat, xTicks };
    }

    $scope.$watch('pairwiseDependencies', (pairwiseDependencies) => {
        let dependenciesByY = pairwiseDependencies.reduce(groupDependenciesByY, new Map());
        let columnLabelCounts = {};
        let plotsDataByYDict = {};
        for (const [yLabel, dependencies] of dependenciesByY) {
            for (const dependency of dependencies) {
                let plotData = {};
                if (dependency.xCategorical && dependency.yCategorical) {
                    plotData.plotType = 'CATEGORIES-HEATMAP';
                    plotData.x = dependency.x;
                    plotData.y = dependency.y;
                    plotData.xLabel = dependency.xLabel;
                    plotData.yLabel = dependency.yLabel;
                    plotData.score = dependency.score;
                } else if (dependency.xCategorical || dependency.yCategorical) {
                    plotData.plotType = 'MULTILINE';
                    plotData.legend = [...new Set(dependency.y)];
                    plotData.legendLabel = dependency.yLabel;
                    plotData.colors = $scope.colors.slice(0, plotData.legend.length);

                    const { xScale, xScaleFormat, xTicks } = setXScaleMultiline(dependency.x);
                    plotData.xScale = xScale;
                    plotData.xTicks = xTicks;

                    const categories = dependency.y;
                    plotData.x = Object.values(dependency.x.reduce(groupByCategoryReducer(categories), {}));
                    plotData.xLabel = dependency.xLabel;
                    plotData.score = Object.values(dependency.score.reduce(groupByCategoryReducer(categories), {}));

                    // Sort by ascending order for x
                    plotData.x.forEach(function (_, idx) {
                        let combinedArray = [];
                        for (let j = 0; j < plotData.x[idx].length; j++) {
                            combinedArray.push({ x: plotData.x[idx][j], score: plotData.score[idx][j] });
                        }

                        combinedArray.sort(function (a, b) {
                            return a.x < b.x ? -1 : a.x == b.x ? 0 : 1;
                        });

                        for (let k = 0; k < combinedArray.length; k++) {
                            plotData.x[idx][k] = combinedArray[k].x;
                            plotData.score[idx][k] = combinedArray[k].score;
                        }
                    });

                    /* If the scores are between 0 and 1 and the difference is big, force the scale to [0, 1].
                     * Else, use a scale that goes just a bit beyond the boundaries so that the data does not
                     * align to the angles of the chart (because it looks a bit ugly)
                     */
                    let minScore = Math.min(...dependency.score);
                    let maxScore = Math.max(...dependency.score);
                    plotData.scale = (function () {
                        if (minScore >= 0 && maxScore <= 1) {
                            if (maxScore - minScore >= 0.2) {
                                return [0, 1];
                            } else {
                                const factor = Math.abs(maxScore - minScore) * 0.1;
                                const furtherMax = maxScore > 0 ? maxScore + factor : maxScore - factor;
                                const furtherMin = minScore > 0 ? minScore - factor : minScore + factor;
                                return [Math.max(0, furtherMin), Math.min(1, furtherMax)];
                            }
                        } else {
                            const furtherMax = maxScore > 0 ? maxScore * 1.05 : maxScore * 0.95;
                            const furtherMin = minScore > 0 ? minScore * 0.95 : minScore * 1.05;
                            return [furtherMin, furtherMax];
                        }
                    })();

                    let yScaleFormat = MLChartsCommon.makeAxisNumericFormatter(minScore, maxScore, 3, 1);

                    plotData.format = [xScaleFormat, yScaleFormat];

                    plotData.svgCallback = function (svg, scope) {
                        scope.chart.yAxis.axisLabelDistance(-15);
                        scope.chart.interpolate(gaussianSmooth);

                        let tooltipNumericFormat = MLChartsCommon.makeTooltipNumericFormatter(3, 4);
                        scope.chart.tooltip.contentGenerator(function (d) {
                            return `
                                <table class="mlchart-tooltip__table">
                                    <tr>
                                        <td class="mlchart-tooltip__label">Score ${$scope.uiState.scoreMetric}</td>
                                        <td class="mlchart-tooltip__value">${tooltipNumericFormat(
                                            d.series[0].value
                                        )}</td>
                                    </tr>
                                    <tr>
                                        <td class="mlchart-tooltip__label">${plotData.xLabel}</td>
                                        <td class="mlchart-tooltip__value">${tooltipNumericFormat(d.value)}</td>
                                    </tr>
                                    <tr>
                                        <td class="mlchart-tooltip__label">${plotData.legendLabel}</td>
                                        <td class="mlchart-tooltip__value">${d.series[0].key}</td>
                                    </tr>
                                </table>`;
                        });

                        // Redraw after axis modification
                        svg.datum(scope.theData).call(scope.chart);
                    };
                } else {
                    plotData.plotType = 'CONTOUR';
                    plotData.x = dependency.x;
                    plotData.y = dependency.y;
                    plotData.xLabel = dependency.xLabel;
                    plotData.yLabel = dependency.yLabel;
                    plotData.score = dependency.score;
                }
                columnLabelCounts[plotData.xLabel] = (columnLabelCounts[plotData.xLabel] || 0) + 1;
                plotsDataByYDict[yLabel] = plotsDataByYDict[yLabel] || [];
                plotsDataByYDict[yLabel].push(plotData);
            }
        }

        // We sort the columns and rows so that we are the closest possible to a triangular matrix:
        // - Rows sorted by descending length
        // - Columns sorted by descending number of rows where they are present
        // We also fill empty cells with 'EMPTY' plots not to break the tabular shape in case it is not
        // possible to obtain a triangular matrix
        $scope.orderedColumns = Object.keys(columnLabelCounts).sort(
            (a, b) => -columnLabelCounts[a] + columnLabelCounts[b]
        );
        $scope.plotsDataByY = Object.entries(plotsDataByYDict)
            .sort((a, b) => -a[1].length + b[1].length)
            .map(([yLabel, plotsData]) => {
                const newPlotsData = $scope.orderedColumns.map((xLabel) => {
                    let plotsWithXLabel = plotsData.filter((plotData) => {
                        return plotData.xLabel === xLabel;
                    });
                    return plotsWithXLabel.length ? plotsWithXLabel[0] : { plotType: 'EMPTY' };
                });
                return [yLabel, newPlotsData];
            });

        // Compute plot width so that they all approximately fit on a 1200px wide screen
        $scope.maxRowLength = Math.max(...$scope.plotsDataByY.map((_) => _[1].length));
        $scope.plotWidth = Math.max(
            Math.min(
                Math.floor(900 / $scope.maxRowLength), // Aim at a total width of ~900px
                350 // Maximum width is 350
            ),
            250 // Minimum width is 250
        );
    });

    // Gaussian kernel interpolation function for cat/num plots
    function gaussianSmooth(points) {
        let gaussian = function (a, b, bandwidth) {
            return Math.exp(-Math.pow(a - b, 2) / (2 * bandwidth * bandwidth));
        };

        if (points.length <= 2) {
            return points.join('L'); // Linear interpolation for <= 2 points
        }

        const x = points.map((_) => _[0]);
        const bandwidth = (Math.max(...x) - Math.min(...x)) / Math.min(points.length, 7);

        // The interpolated line is defined by 50 x values
        return MLChartsCommon.linspace(Math.min(...x), Math.max(...x), 50)
            .map(function (xValue) {
                var numerator = d3.sum(points, (point) => gaussian(point[0], xValue, bandwidth) * point[1]);
                var denominator = d3.sum(points, (point) => gaussian(point[0], xValue, bandwidth));
                return [xValue, numerator / denominator];
            })
            .join('L');
    }
}]);

app.component('reportPaneTitleBar', {
    restrict: 'E',
    transclude: true,
    template: `
        <div class="model-report-title" ng-if="showTitle()" ng-class="{'mbot0': noBottomMargin}">
            <h2 class="model-report-title__tab-name">{{ $ctrl.paneTitle }}</h2>
            <post-train-weights-explanation sample-weight-variable="$ctrl.sampleWeightVariable"/>
            <post-train-inverse-propensity-weights-explanation ng-if="$ctrl.isInversePropensityWeighted"/>
            <div class="model-report-title__content" ng-transclude></div>
        </div>`,
    bindings: { paneTitle: '@', sampleWeightVariable: '<', isInversePropensityWeighted: '<' },
    controller: function($scope, $attrs, $transclude, $stateParams) {
        this.$onInit = () => {
            $transclude(content => $scope.hasTranscluded = !!content.length);
            $scope.showTitle = () => $scope.hasTranscluded || this.sampleWeightVariable !== undefined || this.isInversePropensityWeighted || !$stateParams.dashboardId; // it's ok to display on insights, but not on dashboards
            // By default, the title has a bottom margin, but it can be overridden with this prop.
            $scope.noBottomMargin = $attrs.noBottomMargin !== undefined;
        };
    }
});

app.component('postTrainWeightsExplanation', {
    template: `
        <span ng-if="$ctrl.sampleWeightVariable !== undefined"
              class="post-train-weights-explanation"
              toggle="tooltip-bottom"
              container="body"
              title="Metrics and aggregates (averages, quantiles etc.) are computed with the variable '{{$ctrl.sampleWeightVariable}}' as sample weight, unless explicitly stated otherwise.">
            <i class="post-train-weights-explanation__icon icon-dku-weight"></i>
            <span class="post-train-weights-explanation__text">{{$ctrl.sampleWeightVariable}}</span>
        </span>`,
    bindings: { sampleWeightVariable: '<' }
});

app.component('postTrainInversePropensityWeightsExplanation', {
    template: `
        <span
              class="post-train-weights-explanation"
              toggle="tooltip-bottom"
              container="body"
              title="Metrics and aggregates (averages, quantiles etc.) are computed with inverse propensity weights.">
            <i class="post-train-weights-explanation__icon icon-dku-weight"></i>
            <span class="post-train-weights-explanation__text">Inverse Propensity Weighting</span>
        </span>`,
});

})();
