/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.azurebfs;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.AccessDeniedException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import javax.annotation.Nullable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.classification.VisibleForTesting;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.CreateFlag;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileAlreadyExistsException;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocatedFileStatus;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.PathIOException;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.XAttrSetFlag;
import org.apache.hadoop.fs.azurebfs.AbfsConfiguration;
import org.apache.hadoop.fs.azurebfs.AbfsCountersImpl;
import org.apache.hadoop.fs.azurebfs.AbfsStatistic;
import org.apache.hadoop.fs.azurebfs.AzureBlobFileSystemStore;
import org.apache.hadoop.fs.azurebfs.commit.ResilientCommitByRename;
import org.apache.hadoop.fs.azurebfs.constants.FSOperationType;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AbfsRestOperationException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.AzureBlobFileSystemException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.FileSystemOperationUnhandledException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriAuthorityException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.InvalidUriException;
import org.apache.hadoop.fs.azurebfs.contracts.exceptions.SASTokenProviderException;
import org.apache.hadoop.fs.azurebfs.contracts.services.AzureServiceErrorCode;
import org.apache.hadoop.fs.azurebfs.security.AbfsDelegationTokenManager;
import org.apache.hadoop.fs.azurebfs.services.AbfsClient;
import org.apache.hadoop.fs.azurebfs.services.AbfsCounters;
import org.apache.hadoop.fs.azurebfs.services.AbfsInputStream;
import org.apache.hadoop.fs.azurebfs.services.AbfsListStatusRemoteIterator;
import org.apache.hadoop.fs.azurebfs.services.AbfsLocatedFileStatus;
import org.apache.hadoop.fs.azurebfs.utils.Listener;
import org.apache.hadoop.fs.azurebfs.utils.TracingContext;
import org.apache.hadoop.fs.azurebfs.utils.TracingHeaderFormat;
import org.apache.hadoop.fs.impl.AbstractFSBuilderImpl;
import org.apache.hadoop.fs.impl.BackReference;
import org.apache.hadoop.fs.impl.OpenFileParameters;
import org.apache.hadoop.fs.impl.PathCapabilitiesSupport;
import org.apache.hadoop.fs.permission.AclEntry;
import org.apache.hadoop.fs.permission.AclStatus;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.fs.statistics.IOStatistics;
import org.apache.hadoop.fs.statistics.IOStatisticsLogging;
import org.apache.hadoop.fs.statistics.IOStatisticsSource;
import org.apache.hadoop.fs.store.DataBlocks;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.security.AccessControlException;
import org.apache.hadoop.security.ProviderUtils;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.security.token.Token;
import org.apache.hadoop.util.DurationInfo;
import org.apache.hadoop.util.LambdaUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.RateLimiting;
import org.apache.hadoop.util.RateLimitingFactory;
import org.apache.hadoop.util.functional.RemoteIterators;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceStability.Evolving
public class AzureBlobFileSystem
extends FileSystem
implements IOStatisticsSource {
    public static final Logger LOG = LoggerFactory.getLogger(AzureBlobFileSystem.class);
    private URI uri;
    private Path workingDir;
    private AzureBlobFileSystemStore abfsStore;
    private boolean isClosed;
    private final String fileSystemId = UUID.randomUUID().toString();
    private boolean delegationTokenEnabled = false;
    private AbfsDelegationTokenManager delegationTokenManager;
    private AbfsCounters abfsCounters;
    private String clientCorrelationId;
    private TracingHeaderFormat tracingHeaderFormat;
    private Listener listener;
    private String blockOutputBuffer;
    private DataBlocks.BlockFactory blockFactory;
    private int blockOutputActiveBlocks;
    private RateLimiting rateLimiting;
    private URI fullPathUri;

    public void initialize(URI uri, Configuration configuration) throws IOException {
        configuration = ProviderUtils.excludeIncompatibleCredentialProviders((Configuration)configuration, AzureBlobFileSystem.class);
        uri = this.ensureAuthority(uri, configuration);
        super.initialize(uri, configuration);
        this.setConf(configuration);
        LOG.debug("Initializing AzureBlobFileSystem for {}", (Object)uri);
        this.fullPathUri = uri;
        this.uri = URI.create(uri.getScheme() + "://" + uri.getAuthority());
        this.abfsCounters = new AbfsCountersImpl(uri);
        this.blockOutputBuffer = configuration.getTrimmed("fs.azure.data.blocks.buffer", "bytebuffer");
        this.blockFactory = DataBlocks.createFactory((String)"fs.azure.buffer.dir", (Configuration)configuration, (String)this.blockOutputBuffer);
        this.blockOutputActiveBlocks = configuration.getInt("fs.azure.block.upload.active.blocks", 20);
        if (this.blockOutputActiveBlocks < 1) {
            this.blockOutputActiveBlocks = 1;
        }
        AzureBlobFileSystemStore.AzureBlobFileSystemStoreBuilder systemStoreBuilder = new AzureBlobFileSystemStore.AzureBlobFileSystemStoreBuilder().withUri(uri).withSecureScheme(this.isSecureScheme()).withConfiguration(configuration).withAbfsCounters(this.abfsCounters).withBlockFactory(this.blockFactory).withBlockOutputActiveBlocks(this.blockOutputActiveBlocks).withBackReference(new BackReference((Object)this)).build();
        this.abfsStore = new AzureBlobFileSystemStore(systemStoreBuilder);
        LOG.info("AzureBlobFileSystemStore init complete");
        AbfsConfiguration abfsConfiguration = this.abfsStore.getAbfsConfiguration();
        this.clientCorrelationId = TracingContext.validateClientCorrelationID(abfsConfiguration.getClientCorrelationId());
        this.tracingHeaderFormat = abfsConfiguration.getTracingHeaderFormat();
        this.setWorkingDirectory(this.getHomeDirectory());
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.CREATE_FILESYSTEM, this.tracingHeaderFormat, this.listener);
        if (abfsConfiguration.getCreateRemoteFileSystemDuringInitialization() && this.tryGetFileStatus(new Path("/"), tracingContext) == null) {
            try {
                this.createFileSystem(tracingContext);
            }
            catch (AzureBlobFileSystemException ex) {
                AzureBlobFileSystem.checkException(null, ex, AzureServiceErrorCode.FILE_SYSTEM_ALREADY_EXISTS);
            }
        }
        if ((this.isEncryptionContextCPK(abfsConfiguration) || this.isGlobalKeyCPK(abfsConfiguration)) && !this.getIsNamespaceEnabled(new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.CREATE_FILESYSTEM, this.tracingHeaderFormat, this.listener))) {
            this.close();
            throw new PathIOException(uri.getPath(), "Non hierarchical-namespace account can not have configs enabled for Customer Provided Keys. Following configs can not be given with non-hierarchical-namespace account:fs.azure.encryption.context.provider.type, fs.azure.encryption.encoded.client-provided-key, fs.azure.encryption.encoded.client-provided-key-sha");
        }
        LOG.trace("Initiate check for delegation token manager");
        if (UserGroupInformation.isSecurityEnabled()) {
            this.delegationTokenEnabled = abfsConfiguration.isDelegationTokenManagerEnabled();
            if (this.delegationTokenEnabled) {
                LOG.debug("Initializing DelegationTokenManager for {}", (Object)uri);
                this.delegationTokenManager = abfsConfiguration.getDelegationTokenManager();
                this.delegationTokenManager.bind(this.getUri(), configuration);
                LOG.debug("Created DelegationTokenManager {}", (Object)this.delegationTokenManager);
            }
        }
        this.rateLimiting = RateLimitingFactory.create((int)abfsConfiguration.getRateLimit());
        LOG.debug("Initializing AzureBlobFileSystem for {} complete", (Object)uri);
    }

    private boolean isGlobalKeyCPK(AbfsConfiguration abfsConfiguration) {
        return StringUtils.isNotEmpty((CharSequence)abfsConfiguration.getEncodedClientProvidedEncryptionKey());
    }

    private boolean isEncryptionContextCPK(AbfsConfiguration abfsConfiguration) {
        return abfsConfiguration.createEncryptionContextProvider() != null;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("AzureBlobFileSystem{");
        sb.append("uri=").append(this.fullPathUri);
        sb.append(", user='").append(this.abfsStore.getUser()).append('\'');
        sb.append(", primaryUserGroup='").append(this.abfsStore.getPrimaryGroup()).append('\'');
        sb.append("[fs.azure.capability.readahead.safe]");
        sb.append('}');
        return sb.toString();
    }

    public boolean isSecureScheme() {
        return false;
    }

    public URI getUri() {
        return this.uri;
    }

    public void registerListener(Listener listener1) {
        this.listener = listener1;
    }

    public FSDataInputStream open(Path path, int bufferSize) throws IOException {
        LOG.debug("AzureBlobFileSystem.open path: {} bufferSize: {}", (Object)path, (Object)bufferSize);
        return this.open(path, Optional.empty());
    }

    private FSDataInputStream open(Path path, Optional<OpenFileParameters> parameters) throws IOException {
        this.statIncrement(AbfsStatistic.CALL_OPEN);
        Path qualifiedPath = this.makeQualified(path);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.OPEN, this.tracingHeaderFormat, this.listener);
            AbfsInputStream inputStream = this.getAbfsStore().openFileForRead(qualifiedPath, parameters, this.statistics, tracingContext);
            return new FSDataInputStream((InputStream)((Object)inputStream));
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    protected CompletableFuture<FSDataInputStream> openFileWithOptions(Path path, OpenFileParameters parameters) throws IOException {
        LOG.debug("AzureBlobFileSystem.openFileWithOptions path: {}", (Object)path);
        AbstractFSBuilderImpl.rejectUnknownMandatoryKeys((Set)parameters.getMandatoryKeys(), (Collection)Options.OpenFileOptions.FS_OPTION_OPENFILE_STANDARD_OPTIONS, (String)("for " + path));
        return LambdaUtils.eval(new CompletableFuture(), () -> this.open(path, Optional.of(parameters)));
    }

    public FSDataOutputStream create(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        LOG.debug("AzureBlobFileSystem.create path: {} permission: {} overwrite: {} bufferSize: {}", new Object[]{f, permission, overwrite, blockSize});
        this.statIncrement(AbfsStatistic.CALL_CREATE);
        this.trailingPeriodCheck(f);
        if (f.isRoot()) {
            throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "Cannot create file over root path", null);
        }
        Path qualifiedPath = this.makeQualified(f);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.CREATE, overwrite, this.tracingHeaderFormat, this.listener);
            OutputStream outputStream = this.getAbfsStore().createFile(qualifiedPath, this.statistics, overwrite, permission == null ? FsPermission.getFileDefault() : permission, FsPermission.getUMask((Configuration)this.getConf()), tracingContext);
            this.statIncrement(AbfsStatistic.FILES_CREATED);
            return new FSDataOutputStream(outputStream, this.statistics);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        TracingContext tracingContext;
        this.statIncrement(AbfsStatistic.CALL_CREATE_NON_RECURSIVE);
        if (f.isRoot()) {
            throw new AbfsRestOperationException(409, AzureServiceErrorCode.PATH_CONFLICT.getErrorCode(), "Cannot create file over root path", null);
        }
        Path parent = f.getParent();
        FileStatus parentFileStatus = this.tryGetFileStatus(parent, tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.CREATE_NON_RECURSIVE, this.tracingHeaderFormat, this.listener));
        if (parentFileStatus == null) {
            throw new FileNotFoundException("Cannot create file " + f.getName() + " because parent folder does not exist.");
        }
        return this.create(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream createNonRecursive(Path f, FsPermission permission, EnumSet<CreateFlag> flags, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        EnumSet<CreateFlag> createflags = EnumSet.of(CreateFlag.CREATE, CreateFlag.OVERWRITE);
        boolean overwrite = flags.containsAll(createflags);
        return this.createNonRecursive(f, permission, overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream createNonRecursive(Path f, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        return this.createNonRecursive(f, FsPermission.getFileDefault(), overwrite, bufferSize, replication, blockSize, progress);
    }

    public FSDataOutputStream append(Path f, int bufferSize, Progressable progress) throws IOException {
        LOG.debug("AzureBlobFileSystem.append path: {} bufferSize: {}", (Object)f.toString(), (Object)bufferSize);
        this.statIncrement(AbfsStatistic.CALL_APPEND);
        Path qualifiedPath = this.makeQualified(f);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.APPEND, this.tracingHeaderFormat, this.listener);
            OutputStream outputStream = this.abfsStore.openFileForWrite(qualifiedPath, this.statistics, false, tracingContext);
            return new FSDataOutputStream(outputStream, this.statistics);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public boolean rename(Path src, Path dst) throws IOException {
        LOG.debug("AzureBlobFileSystem.rename src: {} dst: {}", (Object)src, (Object)dst);
        this.statIncrement(AbfsStatistic.CALL_RENAME);
        this.trailingPeriodCheck(dst);
        Path parentFolder = src.getParent();
        if (parentFolder == null) {
            return false;
        }
        Path qualifiedSrcPath = this.makeQualified(src);
        Path qualifiedDstPath = this.makeQualified(dst);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.RENAME, true, this.tracingHeaderFormat, this.listener);
        if (this.makeQualified(parentFolder).equals((Object)qualifiedDstPath)) {
            return this.tryGetFileStatus(qualifiedSrcPath, tracingContext) != null;
        }
        FileStatus dstFileStatus = null;
        if (qualifiedSrcPath.equals((Object)qualifiedDstPath)) {
            dstFileStatus = this.tryGetFileStatus(qualifiedDstPath, tracingContext);
            if (dstFileStatus == null) {
                return false;
            }
            return !dstFileStatus.isDirectory();
        }
        if (!this.getIsNamespaceEnabled(tracingContext) && dstFileStatus == null) {
            dstFileStatus = this.tryGetFileStatus(qualifiedDstPath, tracingContext);
        }
        try {
            String sourceFileName = src.getName();
            Path adjustedDst = dst;
            if (dstFileStatus != null) {
                if (!dstFileStatus.isDirectory()) {
                    return qualifiedSrcPath.equals((Object)qualifiedDstPath);
                }
                adjustedDst = new Path(dst, sourceFileName);
            }
            qualifiedDstPath = this.makeQualified(adjustedDst);
            this.abfsStore.rename(qualifiedSrcPath, qualifiedDstPath, tracingContext, null);
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            LOG.debug("Rename operation failed. ", (Throwable)ex);
            AzureBlobFileSystem.checkException(src, ex, AzureServiceErrorCode.PATH_ALREADY_EXISTS, AzureServiceErrorCode.INVALID_RENAME_SOURCE_PATH, AzureServiceErrorCode.SOURCE_PATH_NOT_FOUND, AzureServiceErrorCode.INVALID_SOURCE_OR_DESTINATION_RESOURCE_TYPE, AzureServiceErrorCode.RENAME_DESTINATION_PARENT_PATH_NOT_FOUND, AzureServiceErrorCode.INTERNAL_OPERATION_ABORT);
            return false;
        }
    }

    @InterfaceAudience.Private
    public ResilientCommitByRename createResilientCommitSupport(Path path) throws IOException {
        if (!this.hasPathCapability(path, "fs.capability.etags.preserved.in.rename")) {
            throw new UnsupportedOperationException("Resilient commit support not available for " + path);
        }
        return new ResilientCommitByRenameImpl();
    }

    public boolean delete(Path f, boolean recursive) throws IOException {
        LOG.debug("AzureBlobFileSystem.delete path: {} recursive: {}", (Object)f.toString(), (Object)recursive);
        this.statIncrement(AbfsStatistic.CALL_DELETE);
        Path qualifiedPath = this.makeQualified(f);
        if (f.isRoot()) {
            if (!recursive) {
                return false;
            }
            return this.deleteRoot();
        }
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.DELETE, this.tracingHeaderFormat, this.listener);
            this.abfsStore.delete(qualifiedPath, recursive, tracingContext);
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, AzureServiceErrorCode.PATH_NOT_FOUND);
            return false;
        }
    }

    public FileStatus[] listStatus(Path f) throws IOException {
        LOG.debug("AzureBlobFileSystem.listStatus path: {}", (Object)f.toString());
        this.statIncrement(AbfsStatistic.CALL_LIST_STATUS);
        Path qualifiedPath = this.makeQualified(f);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.LISTSTATUS, true, this.tracingHeaderFormat, this.listener);
            FileStatus[] result = this.abfsStore.listStatus(qualifiedPath, tracingContext);
            return result;
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    private void statIncrement(AbfsStatistic statistic) {
        this.incrementStatistic(statistic);
    }

    private void incrementStatistic(AbfsStatistic statistic) {
        if (this.abfsCounters != null) {
            this.abfsCounters.incrementCounter(statistic, 1L);
        }
    }

    private void trailingPeriodCheck(Path path) throws IllegalArgumentException {
        String pathToString;
        while (!path.isRoot() && (pathToString = path.toString()).length() != 0) {
            if (pathToString.charAt(pathToString.length() - 1) == '.') {
                throw new IllegalArgumentException("ABFS does not allow files or directories to end with a dot.");
            }
            path = path.getParent();
        }
    }

    public boolean mkdirs(Path f, FsPermission permission) throws IOException {
        LOG.debug("AzureBlobFileSystem.mkdirs path: {} permissions: {}", (Object)f, (Object)permission);
        this.statIncrement(AbfsStatistic.CALL_MKDIRS);
        this.trailingPeriodCheck(f);
        Path parentFolder = f.getParent();
        if (parentFolder == null) {
            return true;
        }
        Path qualifiedPath = this.makeQualified(f);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.MKDIR, false, this.tracingHeaderFormat, this.listener);
            this.abfsStore.createDirectory(qualifiedPath, permission == null ? FsPermission.getDirDefault() : permission, FsPermission.getUMask((Configuration)this.getConf()), tracingContext);
            this.statIncrement(AbfsStatistic.DIRECTORIES_CREATED);
            return true;
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, new AzureServiceErrorCode[0]);
            return true;
        }
    }

    public synchronized void close() throws IOException {
        if (this.isClosed) {
            return;
        }
        super.close();
        LOG.info("AzureBlobFileSystem.close");
        if (this.getConf() != null) {
            String iostatisticsLoggingLevel = this.getConf().getTrimmed("fs.iostatistics.logging.level", "debug");
            IOStatisticsLogging.logIOStatisticsAtLevel((Logger)LOG, (String)iostatisticsLoggingLevel, (Object)this.getIOStatistics());
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{this.abfsStore, this.delegationTokenManager, this.getAbfsClient()});
        this.isClosed = true;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Closing Abfs: {}", (Object)this.toString());
        }
    }

    public FileStatus getFileStatus(Path f) throws IOException {
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.GET_FILESTATUS, this.tracingHeaderFormat, this.listener);
        return this.getFileStatus(f, tracingContext);
    }

    private FileStatus getFileStatus(Path path, TracingContext tracingContext) throws IOException {
        LOG.debug("AzureBlobFileSystem.getFileStatus path: {}", (Object)path);
        this.statIncrement(AbfsStatistic.CALL_GET_FILE_STATUS);
        Path qualifiedPath = this.makeQualified(path);
        try {
            return this.abfsStore.getFileStatus(qualifiedPath, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public void breakLease(Path f) throws IOException {
        LOG.debug("AzureBlobFileSystem.breakLease path: {}", (Object)f);
        Path qualifiedPath = this.makeQualified(f);
        try (DurationInfo ignored = new DurationInfo(LOG, false, "Break lease for %s", new Object[]{qualifiedPath});){
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.BREAK_LEASE, this.tracingHeaderFormat, this.listener);
            this.abfsStore.breakLease(qualifiedPath, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(f, ex, new AzureServiceErrorCode[0]);
        }
    }

    public Path makeQualified(Path path) {
        if (path != null) {
            String uriPath = path.toUri().getPath();
            path = uriPath.isEmpty() ? path : new Path(uriPath);
        }
        return super.makeQualified(path);
    }

    public Path getWorkingDirectory() {
        return this.workingDir;
    }

    public void setWorkingDirectory(Path newDir) {
        this.workingDir = newDir.isAbsolute() ? newDir : new Path(this.workingDir, newDir);
    }

    public String getScheme() {
        return "abfs";
    }

    public Path getHomeDirectory() {
        return this.makeQualified(new Path("/user/" + this.abfsStore.getUser()));
    }

    public BlockLocation[] getFileBlockLocations(FileStatus file, long start, long len) {
        if (file == null) {
            return null;
        }
        if (start < 0L || len < 0L) {
            throw new IllegalArgumentException("Invalid start or len parameter");
        }
        if (file.getLen() < start) {
            return new BlockLocation[0];
        }
        String blobLocationHost = this.abfsStore.getAbfsConfiguration().getAzureBlockLocationHost();
        String[] name = new String[]{blobLocationHost};
        String[] host = new String[]{blobLocationHost};
        long blockSize = file.getBlockSize();
        if (blockSize <= 0L) {
            throw new IllegalArgumentException("The block size for the given file is not a positive number: " + blockSize);
        }
        int numberOfLocations = (int)(len / blockSize) + (len % blockSize == 0L ? 0 : 1);
        BlockLocation[] locations = new BlockLocation[numberOfLocations];
        for (int i = 0; i < locations.length; ++i) {
            long currentOffset = start + (long)i * blockSize;
            long currentLength = Math.min(blockSize, start + len - currentOffset);
            locations[i] = new BlockLocation(name, host, currentOffset, currentLength);
        }
        return locations;
    }

    protected void finalize() throws Throwable {
        LOG.info("AzureBlobFileSystem finalize() called.");
        this.close();
        super.finalize();
    }

    public String getOwnerUser() {
        return this.abfsStore.getUser();
    }

    public String getOwnerUserPrimaryGroup() {
        return this.abfsStore.getPrimaryGroup();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean deleteRoot() throws IOException {
        LOG.debug("Deleting root content");
        ExecutorService executorService = Executors.newFixedThreadPool(10);
        try {
            FileStatus[] ls = this.listStatus(this.makeQualified(new Path(File.separator)));
            ArrayList<Future<Void>> deleteTasks = new ArrayList<Future<Void>>();
            for (final FileStatus fs : ls) {
                Future<Void> deleteTask = executorService.submit(new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        AzureBlobFileSystem.this.delete(fs.getPath(), fs.isDirectory());
                        if (fs.isDirectory()) {
                            AzureBlobFileSystem.this.statIncrement(AbfsStatistic.DIRECTORIES_DELETED);
                        } else {
                            AzureBlobFileSystem.this.statIncrement(AbfsStatistic.FILES_DELETED);
                        }
                        return null;
                    }
                });
                deleteTasks.add(deleteTask);
            }
            for (final Future future : deleteTasks) {
                this.execute("deleteRoot", new Callable<Void>(){

                    @Override
                    public Void call() throws Exception {
                        future.get();
                        return null;
                    }
                });
            }
        }
        finally {
            executorService.shutdownNow();
        }
        return true;
    }

    public void setOwner(Path path, String owner, String group) throws IOException {
        LOG.debug("AzureBlobFileSystem.setOwner path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.SET_OWNER, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            super.setOwner(path, owner, group);
            return;
        }
        if ((owner == null || owner.isEmpty()) && (group == null || group.isEmpty())) {
            throw new IllegalArgumentException("A valid owner or group must be specified.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.setOwner(qualifiedPath, owner, group, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void setXAttr(Path path, String name, byte[] value, EnumSet<XAttrSetFlag> flag) throws IOException {
        LOG.debug("AzureBlobFileSystem.setXAttr path: {}", (Object)path);
        if (name == null || name.isEmpty() || value == null) {
            throw new IllegalArgumentException("A valid name and value must be specified.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.SET_ATTR, true, this.tracingHeaderFormat, this.listener);
            Hashtable<String, String> properties = this.abfsStore.getPathStatus(qualifiedPath, tracingContext);
            String xAttrName = AzureBlobFileSystem.ensureValidAttributeName(name);
            boolean xAttrExists = properties.containsKey(xAttrName);
            XAttrSetFlag.validate((String)name, (boolean)xAttrExists, flag);
            String xAttrValue = this.abfsStore.decodeAttribute(value);
            properties.put(xAttrName, xAttrValue);
            this.abfsStore.setPathProperties(qualifiedPath, properties, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public byte[] getXAttr(Path path, String name) throws IOException {
        LOG.debug("AzureBlobFileSystem.getXAttr path: {}", (Object)path);
        if (name == null || name.isEmpty()) {
            throw new IllegalArgumentException("A valid name must be specified.");
        }
        Path qualifiedPath = this.makeQualified(path);
        byte[] value = null;
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.GET_ATTR, true, this.tracingHeaderFormat, this.listener);
            Hashtable<String, String> properties = this.abfsStore.getPathStatus(qualifiedPath, tracingContext);
            String xAttrName = AzureBlobFileSystem.ensureValidAttributeName(name);
            if (properties.containsKey(xAttrName)) {
                String xAttrValue = properties.get(xAttrName);
                value = this.abfsStore.encodeAttribute(xAttrValue);
            }
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
        return value;
    }

    private static String ensureValidAttributeName(String attribute) {
        return attribute.replace('.', '_');
    }

    public void setPermission(Path path, FsPermission permission) throws IOException {
        LOG.debug("AzureBlobFileSystem.setPermission path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.SET_PERMISSION, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            super.setPermission(path, permission);
            return;
        }
        if (permission == null) {
            throw new IllegalArgumentException("The permission can't be null");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.setPermission(qualifiedPath, permission, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void modifyAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.modifyAclEntries path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.MODIFY_ACL, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("modifyAclEntries is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.isEmpty()) {
            throw new IllegalArgumentException("The value of the aclSpec parameter is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.modifyAclEntries(qualifiedPath, aclSpec, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeAclEntries(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeAclEntries path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.REMOVE_ACL_ENTRIES, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("removeAclEntries is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.isEmpty()) {
            throw new IllegalArgumentException("The aclSpec argument is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.removeAclEntries(qualifiedPath, aclSpec, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeDefaultAcl(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeDefaultAcl path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.REMOVE_DEFAULT_ACL, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("removeDefaultAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.removeDefaultAcl(qualifiedPath, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void removeAcl(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.removeAcl path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.REMOVE_ACL, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("removeAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.removeAcl(qualifiedPath, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public void setAcl(Path path, List<AclEntry> aclSpec) throws IOException {
        LOG.debug("AzureBlobFileSystem.setAcl path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.SET_ACL, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("setAcl is only supported by storage accounts with the hierarchical namespace enabled.");
        }
        if (aclSpec == null || aclSpec.size() == 0) {
            throw new IllegalArgumentException("The aclSpec argument is invalid.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            this.abfsStore.setAcl(qualifiedPath, aclSpec, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
        }
    }

    public AclStatus getAclStatus(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.getAclStatus path: {}", (Object)path);
        TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.GET_ACL_STATUS, true, this.tracingHeaderFormat, this.listener);
        if (!this.getIsNamespaceEnabled(tracingContext)) {
            throw new UnsupportedOperationException("getAclStatus is only supported by storage account with the hierarchical namespace enabled.");
        }
        Path qualifiedPath = this.makeQualified(path);
        try {
            return this.abfsStore.getAclStatus(qualifiedPath, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(path, ex, new AzureServiceErrorCode[0]);
            return null;
        }
    }

    public void access(Path path, FsAction mode) throws IOException {
        LOG.debug("AzureBlobFileSystem.access path : {}, mode : {}", (Object)path, (Object)mode);
        Path qualifiedPath = this.makeQualified(path);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.ACCESS, this.tracingHeaderFormat, this.listener);
            this.abfsStore.access(qualifiedPath, mode, tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            this.checkCheckAccessException(path, ex);
        }
    }

    public boolean exists(Path f) throws IOException {
        this.statIncrement(AbfsStatistic.CALL_EXIST);
        return super.exists(f);
    }

    public RemoteIterator<FileStatus> listStatusIterator(Path path) throws IOException {
        LOG.debug("AzureBlobFileSystem.listStatusIterator path : {}", (Object)path);
        if (this.abfsStore.getAbfsConfiguration().enableAbfsListIterator()) {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.LISTSTATUS, true, this.tracingHeaderFormat, this.listener);
            AbfsListStatusRemoteIterator abfsLsItr = new AbfsListStatusRemoteIterator(path, this.abfsStore, tracingContext);
            return RemoteIterators.typeCastingRemoteIterator((RemoteIterator)abfsLsItr);
        }
        return super.listStatusIterator(path);
    }

    protected RemoteIterator<LocatedFileStatus> listLocatedStatus(Path path, PathFilter filter) throws FileNotFoundException, IOException {
        LOG.debug("AzureBlobFileSystem.listStatusIterator path : {}", (Object)path);
        RemoteIterator sourceEntries = RemoteIterators.filteringRemoteIterator(this.listStatusIterator(path), st -> filter.accept(st.getPath()));
        return RemoteIterators.mappingRemoteIterator((RemoteIterator)sourceEntries, st -> new AbfsLocatedFileStatus((FileStatus)st, st.isFile() ? this.getFileBlockLocations((FileStatus)st, 0L, st.getLen()) : null));
    }

    private FileStatus tryGetFileStatus(Path f, TracingContext tracingContext) {
        try {
            return this.getFileStatus(f, tracingContext);
        }
        catch (IOException ex) {
            LOG.debug("File not found {}", (Object)f);
            this.statIncrement(AbfsStatistic.ERROR_IGNORED);
            return null;
        }
    }

    private boolean fileSystemExists() throws IOException {
        LOG.debug("AzureBlobFileSystem.fileSystemExists uri: {}", (Object)this.uri);
        try {
            TracingContext tracingContext = new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.TEST_OP, this.tracingHeaderFormat, this.listener);
            this.abfsStore.getFilesystemProperties(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            try {
                AzureBlobFileSystem.checkException(null, ex, new AzureServiceErrorCode[0]);
            }
            catch (FileNotFoundException e) {
                this.statIncrement(AbfsStatistic.ERROR_IGNORED);
                return false;
            }
        }
        return true;
    }

    private void createFileSystem(TracingContext tracingContext) throws IOException {
        LOG.debug("AzureBlobFileSystem.createFileSystem uri: {}", (Object)this.uri);
        try {
            this.abfsStore.createFilesystem(tracingContext);
        }
        catch (AzureBlobFileSystemException ex) {
            AzureBlobFileSystem.checkException(null, ex, new AzureServiceErrorCode[0]);
        }
    }

    private URI ensureAuthority(URI uri, Configuration conf) {
        URI defaultUri;
        Preconditions.checkNotNull((Object)uri, (Object)"uri");
        if (uri.getAuthority() == null && (defaultUri = FileSystem.getDefaultUri((Configuration)conf)) != null && this.isAbfsScheme(defaultUri.getScheme())) {
            try {
                uri = new URI(uri.getScheme(), defaultUri.getAuthority(), uri.getPath(), uri.getQuery(), uri.getFragment());
            }
            catch (URISyntaxException e) {
                throw new IllegalArgumentException(new InvalidUriException(uri.toString()));
            }
        }
        if (uri.getAuthority() == null) {
            throw new IllegalArgumentException(new InvalidUriAuthorityException(uri.toString()));
        }
        return uri;
    }

    private boolean isAbfsScheme(String scheme) {
        if (scheme == null) {
            return false;
        }
        return scheme.equals("abfs") || scheme.equals("abfss");
    }

    @VisibleForTesting
    <T> FileSystemOperation<T> execute(String scopeDescription, Callable<T> callableFileOperation) throws IOException {
        return this.execute(scopeDescription, callableFileOperation, null);
    }

    @VisibleForTesting
    <T> FileSystemOperation<T> execute(String scopeDescription, Callable<T> callableFileOperation, T defaultResultValue) throws IOException {
        try {
            T executionResult = callableFileOperation.call();
            return new FileSystemOperation<T>(executionResult, null);
        }
        catch (AbfsRestOperationException abfsRestOperationException) {
            return new FileSystemOperation<T>(defaultResultValue, abfsRestOperationException);
        }
        catch (AzureBlobFileSystemException azureBlobFileSystemException) {
            throw new IOException(azureBlobFileSystemException);
        }
        catch (Exception exception) {
            if (exception instanceof ExecutionException) {
                exception = (Exception)this.getRootCause(exception);
            }
            FileSystemOperationUnhandledException fileSystemOperationUnhandledException = new FileSystemOperationUnhandledException(exception);
            throw new IOException(fileSystemOperationUnhandledException);
        }
    }

    private void checkCheckAccessException(Path path, AzureBlobFileSystemException exception) throws IOException {
        AbfsRestOperationException ere;
        if (exception instanceof AbfsRestOperationException && (ere = (AbfsRestOperationException)exception).getStatusCode() == 403) {
            throw (IOException)new AccessControlException(ere.getMessage()).initCause((Throwable)exception);
        }
        AzureBlobFileSystem.checkException(path, exception, new AzureServiceErrorCode[0]);
    }

    @VisibleForTesting
    public static void checkException(Path path, AzureBlobFileSystemException exception, AzureServiceErrorCode ... allowedErrorCodesList) throws IOException {
        if (exception instanceof AbfsRestOperationException) {
            AbfsRestOperationException ere = (AbfsRestOperationException)exception;
            if (ArrayUtils.contains((Object[])allowedErrorCodesList, (Object)((Object)ere.getErrorCode()))) {
                return;
            }
            String message = ere.getMessage();
            switch (ere.getStatusCode()) {
                case 404: {
                    throw (IOException)new FileNotFoundException(message).initCause(exception);
                }
                case 409: {
                    throw (IOException)new FileAlreadyExistsException(message).initCause((Throwable)exception);
                }
                case 401: 
                case 403: {
                    throw (IOException)new AccessDeniedException(message).initCause(exception);
                }
            }
            throw ere;
        }
        if (exception instanceof SASTokenProviderException) {
            throw exception;
        }
        if (path == null) {
            throw exception;
        }
        throw new PathIOException(path.toString(), (Throwable)exception);
    }

    private Throwable getRootCause(Throwable throwable) {
        if (throwable == null) {
            throw new IllegalArgumentException("throwable can not be null");
        }
        Throwable result = throwable;
        while (result.getCause() != null) {
            result = result.getCause();
        }
        return result;
    }

    public synchronized Token<?> getDelegationToken(String renewer) throws IOException {
        this.statIncrement(AbfsStatistic.CALL_GET_DELEGATION_TOKEN);
        return this.delegationTokenEnabled ? this.delegationTokenManager.getDelegationToken(renewer) : super.getDelegationToken(renewer);
    }

    public String getCanonicalServiceName() {
        String name = null;
        if (this.delegationTokenManager != null) {
            name = this.delegationTokenManager.getCanonicalServiceName();
        }
        return name != null ? name : super.getCanonicalServiceName();
    }

    @VisibleForTesting
    FileSystem.Statistics getFsStatistics() {
        return this.statistics;
    }

    @VisibleForTesting
    void setListenerOperation(FSOperationType operation) {
        this.listener.setOperation(operation);
    }

    @VisibleForTesting
    public AzureBlobFileSystemStore getAbfsStore() {
        return this.abfsStore;
    }

    @VisibleForTesting
    AbfsClient getAbfsClient() {
        return this.abfsStore.getClient();
    }

    @VisibleForTesting
    AbfsDelegationTokenManager getDelegationTokenManager() {
        return this.delegationTokenManager;
    }

    @VisibleForTesting
    boolean getIsNamespaceEnabled(TracingContext tracingContext) throws AzureBlobFileSystemException {
        return this.abfsStore.getIsNamespaceEnabled(tracingContext);
    }

    @VisibleForTesting
    Map<String, Long> getInstrumentationMap() {
        return this.abfsCounters.toMap();
    }

    @VisibleForTesting
    String getFileSystemId() {
        return this.fileSystemId;
    }

    @VisibleForTesting
    String getClientCorrelationId() {
        return this.clientCorrelationId;
    }

    public boolean hasPathCapability(Path path, String capability) throws IOException {
        Path p = this.makeQualified(path);
        switch (PathCapabilitiesSupport.validatePathCapabilityArgs((Path)p, (String)capability)) {
            case "fs.capability.paths.permissions": 
            case "fs.capability.paths.append": 
            case "fs.capability.virtual.block.locations": {
                return true;
            }
            case "fs.capability.etags.available": {
                return true;
            }
            case "fs.capability.etags.preserved.in.rename": 
            case "fs.capability.paths.acls": {
                return this.getIsNamespaceEnabled(new TracingContext(this.clientCorrelationId, this.fileSystemId, FSOperationType.HAS_PATH_CAPABILITY, this.tracingHeaderFormat, this.listener));
            }
            case "fs.azure.capability.readahead.safe": {
                return true;
            }
        }
        return super.hasPathCapability(p, capability);
    }

    public IOStatistics getIOStatistics() {
        return this.abfsCounters != null ? this.abfsCounters.getIOStatistics() : null;
    }

    @VisibleForTesting
    static class FileSystemOperation<T> {
        private final T result;
        private final AbfsRestOperationException exception;

        FileSystemOperation(T result, AbfsRestOperationException exception) {
            this.result = result;
            this.exception = exception;
        }

        public boolean failed() {
            return this.exception != null;
        }
    }

    public class ResilientCommitByRenameImpl
    implements ResilientCommitByRename {
        @Override
        public Pair<Boolean, Duration> commitSingleFileByRename(Path source, Path dest, @Nullable String sourceEtag) throws IOException {
            LOG.debug("renameFileWithEtag source: {} dest: {} etag {}", new Object[]{source, dest, sourceEtag});
            AzureBlobFileSystem.this.statIncrement(AbfsStatistic.CALL_RENAME);
            AzureBlobFileSystem.this.trailingPeriodCheck(dest);
            Path qualifiedSrcPath = AzureBlobFileSystem.this.makeQualified(source);
            Path qualifiedDstPath = AzureBlobFileSystem.this.makeQualified(dest);
            TracingContext tracingContext = new TracingContext(AzureBlobFileSystem.this.clientCorrelationId, AzureBlobFileSystem.this.fileSystemId, FSOperationType.RENAME, true, AzureBlobFileSystem.this.tracingHeaderFormat, AzureBlobFileSystem.this.listener);
            if (qualifiedSrcPath.equals((Object)qualifiedDstPath)) {
                throw new PathIOException(qualifiedSrcPath.toString(), "cannot rename object onto self");
            }
            Duration waitTime = AzureBlobFileSystem.this.rateLimiting.acquire(1);
            try {
                boolean recovered = AzureBlobFileSystem.this.abfsStore.rename(qualifiedSrcPath, qualifiedDstPath, tracingContext, sourceEtag);
                return Pair.of((Object)recovered, (Object)waitTime);
            }
            catch (AzureBlobFileSystemException ex) {
                LOG.debug("Rename operation failed. ", (Throwable)ex);
                AzureBlobFileSystem.checkException(source, ex, new AzureServiceErrorCode[0]);
                return null;
            }
        }
    }
}

