(function(){
'use strict';
const app = angular.module('dataiku.agents', ['dataiku.ml.report', 'dataiku.lambda']);

// todo @lavish-agents move to type_mappings.js or something similar
app.constant('AGENT_BLOCK_TYPES', [
    {type: 'STANDARD_REACT', label: 'Core Loop', defaultName: 'core_loop', icon: 'dku-icon-ai-agent', description: 'The core block for most Agents. Use a LLM to iteratively reason and call tools to reach a goal', group: "COREAI"},
    {type: 'LLM_REQUEST', label: 'LLM Request', icon: 'dku-icon-llm', description: 'Make a single LLM call with no tools', group: "COREAI"},
    {type: 'DELEGATE_TO_OTHER_AGENT', label: 'Delegate to Other Agent', icon: 'dku-icon-ai-agent-delegation', description: 'Delegate the conversation to another agent', group: "COREAI"},
    {type: 'MANDATORY_TOOL_CALL', label: 'Mandatory Tool Call', icon: 'dku-icon-tool-wrench', description: 'Force a LLM to execute a specific tool', group: "COREAI"},
    {type: 'MANUAL_TOOL_CALL', label: 'Manual Tool Call', icon: 'dku-icon-hand', description: 'Execute a specific tool call, with no LLM intervention (specify all arguments)', group: "COREAI"},
    {type: 'ROUTING', label: 'Routing', icon: 'dku-icon-arrow-split', description: 'Use conditional logic to direct the flow to different branches', group: "LOGIC"},
    {type: 'FOR_EACH', label: 'For Each', icon: 'dku-icon-arrow-clockwise', description: 'Run a block multiple times for each item in a list', group: "LOGIC"},
    {type: 'PARALLEL', label: 'Parallel', icon: 'dku-icon-arrow-parallel', description: 'Execute several different blocks simultaneously', group: "LOGIC"},
    {type: 'REFLECTION', label: 'Reflection', icon: 'dku-icon-deep-learning', description: 'Loop over a block using various LLM-as-a-judge techniques to verify, or refine the output', group: "LOGIC"},
    {type: 'SET_STATE_ENTRIES', label: 'Set State Entries', icon: 'dku-icon-book', description: 'Configure conversation-scope short-term memory', group: "MEMORY"},
    {type: 'SET_SCRATCHPAD_ENTRIES', label: 'Set Scratchpad Entries', icon: 'dku-icon-note-sticky', description: 'Configure very-short-term memory for the current turn or parallel/foreach/reflection iteration', group: "MEMORY"},
    {type: 'CONTEXT_COMPRESSION', label: 'Context Compression', icon: 'dku-icon-arrow-expand-exit', description: 'Summarize earlier conversation history to keep the active context small', group: "MEMORY"},
    {type: 'EMIT_OUTPUT', label: 'Emit Output', icon: 'dku-icon-send', description: 'Send an output directly to the user', group: "INPUT_OUTPUT"},
    {type: 'GENERATE_ARTIFACT', label: 'Generate Artifact', icon: 'dku-icon-document-card', description: 'Generate an artifact from a text or DOCX template', group: "INPUT_OUTPUT"},
    /*{type: 'EDIT_LAST_USER_MESSAGE', label: 'Edit Last User Message', icon: 'dku-icon-edit-note', description: 'Rephrase the previous user input', group: "INPUT_OUTPUT"},*/
    {type: 'PYTHON_CODE', label: 'Python Code', icon: 'dku-icon-python', description: 'Execute custom logic using Python', group: "CUSTOM"},
]);

app.constant("AGENT_BLOCK_GROUPS", {
    COREAI: "CoreAI",
    LOGIC: "Logic",
    MEMORY: "Memory",
    INPUT_OUTPUT:"Input/Output",
    CUSTOM: "Custom",
});

app.constant("AGENT_OUTPUT_MODES", [
    {
        type: 'ADD_TO_MESSAGES',
        label: 'Add to internal conversation history',
        shortLabel: 'Add to internal history',
        description: 'Stores output in the agent\'s internal message history for later blocks and turns. Not displayed to the user.',
    },
    {
        type: 'SAVE_TO_STATE',
        label: 'Save to state',
        description: 'Stores output in the persistent state. Can be reused across subsequent blocks and conversation turns.',
    },
    {
        type: 'SAVE_TO_SCRATCHPAD',
        label: 'Save to scratchpad',
        description: 'Stores output in temporary memory for the current turn or sequence of blocks only. Not preserved across turns.',
    },
    {
        type: 'NONE',
        label: 'None',
        description: '',
    },
]);

app.controller("AgentSavedModelController", function($scope, $rootScope, $state, $controller, $stateParams, DataikuAPI, ActiveProjectKey, FullModelLikeIdUtils, $q,
    DKUtils, Dialogs, WT1, Debounce) {
    $controller("_SavedModelReportController", {$scope});

    DataikuAPI.savedmodels.llmCommon.getStatus(ActiveProjectKey.get(), $stateParams.smId).success(function(data){
        $scope.fillVersionSelectorStuff(data);
    });

    const currentVersionId = FullModelLikeIdUtils.parse($stateParams.fullModelId).versionId;
    $scope.currentVersionId = currentVersionId;
    $scope.warningMsg = '';

    $scope.$watch("savedModel", function(nv, ov) {
        if (!nv) return;

        $scope.currentVersionIdx = nv.inlineVersions.findIndex(v => v.versionId === currentVersionId);
        if ($scope.currentVersionIdx < 0) {
            return;
        }
        $scope.currentlySavedInlineModel = angular.copy(nv.inlineVersions[$scope.currentVersionIdx]);
        $scope.currentVersionData = nv.inlineVersions[$scope.currentVersionIdx];
    });

    // to trigger change detection in child components
    $scope.currentVersionUpdateCounter = 0;
    $scope.$watch('currentVersionData', Debounce().withDelay(200, 200).wrap(() => $scope.currentVersionUpdateCounter++), true);

    $scope.save = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        const deferred = $q.defer();
        const saveAfterConflictCheck = function() {
            DataikuAPI.savedmodels.agents.save($scope.savedModel, $scope.currentVersionId).success(function(data) {
                WT1.event(
                    'agent-save', {
                        savedModelType: data.savedModelType,
                    });
                $scope.savedModel = data;
                $scope.currentlySavedInlineModel = angular.copy($scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
                deferred.resolve('agent saved');
            }, () => deferred.reject()).error(setErrorInScope.bind($scope));
        };

        DataikuAPI.savedmodels.agents.checkSaveConflict($scope.savedModel).success(function(conflictResult) {
            if (!conflictResult.canBeSaved) {
                Dialogs.openConflictDialog($scope, conflictResult).then(
                    function(resolutionMethod) {
                        if (resolutionMethod === 'erase') {
                            saveAfterConflictCheck();
                        } else if (resolutionMethod === 'ignore') {
                            deferred.reject();
                            DKUtils.reloadState();
                        }
                    }
                );
            } else {
                saveAfterConflictCheck();
            }
        }).error(setErrorInScope.bind($scope));

        return deferred.promise;
    };
    $rootScope.saveAgentModel = $scope.save;

    $scope.internalCodeEnvsHRef = function() {
        if ($scope.appConfig.isAutomation) {
            return $state.href("admin.codeenvs-automation.internal");
        } else {
            return $state.href("admin.codeenvs-design.internal");
        }
    }

    $scope.saveIgnoringQuickTestQuery = function() {
        if (!$scope.isDirty()) {
            return $q.when('agent not dirty');
        }

        if (!$scope.isConfigDirty()) {
            return $q.when('only difference is the quicktest query, not saving');
        }

        return $scope.save();
    };

    $scope.isConfigDirty = function() {
        let frankenVersion = angular.copy($scope.currentlySavedInlineModel);
        frankenVersion["quickTestQueryStr"] = $scope.savedModel.inlineVersions[$scope.currentVersionIdx]["quickTestQueryStr"];

        return !angular.equals(frankenVersion, $scope.savedModel.inlineVersions[$scope.currentVersionIdx]);
    };

    $scope.isDirty = function() {
        if (!$scope.savedModel) return false;
        return !angular.equals($scope.savedModel.inlineVersions[$scope.currentVersionIdx], $scope.currentlySavedInlineModel);
    };
    $scope.getCurrentFile = function() {
        if (!$scope.savedModel) return false;
        return $scope.savedModel.id + "_" + $scope.currentVersionId + ".py";
    };
    $rootScope.agentModelIsDirty = $scope.isDirty;
    $rootScope.agentModelCurrentFile = $scope.getCurrentFile;

    function allowedTransitionsFn(data) {
        return data.toState?.name?.startsWith('projects.project.savedmodels.savedmodel.agent')
            && data.fromState?.name?.startsWith('projects.project.savedmodels.savedmodel.agent')
            && data.toParams?.fullModelId === data.fromParams?.fullModelId;
    }

    checkChangesBeforeLeaving($scope, $scope.isDirty, null, allowedTransitionsFn);
});

app.factory("AgentBlockTypeUtils", function($rootScope, AGENT_BLOCK_TYPES, PluginConfigUtils) {
    let svc = {}

     const CUSTOM_BLOCK_TYPE = 'CUSTOM';

    const DEFAULT_CUSTOM_BLOCK_ICON = 'dku-icon-puzzle-piece';

    function getCustomBlockComponentId(customBlock) {
        return customBlock?.blockType || customBlock?.desc?.id || customBlock?.componentId || customBlock?.id;
    }

    function normalizeCustomBlockIcon(icon) {
        if (!icon) return DEFAULT_CUSTOM_BLOCK_ICON;
        if (icon.startsWith('icon-')) return DEFAULT_CUSTOM_BLOCK_ICON;
        return icon.replace(/-\d+$/, '');
    }

    function buildBlockTypeKey(type, componentId) {
        return componentId ? `${type}:${componentId}` : type;
    }

    function buildAgentBlockByTypes(blockTypes) {
        return (blockTypes || []).reduce((acc, entry) => {
            acc[buildBlockTypeKey(entry.type, entry.componentId)] = entry;
            if (!entry.componentId && !acc[entry.type]) {
                acc[entry.type] = entry;
            }
            return acc;
        }, {});
    }

    const customBlocksGraphBlocks = $rootScope.appConfig.customBlocksGraphBlocks || [];
    const customBlocksGraphBlocksMap = customBlocksGraphBlocks.reduce((acc, block) => {
            const componentId = getCustomBlockComponentId(block);
            if (componentId) {
                acc[componentId] = block;
            }
            return acc;
        }, {});

    const customBlocksGraphBlocksVisible = customBlocksGraphBlocks.filter(
        PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins, null, getCustomBlockComponentId)
    );

    const customBlocksGraphBlockTypesVisible = customBlocksGraphBlocksVisible.map(block => {
        const meta = block.desc?.meta || {};
        const label = meta.label || block.id || block.desc?.id || block.blockType || 'Custom block';
        return {
            type: CUSTOM_BLOCK_TYPE,
            componentId: getCustomBlockComponentId(block),
            label,
            description: meta.description || '',
            icon: normalizeCustomBlockIcon(meta.icon),
            group: 'CUSTOM',
            blockIdBase: label,
        };
    });

    const customBlocksGraphBlockTypesAll = customBlocksGraphBlocks.map(block => {
        const meta = block.desc?.meta || {};
        const label = meta.label || block.id || block.desc?.id || block.blockType || 'Custom block';
        return {
            type: CUSTOM_BLOCK_TYPE,
            componentId: getCustomBlockComponentId(block),
            label,
            description: meta.description || '',
            icon: normalizeCustomBlockIcon(meta.icon),
            group: 'CUSTOM',
            blockIdBase: label,
        };
    });

    svc.agentBlockTypes = AGENT_BLOCK_TYPES.concat(customBlocksGraphBlockTypesVisible);
    svc.agentBlockByTypes = buildAgentBlockByTypes(AGENT_BLOCK_TYPES.concat(customBlocksGraphBlockTypesAll));

    return svc;
});

app.controller("AgentSavedModelDesignController", function($scope, $rootScope, $state, $stateParams, $timeout, Debounce, ActivityIndicator, CodeBasedEditorUtils,
                                                           CreateModalFromTemplate, DataikuAPI, PluginConfigUtils, PluginsService, TopNav, SmartId,
                                                           AGENT_BLOCK_TYPES, AGENT_BLOCK_GROUPS, AGENT_OUTPUT_MODES, BlockAgentDiagramUpdateService) {
    const CUSTOM_BLOCK_TYPE = 'CUSTOM';
    const DEFAULT_CUSTOM_BLOCK_ICON = 'dku-icon-puzzle-piece';
    $scope.outputModeOptions = AGENT_OUTPUT_MODES;
    $scope.addToMessagesOption = AGENT_OUTPUT_MODES.find((option) => (option.type === 'ADD_TO_MESSAGES'));

    function getCustomBlockComponentId(customBlock) {
        return customBlock?.blockType || customBlock?.desc?.id || customBlock?.componentId || customBlock?.id;
    }

    function normalizeCustomBlockIcon(icon) {
        if (!icon) return DEFAULT_CUSTOM_BLOCK_ICON;
        if (icon.startsWith('icon-')) return DEFAULT_CUSTOM_BLOCK_ICON;
        return icon.replace(/-\d+$/, '');
    }

    function buildBlockTypeKey(type, componentId) {
        return componentId ? `${type}:${componentId}` : type;
    }

    function buildAgentBlockByTypes(blockTypes) {
        return (blockTypes || []).reduce((acc, entry) => {
            acc[buildBlockTypeKey(entry.type, entry.componentId)] = entry;
            if (!entry.componentId && !acc[entry.type]) {
                acc[entry.type] = entry;
            }
            return acc;
        }, {});
    }

    $scope.editorOptions = CodeBasedEditorUtils.editorOptions('text/x-python', $scope, true);
    CodeBasedEditorUtils.registerBroadcastSelectionHandler($scope.editorOptions);
    CodeBasedEditorUtils.setRecipeScript($scope.script);
    $scope.testPanelDisplayMode = '';

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        if ($scope.savedModel.savedModelType === "PLUGIN_AGENT") {
            $scope.loadedDesc =  $rootScope.appConfig.customAgents.find(x => x.desc.id === $scope.currentVersionData.pluginAgentType);
            $scope.desc = $scope.loadedDesc.desc;
            $scope.pluginDesc = PluginsService.getOwnerPluginDesc($scope.loadedDesc);

            if ($scope.pluginDesc && $scope.desc && $scope.desc.params) {
                PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.currentVersionData.pluginAgentConfig);
            }
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            $scope.customBlocksGraphBlocks = $rootScope.appConfig.customBlocksGraphBlocks || [];
            $scope.customBlocksGraphBlocksMap = $scope.customBlocksGraphBlocks.reduce((acc, block) => {
                const componentId = getCustomBlockComponentId(block);
                if (componentId) {
                    acc[componentId] = block;
                }
                return acc;
            }, {});

            const customBlocksGraphBlocksVisible = $scope.customBlocksGraphBlocks.filter(
                PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins, null, getCustomBlockComponentId)
            );

            const customBlocksGraphBlockTypesVisible = customBlocksGraphBlocksVisible.map(block => {
                const meta = block.desc?.meta || {};
                const label = meta.label || block.id || block.desc?.id || block.blockType || 'Custom block';
                return {
                    type: CUSTOM_BLOCK_TYPE,
                    componentId: getCustomBlockComponentId(block),
                    label,
                    description: meta.description || '',
                    icon: normalizeCustomBlockIcon(meta.icon),
                    group: 'CUSTOM',
                    blockIdBase: label,
                };
            });

            const customBlocksGraphBlockTypesAll = $scope.customBlocksGraphBlocks.map(block => {
                const meta = block.desc?.meta || {};
                const label = meta.label || block.id || block.desc?.id || block.blockType || 'Custom block';
                return {
                    type: CUSTOM_BLOCK_TYPE,
                    componentId: getCustomBlockComponentId(block),
                    label,
                    description: meta.description || '',
                    icon: normalizeCustomBlockIcon(meta.icon),
                    group: 'CUSTOM',
                    blockIdBase: label,
                };
            });

            $scope.agentBlockTypes = AGENT_BLOCK_TYPES.concat(customBlocksGraphBlockTypesVisible);
            $scope.agentBlockByTypes = buildAgentBlockByTypes(AGENT_BLOCK_TYPES.concat(customBlocksGraphBlockTypesAll));
            $scope.onToolLLMChange = function() {
                if ($scope.currentVersionData.toolsUsingAgentSettings.llmId && $scope.availableLLMs) {
                    $scope.activeLLM = $scope.availableLLMs.find(l => l.id === $scope.currentVersionData.toolsUsingAgentSettings.llmId);
                }
            };

            DataikuAPI.agentTools.listAvailable($stateParams.projectKey).success(function(data) {
                $scope.availableTools = data;
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.savedmodels.listWithAccessible($stateParams.projectKey).success(function(data) {
                $scope.availableAgents = data
                    .filter(sm => sm.savedModelType === "TOOLS_USING_AGENT" || sm.savedModelType === "PLUGIN_AGENT" || sm.savedModelType === "PYTHON_AGENT")
                    .map(sm => {
                        sm.ref = (sm.projectKey == $stateParams.projectKey) ? sm.id : (sm.projectKey + "." + sm.id);
                        return sm;
                    });
            })

            DataikuAPI.codeenvs.checkAgentsCodeEnv($stateParams.projectKey)
                .then(function ({data}) {
                    $scope.codeEnvType = data["codeEnvType"];
                    $scope.codeEnvError = data["codeEnvError"];
                }).catch(setErrorInScope.bind($scope));

            $scope.availableToolsParams = $scope.availableToolsParams || {};
            $scope.availableToolsSubtools = $scope.availableToolsSubtools || {};
            $scope.loadToolDescription = function(agentTool) {
                if (!agentTool || !agentTool["toolRef"]) {
                    return;
                }
                if (!$scope.availableToolsSubtools[agentTool["toolRef"]]) {
                    $scope.availableToolsSubtools[agentTool["toolRef"]] = [];
                    const smartId = SmartId.resolve(agentTool["toolRef"], $stateParams.projectKey);
                    DataikuAPI.agentTools.getDescriptor(smartId.projectKey, smartId.id).success(descriptor => {
                        $scope.availableToolsSubtools[agentTool["toolRef"]] = (descriptor?.subtools || []).filter(st => st.enabled).map(st => st.name);
                        if (descriptor?.inputSchema) {
                            $scope.availableToolsParams[$scope.toolsParamsKey(agentTool["toolRef"], null)] = descriptor?.inputSchema?.properties ? Object.keys(descriptor?.inputSchema?.properties) : [];
                        }
                        for (const subtoolDescriptor of descriptor?.subtools || []) {
                            if (subtoolDescriptor.enabled) {
                                $scope.availableToolsParams[$scope.toolsParamsKey(agentTool["toolRef"], subtoolDescriptor.name)] = subtoolDescriptor?.inputSchema?.properties ? Object.keys(subtoolDescriptor?.inputSchema?.properties) : [];
                            }
                        }
                        if (!$scope.availableToolsSubtools[agentTool["toolRef"]].includes(agentTool.subtoolName)) {
                            agentTool.subtoolName = null;
                        }
                    }).error(setErrorInScope.bind($scope));
                } else {
                    if (!$scope.availableToolsSubtools[agentTool["toolRef"]].includes(agentTool.subtoolName)) {
                        agentTool.subtoolName = null;
                    }
                }
            };

            $scope.toolsParamsKey = function(toolRef, subtoolName) {
                if (subtoolName) {
                    return toolRef + "_" + subtoolName;
                } else {
                    return toolRef;
                }
            }

            $scope.generateArtifactOutputFormatsByTemplateType = {
                CEL_EXPANSION: [
                    {type: 'TEXT', label: 'Text'},
                    {type: 'MARKDOWN', label: 'Markdown'},
                ],
                JINJA: [
                    {type: 'TEXT', label: 'Text'},
                    {type: 'MARKDOWN', label: 'Markdown'},
                ],
                DOCX_JINJA: [
                    {type: 'DOCX', label: 'DOCX'},
                    {type: 'PDF', label: 'PDF'},
                ],
            };

            $scope.ensureGenerateArtifactOutputFormat = function(block) {
                if (!block || block.type !== 'GENERATE_ARTIFACT') return;
                const availableFormats = $scope.generateArtifactOutputFormatsByTemplateType[block.templateType];
                if (!availableFormats || !availableFormats.length) return;
                const isValid = availableFormats.some(format => format.type === block.outputFormat);
                if (!isValid) {
                    block.outputFormat = availableFormats[0].type;
                }
            };
        }

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
            $scope.availableLLMs = data.identifiers;

            if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
                $scope.availableLLMs = $scope.availableLLMs.filter(llm => (
                    !llm.id.startsWith("agent:" + $scope.savedModel.id)  // Don't allow selecting myself as the LLM for this agent, to avoid infinite loop
                    && llm.type !== "RETRIEVAL_AUGMENTED"
                    && (
                        $scope.currentVersionData.toolsUsingAgentSettings.allowAgentsAsLLM 
                        || (llm.type !== "SAVED_MODEL_AGENT")
                    )
                ));
                $scope.onToolLLMChange();
            }
        }).error(setErrorInScope.bind($scope));

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "design");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "design");
            }
        }
    });

    $scope.blocksGraphBlockIds = [];
    $scope.blocksGraphUiState = $scope.blocksGraphUiState || [];

    $scope.getCustomBlocksGraphBlocks = function(selectedComponentId) {
        const visibilityFilter = PluginConfigUtils.shouldComponentBeVisible(
            $rootScope.appConfig.loadedPlugins,
            selectedComponentId,
            getCustomBlockComponentId
        );
        return ($scope.customBlocksGraphBlocks || []).filter(visibilityFilter);
    };

    $scope.getCustomBlockDesc = function(componentId) {
        return $scope.customBlocksGraphBlocksMap?.[componentId] || null;
    };

    $scope.getCustomBlockPluginDesc = function(componentId) {
        const customBlock = $scope.getCustomBlockDesc(componentId);
        return customBlock ? PluginsService.getOwnerPluginDesc(customBlock) : null;
    };

    $scope.getCustomBlockLabel = function(customBlock) {
        const meta = customBlock?.desc?.meta || {};
        return meta.label || customBlock?.id || customBlock?.desc?.id || customBlock?.blockType || 'Custom block';
    };

    $scope.getCustomBlockCategory = function(customBlock) {
        return customBlock?.desc?.meta?.category || 'Custom';
    };

    $scope.customBlockComponentId = getCustomBlockComponentId;

    $scope.hasCustomBlockParams = function(componentId) {
        const customBlock = $scope.getCustomBlockDesc(componentId);
        return !!(customBlock && customBlock.desc && customBlock.desc.params && customBlock.desc.params.length);
    };

    $scope.hasCustomBlockSettings = function(componentId) {
        const pluginDesc = $scope.getCustomBlockPluginDesc(componentId);
        return !!(pluginDesc && pluginDesc.hasSettings) || $scope.hasCustomBlockParams(componentId);
    };

    $scope.onCustomBlockComponentChange = function(block) {
        if (!block) return;
        if (!block.config) {
            block.config = {};
        }
        const customBlock = $scope.getCustomBlockDesc(block.componentId);
        if (customBlock?.desc?.params) {
            PluginConfigUtils.setDefaultValues(customBlock.desc.params, block.config);
        }
    };

    $scope.getBlockTypeDefinition = function(block) {
        if (!block) return {};
        const key = buildBlockTypeKey(block.type, block.componentId);
        return $scope.agentBlockByTypes?.[key] || $scope.agentBlockByTypes?.[block.type] || {};
    };

    $scope.getBlockTypeGroup = function(block) {
        return $scope.getBlockTypeDefinition(block).group || 'CUSTOM';
    };

    $scope.getBlockTypeContainerClass = function(block, blockUiState) {
        const classes = [];
        const group = ($scope.getBlockTypeGroup(block) || '').toLowerCase();
        if (blockUiState?.open && group) {
            classes.push(`block-agent-type_open--${group}`);
        }
        if ($scope.isBlockInDiagram(block.id) && block.id === $scope.highlightedBlockId) {
            classes.push('repeatable-config-block--highlighted-agent');
        }
        return classes.join(' ');
    };

    function refreshBlocksGraphBlockIds() {
        const settings = $scope.currentVersionData?.toolsUsingAgentSettings;
        const blocks = settings?.blocks || [];
        const ids = blocks.map(block => block?.id).filter(Boolean);

        $scope.blocksGraphBlockIds = Array.from(new Set(ids));

        if (settings?.startingBlockId && !$scope.blocksGraphBlockIds.includes(settings.startingBlockId)) {
            settings.startingBlockId = null;
            $scope.triggerDiagramUpdate();
        }

        if ($scope.blocksGraphBlockIds.length === 1 && !settings?.startingBlockId) {
            settings.startingBlockId = $scope.blocksGraphBlockIds[0];
            $scope.triggerDiagramUpdate();
        }
    }

    function replaceBlocksGraphIdReferences(settings, oldId, newId) {
        if (!settings || !oldId || oldId === newId) return;

        if (settings.startingBlockId === oldId) {
            settings.startingBlockId = newId;
            $scope.triggerDiagramUpdate();
        }

        (settings.blocks || []).forEach(block => {
            if (!block) return;

            if (block.defaultNextBlock === oldId) {
                block.defaultNextBlock = newId;
            }
            if (block.nextBlock === oldId) {
                block.nextBlock = newId;
            }
            if (block.blockIdToRepeat === oldId) {
                block.blockIdToRepeat = newId;
            }
            if (block.generatorBlockId === oldId) {
                block.generatorBlockId = newId;
            }
            if (Array.isArray(block.blockIds)) {
                block.blockIds = block.blockIds.map(id => id === oldId ? newId : id);
            }

            (block.clausesBasedDecisions || []).forEach(decision => {
                if (decision?.nextBlock === oldId) {
                    decision.nextBlock = newId;
                }
            });

            (block.resultDispatch || []).forEach(entry => {
                if (entry?.value === oldId) {
                    entry.value = newId;
                }
            });

            (block.exitConditions || []).forEach(condition => {
                if (!condition) return;
                if (condition.nextBlock === oldId) {
                    condition.nextBlock = newId;
                }
            });
        });
    }

    function collectBlockIds(blocks) {
        return (blocks || []).map(block => block?.id);
    }

    function isReorderOnly(oldIds, newIds) {
        if (oldIds.length !== newIds.length) return false;
        if (oldIds.some(id => !id) || newIds.some(id => !id)) return false;
        const oldSet = new Set(oldIds);
        if (oldSet.size !== oldIds.length) return false;
        for (const id of newIds) {
            if (!oldSet.has(id)) return false;
        }
        for (let i = 0; i < oldIds.length; i++) {
            if (oldIds[i] !== newIds[i]) return true;
        }
        return false;
    }

    $scope.hasLLMBasedClauses = function(block) {
        for (const clauseDecision of block.clausesBasedDecisions) {
            if (isLLMBasedClause(clauseDecision.clause)) {
                return true;
            }
        }
        return false;
    }

    function isLLMBasedClause(clause) {
        if (clause.type === 'LLM_BASED') {
            return true;
        } else if (['AND', 'OR'].includes(clause.type)) {
            for (const subClause of clause.clauses) {
                if (isLLMBasedClause(subClause)) {
                    return true;
                }
            }
        }
        return false;
    }

    $scope.$watch('currentVersionData.toolsUsingAgentSettings.blocks', function(newBlocks, oldBlocks) {
        if (!newBlocks || !oldBlocks) return;
        const settings = $scope.currentVersionData?.toolsUsingAgentSettings;
        const oldIds = collectBlockIds(oldBlocks);
        const newIds = collectBlockIds(newBlocks);
        if (isReorderOnly(oldIds, newIds)) return;
        if (oldIds.length !== newIds.length) return;
        const maxLen = Math.min(oldBlocks.length, newBlocks.length);
        for (let i = 0; i < maxLen; i++) {
            const oldId = oldIds[i];
            const newId = newIds[i];
            if (oldId && newId && oldId !== newId) {
                replaceBlocksGraphIdReferences(settings, oldId, newId);
                $scope.triggerDiagramUpdate();
            }
        }
    }, true);

    $scope.$watch('currentVersionData.toolsUsingAgentSettings.blocks', refreshBlocksGraphBlockIds, true);

    $scope.$watch('currentVersionData.toolsUsingAgentSettings.blocks', function(newBlocks) {
        if (!newBlocks) return;
        newBlocks.forEach(block => {
            if (block?.type === CUSTOM_BLOCK_TYPE) {
                $scope.onCustomBlockComponentChange(block);
            }
        });
    }, true);

    $scope.onNextTurnBehaviorChange = function() {
        if ($scope.currentVersionData.toolsUsingAgentSettings.nextTurnBehaviour === 'SMART' && !$scope.currentVersionData.toolsUsingAgentSettings.nextTurnSmartModeLLMId) {
            $scope.currentVersionData.toolsUsingAgentSettings.nextTurnSmartModeLLMId = $scope.appConfig.agentDefaultLlmId;
        }
    }

    $scope.collapseAllBlocksGraphRows = function () {
        $scope.blocksGraphUiState.forEach(state => {
            if (state) state.open = false;
        });
    }

    $scope.blocksGraphSortableOptions = {
        axis: 'y',
        handle: '.blocks-graph-block__drag-handle',
        cancel: 'input,textarea,select,option,a',
        helper: 'clone',
        start: $scope.collapseAllBlocksGraphRows,
    };

    const blockIdCleanupRegexFallback = /[^A-Za-z0-9 _-]+/g;
    let blockIdRegex;
    let blockIdCleanupRegex;

    try {
        blockIdRegex = new RegExp('^[\\p{L}\\p{N} _<>:;-]+$', 'u');
        blockIdCleanupRegex = new RegExp('[^\\p{L}\\p{N} _<>:;-]+', 'gu');
    } catch (e) {
        blockIdRegex = /^[A-Za-z0-9 _<>:;-]+$/;
        blockIdCleanupRegex = /[^A-Za-z0-9 _<>:;-]+/g;
    }

    $scope.blockIdPattern = new RegExp(`^(?:${blockIdRegex.source})?$`, blockIdRegex.flags);

    function sanitizeBlockIdValue(value) {
        if (!value) return value;
        try {
            return value.replace(blockIdCleanupRegex, '');
        } catch (e) {
            return value.replace(blockIdCleanupRegexFallback, '');
        }
    }

    function blockIdBaseFromType(type) {
        const base = (type || 'block').toLowerCase().replace(/ /g, '_').trim();
        return base || 'block';
    }

    $scope.generateBlockId = function(type, blocks) {
        const usedIds = new Set(blocks.map((block) => block?.id).filter(Boolean));
        let baseFromType = blockIdBaseFromType(type);
        if($scope.agentBlockByTypes[type]?.defaultName) {
            baseFromType = $scope.agentBlockByTypes[type].defaultName
        }

        const base = sanitizeBlockIdValue(baseFromType);
        let candidate = base;
        let index = 2;
        while (usedIds.has(candidate)) {
            candidate = `${base} ${index}`;
            index += 1;
        }
        return candidate;
    };

    $scope.getCurrentBlocks = function () {
        const settings = $scope.currentVersionData?.toolsUsingAgentSettings;
        if (!settings) return;
        if (!Array.isArray(settings.blocks)) {
            settings.blocks = [];
        }
        return settings.blocks;
    };

    const scrollBlockIntoView = ({ element, behavior, block, onlyIfNotVisible }) => {
        if (!element) return;
        const scrollContainer = element.closest('[data-scroll-container="blocks-graph"]');
        const headerHeight = scrollContainer.querySelector('[data-scroll="block-graph-actions"]').getBoundingClientRect().height;
        const containerRect = scrollContainer.getBoundingClientRect();
        const elementRect = element.getBoundingClientRect();
        if (onlyIfNotVisible) {
            const visibleTop = containerRect.top + headerHeight;
            const visibleBottom = containerRect.bottom;
            const isFullyVisible = elementRect.top >= visibleTop && elementRect.bottom <= visibleBottom;
            if (isFullyVisible) return;
        }
        const targetTop = elementRect.top - containerRect.top + scrollContainer.scrollTop - headerHeight;
        scrollContainer.scrollTo({ top: targetTop, behavior });
    };

    $scope.addBlocksGraphBlock = function (type, newBlockId, options) {
        const blocks = $scope.getCurrentBlocks();
        const newBlock = { type, id: newBlockId };
        if (type === CUSTOM_BLOCK_TYPE) {
            newBlock.componentId = options?.componentId || null;
            newBlock.config = {};
            $scope.onCustomBlockComponentChange(newBlock);
        }
        // todo @lavish-agents: we need to clean this logic!
        if (type === 'EMIT_OUTPUT') {
            newBlock.templateType = "CEL_EXPANSION";
            newBlock.template = "";
            newBlock.addToMessages = true;
        } else if (type === 'CONTEXT_COMPRESSION') {
            newBlock.activeBufferSize = 5;
            newBlock.compressionTriggerChars = 15000;
        } else if (type === 'STANDARD_REACT' || type === 'LLM_REQUEST' || type === 'DELEGATE_TO_OTHER_AGENT') {
            newBlock.passConversationHistory = true;
            newBlock.streamOutput = true;
            newBlock.completionSettings = {};
            newBlock.outputMode = 'ADD_TO_MESSAGES';
        } else if (type === 'REFLECTION') {
            newBlock.mode = 'CRITIQUE_AND_IMPROVE';
            newBlock.critiqueMaxIterations = 3;
            newBlock.synthesizeIterations = 3;
            newBlock.failOnMaxIterations = true;
            newBlock.streamOutput = true;
            newBlock.maxThreads = 32;
            newBlock.outputMode = 'ADD_TO_MESSAGES';
        } else if (type === 'PARALLEL') {
            newBlock.maxThreads = 32;
        } else if (type === 'GENERATE_ARTIFACT') {
            newBlock.templateType = "CEL_EXPANSION";
            newBlock.outputFormat = "TEXT";
            newBlock.template = "";
        } else if (type === 'PYTHON_CODE') {
            newBlock.code = `from dataiku.llm.python.blocks_graph import NextBlock

def process(trace):
    # The function can be a generator that yields multiple output chunks
    # Each such chunk is streamed to the user as soon as it's produced

    # Chunks can be:
    #   - a string
    yield "Hello from Python code block"

    #   - a dict like this
    yield {
        "chunk": {
            "text": "Hello from Python code block"
        }
    }

    # A NextBlock object to indicate which block to execute next
    yield NextBlock("a_block_id")

    # You can access the state and scratchpad as Python dictionaries
    # state["Hello"] = scratchpad["World"]

    # The last value output form the previous block is stored in the last_output variable
    # yield last_output

# The function can also be a simple function that returns either a single output chunk
# or a NextBlock object to indicate which block to execute next
#def process_simple(trace):
#    return "Hello from Python code block"
#
#def process_simple_going_to_next_block(trace):
#    return NextBlock("a_block_id")
`;
//                +
//                + '    yield {\n'
//                + '        "chunk": {\n'
//                + '            "text": "Hello from Python code block"\n'
//                + '        }\n'
//                + '    }\n';
            newBlock.functionName = "process";
        }
        if (type === 'MANDATORY_TOOL_CALL') {
            newBlock.completionSettings = {};
            newBlock.outputMode = 'ADD_TO_MESSAGES';
        }
        if (type === 'MANUAL_TOOL_CALL') {
            newBlock.outputMode = 'ADD_TO_MESSAGES';
        }
        if (type === 'STANDARD_REACT') {
            newBlock.maxLoopIterations = 25;
            newBlock.maxParallelToolExecutions = 2;
        }
        if (type === 'REFLECTION') {
            newBlock.critiqueCompletionSettings = {};
            newBlock.synthesizeCompletionSettings = {};
        }
        if (type === 'FOR_EACH') {
            newBlock.forEachInputKey = "for_each_input";
        }
        if (['STANDARD_REACT', 'LLM_REQUEST', 'LLM_DISPATCH', 'CONTEXT_COMPRESSION', 'MANDATORY_TOOL_CALL', 'ROUTING', 'REFLECTION'].includes(type)) {
            newBlock.llmId = $scope.appConfig.agentDefaultLlmId;
        }
        blocks.push(newBlock);
        const newIndex = blocks.length - 1;
        $scope.blocksGraphUiState[newIndex] = { open: true };
        refreshBlocksGraphBlockIds();
        $timeout(function () {
            const list = document.querySelector('.blocks-graph-list');
            if (!list) return;
            const items = list.querySelectorAll('.blocks-graph-block');
            const newItem = items[items.length - 1];
            scrollBlockIntoView({
                element: newItem,
                behavior: 'smooth',
                block: 'nearest',
                onlyIfNotVisible: true,
            });
        });
    };

    $scope.openBlocksGraphAddBlockModal = function(options) {
        const modalScope = $scope.$new(true);
        modalScope.selection = {
            filterParams: {
                userQueryTargets: ['label'],
            },
        };
        modalScope.agentBlockTypes = $scope.agentBlockTypes;
        modalScope.agentBlockGroups = AGENT_BLOCK_GROUPS;
        CreateModalFromTemplate('/templates/savedmodels/agents/agent-add-block-modal.html', modalScope, null, function (newScope) {
            let didCreate = false;
            const originalDismiss = newScope.dismiss;
            newScope.dismiss = function () {
                if (!didCreate && options?.onCancel && typeof options.onCancel === 'function') {
                    options.onCancel();
                }
                if (originalDismiss && typeof originalDismiss === 'function') {
                    originalDismiss.call(newScope);
                }
            };

            newScope.blockId = '';

            newScope.create = function () {
                if (!newScope.blockId) {
                    newScope.blockId = newScope.defaultBlockId;
                }
                const selected = newScope.selection?.selectedObject;
                if (!selected) return;
                $scope.addBlocksGraphBlock(selected.type, newScope.blockId, { componentId: selected.componentId });
                if (options?.onCreate && typeof options.onCreate === 'function') {
                    options.onCreate(newScope.blockId, selected.type);
                }
                didCreate = true;
                newScope.dismiss();
            };
            newScope.$watch('selection.selectedObject', function (selectedObject) {
                if (selectedObject) {
                    const blocks = $scope.getCurrentBlocks();
                    newScope.defaultBlockId = $scope.generateBlockId(selectedObject.blockIdBase || selectedObject.label || selectedObject.type, blocks);
                }
            });
        });
    };

    $scope.deleteBlockAtIndex = function(index) {
        $scope.currentVersionData.toolsUsingAgentSettings.blocks.splice(index, 1);
        $scope.triggerDiagramUpdate();
    };

    $scope.removeAndUpdateDiagram = function(removeFn, index, block) {
        removeFn(index, null, () => $scope.triggerDiagramUpdate(block));
    };

    $scope.highlightBlock = function({ blockId }) {
        $timeout(() => {
            $scope.setHighlightedBlock(blockId, false);
        });

        $timeout(() => {
            const selectedElement = document.querySelector('.repeatable-config-block--highlighted-agent');
            scrollBlockIntoView({
                element: selectedElement,
                behavior: 'smooth',
                block: 'start',
            });
        });
    };

    $scope.setHighlightedBlock = function(blockId, updateDiagram = false) {
        const viewTabActive = $scope.isDiagramActive();
        $scope.highlightedBlockId = blockId;
        $scope.testPanelDisplayMode = 'view';
        if (updateDiagram) {
            $timeout(() => {
                BlockAgentDiagramUpdateService.setSelectedBlock(blockId);
            }, viewTabActive ? 0 : 200);
        }
    };

    let usedDiagramBlocks = {};
    $scope.updateUsedDiagramBlocks = function({ blocks }) {
        usedDiagramBlocks = blocks || {};
    };

    $scope.isDiagramActive = function() {
        return $scope.testPanelDisplayMode === 'view';
    };

    $scope.isBlockInDiagram = function(blockId) {
        return !!usedDiagramBlocks[blockId];
    };

    $scope.setTestPanelDisplayMode = function({ mode }) {
        $scope.testPanelDisplayMode = mode;
    };

    // update diagram functions
    function triggerDiagramUpdate(block) {
        // if block is not in diagram, don't update diagram
        if (block?.id && !$scope.isBlockInDiagram(block?.id)) {
            return;
        }

        BlockAgentDiagramUpdateService.update();
        $scope.highlightedBlockId = '';
    }

    $scope.triggerDiagramUpdate = Debounce().withDelay(500, 500).wrap(triggerDiagramUpdate);

    $scope.$watch('currentVersionData.toolsUsingAgentSettings.tools', Debounce().withDelay(500, 500).wrap(() => {
        if ($scope.availableTools && $scope.currentVersionData.toolsUsingAgentSettings.tools.length) {
            const kbTools = $scope.availableTools.filter(tool => tool.type === 'VectorStoreSearch').map(tool => SmartId.fromTor(tool));
            $scope.hasKbTools = $scope.currentVersionData.toolsUsingAgentSettings.tools.some(tool => !tool.disabled && kbTools.includes(tool.toolRef));
        }
    }), true);

    $scope.isToolAvailable = function(usedTool) {
        if ($scope.availableTools && usedTool?.toolRef) {
            return $scope.availableTools.some((l) => SmartId.fromTor(l) === usedTool.toolRef);
        }
        return false;
    };

    $scope.createNewTool = function ($event) {
        $event.preventDefault();
        const stateName = 'projects.project.agenttools.list';
        if ($scope.isDirty()) {
            $scope.save().then(function () {
                ActivityIndicator.success("Saved");
                $scope.$state.go(stateName, { createTool: true });
            }).catch(setErrorInScope.bind($scope));
        } else {
            $scope.$state.go(stateName, { createTool: true });
        }
    };

    $scope.toggleTool = function(usedTool) {
        usedTool.disabled = !usedTool.disabled;
    };

    $scope.editToolDescriptionModal = function(usedTool) {
        if (!usedTool.toolRef) {
            return;
        }
        const smartId = SmartId.resolve(usedTool.toolRef, $stateParams.projectKey);

        DataikuAPI.agentTools.get(smartId.projectKey, smartId.id).success(agentTool => {
            const modalScope = $scope.$new(true);
            modalScope.tool = agentTool;
            modalScope.usedTool = usedTool;
            modalScope.toolAdditionalDescription = usedTool.additionalDescription;

            CreateModalFromTemplate("/templates/savedmodels/agents/agent-design-edit-tool-description-modal.html", modalScope, null, function(modalScope) {
                modalScope.accept = function() {
                    if (modalScope.toolAdditionalDescription) {
                        usedTool.additionalDescription = modalScope.toolAdditionalDescription;
                    } else {
                        delete usedTool.additionalDescription;
                    }
                };

                modalScope.showToolDescriptionModal = function() {
                    DataikuAPI.agentTools.getDescriptor(agentTool.projectKey, agentTool.id).success(descriptor => {
                        const innerModalScope = $scope.$new(true);
                        innerModalScope.toolName = agentTool.name;
                        // remove the global description that was appended by the back-end in get-descriptor to show it in another field
                        innerModalScope.internalDescription = descriptor.description.replace(agentTool.additionalDescriptionForLLM, "").trimEnd();
                        innerModalScope.globalDescription = agentTool.additionalDescriptionForLLM;
                        innerModalScope.agentSpecificDescription = modalScope.toolAdditionalDescription || "";
                        innerModalScope.agentToolURL = $state.href("projects.project.agenttools.agenttool", {projectKey: agentTool.projectKey, agentToolId: agentTool.id});
                        CreateModalFromTemplate('/templates/savedmodels/agents/agent-tool-description-modal.html', innerModalScope);
                    }).error(setErrorInScope.bind($scope));
                };
            });
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("AgentSavedModelSettingsController", function($scope, $controller, $stateParams, DataikuAPI, TopNav, FLOW_COMPUTABLE_DEPENDENCIES, SavedModelHelperService) {
    $scope.availableDependencies = undefined;
    $scope.currentDependencies = undefined;

    $scope.isDependenciesEnabled = function(){
        return $scope.savedModel?.savedModelType === 'PYTHON_AGENT';
    }

    $scope.$watch("savedModel", (nv, ov) => {
        if (!nv) return;

        switch ($scope.savedModel.savedModelType) {
            case "PYTHON_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pythonAgentSettings;
                break;
            case "PLUGIN_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.pluginAgentSettings;
                break;
            case "TOOLS_USING_AGENT":
                $scope.currentAgentSettings = $scope.currentVersionData.toolsUsingAgentSettings;
                break;
        }

        if ($scope.savedModel.savedModelType === "TOOLS_USING_AGENT") {
            DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
                if ($scope.currentAgentSettings.llmId && data.identifiers) {
                    $scope.activeLLM = data.identifiers.find(l => l.id === $scope.currentAgentSettings.llmId);
                }
            }).error(setErrorInScope.bind($scope));

            DataikuAPI.codeenvs.checkAgentsCodeEnv($stateParams.projectKey)
                .then(function ({data}) {
                    $scope.codeEnvType = data["codeEnvType"];
                    $scope.codeEnvError = data["codeEnvError"];
                }).catch(setErrorInScope.bind($scope));

            $scope.ImageSupportMode = [
                {value: "AUTOMATIC", label: "Automatic", annotation: "Detect from base LLM, on failure, assume supported."},
                {value: "ENABLED", label: "Enabled", annotation: "Force on (may fail if unsupported)."},
                {value: "DISABLED", label: "Disabled", annotation: "Never allow image inputs."},
            ]
        }

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "settings");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "settings");
            }
        }

        $scope.addDependency = function(dependencyToAdd){
            const pythonSettings = $scope.currentVersionData.pythonAgentSettings;
            if (!pythonSettings.dependencies) {
                pythonSettings.dependencies = [];
            }

            const isAlreadyAdded = pythonSettings.dependencies.some(dep => dep.ref === dependencyToAdd.smartId);
            if (isAlreadyAdded) {
                $scope.warningMsg = "Object already added as dependency.";
                return;
            }

            const newDependency = {
                type: dependencyToAdd.model?.savedModelType || dependencyToAdd.type,
                ref: dependencyToAdd.smartId
            };

            pythonSettings.dependencies.push(newDependency);
            $scope.currentDependencies.push(dependencyToAdd);
            $scope.warningMsg = "";
        };

        $scope.removeDependency = function(index) {
            if (index < 0) { return; }
            const pythonSettings = $scope.currentVersionData.pythonAgentSettings;
            if (pythonSettings?.dependencies) {
                pythonSettings.dependencies.splice(index, 1);
                $scope.currentDependencies.splice(index, 1);
            }
        };

        // only load this once as it depends on the project not the model
        if ($scope.isDependenciesEnabled() && $scope.availableDependencies === undefined) {
            $scope.availableDependencies = [];
            $scope.currentDependencies = [];
            DataikuAPI.taggableObjects.listAccessibleObjects($stateParams.projectKey).then(function(resp) {
                const dependencies = resp.data;
                $scope.currentDependencies = SavedModelHelperService.enrichDependencies($scope.currentAgentSettings?.dependencies, dependencies)
                $scope.availableDependencies = SavedModelHelperService.filterDependencies(dependencies, FLOW_COMPUTABLE_DEPENDENCIES);
            }).catch(setErrorInScope.bind($scope));
        }
    });
});

app.controller("AgentVersionHistoryController", function($scope, SavedModelRefService, StateUtils, $controller, TopNav) {
    $scope.redirectCallback = function() {
        StateUtils.go.savedModel(SavedModelRefService.smartId);
    };
    TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "history");
});


app.controller("AgentSavedModelLogsController", function($scope, $stateParams, $controller, DataikuAPI, Dialogs, TopNav) {
    $scope.listLogs = function() {
        DataikuAPI.savedmodels.agents.listLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId).success(function(data) {
            $scope.agentLogs = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.clearLogs = function() {
        const title = 'Confirm logs deletion';
        const message = 'Are you sure you want to clear the logs for this agent?';

        Dialogs.confirm($scope, title, message).then(() => {
            DataikuAPI.savedmodels.agents.clearLogs($stateParams.projectKey, $scope.savedModel.id, $scope.currentVersionId)
                .error(setErrorInScope.bind($scope))
                .finally($scope.listLogs);
        });
    }

    $scope.deleteLog = function(projectKey, savedModelId, versionId, logName) {
        return DataikuAPI.savedmodels.agents.deleteLog(projectKey, savedModelId, versionId, logName)
                .error(setErrorInScope.bind($scope))
                .finally($scope.listLogs);
    }

    $scope.getLog = DataikuAPI.savedmodels.agents.getLog;

    $scope.logDownloadURL = (projectKey, savedModelId, versionId, logName) => {
        const params = new URLSearchParams({
            projectKey,
            smId: savedModelId,
            versionId,
            logName
        });
        return `/dip/api/savedmodels/agents/stream-log?${params}`;
    };

    $scope.$watch("savedModel", (nv) => {
        if (!nv) return;

        $scope.listLogs();

        if (!$scope.noSetLoc) {
            if ($scope.savedModel.savedModelType === "PLUGIN_AGENT"){
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "PLUGIN_AGENT-SAVED_MODEL-VERSION", "logs");
            } else{
                TopNav.setLocation(TopNav.TOP_GENAI_MODELS, TopNav.LEFT_GENAI_MODELS, "AGENT-SAVED_MODEL-VERSION", "logs");
            }
        }
    });
});

app.directive('blocksGraphNextBlockSelect', function() {
    return {
        restrict: 'E',
        scope: {
            model: '=',
            ids: '<',
            currentBlockId: '<?',
            onChange: '&?',
            helpText: '@?',
            placeholder: '@?',
            selectClass: '@?',
            wrapperClass: '@?',
            createBlock: '<?',
            allowNone: '<?',
        },
        template: `
            <div class="blocks-graph-next-block-select" ng-class="wrapperClass">
                <basic-select items="optionItems"
                              ng-model="model"
                              ng-change="handleChange()"
                              bind-label="label"
                              bind-value="id"
                              searchable="true"
                              placeholder="{{placeholder}}"
                              class="{{selectClass}}">
                </basic-select>
                <span class="help-inline" ng-if="helpText">{{helpText}}</span>
                <span class="help-inline text-warning" ng-if="currentBlockId && model == currentBlockId">Warning: this is the current block.</span>
            </div>
        `,
        link: function ($scope) {
            const createBlockOptionId = '__create_block__';
            const createBlockLabel = '+ Create new block';

            $scope.optionItems = [];

            if ($scope.allowNone === undefined) {
                $scope.allowNone = true;
            }

            if (!$scope.placeholder) {
                $scope.placeholder = $scope.allowNone ? 'None' : 'Select next block';
            }

            $scope.lastSelected = $scope.model;

            function rebuildOptions() {
                const safeIds = Array.isArray($scope.ids) ? $scope.ids.slice() : [];
                const filteredIds = safeIds.filter((blockId) => blockId !== $scope.currentBlockId);
                const items = filteredIds.map((blockId) => ({ id: blockId, label: blockId }));

                if ($scope.allowNone) {
                    items.unshift({ id: null, label: 'None' });
                }

                if ($scope.createBlock) {
                    items.unshift({ id: createBlockOptionId, label: createBlockLabel });
                }
                $scope.optionItems = items;
                if ($scope.model && !filteredIds.includes($scope.model)) {
                    $scope.model = null;
                    $scope.lastSelected = null;
                    if ($scope.onChange) {
                        $scope.onChange();
                    }
                }
            }

            $scope.$watchGroup(['ids', 'currentBlockId', 'forbidSelectingCurrentBlock'], rebuildOptions);

            $scope.$watch('model', function (newValue) {
                if (newValue !== createBlockOptionId) {
                    $scope.lastSelected = newValue;
                }
            });

            $scope.handleChange = function () {
                if ($scope.model === createBlockOptionId) {
                    $scope.model = $scope.lastSelected;
                    if (typeof $scope.createBlock === 'function') {
                        $scope.createBlock({
                            onCreate: function (newBlockId) {
                                $scope.model = newBlockId;
                                if ($scope.onChange) {
                                    $scope.onChange();
                                }
                            },
                            onCancel: function () {
                                $scope.$evalAsync(() => {
                                    $scope.model = $scope.lastSelected;
                                });
                            },
                        });
                    }
                    return;
                }
                if ($scope.onChange) {
                    $scope.onChange();
                }
            };
        },
    };
});

app.component('agentArtifacts', {
    bindings: {
        artifacts: '<',
        expanded: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifacts.html',

    controller: function() {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
        };

        $ctrl.hasDisplayableArtifacts = function() {
            // If the agent returns no results, we could have a sources section with empty `items`
            return $ctrl.artifacts?.some(artifact => artifact.parts.length > 0);
        };

        $ctrl.shouldExpandArtifact = function(artifact) {
            // Always expand artifacts that do not have a hierarchy field
            // Expand thinking artifacts to level 1
            // Expand all other artifacts to level 2
            if (!artifact.hierarchy || !artifact.hierarchy.length) {
                return true;
            }
            if (artifact.type === "REASONING") {
                return artifact.hierarchy.length <= 1;
            }
            return artifact.hierarchy.length <= 2;
        };
    }
});

app.component('agentArtifact', {
    bindings: {
        artifact: '<',
        expanded: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifact.html',

    controller: function($dkuSanitize, Logger) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
        };

        $ctrl.convertToMarkdown = function(reasoningParts) {
            for (let reasoning of reasoningParts) {
                if (reasoning.type === "TEXT") {
                    try {
                        reasoning.reasoning  = $dkuSanitize(marked(reasoning.text, { breaks: true}));
                    }
                    catch(e) {
                        Logger.error('Error parsing markdown HTML, switching to plain text', e);
                    }
                }
            }
            return reasoningParts;
        };

        $ctrl.hasHierarchy = function() {
            return $ctrl.artifact.hierarchy && $ctrl.artifact.hierarchy.length;
        };
    },
});

app.component('agentHierarchy', {
    bindings: {
        hierarchy: '<',
    },
    templateUrl: '/templates/savedmodels/agents/hierarchy.html',
});

app.component('agentToolValidationRequests', {
    bindings: {
        toolValidationRequests: '=',
        sendResponses: '<',
        expanded: '<',
        disabled: '<',
    },
    templateUrl: '/templates/savedmodels/agents/tool-validation-requests.html',

    controller: function() {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }
            $ctrl.setFinalized();
        };

        $ctrl.setFinalized = function() {
            if (!$ctrl.finalized) {
                $ctrl.finalized = $ctrl.toolValidationRequests.every(tvr => tvr.validated != null);
            }
        }

        $ctrl.respond = function(toolValidationRequest, response, editedArgs) {
            if ($ctrl.sendResponses == null) {
                return;
            }

            toolValidationRequest.validated = response;
            toolValidationRequest.arguments = editedArgs;
            $ctrl.setFinalized();
            if ($ctrl.finalized) {
                $ctrl.sendResponses()
            }
        }
    }
});

app.component('agentToolValidationRequest', {
    bindings: {
        toolValidationRequest: '<',
        expanded: '<',
        onRespond: '&',
        finalized: '<',
        disabled: '<',
    },
    templateUrl: '/templates/savedmodels/agents/tool-validation-request.html',

    controller: function($dkuSanitize, $scope, CodeMirrorSettingService, CreateModalFromTemplate, Logger, StateUtils) {
        const $ctrl = this;

        $ctrl.getCodeMirrorSettings = function() {
            return CodeMirrorSettingService.get('application/json');
        }

        $ctrl.$onInit = function() {
            if ($ctrl.disabled !== true) {
                $ctrl.disabled = false;
            }

            if ($ctrl.expanded !== false) {
                $ctrl.expanded = true;
            }

            if ($ctrl.finalized && $ctrl.toolValidationRequest.arguments != null) {
                $ctrl.currentJsonArgs = JSON.parse($ctrl.toolValidationRequest.arguments);
            } else {
                $ctrl.currentJsonArgs = JSON.parse($ctrl.toolValidationRequest.toolCall.function.arguments);
            }
            $ctrl.initialArgs = JSON.stringify($ctrl.currentJsonArgs, null, 4);
            $ctrl.currentArgs = $ctrl.initialArgs;

            $ctrl.editMode = false;

            $ctrl.message = $ctrl.toolValidationRequest.message ?? "Do you want to execute this tool call?"
            try {
                $ctrl.markdownMessage = $dkuSanitize(marked($ctrl.message, {breaks: true}));
            } catch(e) {
                Logger.error('Error parsing markdown HTML, switching to plain text', e);
            }
        };

        $ctrl.hasHierarchy = function() {
            return $ctrl.toolValidationRequest.hierarchy && $ctrl.toolValidationRequest.hierarchy.length;
        };

        $ctrl.editingAllowed = function() {
            return $ctrl.toolValidationRequest.allowEditingInputs;
        }

        $ctrl.validatingDisabledTooltipMessage = function() {
            return 'Validating tool calls is only supported in chat mode';
        }

        $ctrl.editingDisabledToolTipMessage = function() {
            if ($ctrl.disabled) {
                return $ctrl.validatingDisabledTooltipMessage();
            }
            if (!$ctrl.editingAllowed()) {
                return 'Editing inputs is not allowed for this tool call'
            }
        }

        $ctrl.isEditMode = function() {
            return $ctrl.editMode && !$ctrl.finalized;
        }

        $ctrl.enableEditMode = function() {
            if (!$ctrl.editingAllowed()) {
                return;
            }

            $ctrl.editedArgs = $ctrl.currentArgs;
            $ctrl.editedArgsError = null;
            $ctrl.validEditedArgs = null;
            $ctrl.validEditedJsonArgs = null;

            const resetResponseEvent = {toolValidationRequest: $ctrl.toolValidationRequest}
            $ctrl.onRespond(resetResponseEvent);

            $ctrl.editMode = true;
        }

        $ctrl.disableEditMode = function() {
            $ctrl.editMode = false;
        }

        $ctrl.validateEditedArgs = function() {
            try {
                $ctrl.validEditedJsonArgs = JSON.parse($ctrl.editedArgs);
                $ctrl.validEditedArgs = $ctrl.editedArgs;
                $ctrl.editedArgsError = null;
            } catch (e) {
                $ctrl.validEditedJsonArgs = null;
                $ctrl.validEditedArgs = null;
                $ctrl.editedArgsError = "Invalid arguments, it should be a JSON object: " + e;
            }
        };

        $ctrl.accept = function() {
            if ($ctrl.editMode) {
                if ($ctrl.validEditedArgs != null) {
                    $ctrl.currentJsonArgs = $ctrl.validEditedJsonArgs;
                    $ctrl.currentArgs = $ctrl.validEditedArgs;
                }
            }
            const acceptEvent = {
                toolValidationRequest: $ctrl.toolValidationRequest,
                response: true,
            };
            if ($ctrl.currentArgs !== $ctrl.initialArgs) {
                acceptEvent["editedArgs"] = $ctrl.currentArgs;
            }
            $ctrl.onRespond(acceptEvent);
            $ctrl.editMode = false;
        }

        $ctrl.reject = function() {
            const rejectEvent = {
                toolValidationRequest: $ctrl.toolValidationRequest,
                response: false,
            };
            $ctrl.onRespond(rejectEvent);
            $ctrl.editMode = false;
        }

        $ctrl.showToolDescriptionModal = function() {
            const modalScope = $scope.$new(true);
            modalScope.toolName = $ctrl.toolValidationRequest.toolName;
            modalScope.toolDescription = $ctrl.toolValidationRequest.toolDescription;
            modalScope.toolInputSchema = $ctrl.toolValidationRequest.toolInputSchema;

            if ($ctrl.toolValidationRequest.toolRef) {
                modalScope.agentToolURL = StateUtils.href.agentTool($ctrl.toolValidationRequest.toolRef);
            }

            CreateModalFromTemplate('/templates/savedmodels/agents/agent-tool-full-description-modal.html', modalScope);
        };
    }
});

app.component('artifactRecordsTable', {
    bindings: {
        records: '<',
    },
    template: `
        <div class="export-btn pull-right">
            <button class="btn btn--secondary" ng-click="$ctrl.exportData()">Export</button>
        </div>
        <div id="artifact-records-table" class="agent-artifact-records-table ag-theme-alpine" style="clear: both;"></div>
    `,
    controller: function(AgGridLoader, $element, ExportUtils, $scope) {
        const $ctrl = this;
        let gridApi;

        $ctrl.$onInit = function() {
            const $container = $element.find('#artifact-records-table')[0];
            const {headers, rows} = formatData($ctrl.records);
            AgGridLoader.load().then(AgGrid => gridApi = AgGrid.createGrid($container, initGridOptions(headers, rows)));
        };

        $ctrl.$onChanges = (changes) => {
            if (!gridApi) return;

            if (changes && changes.records) {
                const {headers, rows} = formatData($ctrl.records);
                gridApi.setGridOption('columnDefs', headers);
                gridApi.setGridOption('rowData', rows);
            }
        };

        $ctrl.exportData = function() {
            ExportUtils.exportUIData($scope, {
                name : "Export artifact records table data",
                columns: $ctrl.records.columns.map(colName => { return { name: colName, type: "String" } }),
                data : $ctrl.records.data,
            }, "Export data", { hideAdvancedParameters: true });
        };

        function formatData(records) {
            const headers = records.columns.map(colName => { return { field: colName} });
            const rows = records.data.map(rowData => {
                const row = {};
                for (let i = 0; i < rowData.length; i++) {
                    row[headers[i].field] = rowData[i];
                }
                return row;
            });
            return {headers, rows}
        }

        function initGridOptions(headers, rows) {
            return {
                rowData: rows,
                columnDefs: headers,
                defaultColDef: {
                    sortable: true,
                    resizable: true,
                    filter: false
                },
                sideBar: {
                    toolPanels: [
                        {
                            id: 'columns',
                            labelDefault: 'Columns',
                            labelKey: 'columns',
                            iconKey: 'columns',
                            toolPanel: 'agColumnsToolPanel',
                            toolPanelParams: {
                                suppressRowGroups: true,
                                suppressValues: true,
                                suppressPivots: true,
                                suppressPivotMode: true,
                            }
                        }
                    ]
                },
                suppressPropertyNamesCheck: true,
                alwaysMultiSort: false,
                suppressCsvExport: true,
                suppressExcelExport: true,
                pinnedBottomRowData: [],
                tooltipShowDelay: 0,
                enableCellTextSelection: true,
                onFirstDataRendered: () => {
                    setTimeout(() => {
                        gridApi.autoSizeAllColumns();

                        const MAX_WIDTH = 600;
                        gridApi.getColumns().forEach(col => {
                            const width = col.getActualWidth();
                            if (width > MAX_WIDTH) {
                                gridApi.setColumnWidths([{
                                    key: col,
                                    newWidth: MAX_WIDTH
                                }]);
                            }
                        });
                    })
                },
            }
        }
    }
});

app.component('artifactDocument', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifact-document.html',
    controller: function (AnyLoc, SmartId, $dkuSanitize) {
        const $ctrl = this;
        $ctrl.fullFolderId = null;
        $ctrl.anyLocFolder = null;
        $ctrl.$onChanges = function (changes) {
            if (changes.part) {
                if ($ctrl.part.type === 'FILE_BASED_DOCUMENT') {
                    if ($ctrl.part.imageRefs && $ctrl.part.imageRefs.length) {
                        $ctrl.fullFolderId = $ctrl.part.imageRefs[0].folderId;
                        $ctrl.anyLocFolder = AnyLoc.getLocFromFull($ctrl.fullFolderId);
                        $ctrl.imagePaths = $ctrl.part.imageRefs.map(image => image['path']);
                    } else if ($ctrl.part.fileRef) {
                        const { projectKey, id } = SmartId.resolve($ctrl.part.fileRef.folderId);
                        const fullFolderId = projectKey + '.' + id;
                        $ctrl.anyLocFolder = AnyLoc.getLocFromFull(fullFolderId);
                    }
                }
                if ($ctrl.part.markdownSnippet) {
                    // avoid performance issue with large payloads
                    if ($ctrl.part.markdownSnippet.length < 10000) {
                        $ctrl.renderedMarkdownSnippet = $dkuSanitize(marked($ctrl.part.markdownSnippet, { breaks: true}))
                    } else {
                        $ctrl.renderedMarkdownSnippet = $dkuSanitize($ctrl.part.markdownSnippet)
                    }
                }
                if ($ctrl.part.htmlSnippet) {
                    $ctrl.renderedHTMLSnippet = $dkuSanitize($ctrl.part.renderedHTMLSnippet)
                }
                if ($ctrl.part.jsonSnippet) {
                    try {
                        $ctrl.parsedJsonSnippet = JSON.parse($ctrl.part.jsonSnippet);
                    } catch (error) {
                        $ctrl.parsedJsonSnippet = {
                            parsingError: "Invalid JSON format.",
                            originalValue: $ctrl.part.jsonSnippet
                        };
                    }
                }
            }
        };
    },
});

app.component('artifactDataInline', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/artifact-data-inline.html',
    controller: function ($sce) {
        const $ctrl = this;
        let objectUrl = null;

        function revokeObjectUrl() {
            if (objectUrl) {
                URL.revokeObjectURL(objectUrl);
                objectUrl = null;
            }
        }

        $ctrl.$onChanges = function (changes) {
            if (!changes.part || !$ctrl.part) return;
            revokeObjectUrl();
            const mimeType = $ctrl.part.mimeType || "application/octet-stream";
            if ($ctrl.part.dataBase64) {
                const binary = atob($ctrl.part.dataBase64);
                const bytes = new Uint8Array(binary.length);
                for (let i = 0; i < binary.length; i++) {
                    bytes[i] = binary.charCodeAt(i);
                }
                const blob = new Blob([bytes], { type: mimeType });
                objectUrl = URL.createObjectURL(blob);
                $ctrl.downloadHref = $sce.trustAsResourceUrl(objectUrl);
            } else {
                $ctrl.downloadHref = null;
            }
            $ctrl.downloadFilename = $ctrl.part.filename || "artifact";
        };

        $ctrl.$onDestroy = function () {
            revokeObjectUrl();
        };

        $ctrl.niceMimeType = function(mimeType) {
            switch (mimeType) {
                case "application/vnd.openxmlformats-officedocument.wordprocessingml.document":
                    return "Docx document";
                case "application/pdf":
                    return "PDF document";
                case "text/html":
                    return "HTML";
                case "text/plain":
                    return "Plain text";
                default:
                    return mimeType;
            }
        };
    },
});

app.component('agentToolMultipartOutput', {
    bindings: {
        parts: '<',
    },
    templateUrl: '/templates/savedmodels/agents/agent-tool-multipart-output.html',
});

app.component('agentToolImageInline', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/agent-tool-image-inline.html',
});

app.component('agentToolImageUri', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/agent-tool-image-uri.html',
});

app.component('agentToolImageRef', {
    bindings: {
        part: '<',
    },
    templateUrl: '/templates/savedmodels/agents/agent-tool-image-ref.html',
    controller: function (AnyLoc) {
        const $ctrl = this;
        $ctrl.anyLocFolder = null;
        $ctrl.$onChanges = function (changes) {
            if (changes.part) {
                if ($ctrl.part.folderId && $ctrl.part.path) {
                    $ctrl.anyLocFolder = AnyLoc.getLocFromFull($ctrl.part.folderId);
                    $ctrl.imagePaths = [$ctrl.part.path];
                }
            }
        };
    },
});

app.component('agentTemplatingHelper', {
    bindings: {
        templatingType: '<', // CEL_EXPRESSION / CEL_EXPANSION / JINJA
    },
    template: `
    <i ng-if="$ctrl.tooltipHelpText" 
       class="dku-icon-info-circle-outline-16 text-prompt mleft4 vab"
       title="{{$ctrl.tooltipHelpText}}"
       toggle="tooltip">
    </i>                                                
    `,
    controller: function () {
        const $ctrl = this;
        $ctrl.tooltipHelpText = '';
        $ctrl.$onChanges = function (changes) {
            if (changes.templatingType) {
                if ($ctrl.templatingType=='CEL_EXPRESSION') {
                    $ctrl.tooltipHelpText = "Value must be a CEL expression.";
                } else if ($ctrl.templatingType=='CEL_EXPANSION' || $ctrl.templatingType=='JINJA') {
                    $ctrl.tooltipHelpText = 'Evaluate expressions with {{ }}.';
                }
                $ctrl.tooltipHelpText += ' Available variables: context, state, scratchpad, last_output.';
            }
        };
    },
});

})();
