import json
import logging

from mcp.types import ImageContent, TextContent, CallToolResult

from dataiku.llm.agent_tools import BaseAgentTool
from dataiku.llm.agent_tools.mcp.client import FastMCPClient
from dataiku.llm.agent_tools.types import AgentToolOutputImageHandlingMode


logger = logging.getLogger(__name__)

class GenericStdioMCPClient(BaseAgentTool):
    def __init__(self):
        super().__init__()
        logger.info("MCP Client __init__")
        self.client = None

    def set_config(self, config, plugin_config):
        logger.info("MCP Client set_config: command %s, args %s", config.get("command", ""), config["args"])
        self.client = FastMCPClient(config)

    def get_descriptor(self, _):
        logger.info("Get descriptor")

        descriptor = {
            "multiple": True,
            "description": "", # no global description, each subtool has one
            "subtools" : []
        }
        result = self.client.list_tools()
        logger.info("Got subtools: %s" % result)

        for tool in result:
            # Shortcut MCP explodes here because one of the tool has a property with multiple types
            # {'type': ['number', 'null'], 'description': 'The epic id of the epic the story belongs to, or null to unset'}
            # this sounds valid according to https://json-schema.org/draft/2020-12/draft-bhutton-json-schema-01#name-assertions-and-instance-pri
            # but is hard to implement on the Java side
            for name, property in tool.inputSchema["properties"].items():
                if property.get("type") is None:
                    logger.warning("No type set for property '%s' of tool '%s'", name, tool.name)
                elif isinstance(property["type"], list):
                    logger.warning("Unsupported array type '%s' for property '%s' of tool '%s'", property["type"], name, tool.name)
                    property["type"] = property["type"][0]
            descriptor["subtools"].append({
                "name": tool.name,
                "description": tool.description.strip() if tool.description else "",
                "inputSchema": tool.inputSchema
            })

        return descriptor

    def describe_tool_call(self, tool, descriptor, input):
        logger.info("Get tool call description")
        return f"""I'm about to call subtool <b>{input["subtoolName"]}</b> of the local MCP server defined in tool <b>{tool["name"]}</b>, with the following input.

Do you want to proceed?"""

    def invoke(self, input, trace):
        logger.info("MCP invoking tool: %s" % input)

        if not "subtoolName" in input:
            raise Exception("missing subtool name, please set 'subtoolName' in the payload")

        with trace.subspan("PYTHON_AGENT_MCP_SUBTOOL_CALL") as subtool_subspan:
            subtool_subspan.attributes["subtool_name"] = input["subtoolName"]
            subtool_subspan.attributes["subtool_args"] = input["input"]

            result: CallToolResult = self.client.call_tool(input["subtoolName"], input["input"])
            logger.info("Got tool invocation result: %s" % result)

        parts = []
        artifact_parts = []
        image_handling_mode = self.client.tool_config.get("imageHandlingMode", AgentToolOutputImageHandlingMode.ONLY_ADD_AS_ARTIFACT)

        content_found = False
        # Backward compatibility checks with fastmcp<2.11 (using pydantic 2.9, it breaks but differently on more recent pydantic).
        if hasattr(result, "structured_content") and result.structured_content:
            content_found = True
            parts.append({
                "type": "TEXT",
                "text": json.dumps(result.structured_content),
            })
        if hasattr(result, "content"):
            returned_content = result.content
        elif isinstance(result, list):
            returned_content = result
        else:
            raise Exception("Unexpected result from tool: {}".format(result))
        for content in returned_content:
            if isinstance(content, TextContent) and not content_found:
                parts.append({
                    "type": "TEXT",
                    "text": content.text,
                })
            elif isinstance(content, ImageContent):
                if image_handling_mode in [
                    AgentToolOutputImageHandlingMode.ONLY_ADD_AS_ARTIFACT.value,
                    AgentToolOutputImageHandlingMode.ADD_AS_ARTIFACT_AND_SEND_TO_LLM.value,
                ]:
                    artifact_parts.append({
                        "type": "DATA_INLINE",
                        "dataBase64": content.data,
                        "mimeType": content.mimeType,
                    })
                if image_handling_mode in [
                    AgentToolOutputImageHandlingMode.ONLY_SEND_TO_LLM.value,
                    AgentToolOutputImageHandlingMode.ADD_AS_ARTIFACT_AND_SEND_TO_LLM.value,
                ]:
                    parts.append({
                        "type": "IMAGE_INLINE",
                        "imageMimeType": content.mimeType,
                        "inlineImage": content.data,
                    })
                if image_handling_mode == "IGNORE":
                    logger.debug("Ignoring image part")
            else:
                logger.warning("ignoring unsupported response content %s", type(content).__name__)

        output = {}
        if len(parts) == 0:
            raise RuntimeError("No supported content returned by tool call, this might be solved by using another image handling mode.")
        elif len(parts) == 1 and parts[0]["type"] == "TEXT":
            # use the string output field in the general case of a single text part output, kept for backward compatibility
            output["output"] = {
                "text": parts[0]["text"]
            }
        else:
            output["parts"] = parts

        if artifact_parts:
            output["artifacts"] = [{
                "type": "DATA_INLINE",
                "name": "Images from local MCP tool",
                "parts": artifact_parts,
            }]

        return output